// Copyright (c) ppy Pty Ltd <contact@ppy.sh>. Licensed under the MIT Licence.
// See the LICENCE file in the repository root for full licence text.

using System;
using System.Collections.Generic;
using osu.Game.Audio;

namespace osu.Game.Rulesets.Taiko.Skinning.Argon
{
    public class VolumeAwareHitSampleInfo : HitSampleInfo
    {
        public const int SAMPLE_VOLUME_THRESHOLD_HARD = 90;
        public const int SAMPLE_VOLUME_THRESHOLD_MEDIUM = 60;

        public VolumeAwareHitSampleInfo(HitSampleInfo sampleInfo, bool isStrong = false)
            : base(sampleInfo.Name, isStrong ? BANK_STRONG : getBank(sampleInfo.Bank, sampleInfo.Name, sampleInfo.Volume), sampleInfo.Suffix, sampleInfo.Volume)
        {
        }

        public override IEnumerable<string> LookupNames
        {
            get
            {
                foreach (string name in base.LookupNames)
                    yield return name.Insert(name.LastIndexOf('/') + 1, "Argon/taiko-");
            }
        }

        private static string getBank(string originalBank, string sampleName, int volume)
        {
            // So basically we're overwriting mapper's bank intentions here.
            // The rationale is that most taiko beatmaps only use a single bank, but regularly adjust volume.

            switch (sampleName)
            {
                case HIT_NORMAL:
                case HIT_CLAP:
                {
                    if (volume >= SAMPLE_VOLUME_THRESHOLD_HARD)
                        return BANK_DRUM;

                    if (volume >= SAMPLE_VOLUME_THRESHOLD_MEDIUM)
                        return BANK_NORMAL;

                    return BANK_SOFT;
                }

                default:
                    return originalBank;
            }
        }

        public override bool Equals(HitSampleInfo? other) => other is VolumeAwareHitSampleInfo && base.Equals(other);

        /// <remarks>
        /// <para>
        /// This override attempts to match the <see cref="Equals"/> override above, but in theory it is not strictly necessary.
        /// Recall that <see cref="GetHashCode"/> <a href="https://learn.microsoft.com/en-us/dotnet/api/system.object.gethashcode?view=net-8.0#notes-to-inheritors">must meet the following requirements</a>:
        /// </para>
        /// <para>
        /// "If two objects compare as equal, the <see cref="GetHashCode"/> method for each object must return the same value.
        /// However, if two objects do not compare as equal, <see cref="GetHashCode"/> methods for the two objects do not have to return different values."
        /// </para>
        /// <para>
        /// Making this override combine the value generated by the base <see cref="GetHashCode"/> implementation with a constant means
        /// that <see cref="HitSampleInfo"/> and <see cref="VolumeAwareHitSampleInfo"/> instances which have the same values of their members
        /// will not have equal hash codes, which is slightly more efficient when these objects are used as dictionary keys.
        /// </para>
        /// </remarks>
        public override int GetHashCode() => HashCode.Combine(base.GetHashCode(), 1);
    }
}
