// $Id$

// Fish Supper
// Copyright 2006, 2007, 2009, 2010 Matthew Clarke <mafferyew@googlemail.com>
//
// This file is part of Fish Supper.
//
// Fish Supper is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Fish Supper is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Fish Supper.  If not, see <http://www.gnu.org/licenses/>.




#include "Tutorial.h"




// *******************
// *** CONSTRUCTOR ***
// *******************
FS::Tutorial::Tutorial() : current_task(0)
{
    // nothing...

} // FS::Tutorial::Tutorial

// ******************
// *** DESTRUCTOR ***
// ******************
FS::Tutorial::~Tutorial()
{
    delete_tasks();

} // FS::Tutorial::~Tutorial




// ************************
// *** MEMBER FUNCTIONS ***
// ************************

// **************************************************

void FS::Tutorial::load_task_list(int level)
{
    // FIXME: old list may need deleting first!
    delete_tasks();
    
    std::string my_line;
    char my_filename[256];
    sprintf(my_filename, PKG_DATA_DIR "/tutorials/task_list_%02d.txt", level);
    std::ifstream fin(my_filename);
    
    // Number of tasks.
    getline(fin, my_line);
    int num_tasks = atoi(my_line.c_str());
    
    for (int i = 0; i < num_tasks; ++i)
    {
        Task* my_task = new Task;
        
        // Task type.
        getline(fin, my_line);
        
        if ( my_line.compare("READ") == 0 )
        {
            my_task->type = READ_TASK;
            
            getline(fin, my_line);
            my_task->msg = my_line;
            
            // once only?
            getline(fin, my_line);
            int once_only = atoi(my_line.c_str());
            my_task->do_once_only = ((once_only) ? true : false);
        } 
        else // must be PLAY
        {
            my_task->type = PLAY_TASK;
            
            // rowUTORIAL_HH_

            getline(fin, my_line);
            my_task->row_desc = (RowDescription) atoi(my_line.c_str());
            
            // log color
            getline(fin, my_line);
            my_task->lc = (LogColor) atoi(my_line.c_str());
            
            // num crystals
            getline(fin, my_line);
            int n = atoi(my_line.c_str());
            my_task->num_required_crystals = n;
            
            for (int j = 0; j < n; ++j)
            {
                getline(fin, my_line);
                my_task->required_crystals[j] = (Crystal::Color) atoi(my_line.c_str());
            } // for
            
            // message
            getline(fin, my_line);
            my_task->msg = my_line;
            
            // play tasks must always be repeated if restart 
            my_task->do_once_only = false;
        } // if ... else
        
        task_list.push_back(my_task);
    } // for
    
    fin.close();
    
    // Make ready for first use.
    reset_task_list();
    
} // FS::Tutorial::load_task_list

// **************************************************

void FS::Tutorial::reset_task_list()
{
    assert( !task_list.empty() );
    
    std::vector<Task*>::iterator iter = task_list.begin();
    do
    {
        (*iter)->done_once_already = false;
        ++iter;
    } while (iter != task_list.end());
    
    current_task = 0;
        
} // FS::Tutorial::reset_task_list

// **************************************************

void FS::Tutorial::print_task_list()
{
    assert( !task_list.empty() );
    
    std::vector<Task*>::iterator iter = task_list.begin();
    do
    {
        Task* my_task = *iter;
        if (my_task->type == READ_TASK)
        {
            std::cout << "---READ_TASK---\n";
            std::cout << "message=" << my_task->msg << std::endl;
            std::cout << "do_once_only=" << my_task->do_once_only << std::endl;
        }
        else
        {
            std::cout << "---PLAY_TASK---\n";
            std::cout << "row_desc=" << my_task->row_desc << std::endl;
            std::cout << "lc=" << my_task->lc << std::endl;
            std::cout << "num_required_crystals=" << my_task->num_required_crystals << 
                    std::endl;
            for (int i = 0; i < my_task->num_required_crystals; ++i)
            {
                std::cout << my_task->required_crystals[i] << " ";
            } // for
            std::cout << std::endl << "message=" << my_task->msg << std::endl;
        } // if ... else
    } while (++iter != task_list.end());
       
} // FS::Tutorial::print_task_list

// **************************************************

void FS::Tutorial::delete_tasks()
{
    if ( !(task_list.empty()) )
    {
        std::vector<Task*>::iterator i = task_list.begin();
        do
        {
            delete *i;
            ++i;
        } while (i != task_list.end());
        
        // Remove pointers so size is 0.
        task_list.clear();
    } // if
    
} // FS::Tutorial::delete_tasks

// **************************************************

const FS::Tutorial::Task* FS::Tutorial::get_current_task() const
{
    if ( current_task < task_list.size() )
    {
        return task_list[current_task];
    }
    else
    {
        return 0;
    } // if ... else
    
} // FS::Tutorial::get_current_task

// **************************************************

const FS::Tutorial::Task* FS::Tutorial::next_task()
{
    // Is function being used properly? - 'current_task' should be 
    // within bounds.
    assert(!task_list.empty());
    assert(current_task < task_list.size());
    
    task_list[current_task]->done_once_already = true;
    ++current_task;
    
    return get_current_task();
    
} // FS::Tutorial::next_task
    
// **************************************************

void FS::Tutorial::restart()
{
    current_task = 0;
    
} // FS::Tutorial::restart

// **************************************************




// ************************************************
// *** RELATED NON-MEMBER, NON-FRIEND FUNCTIONS ***
// ************************************************

// **************************************************
        
void FS::draw_task_message(const std::string& msg)
{
    gfx_ptr->draw_texture( FS_gfx::MESSAGE_BOARD, Tutorial::MSG_BOARD_X, Tutorial::MSG_BOARD_Y );
    gfx_ptr->draw_text( FS_gfx::TUTORIAL_FONT, Tutorial::MSG_BOARD_X, Tutorial::MSG_BOARD_Y + Tutorial::MSG_TEXT_Y_OFFSET,
            msg );

} // FS::draw_task_message

// **************************************************
