// $Id$

// Fish Supper
// Copyright 2006, 2007, 2009, 2010 Matthew Clarke <mafferyew@googlemail.com>
//
// This file is part of Fish Supper.
//
// Fish Supper is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
// 
// Fish Supper is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with Fish Supper.  If not, see <http://www.gnu.org/licenses/>.




#include "Albert.h"
#include <iostream>
#include <cstdlib>




// *******************
// *** CONSTRUCTOR ***
// *******************
FS::Albert::Albert() :
    host_log(0)
{
    width = ALBERT_WIDTH;
    height = ALBERT_HEIGHT;
    
    velocity = ALBERT_VELOCITY;
    
    sprite_dest_rect.w = width;
    sprite_dest_rect.h = height;
    
    erase_rect.w = width;
    erase_rect.h = height;
    
    sprite_src_rect.y = 0;
    sprite_src_rect.w = width;
    sprite_src_rect.h = height;
    
    for (int i = 0; i < NUM_COLL_BOXES; ++i)
    {
        coll_box_x_offsets[i] = COLL_BOX_X_OFFSETS[i];
        coll_box_y_offsets[i] = COLL_BOX_Y_OFFSETS[i];
        
        my_coll_boxes[i].w = COLL_BOX_WIDTHS[i]; 
        my_coll_boxes[i].h = COLL_BOX_HEIGHTS[i];
    } // for
    
    // FIXME: make these literals constants?
    bounding_box.w = width + 20;
    bounding_box.h = height + 20;
            
} // FS::Albert::Albert() 

// ******************
// *** DESTRUCTOR ***
// ******************
FS::Albert::~Albert() {}




// ************************
// *** MEMBER FUNCTIONS ***
// ************************

// **************************************************

void FS::Albert::reset()
{
    my_orientation = NORTH;
    
    actions = 0;
    
    airborne = false;
    
    host_log = 0;
    
    splashed_zapped_out = false;
    left = false;
    next_jump_double = false;
    
    // start coords
    x = old_x = ALBERT_START_X;
    y = old_y = ALBERT_START_Y;
    
    sprite_dest_rect.x = x;
    sprite_dest_rect.y = y;
    
    erase_rect.x = old_x;
    erase_rect.y = old_y;
    
    update_rect.x = x;
    update_rect.y = y;
    update_rect.w = width;
    update_rect.h = height;
    
    current_frame = 0;  // depends on what he's doing!
    
} // FS::Albert::reset()

// **************************************************

void FS::Albert::update(int t, bool is_running)
{
    old_x = x;
    old_y = y;
    
    if (test_action(FLOATING))
    {
        switch (host_log->get_direction())
        {
            case LEFT_TO_RIGHT:
                x += host_log->get_num_pixels_moved();
                start_x += host_log->get_num_pixels_moved();
                break;
            case RIGHT_TO_LEFT:
                x -= host_log->get_num_pixels_moved();
                start_x -= host_log->get_num_pixels_moved();
                break;
        } // switch
    } // if
    
    // If the game is paused (either due to user pressing [ESC] or because
    // we're in TUTORIAL mode and there's a READ_TASK), disallow all 
    // key processing - those keypresses will be processed by the PauseMenu or
    // Tutorial class, as appropriate.
    if ( is_running )
    {
        process_keypress(t);
    } 
    else
    {
        turn_off_action(WALKING);
    } // if ... else
    
    process_animation(t);
    
    calc_strip();

    if (test_action(FLOATING))
    {
        check_host_log();
    } // if
    
    if (stars.is_active())
    {
        stars.update(t, get_coll_box()/*, pd*/);
        //stars.draw_stars(/*pd*/);
    } // if

    sprite_src_rect.x = current_frame * width;
    
    /*    
    pd->request_erase(&erase_rect);
    pd->request_blit(current_strip, &image_section_rect, &sprite_dest_rect);
    pd->request_update(update_rect);
    */
            
} // FS::Albert::update()
 
// **************************************************

void FS::Albert::process_keypress(int t)
{
    User_input::Keyboard_Event ke;
    
    while ( input_ptr->next_event(ke) )
    {
        switch (ke)
        {
            case User_input::UP_KEY_PRESSED:
                if (test_action(JUMPING|DISABLING_ACTIONS))
                {
                    break;
                } // if
                
                turn_on_action(WALKING);
                my_orientation = NORTH;
                start_time = t;
                start_y = y;
                break;
                
            case User_input::RIGHT_KEY_PRESSED:
                if (test_action(JUMPING|DISABLING_ACTIONS))
                {
                    break;
                } // if
            
                turn_on_action(WALKING);
                my_orientation = EAST;
                start_time = t;
                start_x = x;
                break;
            
            case User_input::DOWN_KEY_PRESSED:
                if (test_action(JUMPING|DISABLING_ACTIONS))
                {
                    break;
                } // if
        
                turn_on_action(WALKING);
                my_orientation = SOUTH;
                start_time = t;
                start_y = y;
                break;
            
            case User_input::LEFT_KEY_PRESSED:
                if (test_action(JUMPING|DISABLING_ACTIONS))
                {
                    break;
                } // if
            
                turn_on_action(WALKING);
                my_orientation = WEST;
                start_time = t;
                start_x = x;
                break;
            
            case User_input::JUMP_KEY_PRESSED:
                if ( test_action(JUMPING|SPLASHING|ZAPPING))
                {
                    break;
                }
                else if ( test_action(LEAVING_WITH_FISH|LEAVING_IN_A_HUFF) )
                {
                    // 'Short-circuit' exit-screen routine.
                    left = true;
                    break;
                } // if ... else
                
                turn_on_action(JUMPING);
                start_time = t;
                start_x = x;
                start_y = y;
                break;
            
            case User_input::UP_KEY_RELEASED:
                if (my_orientation == NORTH)
                {
                    turn_off_action(WALKING);
                } // if ... else
                break;
            
            case User_input::RIGHT_KEY_RELEASED:
                if (my_orientation == EAST)
                {
                    turn_off_action(WALKING);
                } // if
                break;
            
            case User_input::DOWN_KEY_RELEASED:
                if (my_orientation == SOUTH)
                {
                    turn_off_action(WALKING);
                } // if
                break;
            
            case User_input::LEFT_KEY_RELEASED:
                if (my_orientation == WEST)
                {
                    turn_off_action(WALKING);
                } // if
                break;
            
            case User_input::ESCAPE_PRESSED:
                turn_off_action(WALKING);
                escape_pressed = true;
                break; 
            
        } // switch
    } // while

} // FS::Albert::process_keypress()    
           
// **************************************************

void FS::Albert::process_animation(int t)
{
    if ( test_action(LEAVING_WITH_FISH) || test_action(LEAVING_IN_A_HUFF) )
    {
        current_frame = ((t - start_time) / ALBERT_DURATION) % WALKING_STRIP_NUM_FRAMES;
        x = (int) (start_x + ((t - start_time) * velocity));
        
        if ( x >= SCREEN_WIDTH )
        {
            left = true;
        } // if
    }
    else if (test_action(SPLASHING))
    {
        current_frame = ((t - start_time) / ALBERT_DURATION);
        
        if (current_frame >= SPLASHING_STRIP_NUM_FRAMES)
        {
            splashed_zapped_out = true;
            turn_off_action(SPLASHING|JUMPING);
        } // if
    }
    else if (test_action(ZAPPING))
    {
        current_frame = ((t - start_time) / ALBERT_DURATION);
        
        if (current_frame >= ZAPPING_STRIP_NUM_FRAMES)
        {
            splashed_zapped_out = true;
            turn_off_action(ZAPPING|JUMPING);
        } // if
    }
    else if (test_action(JUMPING))
    {
        current_frame = ((t - start_time) / ALBERT_DURATION);
        
        if ( (airborne = 
                (current_frame >= AIRBORNE_START_FRAME && current_frame <= AIRBORNE_END_FRAME)) )
        { 
            turn_off_action(FLOATING);
            
            if ( (next_jump_double) && (!stars.is_active()) )
            {
                stars.activate(t, get_coll_box());
            } // if
            
            float vel = ((next_jump_double) ? (2 * ALBERT_JUMPING_VELOCITY) :
                    ALBERT_JUMPING_VELOCITY);
            
            switch (my_orientation)
            {
                case NORTH:
                    y = (int) (start_y - ((t - (start_time + 500)) * 
                            /*ALBERT_JUMPING_VELOCITY*/vel));
                    break;
                    
                case EAST:
                    x = (int) (start_x + ((t - (start_time + 500)) * 
                            /*ALBERT_JUMPING_VELOCITY*/vel));
                    break;
                    
                case SOUTH:
                    y = (int) (start_y + ((t - (start_time + 500)) * 
                            /*ALBERT_JUMPING_VELOCITY*/vel));
                    break;
                    
                case WEST:
                    x = (int) (start_x - ((t - (start_time + 500)) * 
                            /*ALBERT_JUMPING_VELOCITY*/vel));
                    break;
                    
            } // switch
        } 
        else if (current_frame >= JUMPING_STRIP_NUM_FRAMES)
        {
            turn_off_action(JUMPING);
            current_frame = 0;
            start_time = t;
            start_x = x;
            start_y = y;
        }
        // This is to make sure that if Albert was already on a log before jumping,
        // he's allowed to stay on that log if he lands on it again straight afterwards.
        else if ( (current_frame > AIRBORNE_END_FRAME) && (!test_action(FLOATING)) &&
                (host_log != 0) )
        {
            turn_on_action(FLOATING);
            next_jump_double = false;
            if ( stars.is_active() && !stars.is_stopped() )
            {
                stars.stop(t);   // ? double jump will always be from a log...
            } // if
        } // if ... else
    }       
    else if (test_action(WALKING))
    {
        switch (my_orientation)
        {
            case NORTH:
                y = (int) (start_y - ((t - start_time) * velocity));
                break;
            case EAST:
                x = (int) (start_x + ((t - start_time) * velocity));
                break;
            case SOUTH:
                y = (int) (start_y + ((t - start_time) * velocity));
                break;
            case WEST:
                x = (int) (start_x - ((t - start_time) * velocity));
                break;
        } // switch
        
        // Walking strip is looped indefinitely so we need to use modulus
        // operator for 'wraparound'.        
        current_frame = ((t - start_time) / ALBERT_DURATION) % WALKING_STRIP_NUM_FRAMES;
    } // if ... else       
    
    calc_rects();
    
} // FS::Albert::process_animation()
       
// **************************************************

void FS::Albert::calc_rects()
{
    if (!test_action(LEAVING_IN_A_HUFF) && !test_action(LEAVING_WITH_FISH))
    {    
        // check bounds
        if ( x < -ALBERT_X_OFFSCREEN_TOLERANCE )
        {
            x = -ALBERT_X_OFFSCREEN_TOLERANCE;
        }
        else if ( x > (SCREEN_WIDTH - ALBERT_WIDTH + ALBERT_X_OFFSCREEN_TOLERANCE) )
        {
            x = SCREEN_WIDTH - ALBERT_WIDTH + ALBERT_X_OFFSCREEN_TOLERANCE;
        } // if ... else
    
        if (y < -ALBERT_Y_OFFSCREEN_TOLERANCE)
        {
            y = -ALBERT_Y_OFFSCREEN_TOLERANCE;
        }
        else if ( y > (SCREEN_HEIGHT - ALBERT_HEIGHT + ALBERT_Y_OFFSCREEN_TOLERANCE) )
        {
            y = SCREEN_HEIGHT - ALBERT_HEIGHT + ALBERT_Y_OFFSCREEN_TOLERANCE;
        } // if ... else
    } // if
       
    sprite_dest_rect.x = x;
    sprite_dest_rect.y = y;

} // FS::Albert::calc_rects()

// **************************************************

void FS::Albert::calc_strip()
{
    if (test_action(LEAVING_IN_A_HUFF))
    {
        my_strip = FS_gfx::WALKING_0;
    }
    else if (test_action(LEAVING_WITH_FISH))
    {
        my_strip = FS_gfx::WALKING_E_WITH_FISH_0;
    }
    else if (test_action(SPLASHING))
    {
        my_strip = FS_gfx::SPLASHING_0;
    }    
    else if (test_action(ZAPPING))
    {
        my_strip = FS_gfx::ZAPPING_0;
    }
    else if (test_action(JUMPING))
    {
        my_strip = FS_gfx::JUMPING_0;
    }
    else // must be walking
    {
        my_strip = FS_gfx::WALKING_0;
    } // if ... else
    
} // FS::Albert::calc_strip
    
// **************************************************

void FS::Albert::check_host_log()
{
    if ( !Collisions::contains(host_log->get_coll_box(), get_coll_box()) )
    {
        host_log = 0;
        turn_off_action(FLOATING);
    } // if
    
} // FS::Albert::check_host_log()
       
// **************************************************

void FS::Albert::start_splash(int t)
{
    turn_on_action(SPLASHING);
    
    start_time = t;
    start_x = x;
    start_y = y;
    
} // FS::Albert::start_splash()

// **************************************************

void FS::Albert::start_zap(int t)
{
    turn_on_action(ZAPPING);
    
    start_time = t;
    start_x = x;
    start_y = y;
    
} // FS::Albert::start_zap(int t)

// **************************************************

int FS::Albert::is_in_water_or_being_zapped() const
{
    return ( test_action(SPLASHING) || test_action(ZAPPING) );
    
} // FS::Albert::is_in_water_or_being_zapped()
 
// **************************************************

void FS::Albert::start_leave_with_fish(int t)
{
    turn_on_action(LEAVING_WITH_FISH);
    turn_off_action(JUMPING);   // FIXME: this is a bit of a hack!
    
    my_orientation = EAST;
    
    start_time = t;
    start_x = x;
    
} // FS::Albert::start_leave_with_fish()

// **************************************************

void FS::Albert::start_leave_in_a_huff(int t)
{
    turn_on_action(LEAVING_IN_A_HUFF);
    
    my_orientation = EAST;
    
    start_time = t;
    start_x = ALBERT_START_X;
    y = ALBERT_START_Y;
    
} // FS::Albert::start_leave_in_a_huff()

// **************************************************

void FS::Albert::set_new_host_log(Log* l)
{
    host_log = l;
    turn_on_action(FLOATING);
    
} // FS::Albert::set_new_host_log()

// **************************************************

const SDL_Rect& FS::Albert::get_coll_box()
{
    my_coll_boxes[my_orientation].x = sprite_dest_rect.x + coll_box_x_offsets[my_orientation];
    my_coll_boxes[my_orientation].y = sprite_dest_rect.y + coll_box_y_offsets[my_orientation];

    return my_coll_boxes[my_orientation];
    
} // FS::Albert::get_coll_box() 

// **************************************************

const SDL_Rect& FS::Albert::get_bounding_box()
{
    bounding_box.x = sprite_dest_rect.x - 10;
    bounding_box.y = sprite_dest_rect.y - 10;
    
    return bounding_box;
    
} // FS::Albert::get_bounding_box()

// **************************************************

void FS::Albert::schedule_draw()
{
    float rotation = 0.0;
    switch ( my_orientation )
    {
        case NORTH:
            break;
        case EAST:
            rotation = 90.0;
            break;
        case SOUTH:
            rotation = 180.0;
            break;
        case WEST:
            rotation = 270.0; 
            break;
    } // switch

    if ( !splashed_zapped_out )
    {
        gfx_ptr->draw_texture( my_strip + current_frame, sprite_dest_rect.x, sprite_dest_rect.y,
               true, rotation );
    }
    else
    {
        splashed_zapped_out = false;
    } // if ... else

} // FS::Albert::schedule_draw     

// **************************************************

int FS::Albert::get_host_log_row() const
{
    if (host_log)
    {
        return host_log->get_row();
    }
    else
    {
        return -1;
    } // if ... else
    
} // FS::Albert::get_host_log_row

// **************************************************
// **************************************************
