/*
  pipeworks.c - A program to control pipelines

  Copyright 2003 Damien Gregory (dgregory@lightuniverse.no-ip.com)

    This file is part of pipeworks.

    pipeworks is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    pipeworks is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with pipeworks; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <unistd.h>
#include <sys/time.h>
#include "filesize.h"
#include "timemanip.h"

char VERSION[1024]="pipeworks v0.4, Copyright 2003 Damien Gregory dgregory@lightuniverse.no-ip.com";

int VERBOSE=1;
tmanip_t INTERVAL=1.0;
fsize_t BLOCKSIZE=mega(1);
fsize_t BWLIMIT=0;

int showhelp (int rval)
{
  fprintf (stderr, "%s\n", VERSION);
  fprintf (stderr, "Usage:\n\
pipeworks [-vVh] [-l XXX | --limit=XXX] [-b XXX | --block-size=XXX] [-i XXX |\n\
          --interval=XXX] [-q | --quiet] [-h | --help] [--version]\n\
\n\
pipeworks writes it's standard input at the given rate limit (off) per second,\n\
using the given blocksize (1M) to it's standard output.\n\
\n\
Reporting will be done on stderr, at the specified interval (1 second).\n\
\n\
(Note:  It is advisable to ensure that blocksize < limit)\n\
");
  
  exit (rval);
}

void diewith (char *message, int rval)
{
  fprintf (stderr, "Error: %s\n", message);

  exit (rval);
}

long double timetold ()
{
  struct timeval tv;

  gettimeofday (&tv, NULL);

  return tv.tv_sec + (long double) tv.tv_usec / 1000000;
}

void showstats (tmanip_t st, tmanip_t lt, tmanip_t ct, fsize_t total, fsize_t lcount)
{
  if ( VERBOSE >= 1 ) {
    fprintf (stderr, "%s: ", formattime (ct - st, 's', 0, NULL));
    fprintf (stderr, "%s processed, ", formatsize (total,'b', 0, NULL));
    fprintf (stderr, "%sbps ", formatsize ((lcount / (ct - lt)), 'b', 0, NULL));
    fprintf (stderr, "(%sbps average)           \r", formatsize ((total / (ct - st)), 'b', 0, NULL));
  }
}

int gopipe ()
{
  ssize_t rcount, wcount;
  char *buffer;
  fsize_t total=0;
  tmanip_t st, ct, lt, lcount, limcount=0, limtime;

  if (! (buffer = (void*) malloc (BLOCKSIZE+10)) )
    diewith ("Cannot allocate memory for buffer (try a smaller value)...", -4);
  
  st = lt = limtime = timetold ();

  if ( VERBOSE >= 1 )
    fprintf (stderr, "0.000s: 0 processed, 0.000bps (0.000bps average)                           \r");

  while ( (rcount = read (0, buffer, BLOCKSIZE)) ) {
    while ( BWLIMIT && (rcount) / (timetold () - limtime) >= BWLIMIT );
    limtime = timetold (); limcount=0;
    if ( (wcount = write (1, buffer, rcount)) != rcount )
      diewith ("Error writing to stdout...", -5);
    total+=wcount;
    lcount+=wcount;
    ct = timetold ();
    if ( VERBOSE >= 1 && (ct - lt) >= INTERVAL ) {
      showstats (st, lt, ct, total, lcount);
      lt = ct;
      lcount=0;
    }
  }

  fprintf (stderr, "%s: ", formattime (ct - st, 's', 0, NULL));
  fprintf (stderr, "%s processed, ", formatsize (total,'b', 0, NULL));
  fprintf (stderr, "%sbps ", formatsize ((lcount / (ct - lt)), 'b', 0, NULL));
  fprintf (stderr, "(%sbps average)           \n", formatsize ((total / (ct - st)), 'b', 0, NULL));

  
  return 0;
}
  

int main (int argc, char **argv)
{
  int x;
  char *p=argv[0],parmstate=0;
  

  for (x=1; x<argc; x++) {
    p=argv[x];
    switch (parmstate) {
    case 'b':
      BLOCKSIZE = parsesize (p, 'b');
      parmstate=0;
      break;
    case 'l':
      BWLIMIT = parsesize (p, 'b');
      parmstate=0;
      break;
    case 'i':
      INTERVAL = parsetime (p, 's');
      parmstate=0;
      break;
    default:
      switch (*p++) {
      case '-':
	if ( *p == '-' ) {
	  p++;
	  if (! strncmp (p, "block-size=", 11) )
	    BLOCKSIZE = parsesize (p+11, 'b');
	  else if (! strncmp (p, "limit=", 8) )
	    BWLIMIT = parsesize (p+8, 'b');
	  else if (! strcmp (p, "help") )
	    showhelp (-1);
	  else if (! strncmp (p, "interval=", 9) )
	    INTERVAL = atoi (p+9);
	  else if (! strcmp (p, "version") ) {
	    fprintf (stderr, "%s\n", VERSION);
	    return -10;
	  } else if (! strcmp (p, "quiet") )
	    VERBOSE=0;
	  else {
	    fprintf (stderr, "Invalid option: --%s\n", p);
	    showhelp (-9);
	  }
	} else
	  while (*p)
	    switch (*p++) {
	    case 'b':
	      parmstate='b';
	      break;
	    case 'l':
	      parmstate='l';
	      break;
	    case 'i':
	      parmstate='i';
	      break;
	    case 'v':
	      VERBOSE++;
	      break;
	    case 'V':
	      fprintf (stderr, "%s\n", VERSION);
	      return -10;
	    case 'h':
	      showhelp (-1);
	    case 'q':
	      VERBOSE=0;
	      break;
	    default:
	      fprintf (stderr, "Invalid option: -%c\n", *(p-1));
	      showhelp (-2);
	    }
      }
    }
  }

  if ( VERBOSE >= 2 ) {
    fprintf (stderr, "BLOCKSIZE=%s, ", formatsize (BLOCKSIZE, 'b', 0, NULL));
    fprintf (stderr, "BWLIMIT=%s, ", formatsize (BWLIMIT, 'b', 0, NULL));
    fprintf (stderr, "INTERVAL=%s\n", formattime (INTERVAL, 's', 0, NULL));
  }

  gopipe ();

  return 0;
}
