// test_cd_ctrl.cc for BBCD - a CD player for X11 / BlackBox
// Copyright (c) 2002 Bertrand Duret <bertrand.duret at libertysurf.fr>
//
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the "Software"),
// to deal in the Software without restriction, including without limitation
// the rights to use, copy, modify, merge, publish, distribute, sublicense,
// and/or sell copies of the Software, and to permit persons to whom the
// Software is furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
// THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.

#include "CD_Ctrl.hh"
#include <iostream>
#include <strstream>

void usage() {
  std::cerr << "Usage: test_cd_ctrl [options]\n"
    " -t       play tracks\n"
    " -p       pause/resume play\n"
    " -s       stop play\n"
    " -f       play next track\n"
    " -b       play previous track\n"
    " -i       display information on drive/disc\n"
    " -d dev   use device `dev'\n";
}

int main(int argc, char** argv) {
  std::string deviceName("/dev/cdrom");
  unsigned int trackNum = 0;
  bool play = false;
  bool stop = false;
  bool pause = false;
  bool forward = false;
  bool backward = false;
  bool info = false;

  if(argc == 1) {
    usage();
    exit(1);
  }

  int argc_iter = 1;
  while(argc_iter < argc) {
    if(! strcmp(argv[argc_iter], "-d")) {
      if(++argc_iter < argc) {
        std::cout << "Device name set to " << argv[argc_iter] << std::endl;
        deviceName = argv[argc_iter];
      } else {
        std::cerr << "Option -d passed without device name parameter !"
          " Ignoring..." << std::endl;
      }
    } else if(! strcmp(argv[argc_iter], "-t")) {
      play = true;
      stop = false;
    } else if(! strcmp(argv[argc_iter], "-i")) {
      info = true;
    } else if(! strcmp(argv[argc_iter], "-s")) {
      play = false;
      stop = true;
    } else if(! strcmp(argv[argc_iter], "-p")) {
      pause = true;
    } else if(! strcmp(argv[argc_iter], "-f")) {
      play = true;
      forward = true;
    } else if(! strcmp(argv[argc_iter], "-b")) {
      play = true;
      backward = true;
    } else {
      std::istrstream is(argv[argc_iter]);
      is >> trackNum;
      if(! is) {
        usage();
        exit(1);
      }
    }
    ++argc_iter;
  }

  CD_Controler cd(deviceName);
  if(info) {
    std::cout << "Drive status: \n";
    std::cout << "Disc present ? " << (cd.isDiscPresent()?"yes":"no") << "\n";
    std::cout << "Is drive playing ? " << (cd.isPlaying()?"yes":"no") << "\n"
      << "Is drive paused ? " << (cd.isPaused()?"yes":"no") << "\n";
  }

  if(pause) {
    std::cout << "Pausing/Resuming..." << std::endl;
    cd.pause();
  } else if(play) {
    std::cout << "Playing ";
    if(forward) {
      std::cout << "next track (" << cd.playNext() << ")" << std::endl;
    } else if(backward) {
      std::cout << "previous track (" << cd.playPrevious() << ")" << std::endl;
    } else {
      std::cout << "disk (" << cd.play() << ")" << std::endl;
    }
  } else if(stop) {
    std::cout << "Stoping CD..." << std::endl;
    cd.stop();
  }

  return 0;
}
