// bbcd.cc for BBCD - a CD player for X11 / BlackBox
// Copyright (c) 2002 Bertrand Duret <bertrand.duret at libertysurf.fr>
//
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the "Software"),
// to deal in the Software without restriction, including without limitation
// the rights to use, copy, modify, merge, publish, distribute, sublicense,
// and/or sell copies of the Software, and to permit persons to whom the
// Software is furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
// THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.

#include "bbcd.hh"
#include "CD_Ctrl.hh"
#include "CD_GUI.hh"
#include <iostream>
#include <strstream>
#include <cstdio>
#include <cstdlib>


void displayHelp() {
  std::cout<<
"Usage: bbcd [options]\n"
"   -h             Display this message and exit\n"
"   -v             Display version information and exit\n"
"   -b             No BlackBox support\n"
"   -rc file       Alternate configuration file\n"
"                  Default is $HOME/.bbtools/bbcd.{bb,nobb}\n"
"   -bbrc file     Alternate BlackBox configuration file\n"
"                  Default is $HOME/.blackboxrc\n"
"   -w             Withdrawn state (i.e. in the Slit/Dock/Wharf)\n"
"   -c             Compact mode (only 2 buttons)\n"
"   -s             Fake transparent effect (used in conjunction with -w)\n"
"   -z size        Size of buttons (in pixel). Default is 10\n"
"   -l layout      Layout of buttons (Horizontal, Vertical, or Box)\n"
"                  Default is Horizontal\n"
"   -d dev         CD-ROM device. Default is /dev/cdrom\n"
"   -display  d    X server display name. Default is $DISPLAY\n"
"   -geometry g    Geometry of the window. (Only position is considered, and\n"
"                  only when not in the Slit/Dock/Wharf)\n";
}

void displayVersion() {
  std::cout<<
  "BBCD -- a CD player for BlackBox / X11\n"
  "Version " VERSION "\n";
}


int main(int argc, char** argv)
{
  Option opt(argc, argv);
  if(opt.help) {
    displayVersion();
    displayHelp();
    return 0;
  }
  if(opt.version) {
    displayVersion();
    return 0;
  }

  try {
    CD_GUI cdplayer(&opt);
    cdplayer.run();
  } catch(CD_GUI::UnableToOpenDisplay e) {
    std::cerr<<"Unable to open display "<<opt.display<<" ! Exiting..."
      <<std::endl;
    return 1;
  }

  return 0;
}


// ==========================================================================
// Option

Option::Option(int c, char** v)
  : argc(c), argv(v),
    withBB(true), bbcdrcfile(), bbrcfile(),
    withdrawn(false), compact(false), shape(false), buttonSize(10),
    layout(Horizontal), cdrom_dev("/dev/cdrom"),
    display(), geometry(), defaultBbcdrcfile()
{
  help = hasCmdLineOpt("-h");
  version = hasCmdLineOpt("-v");
  // Stop option processing if help or version are required
  if(help || version) return;

  // Make all the stuff...
  init();
}


void Option::init() {
  char * home = getenv("HOME");

  // Determine if we are with BB
  withBB = ! hasCmdLineOpt("-b");
  hasCmdLineParam("-rc", bbcdrcfile);
  hasCmdLineParam("-bbrc", bbrcfile);
  if(withBB) { // -b option is stronger than -bbrc
    if(bbrcfile == "") {
      // Consider first default location for BB rc file
      if(home)
        bbrcfile = std::string(home) + std::string("/.blackboxrc");
      else {
        std::cerr<<"(Option) HOME variable is not set in environment !\n"
            "Looking for .blackboxrc in current directory...\n";
        bbrcfile = std::string(".blackboxrc");
      }
    }
    // Get current BB style filename
    if(! getBBStyleFile()) {
      std::cerr<<"(Option) Can't open "<<bbrcfile
        <<" or style filename is missing !\n"
          "Running without BlackBox...\n";
      withBB = false;
    }
  }

  // Determine name of BBCD rc file, if needed (first default location)
  if(bbcdrcfile == "") {
    if(home)
      bbcdrcfile = std::string(home) + std::string("/.bbtools/bbcd")
          + (withBB ? std::string(".bb") : std::string(".nobb"));
    else {
      std::cerr<<"(Option) HOME variable is not set in environment !\n"
          "BBCD configuration file is not found: "
          "running with default options\n";
    }
  }

  // Depending on support of BB, set global default config file name
  defaultBbcdrcfile = (withBB ? GLOBAL_BB_RC : GLOBAL_NOBB_RC);

  // Now try to read the BBCD rc file
  if(bbcdrcfile != "") {
    if(!readBBCDConfig()) {
//      std::cerr<<"(Option) Can't open "<<bbcdrcfile<<" !\n"
//          "Running with default options...\n";
      bbcdrcfile = "";
    }
  }

  // Now supersede config options/parameters with command line arguments
  parseCmdLineArgs();

  // That's it !
}


bool Option::hasCmdLineOpt(const std::string & opt) {
  std::string param;
  return hasCmdLineParam(opt, param);
}


bool Option::hasCmdLineParam(const std::string & opt, std::string & paramv) {
  int ai = 1;
  while(ai < argc) {
    if(opt == std::string(argv[ai++])) {
      if(ai < argc) {
        paramv = argv[ai++];
      } else {
        paramv = "";
      }
      return true;
    }
  }
  return false;
}


bool Option::getBBStyleFile() {
  if(bbrcfile == "") {
    return false;
  }

  XrmDatabase rc = XrmGetFileDatabase(bbrcfile.c_str());
  if(!rc) {
    return false;
  }
  bool found = readResource(rc, "session.styleFile", "Session.StyleFile",
      bbstylefile);
  XrmDestroyDatabase(rc);

  return found;
}


bool Option::readBBCDConfig() {
  std::string config = (bbcdrcfile == "") ? defaultBbcdrcfile : bbcdrcfile;

  XrmDatabase rc = XrmGetFileDatabase(config.c_str());
  if(!rc) {
    if(config == bbcdrcfile) { // Let's try default one
      rc = XrmGetFileDatabase(defaultBbcdrcfile.c_str());
      if(!rc) {
        std::cerr<<"(Option) Can't open BBCD default rc file: "
          <<defaultBbcdrcfile<<" !\n";
        return false;
      }
    } else {
//      std::cerr<<"(Option) Can't open BBCD rc file !\n";
      return false;
    }
  }

  std::string val;
  readBoolResource(rc, "bbcd.withdrawn", "Bbcd.Withdrawn", withdrawn);
  readBoolResource(rc, "bbcd.compact", "Bbcd.Compact", compact);
  readBoolResource(rc, "bbcd.shape", "Bbcd.Shape", shape);
  if(readResource(rc, "bbcd.buttonSize", "Bbcd.ButtonSize", val)) {
    std::istrstream is(val.c_str());
    is>>buttonSize;
    if(buttonSize < 10 || buttonSize > 20) buttonSize = 10;
  }
  if(readResource(rc, "bbcd.layout", "Bbcd.Layout", val)) {
    if(val == "Horizontal") layout = Horizontal;
    else if(val == "Vertical") layout = Vertical;
    else if(val == "Box") layout = Box;
  }
  readResource(rc, "bbcd.cdromDevice", "Bbcd.CdromDevice", cdrom_dev);
  readResource(rc, "bbcd.display", "Bbcd.Display", display);
  readResource(rc, "bbcd.geometry", "Bbcd.Geometry", geometry),

  XrmDestroyDatabase(rc);
  return true;
}


bool Option::readResource(XrmDatabase & d, const std::string & n,
                          const std::string & c, std::string & v) {
  XrmValue value;
  char * value_type;
  if(XrmGetResource(d, n.c_str(), c.c_str(), &value_type, &value)) {
    v = value.addr;
    return true;
  }

  return false;
}


bool Option::readBoolResource(XrmDatabase & d, const std::string & n,
                              const std::string & c, bool & b) {
  std::string val;
  if(readResource(d, n, c, val)) {
    b = ((val == "True") || (val == "true"));
    return true;
  }

  return false;
}


void Option::parseCmdLineArgs() {
  std::string param;

  withdrawn = withdrawn || hasCmdLineOpt("-w");
  compact = compact || hasCmdLineOpt("-c");
  shape = shape || hasCmdLineOpt("-s");
  if(hasCmdLineParam("-z", param)) {
    std::istrstream p(param.c_str());
    unsigned int bsz;
    p>>bsz;
    if(p) buttonSize = bsz;
    else
      std::cerr<<"(Option) Specified button size is incorrect: "<<param<<"\n";
  }
  if(hasCmdLineParam("-l", param)) {
    if(param == "Horizontal") layout = Horizontal;
    else if(param == "Vertical") layout = Vertical;
    else if(param == "Box") layout = Box;
    else {
      std::cerr<<"(Option) Specified layout is incorrect: "<<param<<"\n";
      layout = Horizontal;
    }
  }
  hasCmdLineParam("-d", cdrom_dev);
  hasCmdLineParam("-display", display);
  hasCmdLineParam("-geometry", geometry);
}
