// Copyright 2019 Drone.IO Inc. All rights reserved.
// Use of this source code is governed by the Drone Non-Commercial License
// that can be found in the LICENSE file.

// +build !oss

package admission

import (
	"context"
	"errors"
	"time"

	"github.com/drone/drone/core"
)

// ErrCannotVerify is returned when attempting to verify the
// user is a human being.
var ErrCannotVerify = errors.New("Cannot verify user authenticity")

// Nobot enforces an admission policy that restricts access to
// users accounts that were recently created and may be bots.
// The policy expects the source control management system will
// identify and remove the bot accounts before they would be
// eligible to use the system.
func Nobot(service core.UserService, age time.Duration) core.AdmissionService {
	return &nobot{service: service, age: age}
}

type nobot struct {
	age     time.Duration
	service core.UserService
}

func (s *nobot) Admit(ctx context.Context, user *core.User) error {
	// this admission policy is only enforced for
	// new users. Existing users are always admitted.
	if user.ID != 0 {
		return nil
	}

	// if the minimum required age is not specified the check
	// is skipped.
	if s.age == 0 {
		return nil
	}
	account, err := s.service.Find(ctx, user.Token, user.Refresh)
	if err != nil {
		return err
	}
	if account.Created == 0 {
		return nil
	}
	now := time.Now()
	if time.Unix(account.Created, 0).Add(s.age).After(now) {
		return ErrCannotVerify
	}
	return nil
}
