// Copyright 2019 Drone.IO Inc. All rights reserved.
// Use of this source code is governed by the Drone Non-Commercial License
// that can be found in the LICENSE file.

// +build !oss

package admission

import (
	"context"
	"errors"
	"testing"

	"github.com/drone/drone/core"
	"github.com/drone/drone/mock"

	"github.com/golang/mock/gomock"
)

var noContext = context.TODO()

func TestMembership_MatchOrg(t *testing.T) {
	controller := gomock.NewController(t)
	defer controller.Finish()

	dummyUser := &core.User{
		Login: "octocat",
	}

	orgs := mock.NewMockOrganizationService(controller)
	orgs.EXPECT().List(gomock.Any(), dummyUser).Return([]*core.Organization{
		{Name: "bar"}, {Name: "baz"}, {Name: "GiThUb"},
	}, nil)

	service := Membership(orgs, []string{"GithuB"})
	err := service.Admit(noContext, dummyUser)
	if err != nil {
		t.Error(err)
	}
}

func TestOrganization_MatchUser(t *testing.T) {
	controller := gomock.NewController(t)
	defer controller.Finish()

	dummyUser := &core.User{
		Login: "octocat",
	}

	service := Membership(nil, []string{"octocat"})
	err := service.Admit(noContext, dummyUser)
	if err != nil {
		t.Error(err)
	}
}

func TestOrganization_MembershipError(t *testing.T) {
	controller := gomock.NewController(t)
	defer controller.Finish()

	dummyUser := &core.User{
		Login: "octocat",
	}

	orgs := mock.NewMockOrganizationService(controller)
	orgs.EXPECT().List(gomock.Any(), dummyUser).Return([]*core.Organization{
		{Name: "foo"}, {Name: "bar"},
	}, nil)

	service := Membership(orgs, []string{"baz"})
	err := service.Admit(noContext, dummyUser)
	if err != ErrMembership {
		t.Errorf("Expect ErrMembership")
	}
}

func TestOrganization_OrganizationListError(t *testing.T) {
	controller := gomock.NewController(t)
	defer controller.Finish()

	dummyUser := &core.User{
		Login: "octocat",
	}

	orgs := mock.NewMockOrganizationService(controller)
	orgs.EXPECT().List(gomock.Any(), dummyUser).Return(nil, errors.New(""))

	service := Membership(orgs, []string{"GithuB"})
	err := service.Admit(noContext, dummyUser)
	if err == nil {
		t.Errorf("Expected error")
	}
}

func TestOrganization_EmptyWhitelist(t *testing.T) {
	controller := gomock.NewController(t)
	defer controller.Finish()

	dummyUser := &core.User{
		Login: "octocat",
	}

	service := Membership(nil, []string{})
	err := service.Admit(noContext, dummyUser)
	if err != nil {
		t.Error(err)
	}
}
