// Copyright 2019 Drone.IO Inc. All rights reserved.
// Use of this source code is governed by the Drone Non-Commercial License
// that can be found in the LICENSE file.

// +build !oss

package secrets

import (
	"encoding/json"
	"net/http"
	"net/http/httptest"
	"testing"

	"github.com/drone/drone/core"
	"github.com/drone/drone/handler/api/errors"
	"github.com/drone/drone/mock"

	"github.com/golang/mock/gomock"
	"github.com/google/go-cmp/cmp"
)

func TestHandleAll(t *testing.T) {
	controller := gomock.NewController(t)
	defer controller.Finish()

	secrets := mock.NewMockGlobalSecretStore(controller)
	secrets.EXPECT().ListAll(gomock.Any()).Return(dummySecretList, nil)

	w := httptest.NewRecorder()
	r := httptest.NewRequest("GET", "/", nil)

	HandleAll(secrets).ServeHTTP(w, r)
	if got, want := w.Code, http.StatusOK; want != got {
		t.Errorf("Want response code %d, got %d", want, got)
	}

	got, want := []*core.Secret{}, dummySecretListScrubbed
	json.NewDecoder(w.Body).Decode(&got)
	if diff := cmp.Diff(got, want); len(diff) != 0 {
		t.Errorf(diff)
	}
}

func TestHandleAll_SecretListErr(t *testing.T) {
	controller := gomock.NewController(t)
	defer controller.Finish()

	secrets := mock.NewMockGlobalSecretStore(controller)
	secrets.EXPECT().ListAll(gomock.Any()).Return(nil, errors.ErrNotFound)

	w := httptest.NewRecorder()
	r := httptest.NewRequest("GET", "/", nil)

	HandleAll(secrets).ServeHTTP(w, r)
	if got, want := w.Code, http.StatusNotFound; want != got {
		t.Errorf("Want response code %d, got %d", want, got)
	}

	got, want := new(errors.Error), errors.ErrNotFound
	json.NewDecoder(w.Body).Decode(got)
	if diff := cmp.Diff(got, want); len(diff) != 0 {
		t.Errorf(diff)
	}
}
