
/*
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: menu_settings.c 2562 2007-07-21 11:35:50Z mschwerin $
 *
 */
#include "config.h"

#include <assert.h>
#include <stdio.h>
#include <xine.h>

#include "aex.h"
#include "environment.h"
#include "heap.h"
#include "i18n.h"
#include "logger.h"
#include "menu_base.h"
#include "menu_help.h"
#include "menu_filelist.h"
#include "menu_main.h"
#include "menu_playback.h"
#include "menu_playlist.h"
#include "menu_settings.h"
#include "menu_weather.h"
#include "oxine.h"
#include "utils.h"

extern oxine_t *oxine;

#define BUTTON_Y                (odk_osd_get_height (oxine->odk) - 55)

#define LABEL_X                 (20)
#define LABEL_W                 ((odk_osd_get_width (oxine->odk) - 40) * 60 / 100)

#define EDIT_X                  (LABEL_X + LABEL_W + 10)
#define EDIT_W                  (odk_osd_get_width (oxine->odk) - 20 - EDIT_X)

#define ENTRY_FIRST_Y           (100)
#define ENTRY_SPACING           (60)
#define ENTRIES_PER_PAGE        ((odk_osd_get_height (oxine->odk) - 175) / ENTRY_SPACING)

#define SETTINGS_BUTTONS_NUM 3
static otk_widget_t *settings_menu_buttons[SETTINGS_BUTTONS_NUM];

static bool changes_to_save = false;

static void display_config_description (xine_cfg_entry_t * entry, int y);
static void display_config_string (xine_cfg_entry_t * entry, int y);
static void display_config_range (xine_cfg_entry_t * entry, int y);
static void display_config_enum (xine_cfg_entry_t * entry, int y);
static void display_config_number (xine_cfg_entry_t * entry, int y);
static void display_config_bool (xine_cfg_entry_t * entry, int y);

static void
save_settings (void)
{
    if (changes_to_save) {
        xine_config_save (oxine->xine, get_file_config ());
        changes_to_save = false;
    }
}


static void
restart_oxine_cb (void *p)
{
    oxine_event_t event;

    oxine->restart = true;

    event.type = OXINE_EVENT_KEY;
    event.source.key = OXINE_KEY_QUIT;

    odk_oxine_event_send (oxine->odk, &event);
}


static void
save_settings_cb (void *p)
{
    otk_cb_t cb = (otk_cb_t) p;
    assert (cb);

    if (changes_to_save) {
        show_message_dialog (restart_oxine_cb, oxine, cb, oxine,
                             DIALOG_YES_NO, "menu_settings.png",
                             _("It may be necessary to restart oxine\n"
                               "for the new configuration to take effect!\n\n"
                               "Restart oxine now?"));
    }
    else {
        cb (oxine);
    }

    save_settings ();
}

/* *********************************************************
 * Airport express settings menu.
 * *********************************************************/
#ifdef HAVE_AEX

static aex_t *
aex_get_by_pos (int pos)
{
    int i = 0;

    aex_t *aex = aex_first ();
    while (aex && (i != pos)) {
        i++;
        aex = aex_next (aex);
    }

    return aex;
}


static int
aex_speakers_change_cb (void *p, int num_current)
{
    if (num_current == 0) {
        xine_cfg_entry_t config;
        config_lookup_entry ("audio.driver.use_aex", &config);
        config.num_value = false;
        config_update_entry (&config);
    }
    else {
        aex_t *aex = aex_get_by_pos (num_current - 1);

        xine_cfg_entry_t config;
        config_lookup_entry ("audio.device.aex_host", &config);
        config.str_value = aex->address;
        config_update_entry (&config);
        config_lookup_entry ("audio.device.aex_port", &config);
        config.num_value = aex->port;
        config_update_entry (&config);
        config_lookup_entry ("audio.driver.use_aex", &config);
        config.num_value = true;
        config_update_entry (&config);
    }
    changes_to_save = true;
    otk_widget_set_enabled (settings_menu_buttons[0], true);

    return num_current;
}


static void
show_menu_aex_settings (void *data_p)
{
    xine_cfg_entry_t config;

    xine_audio_port_t *audio_port = xine_open_audio_driver (oxine->xine,
                                                            "aex", NULL);

    bool use_aex = config_get_bool ("audio.driver.use_aex");
    const char *aex_host = config_get_string ("audio.device.aex_host");

    if (!audio_port) {
        show_message_dialog (save_settings_cb, show_menu_settings, NULL, NULL,
                             DIALOG_OK, "menu_settings.png",
                             _("Could not open AEX audio driver."));
        return;
    }

    create_new_window (false, true);

    int y = ENTRY_FIRST_Y;
    otk_label_new (oxine->otk, LABEL_X, y + 20, LABEL_W,
                   OTK_ALIGN_LEFT | OTK_ALIGN_VCENTER,
                   _("airport express to use"));

    otk_widget_t *aex_speakers_sel =
        otk_selector_new (oxine->otk, EDIT_X, y, EDIT_W,
                          aex_speakers_change_cb, NULL);

    y += ENTRY_SPACING;
    config_lookup_entry ("audio.device.aex_latency", &config);
    display_config_range (&config, y);

    settings_menu_buttons[0] =
        otk_text_button_new (oxine->otk, 20, 545, 180, 35,
                             _("Save"), save_settings_cb,
                             show_menu_aex_settings);
    otk_widget_set_alignment (settings_menu_buttons[0], OTK_ALIGN_CENTER);
    otk_widget_set_enabled (settings_menu_buttons[0], false);

    settings_menu_buttons[1] =
        otk_text_button_new (oxine->otk, 210, 545, 180, 35,
                             _("Next Page"), NULL, NULL);
    otk_widget_set_alignment (settings_menu_buttons[1], OTK_ALIGN_CENTER);
    otk_widget_set_enabled (settings_menu_buttons[1], false);

    settings_menu_buttons[2] =
        otk_text_button_new (oxine->otk, 400, 545, 180, 35,
                             _("Back"), save_settings_cb, show_menu_settings);
    otk_widget_set_alignment (settings_menu_buttons[2], OTK_ALIGN_CENTER);
    otk_widget_set_focused (settings_menu_buttons[2], true);

    otk_selectoritem_new (aex_speakers_sel, _("none"), NULL, NULL);
    int i = 1;
    int position = 0;
    aex_monitor_lock ();
    aex_t *aex = aex_first ();
    while (aex) {
        otk_selectoritem_new (aex_speakers_sel, aex->title, NULL, NULL);
        if (aex_host && (strcasecmp (aex->address, aex_host) == 0)) {
            position = i;
        }
        aex = aex_next (aex);
        i++;
    }
    aex_monitor_unlock ();

    if (use_aex) {
        otk_list_set_pos (aex_speakers_sel, position);
    }
    else {
        otk_list_set_pos (aex_speakers_sel, 0);
    }

    set_backto_menu (show_menu_aex_settings, NULL);
    set_current_menu (show_menu_aex_settings, NULL);

    show_user_interface (NULL);
    show_menu_background ("menu_settings.png");
}
#endif /* HAVE_AEX */


/* *********************************************************
 * Weather settings menu.
 * *********************************************************/
#ifdef HAVE_WEATHER

static otk_widget_t *weather_station_sel = NULL;
static otk_widget_t *weather_country_sel = NULL;

static weather_country_t *
country_get_by_pos (int pos)
{
    int i = 0;

    weather_country_t *country = weather_country_get_first ();
    while (country && (i != pos)) {
        i++;
        country = weather_country_get_next (country);
    }

    return country;
}


static weather_station_t *
station_get_by_pos (weather_country_t * country, int pos)
{
    int i = 0;

    weather_station_t *station;
    station = (weather_station_t *) l_list_first (country->stations);
    while (station && (i != pos)) {
        i++;
        station = (weather_station_t *) l_list_next (country->stations,
                                                     station);
    }

    return station;
}


static void
station_list_update (weather_country_t * country)
{
    assert (country);

    otk_list_clear (weather_station_sel);

    weather_station_t *station;
    station = (weather_station_t *) l_list_first (country->stations);
    while (station) {
        otk_selectoritem_new (weather_station_sel, station->name, NULL, NULL);
        station = (weather_station_t *) l_list_next (country->stations,
                                                     station);
    }

    otk_list_set_pos (weather_station_sel, country->last_station_i);
}


static int
country_change_cb (void *p, int num_current)
{
    weather_country_t *country = country_get_by_pos (num_current);
    weather_station_t *station =
        station_get_by_pos (country, country->last_station_i);

    station_list_update (country);

    xine_cfg_entry_t config;
    config_lookup_entry ("weather.station_id", &config);
    config.str_value = station->id;
    config_update_entry (&config);
    changes_to_save = true;
    otk_widget_set_enabled (settings_menu_buttons[0], true);

    return num_current;
}


static int
station_change_cb (void *p, int num_current)
{
    int country_i = otk_list_get_pos (weather_country_sel);
    weather_country_t *country = country_get_by_pos (country_i);
    weather_station_t *station = station_get_by_pos (country, num_current);

    country->last_station_i = num_current;

    xine_cfg_entry_t config;
    config_lookup_entry ("weather.station_id", &config);
    config.str_value = station->id;
    config_update_entry (&config);
    changes_to_save = true;
    otk_widget_set_enabled (settings_menu_buttons[0], true);

    return num_current;
}


static void
show_menu_weather_settings (void *data_p)
{
    xine_cfg_entry_t config;
    config_lookup_entry ("weather.station_id", &config);

    int station_i = 0;
    int country_i = 0;

    weather_station_t *station = NULL;
    weather_country_t *country = weather_country_get_first ();
    while (country) {
        station_i = 0;
        station = (weather_station_t *) l_list_first (country->stations);
        while (station) {
            if (strcasecmp (config.str_value, station->id) == 0) {
                country->last_station_i = station_i;
                goto cur_station_found;
            }
            station_i++;
            station = (weather_station_t *) l_list_next (country->stations,
                                                         station);
        }
        country_i++;
        country = weather_country_get_next (country);
    }
  cur_station_found:

    create_new_window (false, true);

    int y = ENTRY_FIRST_Y;
    otk_label_new (oxine->otk, LABEL_X, y + 20, LABEL_W,
                   OTK_ALIGN_LEFT | OTK_ALIGN_VCENTER, _("Country"));
    weather_country_sel = otk_selector_new (oxine->otk, EDIT_X, y, EDIT_W,
                                            country_change_cb, NULL);

    y += ENTRY_SPACING;
    otk_label_new (oxine->otk, LABEL_X, y + 20, LABEL_W,
                   OTK_ALIGN_LEFT | OTK_ALIGN_VCENTER, _("Station"));
    weather_station_sel = otk_selector_new (oxine->otk, EDIT_X, y, EDIT_W,
                                            station_change_cb, NULL);

    y += ENTRY_SPACING;
    config_lookup_entry ("weather.metar_url", &config);
    display_config_string (&config, y);

    settings_menu_buttons[0] =
        otk_text_button_new (oxine->otk, 20, BUTTON_Y, 180, 35,
                             _("Save"), save_settings_cb,
                             show_menu_weather_settings);
    otk_widget_set_alignment (settings_menu_buttons[0], OTK_ALIGN_CENTER);
    otk_widget_set_enabled (settings_menu_buttons[0], false);

    settings_menu_buttons[1] =
        otk_text_button_new (oxine->otk, 210, BUTTON_Y, 180, 35,
                             _("Next Page"), NULL, NULL);
    otk_widget_set_alignment (settings_menu_buttons[1], OTK_ALIGN_CENTER);
    otk_widget_set_enabled (settings_menu_buttons[1], false);

    settings_menu_buttons[2] =
        otk_text_button_new (oxine->otk, 400, BUTTON_Y, 180, 35,
                             _("Back"), save_settings_cb, show_menu_settings);
    otk_widget_set_alignment (settings_menu_buttons[2], OTK_ALIGN_CENTER);
    otk_widget_set_focused (settings_menu_buttons[2], true);

    country = weather_country_get_first ();
    while (country) {
        otk_selectoritem_new (weather_country_sel, country->name, NULL, NULL);
        country = weather_country_get_next (country);
    }
    otk_list_set_pos (weather_country_sel, country_i);

    station_list_update (country_get_by_pos (country_i));

    set_backto_menu (show_menu_weather_settings, NULL);
    set_current_menu (show_menu_weather_settings, NULL);

    show_user_interface (NULL);
    show_menu_background ("menu_settings.png");
}
#endif /* HAVE_WEATHER */

/* *********************************************************
 * Generic settings menu.
 * *********************************************************/

typedef struct {
    char *name;
    char *label;
} settings_section_t;

static char *cur_page_first_key = NULL;
static char *cur_page_last_key = NULL;

static l_list_t *cur_page_entries = NULL;

static bool focus_next_page = false;
static bool multipage_section = false;

static char *section = NULL;

static void settings_menu_gettext (void);
static void show_menu_section_settings (void *oxine_p);

static void
free_config_dups (void *entry_p)
{
    xine_cfg_entry_t *entry = (xine_cfg_entry_t *) entry_p;
    ho_free (entry);
}


static xine_cfg_entry_t *
ho_config_dup (xine_cfg_entry_t * entry)
{
    xine_cfg_entry_t *entry_cpy = ho_new (xine_cfg_entry_t);
    memcpy (entry_cpy, entry, sizeof (xine_cfg_entry_t));

    if (!cur_page_entries)
        cur_page_entries = l_list_new ();
    l_list_append (cur_page_entries, entry_cpy);

    return entry_cpy;
}


static void
display_config_description (xine_cfg_entry_t * entry, int y)
{
    otk_label_new (oxine->otk, LABEL_X, y + 20, LABEL_W,
                   OTK_ALIGN_LEFT | OTK_ALIGN_VCENTER, entry->description);
}


static void
config_string_change_cb (void *entry_p, char *text)
{
    xine_cfg_entry_t *entry = (xine_cfg_entry_t *) entry_p;

    if (!text)
        return;

    entry->str_value = text;
    config_update_entry (entry);
    changes_to_save = true;
    otk_widget_set_enabled (settings_menu_buttons[0], true);

    ho_free (text);
}


static void
display_config_string (xine_cfg_entry_t * entry, int y)
{
    display_config_description (entry, y);

    xine_cfg_entry_t *entry_cpy = ho_config_dup (entry);
    otk_widget_t *editbox = otk_editbox_new (oxine->otk, EDIT_X, y,
                                             EDIT_W, 40, 1024,
                                             config_string_change_cb,
                                             entry_cpy,
                                             config_string_change_cb,
                                             entry_cpy);
    otk_editbox_set_text (editbox, entry->str_value);
}


static int
config_range_get_cb (void *entry_p)
{
    xine_cfg_entry_t *entry = (xine_cfg_entry_t *) entry_p;

    return entry->num_value;
}


static int
config_range_change_cb (void *entry_p, int value)
{
    xine_cfg_entry_t *entry = (xine_cfg_entry_t *) entry_p;

    entry->num_value = value;
    if (entry->num_value > entry->range_max) {
        entry->num_value = entry->range_max;
    }
    if (entry->num_value < entry->range_min) {
        entry->num_value = entry->range_min;
    }

    config_update_entry (entry);
    changes_to_save = true;
    otk_widget_set_enabled (settings_menu_buttons[0], true);

    return value;
}


static void
display_config_range (xine_cfg_entry_t * entry, int y)
{
    display_config_description (entry, y);

    xine_cfg_entry_t *entry_cpy = ho_config_dup (entry);

    char *format = "%2.0f";
    if ((entry->range_min == 0) && (entry->range_max == 100)) {
        format = "%2.0f %%";
    }

    otk_widget_t *s;
    s = otk_slider_new (oxine->otk, EDIT_X, y, EDIT_W, 40,
                        5, 36, true, OTK_SLIDER_HORIZONTAL,
                        format, 1, true, true,
                        entry->num_default, 1,
                        entry->range_min, entry->range_max,
                        config_range_get_cb, entry_cpy,
                        config_range_change_cb, entry_cpy);
    otk_widget_set_updated (s, true);
}


static int
config_enum_change_cb (void *entry_p, int num_current)
{
    xine_cfg_entry_t *entry = (xine_cfg_entry_t *) entry_p;

    if (entry->num_value != num_current) {
        entry->num_value = num_current;
        config_update_entry (entry);
        changes_to_save = true;
        otk_widget_set_enabled (settings_menu_buttons[0], true);
    }

    return num_current;
}


static void
display_config_enum (xine_cfg_entry_t * entry, int y)
{
    display_config_description (entry, y);

    xine_cfg_entry_t *entry_cpy = ho_config_dup (entry);
    otk_widget_t *list = otk_selector_new (oxine->otk, EDIT_X, y, EDIT_W,
                                           config_enum_change_cb, entry_cpy);
    int i = 0;
    char **enum_values = entry->enum_values;
    while (enum_values[i]) {
        otk_selectoritem_new (list, _(enum_values[i]), NULL, NULL);
        i++;
    }
    otk_list_set_pos (list, entry->num_value);
}


static void
config_num_change_cb (void *entry_p, char *text)
{
    xine_cfg_entry_t *entry = (xine_cfg_entry_t *) entry_p;

    if (!text)
        return;

    entry->num_value = atoi (text);
    config_update_entry (entry);
    changes_to_save = true;
    otk_widget_set_enabled (settings_menu_buttons[0], true);

    ho_free (text);
}


static void
display_config_number (xine_cfg_entry_t * entry, int y)
{
    display_config_description (entry, y);

    char str[10];
    snprintf (str, 10, "%d", entry->num_value);
    xine_cfg_entry_t *entry_cpy = ho_config_dup (entry);
    otk_widget_t *editbox = otk_editbox_new (oxine->otk, EDIT_X, y,
                                             EDIT_W, 40, 100,
                                             config_num_change_cb, entry_cpy,
                                             config_num_change_cb, entry_cpy);
    otk_editbox_set_text (editbox, str);
}


static void
config_bool_change_cb (void *entry_p)
{
    xine_cfg_entry_t *entry = (xine_cfg_entry_t *) entry_p;

    entry->num_value = (entry->num_value + 1) % 2;
    config_update_entry (entry);
    changes_to_save = true;
    otk_widget_set_enabled (settings_menu_buttons[0], true);
}


static void
display_config_bool (xine_cfg_entry_t * entry, int y)
{
    xine_cfg_entry_t *entry_cpy = ho_config_dup (entry);
    otk_label_new (oxine->otk, LABEL_X, y + 20, LABEL_W + EDIT_W - 30,
                   OTK_ALIGN_LEFT | OTK_ALIGN_VCENTER, entry->description);
    otk_checkbox_new (oxine->otk, EDIT_X + EDIT_W - 30, y + 5, 30,
                      entry->num_value, config_bool_change_cb, entry_cpy);
}


static void
change_section_cb (void *section_p)
{
    if (cur_page_entries) {
        l_list_free (cur_page_entries, free_config_dups);
        cur_page_entries = NULL;
    }

    multipage_section = false;
    focus_next_page = false;

    section = (char *) section_p;
    ho_free (cur_page_first_key);
    cur_page_first_key = NULL;
    ho_free (cur_page_last_key);
    cur_page_last_key = NULL;

#ifdef HAVE_AEX
    if (strcmp (section, "aex") == 0) {
        show_menu_aex_settings (oxine);
    }
    else
#endif
#ifdef HAVE_WEATHER
    if (strcmp (section, "weather") == 0) {
        show_menu_weather_settings (oxine);
    }
    else
#endif
    {
        show_menu_section_settings (oxine);
    }
}


static void
change_page_cb (void *section_p)
{
    if (cur_page_entries) {
        l_list_free (cur_page_entries, free_config_dups);
        cur_page_entries = NULL;
    }

    focus_next_page = true;

    ho_free (cur_page_first_key);
    cur_page_first_key = ho_strdup (cur_page_last_key);
    ho_free (cur_page_last_key);
    cur_page_last_key = NULL;

    show_menu_section_settings (oxine);
}


static bool
is_from_correct_section (xine_cfg_entry_t * entry)
{
    return (strncmp (entry->key, section, strlen (section)) == 0);
}


static bool
is_ok_to_show (xine_cfg_entry_t * entry)
{
    if (strcmp (entry->key, "shutdown.last") == 0) {
        return false;
    }
    else if (strcmp (entry->key, "media.dvb.last_channel") == 0) {
        return false;
    }
    else if (strcmp (entry->key, "misc.ask_missing_mediamarks") == 0) {
        return false;
    }
#ifdef HAVE_AEX
    else if (strcmp (entry->key, "audio.driver.use_aex") == 0) {
        return false;
    }
    else if (strcmp (entry->key, "audio.device.aex_host") == 0) {
        return false;
    }
    else if (strcmp (entry->key, "audio.device.aex_port") == 0) {
        return false;
    }
    else if (strcmp (entry->key, "audio.device.aex_latency") == 0) {
        return false;
    }
#endif

    return true;
}


static void
settings_menu_event_handler (void *oxine_p, oxine_event_t * event)
{
    if (!is_current_menu (show_menu_settings)
#ifdef HAVE_AEX
        && !is_current_menu (show_menu_aex_settings)
#endif
#ifdef HAVE_WEATHER
        && !is_current_menu (show_menu_weather_settings)
#endif
        && !is_current_menu (show_menu_section_settings)) {
        return;
    }
    if (event->type != OXINE_EVENT_KEY) {
        return;
    }

    oxine_key_id_t saved_key = event->source.key;
    event->source.key = OXINE_KEY_NULL;

    switch (saved_key) {
    case OXINE_KEY_SAVE:
        save_settings_cb (get_current_menu ());
        break;
    case OXINE_KEY_BACK:
        if (is_current_menu (show_menu_settings)) {
            show_menu_main (oxine);
        }
        else {
            save_settings_cb (show_menu_settings);
        }
        break;
    case OXINE_KEY_MENU_PLAYBACK:
        if (odk_current_is_playback_mode (oxine->odk)) {
            save_settings_cb (show_menu_playback);
        }
        else {
            save_settings_cb (show_menu_main);
        }
        break;
    case OXINE_KEY_MENU_MAIN:
    case OXINE_KEY_TELEVISION:
        save_settings_cb (show_menu_main);
        break;
    case OXINE_KEY_MENU_MUSIC:
        save_settings_cb (show_menu_music);
        break;
    case OXINE_KEY_MENU_VIDEO:
        save_settings_cb (show_menu_video);
        break;
#ifdef HAVE_IMAGE_PLAYBACK
    case OXINE_KEY_MENU_IMAGE:
        save_settings_cb (show_menu_image);
        break;
#endif /* HAVE_IMAGE_PLAYBACK */
    case OXINE_KEY_MENU_PLAYLIST:
        save_settings_cb (show_menu_playlist);
        break;
    case OXINE_KEY_MENU_HELP:
        save_settings_cb (show_menu_help);
        break;
#ifdef HAVE_WEATHER
    case OXINE_KEY_MENU_WEATHER:
        save_settings_cb (show_menu_weather);
        break;
#endif /* HAVE_WEATHER */
    case OXINE_KEY_MENU_SETTINGS:
        if (odk_current_is_playback_mode (oxine->odk)) {
            save_settings_cb (show_menu_playback);
        }
        else {
            save_settings_cb (show_menu_settings);
        }
        break;
    case OXINE_KEY_QUIT:
    case OXINE_KEY_SHUTDOWN:
        save_settings ();
        free_menu_settings ();
    default:
        event->source.key = saved_key;
        break;
    }
}


static void
show_menu_section_settings (void *oxine_p)
{
    create_new_window (false, true);

    int err = 0;
    xine_cfg_entry_t entry;
    if (cur_page_first_key) {
        err = config_lookup_entry (cur_page_first_key, &entry);
    }
    else {
        err = xine_config_get_first_entry (oxine->xine, &entry);
    }

    int num = 0;
    while ((err || num == 0) && (num < ENTRIES_PER_PAGE)) {
        if (entry.description && is_from_correct_section (&entry)
            && is_ok_to_show (&entry)) {

            if (!cur_page_first_key) {
                cur_page_first_key = ho_strdup (entry.key);
            }

            int y = ENTRY_FIRST_Y + (num * ENTRY_SPACING);
            switch (entry.type) {
            case XINE_CONFIG_TYPE_STRING:
                display_config_string (&entry, y);
                break;
            case XINE_CONFIG_TYPE_RANGE:
                display_config_range (&entry, y);
                break;
            case XINE_CONFIG_TYPE_ENUM:
                display_config_enum (&entry, y);
                break;
            case XINE_CONFIG_TYPE_NUM:
                display_config_number (&entry, y);
                break;
            case XINE_CONFIG_TYPE_BOOL:
                display_config_bool (&entry, y);
                break;
            }
            num += 1;
        }
        err = xine_config_get_next_entry (oxine->xine, &entry);
    }
    ho_free (cur_page_last_key);
    cur_page_last_key = ho_strdup (entry.key);

    if (num >= ENTRIES_PER_PAGE) {
        multipage_section = true;
    }

    settings_menu_buttons[0] =
        otk_text_button_new (oxine->otk, 20, BUTTON_Y, 180, 35,
                             _("Save"), save_settings_cb,
                             show_menu_section_settings);
    otk_widget_set_alignment (settings_menu_buttons[0], OTK_ALIGN_CENTER);
    otk_widget_set_enabled (settings_menu_buttons[0], changes_to_save);

    settings_menu_buttons[1] =
        otk_text_button_new (oxine->otk, 210, BUTTON_Y, 180, 35,
                             _("Next Page"), change_page_cb, section);
    otk_widget_set_alignment (settings_menu_buttons[1], OTK_ALIGN_CENTER);
    otk_widget_set_enabled (settings_menu_buttons[1], multipage_section);
    otk_widget_set_focused (settings_menu_buttons[1], focus_next_page);

    settings_menu_buttons[2] =
        otk_text_button_new (oxine->otk, 400, BUTTON_Y, 180, 35,
                             _("Back"), save_settings_cb, show_menu_settings);
    otk_widget_set_alignment (settings_menu_buttons[2], OTK_ALIGN_CENTER);
    otk_widget_set_focused (settings_menu_buttons[2], !focus_next_page);

    set_backto_menu (show_menu_section_settings, NULL);
    set_current_menu (show_menu_section_settings, NULL);

    show_user_interface (NULL);
    show_menu_background ("menu_settings.png");
}


void
show_menu_settings (void *oxine_p)
{
    settings_section_t sections[] = {
        {"gui", _("User Interface")},
        {"playlist", _("Playlist")},
#ifdef HAVE_HAL
        {"hal", _("Removable Drives")},
#endif
#ifdef HAVE_EXTRACTOR
        {"extractor", _("CD / DVD Extractor")},
#endif
#ifdef HAVE_AEX
        {"aex", _("Apple Airport Express")},
#endif
#ifdef HAVE_WEATHER
        {"weather", _("Weather Report")},
#endif
#if defined (HAVE_SHOUTCAST) \
 || defined (HAVE_YOUTUBE) \
 || defined (HAVE_TVLINKS)
        {"streaming", _("Streaming Media")},
#endif
        {"television", _("Television")},
        {"visual_anim", _("Audio Visualization")},
        {"audio", _("Audio")},
        {"video", _("Video")},
#ifdef HAVE_IMAGE_PLAYBACK
        {"image", _("Images")},
#endif
        {"media", _("Media")},
        {"decoder", _("Decoder")},
        {"subtitles", _("Subtitles")},
        {"effects", _("Effects")},
        {"engine", _("Engine")},
        {"shutdown", _("Shutdown")},
        {"misc", _("Miscellaneous")},
        {NULL, NULL},
    };

    odk_add_event_handler (oxine->odk, settings_menu_event_handler, oxine,
                           EVENT_HANDLER_PRIORITY_NORMAL);

    static otk_widget_t *settings_menu_list = NULL;
    static otk_widget_t *settings_menu_window = NULL;
    if (settings_menu_window) {
        otk_set_current_window (oxine->otk, settings_menu_window);
        goto out_show;
    }
    settings_menu_window = create_new_window (true, true);

    /* We use the same width and height as in the main menu. */
    int w = 370;
    if (oxine->main_menu->w != -1) {
        w = oxine->main_menu->w;
    }
    int x = (odk_osd_get_width (oxine->odk) - w) / 2;
    if (oxine->main_menu->x != -1) {
        x = oxine->main_menu->x;
    }
    int y = 100;
    if (oxine->main_menu->y != -1) {
        y = oxine->main_menu->y;
    }
    int h = (odk_osd_get_height (oxine->odk) - y - 20);
    if (oxine->main_menu->h != -1) {
        h = oxine->main_menu->h;
    }

    settings_menu_list = otk_menulist_new (oxine->otk, x, y, w, h,
                                           45, 55, oxine);
    int i = 0;
    for (; sections[i].name; i++) {
#ifdef HAVE_AEX
        if (strcmp (sections[i].name, "aex") == 0) {
            if (odk_is_available_audio_driver (oxine->odk, "aex")) {
                otk_menuitem_new (settings_menu_list, sections[i].label,
                                  show_menu_aex_settings, oxine);
            }
        }
        else
#endif
#ifdef HAVE_WEATHER
        if (strcmp (sections[i].name, "weather") == 0) {
            otk_menuitem_new (settings_menu_list, sections[i].label,
                              show_menu_weather_settings, oxine);
        }
        else
#endif
        {
            otk_menuitem_new (settings_menu_list, sections[i].label,
                              change_section_cb, sections[i].name);
        }
    }
    {
        otk_menuitem_new (settings_menu_list, _("Mainmenu"),
                          show_menu_main, oxine);
    }

  out_show:
    set_backto_menu (show_menu_settings, NULL);
    set_current_menu (show_menu_settings, NULL);

    show_user_interface (NULL);
    show_menu_background ("menu_settings_list.png");
}


void
free_menu_settings (void)
{
    if (cur_page_entries) {
        l_list_free (cur_page_entries, free_config_dups);
        cur_page_entries = NULL;
    }

    ho_free (cur_page_first_key);
    cur_page_first_key = NULL;
    ho_free (cur_page_last_key);
    cur_page_last_key = NULL;

    if (false) {
        settings_menu_gettext ();
    }
}


/* Here are all the strings used in enumerations. They are here so gettext can
 * find them. This is necessary as they are not translated by xine-lib which
 * is rather sad :-(. To update these run: 
 * grep -B 1 "# {" ~/.oxine/config | grep -v "^--$" | sed -e 's/, default.*$//' -e 's/# { /_("/' -e 's/  /");_("/g' -e 's/ }$/");/' | sed -e 's/^# /    \/\* /' -e 's/$/ \*\//' | sed -e 's/; \*\//;/' >> menu_settings.c
 */
static void
settings_menu_gettext (void)
{
    /* palette (foreground-border-background) to use for subtitles and OSD */
    _("white-black-transparent");
    _("white-none-transparent");
    _("white-none-translucid");
    _("yellow-black-transparent");
    /* speaker arrangement */
    _("Mono 1.0");
    _("Stereo 2.0");
    _("Headphones 2.0");
    _("Stereo 2.1");
    _("Surround 3.0");
    _("Surround 4.0");
    _("Surround 4.1");
    _("Surround 5.0");
    _("Surround 5.1");
    _("Surround 6.0");
    _("Surround 6.1");
    _("Surround 7.1");
    _("Pass Through");
    /* method to sync audio and video */
    _("metronom feedback");
    _("resample");
    /* enable resampling */
    _("auto");
    _("off");
    _("on");
    /* CSS decryption method */
    _("key");
    _("disc");
    _("title");
    /* OpenGL renderer */
    _("2D_Tex_Fragprog");
    _("2D_Tex");
    _("2D_Tex_Tiled");
    _("Image_Pipeline");
    _("Cylinder");
    _("Env_Mapped_Torus");
    /* play mode when title/chapter is given */
    _("entire dvd");
    _("one chapter");
    /* unit for seeking */
    _("seek in program chain");
    _("seek in program");
    /* unit for the skip action */
    _("skip program");
    _("skip part");
    _("skip title");
    /* network bandwidth */
    _("14.4 Kbps (Modem)");
    _("19.2 Kbps (Modem)");
    _("28.8 Kbps (Modem)");
    _("33.6 Kbps (Modem)");
    _("34.4 Kbps (Modem)");
    _("57.6 Kbps (Modem)");
    _("115.2 Kbps (ISDN)");
    _("262.2 Kbps (Cable/DSL)");
    _("393.2 Kbps (Cable/DSL)");
    _("524.3 Kbps (Cable/DSL)");
    _("1.5 Mbps (T1)");
    _("10.5 Mbps (LAN)");
    /* MMS protocol */
    _("auto");
    _("TCP");
    _("HTTP");
    /* VCD default type to use on autoplay */
    _("MPEG track");
    _("entry");
    _("segment");
    _("playback-control item");
    /* VCD position slider range */
    _("auto");
    _("track");
    _("entry");
    /* subtitle size */
    _("tiny");
    _("small");
    _("normal");
    _("large");
    _("very large");
    _("huge");
    /* colorspace conversion method */
    _("Fast but not photorealistic");
    _("Slow but looks better");
    /* media format detection strategy */
    _("default");
    _("reverse");
    _("content");
    _("extension");
    /* memcopy method used by xine */
    _("probe");
    _("libc");
    _("kernel");
    _("mmx");
    _("mmxext");
    _("sse");
    /* when removable disc is inserted */
    _("do nothing");
    _("play disc");
    _("play files on disc");
    _("extract disc");
    /* if playback is running */
    _("do nothing");
    _("replace playlist");
    _("append to playlist");
    /* MPAA movie rating to allow */
    _("G (general audiences)");
    _("PG (parental guidance)");
    _("PG-13 (parents cautioned)");
    _("R (restricted)");
    _("NC-17 (must be 18)");
    /* type of television */
    _("analogue television");
    _("digital television");
    _("video disc recorder");
    /* audio visualization type */
    _("background image");
    _("background animation");
    _("post plugin");
    /* post plugin type */
    _("oscope");
    _("fftscope");
    _("fftgraph");
    _("goom");
    /* encoder to use for encoding audio */
    _("Lossless (FLAC)");
    _("Lossy (OGG)");
    _("Lossy (MP3)");
}
