
/*
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: mediamarks_favorites.c 2621 2007-07-25 12:53:00Z mschwerin $
 *
 */
#include "config.h"

#include <assert.h>
#include <errno.h>
#include <limits.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "codeset.h"
#include "environment.h"
#include "filelist.h"
#include "heap.h"
#include "i18n.h"
#include "logger.h"
#include "mediamarks.h"
#include "mediamarks_favorites.h"
#include "utils.h"

#define OXINE_FAVORITES_XML     "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n"
#define OXINE_FAVORITES_DTD     "<!DOCTYPE oxine_favorites SYSTEM \"http://oxine.sf.net/DTD/favorites.dtd\">\n\n"


void
favorites_add_directory (filelist_t * favorites_filelist,
                         const char *title, const char *mrl)
{
    if (!config_get_bool ("misc.favorites.enable")) {
        return;
    }

    if (!is_directory (mrl)) {
        debug ("This is not a directory (ignoring): '%s'!", mrl);
        return;
    }

    char *rmrl = resolve_softlink (mrl);

    /* This is necessary, so the MRLs are always the same. */
    int len = strlen (rmrl);
    if (rmrl[len - 1] == '/') {
        rmrl[len - 1] = 0;
    }

    /* Try to find an entry with the same MRL. */
    fileitem_t *item = filelist_first (favorites_filelist);
    while (item && !(strcmp (item->mrl, rmrl) == 0)) {
        item = filelist_next (favorites_filelist, item);
    }

    /* If no entry was found, we add a new one. */
    if (!item) {
        item = filelist_add (favorites_filelist,
                             title, rmrl, FILE_TYPE_DIRECTORY);
    }

    ho_free (rmrl);
    assert (item);
    item->listened_count++;

    /* Write the list back to the favorites file. */
    const char *fav_mrl = get_file_favorites ();
    FILE *f = fopen (fav_mrl, "w");
    if (f == NULL) {
        error (_("Could not open '%s': %s!"), fav_mrl, strerror (errno));
        return;
    }

    char *codeset = get_system_encoding ();
    recode_t *xr = recode_init (codeset, "utf8");

    fprintf (f, OXINE_FAVORITES_XML);
    fprintf (f, OXINE_FAVORITES_DTD);

    fprintf (f, "<!--\n\n");
    fprintf (f, "WARNING:\n\n");
    fprintf (f, "This file was automatically created by oxine.\n");
    fprintf (f, "Any changes made to this file will be lost!\n\n");
    fprintf (f, "-->\n\n");

    fprintf (f, "<oxine_favorites>\n");
    item = filelist_first (favorites_filelist);
    while (item) {
        if (item->type == FILE_TYPE_DIRECTORY) {
            char *esc_str0 = xml_escape (item->title,
                                         xr, XML_ESCAPE_NO_QUOTE);
            char *esc_str1 = xml_escape (item->mrl,
                                         xr, XML_ESCAPE_DOUBLE_QUOTE);
            char *esc_str2 = xml_escape (item->thumbnail_mrl,
                                         xr, XML_ESCAPE_DOUBLE_QUOTE);
            fprintf (f, "  <entry>\n");
            fprintf (f, "    <title>%s</title>\n", esc_str0);
            fprintf (f, "    <mrl           href=\"%s\" />\n", esc_str1);
            fprintf (f, "    <thumbnail_mrl href=\"%s\" />\n", esc_str2);
            fprintf (f, "    <listened_count>%d</listened_count>\n",
                     item->listened_count);
            fprintf (f, "  </entry>\n");
            ho_free (esc_str0);
            ho_free (esc_str1);
            ho_free (esc_str2);
        }

        item = filelist_next (favorites_filelist, item);
    }
    fprintf (f, "</oxine_favorites>\n");
    fclose (f);

    recode_done (xr);
    ho_free (codeset);
}
