
/*
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 * Most of this file was copied from the xine-ui project.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: download.h 2486 2007-07-13 16:24:27Z mschwerin $
 *
 */

#ifndef HAVE_DOWNLOAD_H
#define HAVE_DOWNLOAD_H

#include "config.h"

#include <stdio.h>

#include "types.h"


#ifdef HAVE_CURL

/// This contains information about the current download.
typedef struct {
    /// The data of the buffer.
    char *buffer;
    /// The size of the buffer.
    size_t size;
    /// A file to write the data to.
    FILE *file;
} download_t;

/**
 * Creates a new download structure. If the downloaded data is supposed to
 * be written to a file, the filename of that file may be passed. If this is
 * not done, the downloaded data is stored to the buffer in the returned
 * data structure.
 *
 * @param filename              The name of the file the data is to be
 *                              written to. This may be NULL.
 */
download_t *download_new (const char *filename);


/// Free the specified download structure.
void download_free (download_t * download, bool free_buffer);


/**
 * Download the specified URL.
 *
 * @param url                   The URL to download.
 * @param download              A valid download structure.
 *
 * @return                      On success TRUE is returned.
 */
bool network_download (const char *url, download_t * download);


/**
 * Tries to retrieve a redirection URL for the passed URL. The returned
 * string must be freed.
 *
 * @param url                   The URL to search the redirect for.
 */
char *get_redirect_url (const char *url);


#endif /* HAVE_CURL */


/**
 * Downloads specified URL to a disc cache.
 *
 * @param url                   The URL to download.
 * @param filename              The filename in the cache. This may be NULL.
 * @param force_update          Overwrite the file if it already exists.
 * @return                      The complete path of the cached file. This
 *                              string must be freed after use.
 */
char *download_to_cache (const char *url, const char *filename,
                         bool force_update);


/// Returns TRUE if the specified URL can be downloaded.
bool is_downloadable (const char *url);


/// Get the percentage of the current download.
int download_get_percent (void *p);


/// Initialize our downloader.
void downloader_init (void);


/// Free our downloader.
void downloader_free (void);


#endif /* HAVE_DOWNLOAD_H */
