/*
 *                           TERMS AND CONDITIONS
 *                                   FOR
 *                         OPEN SOURCE CODE LICENSE
 *                               Version 1.1
 * 
 * Japan Registry Services Co., Ltd. ("JPRS"), a Japanese corporation
 * having its head office at Chiyoda First Bldg. East 13F 3-8-1 Nishi-Kanda,
 * Chiyoda-ku, Tokyo 101-0065, Japan, grants you the license for open source
 * code specified in EXHIBIT A the "Code" subject to the following Terms and
 * Conditions ("OSCL").
 * 
 * 1. License Grant.
 *   JPRS hereby grants you a worldwide, royalty-free, non-exclusive
 *   license, subject to third party intellectual property claims:
 *   (a) under intellectual property rights (other than patent or
 *       trademark) licensable by JPRS to use, reproduce, modify, display,
 *       perform, sublicense and distribute the Code (or portions thereof)
 *       with or without modifications, and/or as part of a derivative work;
 *       or
 *   (b) under claims of the infringement through the making, using,
 *       offering to sell and/or otherwise disposing the JPRS Revised Code
 *       (or portions thereof);
 *   (c) the licenses granted in this Section 1(a) and (b) are effective on
 *       the date JPRS first distributes the Code to you under the terms of
 *       this OSCL;
 *   (d) Notwithstanding the above stated terms, no patent license is
 *       granted:
 *       1)  for a code that you delete from the Code;
 *       2)  separate from the Code; or
 *       3)  for infringements caused by:
 *            i) modification of the Code; or
 *           ii) combination of the Code with other software or devices.
 * 
 * 2. Consents.
 *   You agree that:
 *   (a) you must include a copy of this OSCL and the notice set forth in
 *       EXHIBIT A with every copy of the Code you distribute;
 *   (b) you must include a copy of this OSCL and the notice set forth in
 *       EXHIBIT A with every copy of binary form of the Code in the
 *       documentation and/or other materials provided with the distribution;
 *   (c) you may not offer or impose any terms on any source code version
 *       that alters or restricts the applicable version of this OSCL or
 *       the recipients' rights hereunder.
 *   (d) If the terms and conditions are set forth in EXHIBIT A, you must
 *       comply with those terms and conditions.
 * 
 * 3. Proprietary Information.
 *   All trademarks, service marks, patents, copyrights, trade secrets, and
 *   other proprietary rights in or related to the Code are and will remain
 *   the exclusive property of JPRS or its licensors, whether or not
 *   specifically recognized or perfected under local law except specified
 *   in this OSCL; provided however you agree and understand that the JPRS
 *   name may not be used to endorse or promote this Code without prior
 *   written approval of JPRS.
 * 
 * 4. WARRANTY DISCLAIMER.
 *   JPRS MAKES NO REPRESENTATIONS AND WARRANTIES REGARDING THE USE OF THE
 *   CODE, NOR DOES JPRS MAKE ANY REPRESENTATIONS THAT THE CODE WILL BECOME
 *   COMMERCIALLY AVAILABLE. JPRS, ITS AFFILIATES, AND ITS SUPPLIERS DO NOT
 *   WARRANT OR REPRESENT THAT THE CODE IS FREE OF ERRORS OR THAT THE CODE
 *   IS SUITABLE FOR TRANSLATION AND/OR LOCALIZATION. THE CODE IS PROVIDED
 *   ON AN "AS IS" BASIS AND JPRS AND ITS SUPPLIERS HAVE NO OBLIGATION TO
 *   CORRECT ERRORS OR TO SUPPORT THE CODE UNDER THIS OSCL FOR ANY REASON.
 *   TO THE FULL EXTENT PERMITTED BY LAW, ALL OBLIGATIONS ARE HEREBY
 *   EXCLUDED WHETHER EXPRESS, STATUTORY OR IMPLIED UNDER LAW, COURSE OF
 *   DEALING, CUSTOM, TRADE USAGE, ORAL OR WRITTEN STATEMENT OR OTHERWISE,
 *   INCLUDING BUT NOT LIMITED TO ANY IMPLIED WARRANTIES OF MERCHANTABILITY
 *   OR FITNESS FOR A PARTICULAR PURPOSE CONCERNING THE CODE.
 * 
 * 5. NO LIABILITY.
 *   UNDER NO CIRCUMSTANCES SHALL JPRS AND/OR ITS AFFILIATES, LICENSORS, OR
 *   REPRESENTATIVES BE LIABLE FOR ANY DAMAGES INCLUDING BUT NOT LIMITED TO
 *   CONSEQUENTIAL, INDIRECT, SPECIAL, PUNITIVE OR INCIDENTAL DAMAGES,
 *   WHETHER FORESEEABLE OR UNFORESEEABLE, BASED ON YOUR CLAIMS, INCLUDING,
 *   BUT NOT LIMITED TO, CLAIMS FOR LOSS OF DATA, GOODWILL, PROFITS, USE OF
 *   MONEY, INTERRUPTION IN USE OR AVAILABILITY OF DATA, STOPPAGE, IMPLIED
 *   WARRANTY, BREACH OF CONTRACT, MISREPRESENTATION, NEGLIGENCE, STRICT
 *   LIABILITY IN TORT, OR OTHERWISE.
 * 
 * 6. Indemnification.
 *   You hereby agree to indemnify, defend, and hold harmless JPRS for any
 *   liability incurred by JRPS due to your terms of warranty, support,
 *   indemnity, or liability offered by you to any third party.
 * 
 * 7. Termination.
 * 7.1 This OSCL shall be automatically terminated in the events that:
 *   (a) You fail to comply with the terms herein and fail to cure such
 *       breach within 30 days of becoming aware of the breach;
 *   (b) You initiate patent or copyright infringement litigation against
 *       any party (including a cross-claim or counterclaim in a lawsuit)
 *       alleging that the Code constitutes a direct or indirect patent or
 *       copyright infringement, in such case, this OSCL to you shall
 *       terminate as of the date such litigation is filed;
 * 7.2 In the event of termination under Sections 7.1(a) or 7.1(b) above,
 *     all end user license agreements (excluding distributors and
 *     resellers) which have been validly granted by You or any distributor
 *     hereunder prior to termination shall survive termination.
 *
 * 
 * 8. General.
 *   This OSCL shall be governed by, and construed and enforced in
 *   accordance with, the laws of Japan. Any litigation or arbitration
 *   between the parties shall be conducted exclusively in Tokyo, Japan
 *   except written consent of JPRS provides other venue.
 * 
 * 
 *                                EXHIBIT A
 * 
 * The original open source code of idnkit-2 is idnkit-1.0 developed and
 * conceived by Japan Network Information Center ("JPNIC"), a Japanese
 * association, Kokusai-Kougyou-Kanda Bldg 6F, 2-3-4 Uchi-Kanda,
 * Chiyoda-ku, Tokyo 101-0047, Japan, and JPRS modifies above original code
 * under following Terms and Conditions set forth by JPNIC.
 * 
 *                                  JPNIC
 * 
 * Copyright (c) 2000-2002 Japan Network Information Center.  All rights reserved.
 * 
 * By using this file, you agree to the terms and conditions set forth bellow.
 * 
 *                       LICENSE TERMS AND CONDITIONS
 * 
 * The following License Terms and Conditions apply, unless a different
 * license is obtained from Japan Network Information Center ("JPNIC"),
 * a Japanese association, Kokusai-Kougyou-Kanda Bldg 6F, 2-3-4 Uchi-Kanda,
 * Chiyoda-ku, Tokyo 101-0047, Japan.
 * 
 * 1. Use, Modification and Redistribution (including distribution of any
 *    modified or derived work) in source and/or binary forms is permitted
 *    under this License Terms and Conditions.
 * 
 * 2. Redistribution of source code must retain the copyright notices as they
 *    appear in each source code file, this License Terms and Conditions.
 * 
 * 3. Redistribution in binary form must reproduce the Copyright Notice,
 *    this License Terms and Conditions, in the documentation and/or other
 *    materials provided with the distribution. For the purposes of binary
 *    distribution the "Copyright Notice" refers to the following language:
 *    "Copyright (c) 2000-2002 Japan Network Information Center.  All rights reserved."
 * 
 * 4. The name of JPNIC may not be used to endorse or promote products
 *    derived from this Software without specific prior written approval of
 *    JPNIC.
 * 
 * 5. Disclaimer/Limitation of Liability: THIS SOFTWARE IS PROVIDED BY JPNIC
 *    "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *    LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 *    PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL JPNIC BE LIABLE
 *    FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 *    CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 *    SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 *    BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 *    WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 *    OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 *    ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
 * 
 * 
 *                        JPRS Public License Notice
 *                                   For
 *                                idnkit-2.
 * 
 * The contents of this file are subject to the Terms and Conditions for
 * the Open Source Code License (the "OSCL"). You may not use this file
 * except in compliance with above terms and conditions. A copy of the OSCL
 * is available at <http://jprs.co.jp/idn/>.
 * The JPRS Revised Code is idnkit-2.
 * The Initial Developer of the JPRS Revised Code is Japan Network
 * Information Center ("JPNIC"), a Japanese association,
 * Kokusai-Kougyou-Kanda Bldg 6F, 2-3-4 Uchi-Kanda, Chiyoda-ku, Tokyo
 * 101-0047, Japan.
 * "Copyright (c) 2000-2002 Japan Network Information Center.  All rights reserved."
 * "Copyright (c) 2010-2012 Japan Registry Services Co., Ltd.  All rights reserved."
 * Contributor(s): ______________________________________.
 * 
 * If you wish to allow use of your version of this file only under the
 * above License(s) and not to allow others to use your version of this
 * file, please indicate your decision by deleting the relevant provisions
 * above and replacing them with the notice and other provisions required
 * by the above License(s). If you do not delete the relevant provisions,
 * a recipient may use your version of this file under either the above
 * License(s).
 */

/*
 * idnconv -- Codeset converter for named.conf and zone files
 */

#include <config.h>

#include <stddef.h>
#include <errno.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#ifdef HAVE_LOCALE_H
#include <locale.h>
#endif

#include <idn/result.h>
#include <idn/punycode.h>
#include <idn/res.h>
#include <idn/resconf.h>
#include <idn/util.h>
#include <idn/utf8.h>
#include <idn/version.h>

#include "strbuf.h"
#include "selectiveencode.h"

/*
 * Domain name registration / lookup procotols.
 */
typedef enum idnconv_protocol {
	idnconv_lookup = 0,
	idnconv_registration = 1
} idnconv_protocol_t;

/*
 * Parsing result of command line arguments.
 */
typedef struct {
	char *in_code;
	char *out_code;
	char *conf_file;
	int no_conf; 
	int reverse;
	idnconv_protocol_t protocol;
	char *localcheck_file;
	int line_flush;
	int whole;
	int test;
	idn_action_t skip_actions;
	char *input_file;
} idnconv_option_t;

/*
 * Quiet mode flag.
 */
int quiet_mode = 0;

static void	idnconv_option_init(idnconv_option_t *option);
static int	parse_command_line(int ac, char **av,
				   idnconv_option_t *option);
static int	parse_skip(const char *s, idn_action_t *action);
static int	create_resconf(idn_resconf_t *resconf, int no_conf,
			       const char *conf_file,
			       const char *localencoding,
			       const char *localcheck_file);
static int	encode_file(idn_resconf_t resconf1, idn_resconf_t resconf2,
			    idn_action_t actions, FILE *fp, int line_flush,
			    int whole);
static int	decode_file(idn_resconf_t resconf1, idn_resconf_t resconf2,
			    idn_action_t actions, FILE *fp, int line_flush,
			    int whole);
static idn_result_t
		encode_line(idnconv_strbuf_t *from, idnconv_strbuf_t *to,
			    idn_resconf_t conf, idn_action_t actions,
			    int whole);
static idn_result_t
		decode_line(idnconv_strbuf_t *from, idnconv_strbuf_t *to,
			    idn_resconf_t conf, idn_action_t actions,
			    int whole);
static int	trim_newline(idnconv_strbuf_t *buf);
static void	print_test_status(idn_resconf_t conf1, idn_resconf_t conf2);
static void	print_version(void);
static void	print_usage(void);
static void	errormsg(const char *fmt, ...);

int
main(int ac, char **av) {
	idn_result_t r;
	int exit_code = 0;
	idnconv_option_t option;
	idn_action_t actions;
	FILE *fp = NULL;
	idn_resconf_t resconf1 = NULL;
	idn_resconf_t resconf2 = NULL;

#ifdef HAVE_SETLOCALE
	setlocale(LC_ALL, "");
#endif

	/*
	 * Parse command line options.
	 */
	idnconv_option_init(&option);
	if (!parse_command_line(ac, av, &option)) {
		errormsg("try 'idnconv2 -help' for more information.\n");
		exit_code = 1;
		goto ret;
	}

	/*
	 * Initialize idnkit library.
	 */
	r = idn_resconf_initialize();
	if (r != idn_success) {
		errormsg("error initializing library\n");
		exit_code = 1;
		goto ret;
	}

	/*
	 * Create two resource contexts; 'resconf1' and 'resconf2'.
	 */
	if (!create_resconf(&resconf1, option.no_conf, option.conf_file,
			    option.in_code, option.localcheck_file)) {
		exit_code = 1;
		goto ret;
	}
	if (!create_resconf(&resconf2, option.no_conf, option.conf_file,
			    option.out_code, option.localcheck_file)) {
		exit_code = 1;
		goto ret;
	}

	/*
	 * Determine main actions.
	 */
	if (option.reverse) {
		if (option.protocol == idnconv_registration) {
			actions = IDN_DECODE_REGIST & \
				  ~IDN_UNICODECONV & \
				  ~IDN_LOCALCONV & \
				  ~option.skip_actions;
		} else {
			actions = IDN_DECODE_LOOKUP & \
				  ~IDN_UNICODECONV & \
				  ~IDN_LOCALCONV & \
				  ~option.skip_actions;
		}
	} else {
		if (option.protocol == idnconv_registration) {
			actions = IDN_ENCODE_REGIST & \
				  ~IDN_UNICODECONV & \
				  ~IDN_LOCALCONV & \
				  ~option.skip_actions;
		} else {
			actions = IDN_ENCODE_LOOKUP & \
				  ~IDN_UNICODECONV & \
				  ~IDN_LOCALCONV & \
				  ~option.skip_actions;
		}
	}
	if (option.localcheck_file != NULL &&
	    !(option.skip_actions & ~IDN_LOCALCHECK)) {
		actions |= IDN_LOCALCHECK;
	}
	if (option.reverse && option.in_code != NULL &&
	    strcmp(option.in_code, IDN__PUNYCODE_ACENAME) != 0) {
		actions &= ~IDN_IDNCONV & ~IDN_RTCHECK;
	}
	if (!option.reverse && option.out_code != NULL &&
	    strcmp(option.out_code, IDN__PUNYCODE_ACENAME) != 0) {
		actions &= ~IDN_IDNCONV & ~IDN_RTCHECK & ~IDN_LENCHECK;
	}

	/*
	 * Test mode.
	 */
	if (option.test) {
		print_test_status(resconf1, resconf2);
		goto ret;
	}

	/*
	 * Open an input file.
	 */
	if (option.input_file != NULL) {
		if ((fp = fopen(option.input_file, "r")) == NULL) {
			errormsg("failed to open file, %s: %s\n",
				 strerror(errno), av[0]);
			exit_code = 1;
			goto ret;
		}
	} else {
		fp = stdin;
	}

	/*
	 * Do the conversion.
	 */
	if (option.reverse) {
		if (!decode_file(resconf1, resconf2, actions, fp,
				 option.line_flush, option.whole)) {
			exit_code = 1;
			goto ret;
		}
	} else {
		if (!encode_file(resconf1, resconf2, actions, fp, 
				 option.line_flush, option.whole)) {
			exit_code = 1;
			goto ret;
		}
	}

ret:
	if (resconf1 != NULL)
		idn_resconf_destroy(resconf1);
	if (resconf2 != NULL)
		idn_resconf_destroy(resconf2);
	if (fp != NULL)
		fclose(fp);

	return (exit_code);
}

static void
idnconv_option_init(idnconv_option_t *option) {
	option->in_code         = NULL;
	option->out_code        = NULL;
	option->conf_file       = NULL;
	option->no_conf         = 0;
	option->reverse         = 0;
	option->protocol        = idnconv_registration;
	option->localcheck_file = NULL;
	option->line_flush      = 0;
	option->whole           = 0;
	option->test            = 0;
	option->skip_actions    = 0;
	option->input_file      = NULL;
}

/*
 * Parse command line arguments.
 */
static int
parse_command_line(int ac, char **av, idnconv_option_t *option) {
	ac--;
	av++;
	while (ac > 0 && **av == '-') {
		if (strcmp(*av, "--") == 0) {
			ac--;
			av++;
			break;
		} else if (strcmp(*av, "-in") == 0 ||
		    strcmp(*av, "-i") == 0) {
			if (ac < 2) {
				errormsg("option '%s' requires an argument.\n", *av);
				return (0);
			}
			option->in_code = av[1];
			ac--;
			av++;
		} else if (strcmp(*av, "-out") == 0 ||
			   strcmp(*av, "-o") == 0) {
			if (ac < 2) {
				errormsg("option '%s' requires an argument.\n", *av);
				return (0);
			}
			option->out_code = av[1];
			ac--;
			av++;
		} else if (strcmp(*av, "-conf") == 0 ||
			   strcmp(*av, "-c") == 0) {
			if (ac < 2) {
				errormsg("option '%s' requires an argument.\n", *av);
				return (0);
			}
			option->conf_file = av[1];
			ac--;
			av++;
		} else if (strcmp(*av, "-noconf") == 0 ||
			   strcmp(*av, "-C") == 0) {
			option->no_conf = 1;
		} else if (strcmp(*av, "-reverse") == 0 ||
			   strcmp(*av, "-r") == 0) {
			option->reverse = 1;
		} else if (strcmp(*av, "-registration") == 0 ||
			   strcmp(*av, "-g") == 0) {
			option->protocol = idnconv_registration;
		} else if (strcmp(*av, "-lookup") == 0 ||
			   strcmp(*av, "-l") == 0) {
			option->protocol = idnconv_lookup;
		} else if (strcmp(*av, "-nomap") == 0 ||
			   strcmp(*av, "-M") == 0) {
			if (!parse_skip("map", &option->skip_actions))
				return (0);
		} else if (strcmp(*av, "-skip") == 0) {
			if (ac < 2) {
				errormsg("option '%s' requires an argument.\n", *av);
				return (0);
			}
			if (!parse_skip(av[1], &option->skip_actions)) {
				errormsg("invalid argument to option '-skip'.\n", *av);
				return (0);
			}
			ac--;
			av++;
		} else if (strcmp(*av, "-localcheck") == 0 ||
			   strcmp(*av, "-e") == 0) {
			if (ac < 2) {
				errormsg("option '%s' requires an argument.\n", *av);
				return (0);
			}
			option->localcheck_file = av[1];
			ac--;
			av++;
		} else if (strcmp(*av, "-flush") == 0) {
			option->line_flush = 1;
		} else if (strcmp(*av, "-whole") == 0 ||
			   strcmp(*av, "-w") == 0) {
			option->whole = 1;
		} else if (strcmp(*av, "-test") == 0 ||
			   strcmp(*av, "-t") == 0) {
			option->test = 1;
		} else if (strcmp(*av, "-version") == 0 ||
			   strcmp(*av, "-v") == 0) {
			print_version();
			exit(0);
		} else if (strcmp(*av, "-help") == 0 ||
			   strcmp(*av, "-h") == 0) {
			print_usage();
			exit(0);
		} else if (strcmp(*av, "-alias-file") == 0 ||
			   strcmp(*av, "-a") == 0 ||
			   strcmp(*av, "-nameprep") == 0 ||
			   strcmp(*av, "-n") == 0 ||
			   strcmp(*av, "-localmap") == 0 ||
			   strcmp(*av, "-delimiter") == 0) {
			if (ac < 2) {
				errormsg("option '%s' requires an argument.\n", *av);
				return (0);
			}
			errormsg("warning: ignore obsolete option '%s'.\n", *av);
			ac--;
			av++;
		} else if (strcmp(*av, "-nonameprep") == 0 ||
			   strcmp(*av, "-N") == 0 ||
			   strcmp(*av, "-nolocalmap") == 0 ||
			   strcmp(*av, "-L") == 0 ||
			   strcmp(*av, "-nounassigncheck") == 0 ||
			   strcmp(*av, "-U") == 0 ||
			   strcmp(*av, "-noasciicheck") == 0 ||
			   strcmp(*av, "-A") == 0 ||
			   strcmp(*av, "-nobidicheck") == 0 ||
			   strcmp(*av, "-B") == 0 ||
			   strcmp(*av, "-noroundtripcheck") == 0 ||
			   strcmp(*av, "-nolengthcheck") == 0) {
			errormsg("ignore obsolete option '%s'.\n", *av);
		} else {
			errormsg("unrecognized option '%s'.\n", *av);
			return (0);
		}
		ac--;
		av++;
	}

	if (ac > 1 && !option->test) {
		errormsg("too many arguments.\n");
		return (0);
	}

	if (ac == 0)
		option->input_file = NULL;
	else
		option->input_file = av[0];

	/*
	 * Resolve conflicts between options.
	 */
	if (option->no_conf && option->conf_file != NULL)
		option->conf_file = NULL;
	if (option->reverse &&
	    option->out_code != NULL &&
	    strcmp(option->out_code, IDN__PUNYCODE_ACENAME) == 0) {
		option->reverse = 0;
	}

	return (1);
}

struct idnconv_actionmap {
	char *name;
	idn_action_t value;
};

static const struct idnconv_actionmap action_maps[] = {
	{"map", 		IDN_MAP}, 
	{"asclower", 		IDN_ASCLOWER}, 
	{"rtconv", 		IDN_RTCONV}, 
	{"prohcheck", 		IDN_PROHCHECK}, 
	{"unascheck", 		IDN_UNASCHECK}, 
	{"nfccheck", 		IDN_NFCCHECK}, 
	{"prefcheck", 		IDN_PREFCHECK}, 
	{"hyphcheck", 		IDN_HYPHCHECK}, 
	{"combcheck", 		IDN_COMBCHECK}, 
	{"ctxjcheck", 		IDN_CTXJCHECK}, 
	{"ctxocheck", 		IDN_CTXOCHECK}, 
	{"ctxolitecheck", 	IDN_CTXOLITECHECK},
	{"bidicheck", 		IDN_BIDICHECK}, 
	{"idnconv",		IDN_IDNCONV}, 
	{"lencheck",		IDN_LENCHECK}, 
	{"rtcheck",		IDN_RTCHECK}, 
	{NULL, 			0}
};

static int
parse_skip(const char *arg, idn_action_t *action) {
	const struct idnconv_actionmap *mapp;
	int found;

	while (*arg != '\0') {
		found = 0;
		for (mapp = action_maps; mapp->name != NULL; mapp++) {
			const char *s1 = arg;
			const char *s2 = mapp->name;

			while (*s1 != ',' && *s1 != '\0') {
				if (*s1 != *s2 || *s2 == '\0')
					break;
				s1++;
				s2++;
			}
			if (*s1 == ',' && *s2 == '\0') {
				if (*(s1 + 1) == '\0')
					return (0);
				*action |= mapp->value;
				arg = s1 + 1;
				found = 1;
				break;
			} else if (*s1 == '\0' && *s2 == '\0') {
				*action |= mapp->value;
				arg = s1;
				found = 1;
				break;
			}
		}
		if (!found)
			return (0);
	}
	return (1);
}

static int
create_resconf(idn_resconf_t *resconf, int no_conf,
	       const char *conf_file, const char *localencoding,
	       const char *localcheck_file) {
	idn_result_t r;

	r = idn_resconf_create(resconf);
	if (r != idn_success) {
		errormsg("failed to initialize configuration contexts, %s\n",
			 idn_result_tostring(r));
		return (0);
	}

	if (!no_conf) {
		r = idn_resconf_loadfile(*resconf, conf_file);
		if (r != idn_success &&
		    (r != idn_nofile || conf_file != NULL)) {
			errormsg("failed to read a configuration file, %s\n",
				 idn_result_tostring(r));
			return (0);
		}
	}

	/*
	 * For backward compatibility to idnkit-1.0, 'Punycode' specified
	 * as local encoding, we assumes it as UTF-8.
	 */
	if (localencoding != NULL &&
	    strcmp(localencoding, IDN__PUNYCODE_ACENAME) == 0)
		localencoding = IDN__UTF8_ENCODINGNAME;
		
	r = idn_resconf_setlocalencoding(*resconf, localencoding);
	if (r != idn_success) {
		errormsg("failed to set the local encoding, %s\n",
			 idn_result_tostring(r));
		return (0);
	}

	r = idn_resconf_setlocalcheckfile(*resconf, localcheck_file);
	if (r != idn_success) {
		errormsg("failed to set the localcheck file, %s\n",
			 idn_result_tostring(r));
		return (0);
	}

	return (1);
}

static int
encode_file(idn_resconf_t resconf1, idn_resconf_t resconf2,
	    idn_action_t actions, FILE *fp, int line_flush, int whole) {
	idn_result_t r;
	idnconv_strbuf_t buf1, buf2;
	int nl_trimmed;
	int lineno;

	strbuf_init(&buf1);
	strbuf_init(&buf2);
	lineno = 0;
	while (strbuf_getline(&buf1, fp) != NULL) {
		lineno++;

		/*
		 * Trim newline at the end.
		 */
		nl_trimmed = trim_newline(&buf1);

		/*
		 * Convert an input line to UTF-8.
		 */
		r = decode_line(&buf1, &buf2, resconf1, IDN_UNICODECONV, 1);
		if (r != idn_success) {
			errormsg("conversion failed at line %d: %s\n",
				 lineno, idn_result_tostring(r));
			goto err;
		}
		if (!idn__utf8_isvalidstring(strbuf_get(&buf2))) {
			errormsg("conversion to utf-8 failed at line %d\n",
				 lineno);
			goto err;
		}

		/*
		 * Perform an encoding process.
		 */
		r = encode_line(&buf2, &buf1, resconf1, actions, whole);
		if (r != idn_success) {
			errormsg("conversion failed at line %d: %s\n",
				 lineno, idn_result_tostring(r));
			goto err;
		}

		/*
		 * Finally, convert the encoded string to the '-out' encoding.
		 */
		r = encode_line(&buf1, &buf2, resconf2, IDN_LOCALCONV, 1);
				 
		if (r != idn_success) {
			errormsg("conversion failed at line %d: %s\n",
				 lineno, idn_result_tostring(r));
			goto err;
		}

		fputs(strbuf_get(&buf2), stdout);
		if (nl_trimmed)
			putc('\n', stdout);

		if (line_flush)
			fflush(stdout);
	}

	strbuf_reset(&buf1);
	strbuf_reset(&buf2);
	return (1);

err:
	strbuf_reset(&buf1);
	strbuf_reset(&buf2);
	return (0);
}

static int
decode_file(idn_resconf_t resconf1, idn_resconf_t resconf2,
	    idn_action_t actions, FILE *fp, int line_flush, int whole) {
	idn_result_t r;
	idnconv_strbuf_t buf1, buf2;
	int nl_trimmed;
	int lineno;

	strbuf_init(&buf1);
	strbuf_init(&buf2);
	lineno = 1;
	while (strbuf_getline(&buf1, fp) != NULL) {
		/*
		 * Trim newline at the end.
		 */
		nl_trimmed = trim_newline(&buf1);

		/*
		 * Convert an input line to UTF-8.
		 */
		r = decode_line(&buf1, &buf2, resconf1, IDN_UNICODECONV, 1);
		if (r != idn_success) {
			errormsg("conversion failed at line %d: %s\n",
				 lineno, idn_result_tostring(r));
			goto err;
		}
		if (!idn__utf8_isvalidstring(strbuf_get(&buf2))) {
			errormsg("conversion to utf-8 failed at line %d\n",
				 lineno);
			goto err;
		}

		/*
		 * Convert internationalized domain names in the line.
		 */
		r = decode_line(&buf2, &buf1, resconf1, actions, whole);
		if (r != idn_success) {
			errormsg("conversion failed at line %d: %s\n",
				 lineno, idn_result_tostring(r));
			goto err;
		}

		/*
		 * Finally, convert the encoded string to the '-out' encoding.
		 */
		r = encode_line(&buf1, &buf2, resconf2, IDN_LOCALCONV, 1);
		if (r != idn_success) {
			errormsg("conversion failed at line %d: %s\n",
				 lineno, idn_result_tostring(r));
			goto err;
		}

		fputs(strbuf_get(&buf2), stdout);
		if (nl_trimmed)
			putc('\n', stdout);

		if (line_flush)
			fflush(stdout);

		lineno++;
	}
	strbuf_reset(&buf1);
	strbuf_reset(&buf2);
	return (1);

err:
	strbuf_reset(&buf1);
	strbuf_reset(&buf2);
	return (0);
}

static int
trim_newline(idnconv_strbuf_t *buf) {
	/*
	 * If the string in BUF ends with a newline, trim it and
	 * return 1.  Otherwise, just return 0 without modifying BUF.
	 */
	char *s = strbuf_get(buf);
	size_t len = strlen(s);

	if (s[len - 1] == '\n') {
		s[len - 1] = '\0';
		return (1);
	}

	return (0);
}

idn_result_t
encode_line(idnconv_strbuf_t *from, idnconv_strbuf_t *to,
	    idn_resconf_t conf, idn_action_t actions, int whole) {
	idn_result_t r = idn_success;
	char *from_str = strbuf_get(from);

	for (;;) {
		char *to_str = strbuf_get(to);
		size_t to_size = strbuf_size(to);

		if (whole) {
			r = idn_res_encodename(conf, actions, from_str,
					       to_str, to_size);
		} else {
			r = selective_encode(conf, actions, from_str,
					     to_str, to_size);
		}
		if (r == idn_buffer_overflow) {
			/*
			 * Conversion is not successful because
			 * the size of the target buffer is not enough.
			 * Double the size and retry.
			 */
			if (strbuf_double(to) == NULL) {
				/* oops. allocation failed. */
				return (idn_nomemory);
			}
		} else {
			break;
		}
	}
	return (r);
}

static idn_result_t
decode_line(idnconv_strbuf_t *from, idnconv_strbuf_t *to,
	    idn_resconf_t conf, idn_action_t actions, int whole) {
	idn_result_t r = idn_success;
	char *from_str = strbuf_get(from);

	for (;;) {
		char *to_str = strbuf_get(to);
		size_t to_size = strbuf_size(to);

		if (whole) {
			r = idn_res_decodename(conf, actions, from_str,
					       to_str, to_size);
		} else {
			r = selective_decode(conf, actions, from_str,
					     to_str, to_size);
		}
		if (r == idn_buffer_overflow) {
			/*
			 * Conversion is not successful because
			 * the size of the target buffer is not enough.
			 * Double the size and retry.
			 */
			if (strbuf_double(to) == NULL) {
				/* oops. allocation failed. */
				return (idn_nomemory);
			}
		} else {
			break;
		}
	}
	return (r);
}

static char *options[] = {
	"Usage: idnconv2 [options..] [file]",
	"  -in INPUT-CODESET   : specifies input codeset name.",
	"  -i INPUT-CODESET    : synonym for -in",
	"  -out OUTPUT-CODESET : specifies output codeset name.",
	"  -o OUTPUT-CODESET   : synonym for -out",
	"  -conf CONF-FILE     : specifies idnkit configuration file.",
	"  -c CONF-FILE        : synonym for -conf",
	"  -noconf             : do not load idnkit configuration file.",
	"  -C                  : synonym for -noconf",
	"  -reverse            : specifies reverse conversion.",
	"                        (i.e. IDN encoding to local encoding)",
	"  -r                  : synonym for -reverse",
	"  -registration       : convert regions with the Domain Name Registration",
	"                        protocol. (default)",
	"  -g                  : synonym for -registration",
	"  -lookup             : convert regions with the Domain Name Lookup",
	"                        protocol.",
	"  -l                  : synonym for -lookup",
	"  -nomap              : do not perform mappings.",
	"  -M                  : synonym for -nomap",
	"  -skip ACTION,...    : do not perform ACTION.",
	"                        the following action names are recognized:",
	"                        map         : mappings",
	"                        asclower    : ASCII uppercase letters to lowercase",
	"                        rtconv      : conversion from Punycode to Unicode",
	"                                      for round trip check",
	"                        prohcheck   : prohibited code point check",
	"                        unascheck   : unassigned code point check",
	"                        nfccheck    : NFC conformance check",
	"                        prefcheck   : ACE prefix check",
	"                        hyphcheck   : hyphen check",
	"                        combcheck   : combining character check",
	"                        ctxjcheck   : CONTEXTJ code point check",
	"                        ctxocheck   : CONTEXTO code point check for the",
	"                                      Domain Name Registration protocol",
	"                        ctxolitecheck",
	"                                    : CONTEXTO code point check for the",
	"                                      Domain Name Lookup protocol",
	"                        bidicheck   : Bidi check",
	"                        idnconv     : conversion from/to Punycode",
	"                        lencheck    : label length check",
	"                        rtcheck     : round trip check",
	"  -localcheck FILE    : perform local check with a code point table FILE.",
	"  -e FILE             : synonym for -localcheck",
	"  -flush              : line-buffering mode.",
	"  -whole              : convert the whole region instead of",
	"                        regions containing non-ascii characters.",
	"  -w                  : synonym for -whole",
	"  -test               : output language and local encoding status,",
	"                        then exit.",
	"  -t                    synonym for -test",
	"  -version            : print version number, then exit.",
	"  -v                  : synonym for -version",
	"  -help               : print this help, then exit.",
	"  -h                  : synonym for -help",
	NULL
};

static void
print_test_status(idn_resconf_t conf1, idn_resconf_t conf2) {
	printf("language = %s\n", idn_resconf_getlanguage(conf1));
	printf("input encoding = %s\n", idn_resconf_getlocalencoding(conf1));
	printf("output encoding = %s\n", idn_resconf_getlocalencoding(conf2));
}

static void
print_version(void) {
	printf("idnconv (idnkit) version: %s\n", IDNKIT_VERSION);
	printf("%s\n", idn_version_getstring());
}

static void
print_usage(void) {
	int i;

	for (i = 0; options[i] != NULL; i++)
		printf("%s\n", options[i]);
}

static void
errormsg(const char *fmt, ...) {
	va_list args;

	if (quiet_mode)
		return;

	va_start(args, fmt);
	vfprintf(stderr, fmt, args);
	va_end(args);
}
