/*
 *                           TERMS AND CONDITIONS
 *                                   FOR
 *                         OPEN SOURCE CODE LICENSE
 *                               Version 1.1
 * 
 * Japan Registry Services Co., Ltd. ("JPRS"), a Japanese corporation
 * having its head office at Chiyoda First Bldg. East 13F 3-8-1 Nishi-Kanda,
 * Chiyoda-ku, Tokyo 101-0065, Japan, grants you the license for open source
 * code specified in EXHIBIT A the "Code" subject to the following Terms and
 * Conditions ("OSCL").
 * 
 * 1. License Grant.
 *   JPRS hereby grants you a worldwide, royalty-free, non-exclusive
 *   license, subject to third party intellectual property claims:
 *   (a) under intellectual property rights (other than patent or
 *       trademark) licensable by JPRS to use, reproduce, modify, display,
 *       perform, sublicense and distribute the Code (or portions thereof)
 *       with or without modifications, and/or as part of a derivative work;
 *       or
 *   (b) under claims of the infringement through the making, using,
 *       offering to sell and/or otherwise disposing the JPRS Revised Code
 *       (or portions thereof);
 *   (c) the licenses granted in this Section 1(a) and (b) are effective on
 *       the date JPRS first distributes the Code to you under the terms of
 *       this OSCL;
 *   (d) Notwithstanding the above stated terms, no patent license is
 *       granted:
 *       1)  for a code that you delete from the Code;
 *       2)  separate from the Code; or
 *       3)  for infringements caused by:
 *            i) modification of the Code; or
 *           ii) combination of the Code with other software or devices.
 * 
 * 2. Consents.
 *   You agree that:
 *   (a) you must include a copy of this OSCL and the notice set forth in
 *       EXHIBIT A with every copy of the Code you distribute;
 *   (b) you must include a copy of this OSCL and the notice set forth in
 *       EXHIBIT A with every copy of binary form of the Code in the
 *       documentation and/or other materials provided with the distribution;
 *   (c) you may not offer or impose any terms on any source code version
 *       that alters or restricts the applicable version of this OSCL or
 *       the recipients' rights hereunder.
 *   (d) If the terms and conditions are set forth in EXHIBIT A, you must
 *       comply with those terms and conditions.
 * 
 * 3. Proprietary Information.
 *   All trademarks, service marks, patents, copyrights, trade secrets, and
 *   other proprietary rights in or related to the Code are and will remain
 *   the exclusive property of JPRS or its licensors, whether or not
 *   specifically recognized or perfected under local law except specified
 *   in this OSCL; provided however you agree and understand that the JPRS
 *   name may not be used to endorse or promote this Code without prior
 *   written approval of JPRS.
 * 
 * 4. WARRANTY DISCLAIMER.
 *   JPRS MAKES NO REPRESENTATIONS AND WARRANTIES REGARDING THE USE OF THE
 *   CODE, NOR DOES JPRS MAKE ANY REPRESENTATIONS THAT THE CODE WILL BECOME
 *   COMMERCIALLY AVAILABLE. JPRS, ITS AFFILIATES, AND ITS SUPPLIERS DO NOT
 *   WARRANT OR REPRESENT THAT THE CODE IS FREE OF ERRORS OR THAT THE CODE
 *   IS SUITABLE FOR TRANSLATION AND/OR LOCALIZATION. THE CODE IS PROVIDED
 *   ON AN "AS IS" BASIS AND JPRS AND ITS SUPPLIERS HAVE NO OBLIGATION TO
 *   CORRECT ERRORS OR TO SUPPORT THE CODE UNDER THIS OSCL FOR ANY REASON.
 *   TO THE FULL EXTENT PERMITTED BY LAW, ALL OBLIGATIONS ARE HEREBY
 *   EXCLUDED WHETHER EXPRESS, STATUTORY OR IMPLIED UNDER LAW, COURSE OF
 *   DEALING, CUSTOM, TRADE USAGE, ORAL OR WRITTEN STATEMENT OR OTHERWISE,
 *   INCLUDING BUT NOT LIMITED TO ANY IMPLIED WARRANTIES OF MERCHANTABILITY
 *   OR FITNESS FOR A PARTICULAR PURPOSE CONCERNING THE CODE.
 * 
 * 5. NO LIABILITY.
 *   UNDER NO CIRCUMSTANCES SHALL JPRS AND/OR ITS AFFILIATES, LICENSORS, OR
 *   REPRESENTATIVES BE LIABLE FOR ANY DAMAGES INCLUDING BUT NOT LIMITED TO
 *   CONSEQUENTIAL, INDIRECT, SPECIAL, PUNITIVE OR INCIDENTAL DAMAGES,
 *   WHETHER FORESEEABLE OR UNFORESEEABLE, BASED ON YOUR CLAIMS, INCLUDING,
 *   BUT NOT LIMITED TO, CLAIMS FOR LOSS OF DATA, GOODWILL, PROFITS, USE OF
 *   MONEY, INTERRUPTION IN USE OR AVAILABILITY OF DATA, STOPPAGE, IMPLIED
 *   WARRANTY, BREACH OF CONTRACT, MISREPRESENTATION, NEGLIGENCE, STRICT
 *   LIABILITY IN TORT, OR OTHERWISE.
 * 
 * 6. Indemnification.
 *   You hereby agree to indemnify, defend, and hold harmless JPRS for any
 *   liability incurred by JRPS due to your terms of warranty, support,
 *   indemnity, or liability offered by you to any third party.
 * 
 * 7. Termination.
 * 7.1 This OSCL shall be automatically terminated in the events that:
 *   (a) You fail to comply with the terms herein and fail to cure such
 *       breach within 30 days of becoming aware of the breach;
 *   (b) You initiate patent or copyright infringement litigation against
 *       any party (including a cross-claim or counterclaim in a lawsuit)
 *       alleging that the Code constitutes a direct or indirect patent or
 *       copyright infringement, in such case, this OSCL to you shall
 *       terminate as of the date such litigation is filed;
 * 7.2 In the event of termination under Sections 7.1(a) or 7.1(b) above,
 *     all end user license agreements (excluding distributors and
 *     resellers) which have been validly granted by You or any distributor
 *     hereunder prior to termination shall survive termination.
 *
 * 
 * 8. General.
 *   This OSCL shall be governed by, and construed and enforced in
 *   accordance with, the laws of Japan. Any litigation or arbitration
 *   between the parties shall be conducted exclusively in Tokyo, Japan
 *   except written consent of JPRS provides other venue.
 * 
 * 
 *                                EXHIBIT A
 * 
 * The original open source code of idnkit-2 is idnkit-1.0 developed and
 * conceived by Japan Network Information Center ("JPNIC"), a Japanese
 * association, Kokusai-Kougyou-Kanda Bldg 6F, 2-3-4 Uchi-Kanda,
 * Chiyoda-ku, Tokyo 101-0047, Japan, and JPRS modifies above original code
 * under following Terms and Conditions set forth by JPNIC.
 * 
 *                                  JPNIC
 * 
 * Copyright (c) 2000-2002 Japan Network Information Center.  All rights reserved.
 * 
 * By using this file, you agree to the terms and conditions set forth bellow.
 * 
 *                       LICENSE TERMS AND CONDITIONS
 * 
 * The following License Terms and Conditions apply, unless a different
 * license is obtained from Japan Network Information Center ("JPNIC"),
 * a Japanese association, Kokusai-Kougyou-Kanda Bldg 6F, 2-3-4 Uchi-Kanda,
 * Chiyoda-ku, Tokyo 101-0047, Japan.
 * 
 * 1. Use, Modification and Redistribution (including distribution of any
 *    modified or derived work) in source and/or binary forms is permitted
 *    under this License Terms and Conditions.
 * 
 * 2. Redistribution of source code must retain the copyright notices as they
 *    appear in each source code file, this License Terms and Conditions.
 * 
 * 3. Redistribution in binary form must reproduce the Copyright Notice,
 *    this License Terms and Conditions, in the documentation and/or other
 *    materials provided with the distribution. For the purposes of binary
 *    distribution the "Copyright Notice" refers to the following language:
 *    "Copyright (c) 2000-2002 Japan Network Information Center.  All rights reserved."
 * 
 * 4. The name of JPNIC may not be used to endorse or promote products
 *    derived from this Software without specific prior written approval of
 *    JPNIC.
 * 
 * 5. Disclaimer/Limitation of Liability: THIS SOFTWARE IS PROVIDED BY JPNIC
 *    "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *    LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 *    PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL JPNIC BE LIABLE
 *    FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 *    CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 *    SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 *    BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 *    WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 *    OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 *    ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
 * 
 * 
 *                        JPRS Public License Notice
 *                                   For
 *                                idnkit-2.
 * 
 * The contents of this file are subject to the Terms and Conditions for
 * the Open Source Code License (the "OSCL"). You may not use this file
 * except in compliance with above terms and conditions. A copy of the OSCL
 * is available at <http://jprs.co.jp/idn/>.
 * The JPRS Revised Code is idnkit-2.
 * The Initial Developer of the JPRS Revised Code is Japan Network
 * Information Center ("JPNIC"), a Japanese association,
 * Kokusai-Kougyou-Kanda Bldg 6F, 2-3-4 Uchi-Kanda, Chiyoda-ku, Tokyo
 * 101-0047, Japan.
 * "Copyright (c) 2000-2002 Japan Network Information Center.  All rights reserved."
 * "Copyright (c) 2010-2012 Japan Registry Services Co., Ltd.  All rights reserved."
 * Contributor(s): ______________________________________.
 * 
 * If you wish to allow use of your version of this file only under the
 * above License(s) and not to allow others to use your version of this
 * file, please indicate your decision by deleting the relevant provisions
 * above and replacing them with the notice and other provisions required
 * by the above License(s). If you do not delete the relevant provisions,
 * a recipient may use your version of this file under either the above
 * License(s).
 */

#include <config.h>

#include <stddef.h>
#include <stdlib.h>

#include <idn/assert.h>
#include <idn/debug.h>
#include <idn/logmacro.h>
#include <idn/result.h>
#include <idn/res_internal.h>
#include <idn/labellist.h>
#include <idn/utf32.h>
#include <idn/auto/bidiclass.h>

static int	check_rule2(const unsigned long *name);
static int	check_rule3(const unsigned long *name);
static int	check_rule4(const unsigned long *name);
static int	check_rule5(const unsigned long *name);
static int	check_rule6(const unsigned long *name);

/*
 * Check if 'label' is an RTL label.
 */
int
idn__res_isrtllabel(idn_resconf_t ctx, idn__labellist_t label) {
	const unsigned long *name;
	const unsigned long *p;
	idn__bidiclass_t bidiclass;
	int result = 0;

	assert(ctx != NULL && label != NULL);

	name = idn__labellist_getname(label);
	TRACE(("idn__res_isrtllabel(label=\"%s\")\n",
	       idn__debug_utf32xstring(name)));

	for (p = name; *p != '\0'; p++) {
		bidiclass = idn__sparsemap_getbidiclass(*p);
		if (bidiclass == idn__bidiclass_R ||
		    bidiclass == idn__bidiclass_AL ||
		    bidiclass == idn__bidiclass_AN) {
			result = 1;
			break;
		}
	}

	TRACE(("idn__res_isrtllabel(): success (label=\"%s\",result=%d)\n",
	       idn__debug_utf32xstring(name), result));
	return (result);
}

/*
 * Perform IDN_BIDICHECK.
 */
idn_result_t
idn__res_bidicheck(idn_resconf_t ctx, idn__labellist_t label) {
	idn_result_t r = idn_success;
	const unsigned long *name;
	idn__bidiclass_t bidiclass;
	const char *reason = NULL;

	assert(ctx != NULL && label != NULL);

	name = idn__labellist_getname(label);
	TRACE(("idn__res_bidicheck(label=\"%s\")\n",
	       idn__debug_utf32xstring(name)));

	bidiclass = idn__sparsemap_getbidiclass(*name);
	if (bidiclass == idn__bidiclass_R || bidiclass == idn__bidiclass_AL) {
		/*
		 * This is an RTL label.
		 */
		if (!check_rule2(name)) {
			r = idn_bidicheck_error;
			reason = "rule 2";
			goto ret;
		}
		if (!check_rule3(name)) {
			r = idn_bidicheck_error;
			reason = "rule 3";
			goto ret;
		}
		if (!check_rule4(name)) {
			r = idn_bidicheck_error;
			reason = "rule 4";
			goto ret;
		}
	} else if (bidiclass == idn__bidiclass_L) {
		/*
		 * This is an LTR label.
		 */
		if (!check_rule5(name)) {
			r = idn_bidicheck_error;
			reason = "rule 5";
			goto ret;
		}
		if (!check_rule6(name)) {
			r = idn_bidicheck_error;
			reason = "rule 6";
			goto ret;
		}
	} else {
		/*
		 * The BIDI rule 1. described in [IDNA2008-Bidi] failed.
		 * 
		 * The first character must be a character with BIDI
		 * property L, R or AL.
		 */
		r = idn_bidicheck_error;
		reason = "rule 1";
		goto ret;
	}

ret:
	if (r == idn_success) {
		TRACE(("idn__res_bidicheck(): success (label=\"%s\")\n",
		       idn__debug_utf32xstring(name)));
	} else if (r == idn_bidicheck_error) {
		TRACE(("idn__res_bidicheck(): %s "
		       "(label=\"%s\", reason=\"%s\")\n",
		       idn_result_tostring(r), idn__debug_utf32xstring(name),
		       reason));
	} else {
		TRACE(("idn__res_bidicheck(): %s\n",
		       idn_result_tostring(r)));
	}

	return (r);
}

/*
 * Perform the BIDI rule 2. described in [IDNA2008-Bidi].
 *
 * In an RTL label, only characters with the BIDI properties R, AL,
 * AN, EN, ES, CS, ET, ON, BN and NSM are allowed.
 */
static int
check_rule2(const unsigned long *name) {
	const unsigned long *p;
	idn__bidiclass_t bidiclass;

	for (p = name; *p != '\0'; p++) {
		bidiclass = idn__sparsemap_getbidiclass(*p);
		if (bidiclass != idn__bidiclass_R   &&
		    bidiclass != idn__bidiclass_AL  &&
		    bidiclass != idn__bidiclass_AN  &&
		    bidiclass != idn__bidiclass_EN  &&
		    bidiclass != idn__bidiclass_ES  &&
		    bidiclass != idn__bidiclass_CS  &&
		    bidiclass != idn__bidiclass_ET  &&
		    bidiclass != idn__bidiclass_ON  &&
		    bidiclass != idn__bidiclass_BN  &&
		    bidiclass != idn__bidiclass_NSM) {
			return (0);
		}
	}

	return (1);
}

/*
 * Test the BIDI rule 3. described in [IDNA2008-Bidi].
 *
 * In an RTL label, the end of the label must be a character with
 * BIDI property R, AL, EN or AN, followed by zero or more
 * characters with BIDI property NSM.
 */
static int
check_rule3(const unsigned long *name) {
	idn__bidiclass_t bidiclass;
	int i;

	for (i = (int)idn__utf32_strlen(name) - 1; i >= 0; i--) {
		bidiclass = idn__sparsemap_getbidiclass(name[i]);
		if (bidiclass == idn__bidiclass_NSM)
			continue;
		if (bidiclass == idn__bidiclass_R ||
		    bidiclass == idn__bidiclass_AL ||
		    bidiclass == idn__bidiclass_EN ||
		    bidiclass == idn__bidiclass_AN) {
			return (1);
		}

		return (0);
	}

	return (0);
}

/*
 * Test the BIDI rule 4. described in [IDNA2008-Bidi].
 *
 * In an RTL label, if an EN is present, no AN may be present, and
 * vice versa.
 */
static int
check_rule4(const unsigned long *name) {
	const unsigned long *p;
	idn__bidiclass_t bidiclass;
	int EN_count = 0;
	int AN_count = 0;

	for (p = name; *p != '\0'; p++) {
		bidiclass = idn__sparsemap_getbidiclass(*p);
		if (bidiclass == idn__bidiclass_EN)
			EN_count++;
		else if (bidiclass == idn__bidiclass_AN)
			AN_count++;
	}

	if (AN_count > 0 && EN_count > 0)
		return (0);
	return (1);
}

/*
 * Test the BIDI rule 5. described in [IDNA2008-Bidi].
 *
 * In an LTR label, only characters with the BIDI properties L, EN,
 * ES, CS.  ET, ON, BN and NSM are allowed.
 */
static int
check_rule5(const unsigned long *name) {
	const unsigned long *p;
	idn__bidiclass_t bidiclass;

	for (p = name; *p != '\0'; p++) {
		bidiclass = idn__sparsemap_getbidiclass(*p);
		if (bidiclass != idn__bidiclass_L   &&
		    bidiclass != idn__bidiclass_EN  &&
		    bidiclass != idn__bidiclass_ES  &&
		    bidiclass != idn__bidiclass_CS  &&
		    bidiclass != idn__bidiclass_ET  &&
		    bidiclass != idn__bidiclass_ON  &&
		    bidiclass != idn__bidiclass_BN  &&
		    bidiclass != idn__bidiclass_NSM) {
			return (0);
		}
	}

	return (1);
}

/*
 * Test the BIDI rule 6. described in [IDNA2008-Bidi].
 *
 * In an LTR label, the end of the label must be a character with
 * BIDI property L or EN, followed by zero or more characters with
 * BIDI property NSM.
 */
static int
check_rule6(const unsigned long *name) {
	idn__bidiclass_t bidiclass;
	int i;

	for (i = (int)idn__utf32_strlen(name) - 1; i >= 0; i--) {
		bidiclass = idn__sparsemap_getbidiclass(name[i]);
		if (bidiclass == idn__bidiclass_NSM)
			continue;
		if (bidiclass == idn__bidiclass_L ||
		    bidiclass == idn__bidiclass_EN) {
			return (1);
		}

		return (0);
	}

	return (0);
}
