
/*

    network.c - network histograms and status for dock applet
    Copyright (C) 2002-2004 Mario Pascucci <ilpettegolo (a) yahoo it>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
    or
    visit the URL http://www.fsf.org/

*/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <sys/time.h>

#include "dockhelper.h"
#include "network.h"

#include "net.xpm"
#include "leds.xpm"




/* pixmap for icons */
static Pixmap net_icons = (Pixmap) NULL;
static Pixmap leds_icon = (Pixmap) NULL;


/* reads net interface status from /proc/net/dev */
void net_update(struct net_monitor *nmon) {
  
  FILE *f;
  char s[255], *ch;
  
  nmon->previntf_in = nmon->intf_in;
  nmon->previntf_out = nmon->intf_out;
  nmon->icon_index = INTF_OFF + nmon->icon_type * DH_ICONX * 2;
  nmon->intf_up = 0;
  if (!(f = fopen ("/proc/net/dev", "r"))) {
    fprintf(stderr, "Open /proc/net/dev failed.");
    return;
  }
  nmon->intf_in = nmon->intf_out = 0;
  /* accumulates bytes for individual interfaces by type */
  while (!feof (f)) {
    fgets (s, 254, f);
    if (strstr (s, nmon->intf_name)) {
      nmon->icon_index = INTF_ON + nmon->icon_type * DH_ICONX * 2;  /* interface is up */
      nmon->intf_up = 1;
      ch = strchr(s,':');
      if (ch) sscanf (++ch, "%Ld %*d %*d %*d %*d %*d %*d %*d %Ld", 
	  &nmon->intf_in, &nmon->intf_out);
      break;
    }
  }
  fclose (f);
  /* sample absolute time */
  gettimeofday(&nmon->tv,NULL);
  /* time interval between sample */
  nmon->time[nmon->tindex] = (nmon->tv.tv_usec - nmon->tvp.tv_usec) + 
    (nmon->tv.tv_sec - nmon->tvp.tv_sec) * 1000000L;
  /* store bytes count */
  nmon->in_bytes[nmon->tindex] = nmon->intf_in - nmon->previntf_in;
  nmon->out_bytes[nmon->tindex] = nmon->intf_out - nmon->previntf_out;
  /* data for interface activity */
  nmon->activity_stat = ACTIVITY_ICO;
  if (nmon->in_bytes[nmon->tindex]) nmon->activity_stat += 3;
  if (nmon->out_bytes[nmon->tindex]) nmon->activity_stat += 6;
  nmon->tindex = (nmon->tindex + 1) % SMOOTH;
  /* old absolute time */
  nmon->tvp.tv_sec = nmon->tv.tv_sec;
  nmon->tvp.tv_usec = nmon->tv.tv_usec;
}



/* set size for ethernet histogram */
int net_size(struct net_monitor *nmon) {

  dh_getsize(nmon->handler, &nmon->xe, &nmon->ye, &nmon->we, &nmon->he);
  /* if size is less than 4x10 pixel is unusable */
  if ((nmon->he < 4) || (nmon->we < 10)) return -1;
  return 0;
}



/* set size for status icons */
int net_icon_size(struct net_monitor *nmon) {

  dh_getsize(nmon->icon_handler, &nmon->xc, &nmon->yc, &nmon->wc, &nmon->hc);
  /* if size is less than 6x10 pixel is unusable */
  if ((nmon->hc < 6) || (nmon->wc < 10)) return -1;
  if (net_icons == (Pixmap) NULL) {
    dh_pixmap(&net_icons,NULL,net_xpm);
  }
  if (leds_icon == (Pixmap) NULL) {
    dh_pixmap(&leds_icon,NULL,leds_xpm);
  }
  return 0;
}



/* draw traffic histogram */
void net_draw(struct net_monitor *nmon) {

  long ix,ox,iw,l;
  int i;

  dh_color(dh_background());
  dh_fillrect(nmon->xe,nmon->ye,nmon->we,nmon->he);
  if (nmon->activity_led) 
    iw = nmon->we - 4;
  else
    iw = nmon->we;
  if (nmon->intf_up != 0) {
    ix = ox = 0;
    for (i=0;i<SMOOTH;i++){
      if (nmon->time[i]) {
	ix += (long) ((long long) nmon->in_bytes[i] * 1000000LL / nmon->time[i]);
	ox += (long) ((long long) nmon->out_bytes[i] * 1000000LL / nmon->time[i]);
      }
    }
    ix /= SMOOTH;
    ox /= SMOOTH; 
    l = (nmon->speedkb >> 3);
    ix /= l; 
    ox /= l;
    if (ix >= 1000) ix = 999;
    if (ox >= 1000) ox = 999;
    if (nmon->log_scale) {
      if (ix >= 2)
        ix = (log10(ix) * (double) iw) / 3;  /* log10(speed_percent) / log10(1000) */
      else 
        ix = 0;
      if (ox >= 2)
	ox = (log10(ox) * (double) iw) / 3;
      else
	ox = 0;
    }
    else {
      ix = (ix * iw) / 1000;
      ox = (ox * iw) / 1000;
    }
    if (ix >= iw) ix = iw-1;
    if (ox >= iw) ox = iw-1;
    dh_color(nmon->in_color);
    dh_fillrect(nmon->xe + ((nmon->activity_led) ? 4 : 0),
	nmon->ye,ix,nmon->he/2);	  /* upper half is input */
    dh_color(nmon->out_color);
    dh_fillrect(nmon->xe + ((nmon->activity_led) ? 4 : 0),
	nmon->ye+nmon->he/2,ox,nmon->he/2);	/* lower half is output */
    if (nmon->activity_led)
    {
      dh_copyarea(leds_icon,nmon->activity_stat,0,nmon->xe,nmon->ye,3,DH_ICONY);
    }
  }
}



/* draw net icon */
void net_icon_draw(struct net_monitor *nmon)
{

  dh_copyarea(net_icons,nmon->icon_index,0,nmon->xc,nmon->yc,DH_ICONX,DH_ICONY);
}


