package licenses

// Type identifies a class of software license.
type Type string

// License types
const (
	// Unknown license type.
	Unknown = Type("")
	// Restricted licenses require mandatory source distribution if we ship a
	// product that includes third-party code protected by such a license.
	Restricted = Type("restricted")
	// Reciprocal licenses allow usage of software made available under such
	// licenses freely in *unmodified* form. If the third-party source code is
	// modified in any way these modifications to the original third-party
	// source code must be made available.
	Reciprocal = Type("reciprocal")
	// Notice licenses contain few restrictions, allowing original or modified
	// third-party software to be shipped in any product without endangering or
	// encumbering our source code. All of the licenses in this category do,
	// however, have an "original Copyright notice" or "advertising clause",
	// wherein any external distributions must include the notice or clause
	// specified in the license.
	Notice = Type("notice")
	// Permissive licenses are even more lenient than a 'notice' license.
	// Not even a copyright notice is required for license compliance.
	Permissive = Type("permissive")
	// Unencumbered covers licenses that basically declare that the code is "free for any use".
	Unencumbered = Type("unencumbered")

	ByExceptionOnly = Type("byExceptionOnly")
	// Forbidden licenses are forbidden to be used.
	Forbidden = Type("forbidden")
)

func (t Type) String() string {
	switch t {
	case Unknown:
		// licenseclassifier uses an empty string to indicate an unknown license
		// type, which is unclear to users when printed as a string.
		return "unknown"
	default:
		return string(t)
	}
}

var typeMap = map[string]Type{
	"BCL":                              Restricted,
	"CC-BY-ND-1.0":                     Restricted,
	"CC-BY-ND-2.0":                     Restricted,
	"CC-BY-ND-2.5":                     Restricted,
	"CC-BY-ND-3.0":                     Restricted,
	"CC-BY-ND-4.0":                     Restricted,
	"CC-BY-SA-1.0":                     Restricted,
	"CC-BY-SA-2.0":                     Restricted,
	"CC-BY-SA-2.5":                     Restricted,
	"CC-BY-SA-3.0":                     Restricted,
	"CC-BY-SA-4.0":                     Restricted,
	"GPL-1.0":                          Restricted,
	"GPL-2.0":                          Restricted,
	"GPL-2.0-with-GCC-exception":       Restricted,
	"GPL-2.0-with-autoconf-exception":  Restricted,
	"GPL-2.0-with-bison-exception":     Restricted,
	"GPL-2.0-with-classpath-exception": Restricted,
	"GPL-2.0-with-font-exception":      Restricted,
	"GPL-3.0":                          Restricted,
	"GPL-3.0-with-GCC-exception":       Restricted,
	"GPL-3.0-with-autoconf-exception":  Restricted,
	"GPL-3.0-with-bison-exception":     Restricted,
	"LGPL-2.0":                         Restricted,
	"LGPL-2.1":                         Restricted,
	"LGPL-3.0":                         Restricted,
	"NPL-1.0":                          Restricted,
	"NPL-1.1":                          Restricted,
	"OSL-1.0":                          Restricted,
	"OSL-1.1":                          Restricted,
	"OSL-2.0":                          Restricted,
	"OSL-2.1":                          Restricted,
	"OSL-3.0":                          Restricted,
	"QPL-1.0":                          Restricted,
	"Sleepycat":                        Restricted,

	"APSL-1.0":  Reciprocal,
	"APSL-1.1":  Reciprocal,
	"APSL-1.2":  Reciprocal,
	"APSL-2.0":  Reciprocal,
	"CDDL-1.0":  Reciprocal,
	"CDDL-1.1":  Reciprocal,
	"CPL-1.0":   Reciprocal,
	"EPL-1.0":   Reciprocal,
	"EPL-2.0":   Reciprocal,
	"FreeImage": Reciprocal,
	"IPL-1.0":   Reciprocal,
	"MPL-1.0":   Reciprocal,
	"MPL-1.1":   Reciprocal,
	"MPL-2.0":   Reciprocal,
	"Ruby":      Reciprocal,

	"AFL-1.1":                       Notice,
	"AFL-1.2":                       Notice,
	"AFL-2.0":                       Notice,
	"AFL-2.1":                       Notice,
	"AFL-3.0":                       Notice,
	"Apache-1.0":                    Notice,
	"Apache-1.1":                    Notice,
	"Apache-2.0":                    Notice,
	"Apache-2.0-Modified":           Notice,
	"Apache-with-LLVM-Exception":    Notice,
	"Apache-with-Runtime-Exception": Notice,
	"Artistic-1.0":                  Notice,
	"Artistic-1.0-Perl":             Notice,
	"Artistic-1.0-cl8":              Notice,
	"Artistic-2.0":                  Notice,
	"BSD-2-Clause":                  Notice,
	"BSD-2-Clause-FreeBSD":          Notice,
	"BSD-2-Clause-NetBSD":           Notice,
	"BSD-2-Clause-Flex":             Notice,
	"BSD-2-Clause-Patent":           Notice,
	"BSD-3-Clause":                  Notice,
	"BSD-3-Clause-Attribution":      Notice,
	"BSD-3-Clause-Clear":            Notice,
	"BSD-3-Clause-LBNL":             Notice,
	"BSD-3-Clause-OpenMPI":          Notice,
	"BSD-4-Clause":                  Notice,
	"BSD-4-Clause-UC":               Notice,
	"BSD-Protection":                Notice,
	"BSL-1.0":                       Notice,
	"CC-BY-1.0":                     Notice,
	"CC-BY-2.0":                     Notice,
	"CC-BY-2.5":                     Notice,
	"CC-BY-3.0":                     Notice,
	"CC-BY-4.0":                     Notice,
	"FTL":                           Notice,
	"ISC":                           Notice,
	"ImageMagick":                   Notice,
	"LPL-1.0":                       Notice,
	"LPL-1.02":                      Notice,
	"Libpng":                        Notice,
	"Lil-1.0":                       Notice,
	"Linux-OpenIB":                  Notice,
	"MIT":                           Notice,
	"MS-PL":                         Notice,
	"NCSA":                          Notice,
	"OpenSSL":                       Notice,
	"PHP-3.0":                       Notice,
	"PHP-3.01":                      Notice,
	"PIL":                           Notice,
	"PostgreSQL":                    Notice,
	"Python-2.0":                    Notice,
	"Python-2.0-complete":           Notice,
	"SGI-B-1.0":                     Notice,
	"SGI-B-1.1":                     Notice,
	"SGI-B-2.0":                     Notice,
	"UPL-1.0":                       Notice,
	"Unicode-DFS-2015":              Notice,
	"Unicode-DFS-2016":              Notice,
	"Unicode-TOU":                   Notice,
	"W3C":                           Notice,
	"W3C-19980720":                  Notice,
	"W3C-20150513":                  Notice,
	"X11":                           Notice,
	"Xnet":                          Notice,
	"ZPL-1.1":                       Notice,
	"ZPL-2.0":                       Notice,
	"ZPL-2.1":                       Notice,
	"Zend-2.0":                      Notice,
	"Zlib":                          Notice,
	"zlib-acknowledgement":          Notice,

	"BSD-0-Clause": Unencumbered,
	"CC0-1.0":      Unencumbered,
	"Unlicense":    Unencumbered,

	"AGPL-1.0":          Forbidden,
	"AGPL-3.0":          Forbidden,
	"CC-BY-NC-1.0":      Forbidden,
	"CC-BY-NC-2.0":      Forbidden,
	"CC-BY-NC-2.5":      Forbidden,
	"CC-BY-NC-3.0":      Forbidden,
	"CC-BY-NC-4.0":      Forbidden,
	"CC-BY-NC-ND-1.0":   Forbidden,
	"CC-BY-NC-ND-2.0":   Forbidden,
	"CC-BY-NC-ND-2.5":   Forbidden,
	"CC-BY-NC-ND-3.0":   Forbidden,
	"CC-BY-NC-ND-4.0":   Forbidden,
	"CC-BY-NC-SA-1.0":   Forbidden,
	"CC-BY-NC-SA-2.0":   Forbidden,
	"CC-BY-NC-SA-2.5":   Forbidden,
	"CC-BY-NC-SA-3.0":   Forbidden,
	"CC-BY-NC-SA-4.0":   Forbidden,
	"Commons-Clause":    Forbidden,
	"Facebook-2-Clause": Forbidden,
	"Facebook-3-Clause": Forbidden,
	"Facebook-Examples": Forbidden,
	"WTFPL":             Forbidden,

	"AML":                                Unknown,
	"AMPAS":                              Unknown,
	"ANTLR":                              Unknown,
	"AdColony-SDK":                       Unknown,
	"Android-SDK":                        Unknown,
	"Atmel":                              Unknown,
	"Autodesk-3D-Studio-File-Toolkit":    Unknown,
	"BLAS":                               Unknown,
	"BSD-FatFs":                          Unknown,
	"BSD-No-Other-Rights":                Unknown,
	"BSD-Rice":                           Unknown,
	"BSD-Source-Code":                    Unknown,
	"BabelstoneIDS":                      Unknown,
	"BeOpen":                             Unknown,
	"Beerware":                           Unknown,
	"BitTorrent-1.1":                     Unknown,
	"Bitstream":                          Unknown,
	"Boost-original":                     Unknown,
	"Business-Source-License-1.1":        Unknown,
	"CAL-1.0":                            Unknown,
	"CDLA-Permissive-1.0":                Unknown,
	"CECILL-2.1":                         Unknown,
	"CERN-OHL-WR-v2":                     Unknown,
	"CERN-OHL-v1.2":                      Unknown,
	"CLIPS":                              Unknown,
	"CNRI-Python-GPL-Compatible":         Unknown,
	"CPAL-1.0":                           Unknown,
	"Cloud-Pre-GA":                       Unknown,
	"DBAD":                               Unknown,
	"DRL-1.0":                            Unknown,
	"EUPL-1.0":                           Unknown,
	"EUPL-1.1":                           Unknown,
	"Elastic-2.0":                        Unknown,
	"Entenssa":                           Unknown,
	"FFT2D":                              Unknown,
	"GD-Graphic-Library":                 Unknown,
	"GIAJWTOU-2.0":                       Unknown,
	"GIF-Encoder":                        Unknown,
	"GNU-All-permissive-Copying-License": Unknown,
	"GUST-Font-License":                  Unknown,
	"GenericIntel":                       Unknown,
	"HDF5":                               Unknown,
	"HPND-sell-variant":                  Unknown,
	"HTK":                                Unknown,
	"ICU":                                Unknown,
	"IDA":                                Unknown,
	"IJG":                                Unknown,
	"Info-ZIP":                           Unknown,
	"InnerNet":                           Unknown,
	"JSON":                               Unknown,
	"JTidy":                              Unknown,
	"JasPer":                             Unknown,
	"KUKA":                               Unknown,
	"Khronos":                            Unknown,
	"Khronos_OpenCL":                     Unknown,
	"LGPLLR":                             Unknown,
	"LPPL-1.3c":                          Unknown,
	"LZMA":                               Unknown,
	"Lattice-Semiconductor":              Unknown,
	"LicenseRef-C-Ares":                  Unknown,
	"LicenseRef-MIT-Lucent":              Unknown,
	"LicenseRef-Nordic-binary":           Unknown,
	"LicenseRef-SendMail":                Unknown,
	"LicenseRef-iccjpeg":                 Unknown,
	"LicenseRef-nacl":                    Unknown,
	"LicenseRef-takuya-ooura":            Unknown,
	"LicenseRef-uthash":                  Unknown,
	"MIT-Modern-Variant":                 Unknown,
	"MPL-2.0-no-copyleft-exception":      Unknown,
	"MS-RL":                              Unknown,
	"MTK":                                Unknown,
	"NAIST-2003":                         Unknown,
	"NCBI":                               Unknown,
	"NGPL":                               Unknown,
	"NREL":                               Unknown,
	"OFL-1.1":                            Unknown,
	"OROMatcher":                         Unknown,
	"Open-Game-License-1.0a":             Unknown,
	"OpenLDAP":                           Unknown,
	"OpenVision":                         Unknown,
	"Oracle-Open-Symphony":               Unknown,
	"PNG":                                Unknown,
	"PPP":                                Unknown,
	"Qhull":                              Unknown,
	"RCSB-PDB":                           Unknown,
	"RSA":                                Unknown,
	"Rijndael-3.0":                       Unknown,
	"SISSL":                              Unknown,
	"SISSL-1.2":                          Unknown,
	"SPL-SQRT-FLOOR":                     Unknown,
	"SQLite":                             Unknown,
	"SSPL-1.0":                           Unknown,
	"Sflow":                              Unknown,
	"Spencer-86":                         Unknown,
	"SunPro":                             Unknown,
	"TPM-2":                              Unknown,
	"Tensilica":                          Unknown,
	"UFL-1.0":                            Unknown,
	"USGovernment":                       Unknown,
	"VMAC":                               Unknown,
	"Vim":                                Unknown,
	"Waymo1P":                            Unknown,
	"Windows-SDK-10":                     Unknown,
	"WordNet-3.0":                        Unknown,
	"X11-Lucent":                         Unknown,
	"XZ":                                 Unknown,
	"Xcode":                              Unknown,
	"Zendesk":                            Unknown,
	"aopalliance":                        Unknown,
	"blessing":                           Unknown,
	"bzip2":                              Unknown,
	"cURL":                               Unknown,
	"dso":                                Unknown,
	"eGenix":                             Unknown,
	"geant4":                             Unknown,
	"getopt":                             Unknown,
	"hdparm":                             Unknown,
	"libtiff":                            Unknown,
	"pffft":                              Unknown,
	"re2c":                               Unknown,
	"tcl_tk":                             Unknown,
	"unicode_org":                        Unknown,
	"wxWindows-3.1":                      Unknown,
}

func LicenseType(license string) Type {
	if t, ok := typeMap[license]; ok {
		return t
	}

	return Unknown
}
