#!/bin/sh

debug_msg()
{
    # This function outputs a stylish debug message if debug is enabled
    # $1 = debuglevel
    # $2 = message

    if test "$debug_level" -gt "$1"; then echo " $0: debug: $2"; fi
}

error()
{
    # This function displays a error message and exits abnormally.
    # Arg_1 = Error Type
    # Arg_2 = Error Message

    debug_msg 0 'in error()'

    case $1 in

	missing)
	    echo " $0: missing parameter(s)"
	    echo
	    usage
	    ;;

	invalid)
	    echo " $0: invalid parameter(s)"
	    echo
	    usage
	    ;;

	abort)
	    echo " $0: installation aborted by user"
	    ;;

	conflict)
	    echo " $0: conflicting parameters"
	    echo
	    usage
	    ;;

	access)
	    echo " $0: You do not have write access to $2, please change to appropriate user."
	    ;;

        uname)
	    echo " $0: OS cannot be determined without the program \"uname\", aborting..."
	    ;;

	sed)
	    echo " $0: Install script will not work without the program \"sed\" aborting..."
	    ;;
	os)
	    echo " $0: Platform and OS was not recogniced, aborting..."
	    echo " $0: Please send output of uname -a to chrisw\@opera.com"
	    ;;

	*)
	    debug_msg 0 "Error message not defined, aborting..."
	;;
    esac
    exit 1
}

guess_os()
{
    # This function tries to guess which OS the install Machine is running.
    # other ideas are $OSTYPE, arch
    # Return = $os

    os=`uname -s`; if test "${?}" != '0'; then error 'uname'; fi
    machine=`uname -m`

    if test ${os} = 'FreeBSD' -o ${os} = 'OpenBSD'; then
	default_prefix='/usr/local'
    else
	default_prefix='/usr'
    fi
}

check_prefixes()
{
    # This function checks system enviroment for install prefix locations
    # and sets prefixed installation paths.

    export prefix exec_prefix wrapperdir docdir sharedir plugindir

    debug_msg 0 "in check_prefixes()"
    debug_msg 1 "enviroment = $prefix $exec_prefix $wrapperdir $docdir $sharedir $plugindir"

    verbose="0"

    guess_os

    if test "$#" -ne 0; then parse_input ${@}; fi
}

set_prefix()
{
    # Arg 1 = new prefix
    prefix="${1}"

    if test ${os} = 'FreeBSD' -o ${os} = 'OpenBSD'; then
        wrapper_dir="${prefix}/bin"
        doc_dir="${prefix}/share/doc/opera"
        share_dir="${prefix}/share/opera"
        exec_dir="${share_dir}/bin"
        plugin_dir="${share_dir}/plugins"
    else
	exec_dir="${prefix}/lib/opera/$opera_version"
	wrapper_dir="${prefix}/bin"
	doc_dir="${prefix}/share/doc/opera"
	share_dir="${prefix}/share/opera"
	plugin_dir="${prefix}/lib/opera/plugins"
    fi
}

usage()
{
    # This function displays the help information.

    debug_msg 0 "in useage()"

    echo 'Usage: install.sh -s'
    echo '  or:  install.sh -f DEST'
    echo '  or:  install.sh -f EXEC_DEST WRAPPER_DEST DOC_DEST SHARE_DEST PLUGIN_DEST'
    echo '  or:  install.sh --prefix=DEST [--exec_prefix=] [--wrapperdir=] [--docdir=] [--sharedir=] [--plugindir=]'
    echo '  or:  install.sh --exec_prefix=EXEC_DEST --wrapperdir=WRAPPER_DEST --docdir=DOC_DEST --sharedir=SHARE_DEST --plugindir=PLUGIN_DEST'
    echo
    echo ' or by enviroment variables:'
    echo '  or:  prefix=DEST install.sh'
    echo '  or:  export $prefix=DEST ; install.sh'
    echo
    echo 'Install Opera files to standard or user defined locations.'
    echo
    echo '  -i, --interactive            Interactive modus, this is now default mode'
    echo
    echo '  -s, --standard               install to standard locations'
    echo '  -f, --force                  install to user defined location(s)'
    echo
    echo '      $prefix'
    echo '      --prefix=                install all files to directory'
    echo
    echo '      $exec_prefix'
    echo '      --exec_prefix=           install Opera binaries to directory'
    echo '      $wrapperdir'
    echo '      --wrapperdir=            install Opera wrapper script to directory'
    echo '      $docdir'
    echo '      --docdir=                install Opera documentation to directory'
    echo '      $sharedir'
    echo '      --sharedir=              install Opera shared files to directory'
    echo '      $plugindir'
    echo '      --plugindir=             install Opera plugin files to directory'
    echo
    echo '  -v, --verbose                output which files are copied'
    echo '  -vv                          output info on each executed command'
    echo
    echo '  -V, --version                output version information and exit'
    echo '  -h, --help                   display this help and exit'
    echo
    echo 'If you choose to do a standard locations install, files will be put into'
    echo '/usr/bin, /usr/share/doc/opera and /usr/share/opera.'
    echo
    echo 'However, if you choose to install to user defined locations, you must either'
    echo '  specify one directory (all files will be put in this directory), or'
    echo '  specify five directories (you may enter the same directory several times).'
}

parse_input()
{
    # This function parses trough command line parameters
    # and sets install flags and selected installation paths.

    debug_msg 0 "in parse_input()"
    debug_msg 1 "args = $@"


while test ! -z "$1" ; do
case $1 in

    -h|--help) usage; exit 0 ;;

    -V|--version) version; exit 0 ;;

    -v|--verbose)
	verbose='1'
	shift
	;;

    -vv)
	verbose='2'
	shift
	;;

    -i|--interactive)
	if test -z "$flag_mode" ; then 
	    flag_mode='--interactive'
	    shift
	  else
	  error 'conflict'
	fi
	;;


    -s|--standard)
	if test -z "$flag_mode" ; then 
	    flag_mode='--standard'
	    shift
	  else
	  error 'conflict'
	fi
	;;

    -f|--force)
	if test -z "$flag_mode" ; then  
	    flag_mode='--force'
	    shift
	    if test -z "$1" ; then error 'missing'
	      elif test -z "$2"; then
		echo "Only one destination parameter found, all files will be installed to $1"
		echo 'Do you want to continue [y/n]?'
		read continue
		if test "${continue}" != 'y' && test "${continue}" != 'Y'
		then
		    error 'abort'
		else
		    set_prefix "${1}"
		    shift
		fi
	      elif test -z "$5"; then error 'missing'
	      else
		exec_dir="$1"
		wrapper_dir="$2"
		doc_dir="$3"
		share_dir="$4"
		plugin_dir="$5"
		shift 5
	    fi
	  else
	  error 'conflict'
	fi
	;;
	

    --prefix=*)
	if test "$flag_mode" = '--prefix=' ||  test -z "$flag_mode" ; then
	    flag_mode='--prefix='

	    param=`echo "$1" | sed -e 's/--prefix=//'`
	    if test $param; then prefix=$param; fi
	    shift
	  else
	  error 'conflict'
	fi
	;;

    --exec_prefix=*)
	if test "$flag_mode" = '--prefix=' || test -z "$flag_mode" ; then
	    flag_mode='--prefix='

	    param=`echo "$1" | sed -e 's/--exec_prefix=//'`
	    if test $param; then exec_prefix=$param; fi
	    shift
	  else
	  error 'conflict'
	fi
	;;

    --wrapperdir=*)
	if test "$flag_mode" = '--prefix=' || test -z "$flag_mode" ; then
	    flag_mode='--prefix='

	    param=`echo "$1" | sed -e 's/--wrapperdir=//'`
	    if test $param; then wrapperdir=$param; fi
	    shift
	  else
	  error 'conflict'
	fi
	;;

    --docdir=*)
	if test "$flag_mode" = '--prefix=' || test -z "$flag_mode" ; then
	    flag_mode='--prefix='

	    param=`echo "$1" | sed -e 's/--docdir=//'`
	    if test $param; then docdir=$param; fi
	    shift
	  else
	  error 'conflict'
	fi
	;;

    --sharedir=*)
	if test "$flag_mode" = '--prefix=' || test -z "$flag_mode" ; then
	    flag_mode='--prefix='

	    param=`echo "$1" | sed -e 's/--sharedir=//'`
	    if test $param; then sharedir=$param; fi
	    shift
	  else
	  error 'conflict'
	fi
	;;

    --plugindir=*)
	if test "$flag_mode" = '--prefix=' || test -z "$flag_mode" ; then
	    flag_mode='--prefix='

	    param=`echo "$1" | sed -e 's/--plugindir=//'`
	    if test $param; then plugindir=$param; fi
	    shift
	  else
	  error 'conflict'
	fi
	;;

     *) error 'invalid';;
esac
done
    debug_msg 1 "flag_mode = $flag_mode"
}

set_os_spesific()
{
    # This function determines which commands and parameters will work on a given platform.

    case "${machine}:${os}" in
	sun*:SunOS*|sparc*:SunOS*|i86pc:SunOS*)
	    cpf=''    # SunOS cp overwrites by default (no -f force option)
	    cpv=''    # SunOS cp (no -v verbose option)
	    chmodv='' # SunOS chmod (no -v verbose option)
	    mkdirv='' # SunOS mkdir (no -v verbose option)
	    mvv=''    # SunOS mv (no -v verbose option)
	;;

	i[3456]86:FreeBSD|i[3456]86:NetBSD)
		cpf='-f'
		if test "$verbose" -gt '1'; then
		    chmodv='-v'
		    mkdirv='-v'
		fi
		if test "$verbose" -gt '0'; then
		    mvv='-v'
		    cpv='-v'
		fi
	;;

	i[3456]86:OpenBSD)
	    cpf='-f'
	    mkdirv=''
	    chmodv=''
	    cpv=''
	    mvv=''
	;;
	
	i[3456]86:Linux|mips:Linux|ppc:Linux|sparc*:Linux)
		cpf='-f'
		if test "$verbose" -gt "1"; then
		    chmodv='-v'
		    mkdirv='--verbose'
		    else # in case of csh
		    chmodv=''
		    mkdirv=''
		fi
		if test "$verbose" -gt "0"; then
		    mvv='-v'
		    cpv='-v'
		    else # in case of csh
		    mvv=''
		    cpv=''
		fi
	;;

	*:*)
		error 'os'
	;;

    esac
    debug_msg 1 "cpf = $cpf"
    debug_msg 1 "cpv = $cpv"
    debug_msg 1 "chmodv = $chmodv"
    debug_msg 1 "mkdirv = $mkdirv"

    # Common
    mkdirp='-p'
    cpp='-p'
    cpR='-R'
    lns='-s'
    lnf='-f'
}

# test_parameter()
# {
    # Test whether a command supports a parameter
    # $1 = command
    # $2 = parameter

#    debug_msg 0 "in test_parameter()"
#    debug_msg 1 "command = $1"
#    debug_msg 1 "parameter = $2"

#    param=` $1 -$2 2>&1 \
#	    | egrep "^$1"': (invalid|illegal) option -- '\
#	    | sed -e "s/$1"': \(invalid\|illegal\) option -- //' `
#    debug_msg 1 "param = $param"

#    if test "$param" = "$2"; then
#	return 1
#	else
#	return 0
#    fi
# }

version()
{
    # This function displays the version information.

    debug_msg 0 'in version()'

    echo 'install.sh (Opera Software ASA) 3.2'
    echo 'Written by Christian Westgaard <chrisw@opera.com>'
    echo
    echo 'Copyright (C) 2001-2003 Opera Software ASA.'
    echo 'This is free software; There is NO warranty; not even'
    echo 'for MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.'
}

select_installation_type()
{
    # This function parses installation flags and sets final installation paths.

    debug_msg 0 'in select_installation_type()'
    debug_msg 1 "flag_mode = ${flag_mode}"

    if test -z "${flag_mode}"; then flag_mode='--interactive'; fi


    if test "${flag_mode}" = '--interactive' -o "${flag_mode}" = '--standard'; then
	set_prefix "${default_prefix}"
    fi

    case $flag_mode in

	--interactive)
	    echo 'Running interactive installation...'
	    interactive_install
	    ;;

	--standard)
	    echo 'Running standard installation...'
	    ;;


	--force)
	    echo 'Running forced installation...'
	    ;;

	--prefix=)
	    echo 'Running prefix installation...'
	    if test "$prefix"; then 
		set_prefix "${prefix}"
	    fi
	    if test "$exec_prefix"; then exec_dir="$exec_prefix"; fi
	    if test "$wrapperdir"; then wrapper_dir="$wrapperdir"; fi
	    if test "$docdir"; then doc_dir="$docdir"; fi
	    if test "$sharedir"; then share_dir="$sharedir"; fi
	    if test "$plugindir"; then plugin_dir="$plugindir"; fi
	    ;;
        *) # no install type defined, check envir
	    if test -z "$prefix" && ( test -z "$exec_prefix" || test -z "$wrapperdir" || test -z "$docdir" || test -z "$sharedir" || test -z "$plugindir")
	      then error 'missing'
	      else flag_mode='--prefix='; select_installation_type
	    fi
	    ;;
    esac

}

can_write_to()
{
    # This function checks write access on paths
    # Returns (0 true writeable) or (1 false unwriteable)

    # Arg1 = directory to test

    debug_msg 0 'in can_write_to()'

    while test "${1}"; do
	debug_msg 1 " ${1}"
	test_folder="${1}"

	# If directory doesn't exist go up once and test again
	while test ! -d "${test_folder}"; do
	    temp_folder=`echo ${test_folder} | sed -e 's/\/[^\/]*$//'`
	    # If nothing removed, avoid infinite loop
	    if test "${temp_folder}" = "${test_folder}"; then break;fi
	    test_folder="${temp_folder}"
	    debug_msg 1 " ${test_folder}"
	done

	test -w "${test_folder}" || { debug_msg 1 "User \"${USERNAME}\" can't write to [${test_folder}]" ; return 1; }
	debug_msg 1 "User \"${USERNAME}\" has write access to [${test_folder}]"
	shift
    done
}

echo_paths()
{
    
    echo
    echo "Files will be intalled as follows:"
    echo "-----------------------------------------------------------"
    echo " Wrapper Script : ${wrapper_dir}"
    echo " Binaries       : ${exec_dir}"
    echo " Plugins        : ${plugin_dir}"
    echo " Shared files   : ${share_dir}"
    echo " Documentation  : ${doc_dir}"
    echo "-----------------------------------------------------------"
    echo "Is this correct [y,n,c | yes,no,cancel]?"
    while true; do
	read confirm
	case "${confirm}" in

	    ''|y|Y|yes|YES)
		break
		;;
		    
	    n|N|no|NO)
		change_paths
		echo_paths
		break
		;;

	    c|C|cancel|CANCEL)
		error "abort"
		;;

	    *)
		echo 'Invalid answer, try again:'
		;;

	esac
    done
}

set ()
{
    # This funtion enables csh syntax for the set command.
    # Read more at http://zsh.sunsite.dk/Intro/intro_12.html
    eval "$1$2$3"
}

not()
{
    # Arg = command/function to revert result

    debug_msg 0 "in not"
    debug_msg 1 "arguments: ${@}"
    
    ${@}
    if test "${?}" = 0; then
	return 1;
    else
	return 0;
    fi
}

prompt_path()
{
    # This function suggests a path and test new if user changes
    # Arg 1 = type of files
    # Arg 2 = path to test
    # Arg 3 = variable to modify

    if test "${1}" = 'prefix'
	then
	    echo "Enter installation prefix [${prefix}]"
	else
	    echo "Enter path for the Opera ${1} [${2}]"
    fi

    read a_path

    if test "${a_path}"
	then
	    while not "can_write_to ${a_path}"; do
		echo "User \"${USERNAME}\" does not have write access to [${a_path}]."
		if test "${1}" = 'prefix'
		    then
			echo "Enter installation prefix [${2}]"
		    else
			echo "Enter path for the Opera ${1} [${2}]"
		fi
		read a_path
		if test -z "${a_path}"; then return; fi
	    done
	    if test "${1}" = 'prefix'
		then
		set_prefix "${a_path}"
		else
		set  "${3}"="${a_path}"
	    fi
    fi
}

change_paths()
{
    prompt_path "prefix" "${prefix}" 'prefix'
    prompt_path "wrapper script" "${wrapper_dir}" "wrapper_dir"
    prompt_path "binaries" "${exec_dir}" "exec_dir"
    prompt_path "plugins" "${plugin_dir}" "plugin_dir"
    prompt_path "shared files" "${share_dir}" "share_dir"
    prompt_path "documentation" "${doc_dir}" "doc_dir"
}

interactive_install()
{
    # This function starts of checking if you have access to
    # prefix location
    # default locations (/usr)
    # ${HOME}

    if test "$flag_mode" = "--prefix=" -o "$flag_mode" = "--force"
	then
	    # Test current prefixed locations.
	    if can_write_to ${exec_dir} ${wrapper_dir} ${doc_dir} ${share_dir} ${plugin_dir}
		then
		    # Prefix location success -> Recommend prefix install
		    echo "User \"${USERNAME}\" has write access to current prefixed locations."
		    echo_paths
		    return
		else
		    # Prefix location failed -> Try default
		    echo "User \"${USERNAME}\" does not have write access to current prefixed locations."
		    echo "Trying default locations, prefix [/usr]..."
	    fi
    fi

    set_prefix "${default_prefix}"

    if can_write_to ${exec_dir} ${wrapper_dir} ${doc_dir} ${share_dir} ${plugin_dir}
	then 
	    # Default location success -> Recommend standard install
	    echo "User \"${USERNAME}\" has write access to default locations. Prefix [${prefix}]"
	else
	    # Default location failed -> Try other prefixes
	    echo "User \"${USERNAME}\" does not have write access to default locations. Prefix [${prefix}]"
	    if test -z "${HOME}"
		then
		    echo "Enviroment variable \"${HOME}\" not set, user \"${USERNAME}\" has no homefolder?"
		    echo "Not trying locations with prefix [~/.]"
		else
		    echo "Trying prefix [${HOME}]..."
		    set_prefix "${HOME}"

		    if can_write_to ${exec_dir} ${wrapper_dir} ${doc_dir} ${share_dir} ${plugin_dir}
			then 
			    # Prefix [~/.] success -> Recommend home install
			    echo "User \"${USERNAME}\" has write access to locations with prefix [${prefix}]."
		    fi

	    fi
    fi
    echo_paths
}


backup(){

    # This function makes a backup of a file or directory to use in future rollback
    # This function is in development
    # 1 = file or directory
    # 2 = location in backup
    # 3 = folder to create

    debug_msg 0 "in backup()"

    rollback_dir="${share_dir}/uninst/rollback_${opera_version}"
    if test -f "${1}" -o -d "${1}"
    then
	if not "can_write_to ${rollback_dir}"
	then
	    echo "Can't make backup of [${1}] for future rollback." 
	else
	    if test ! -d ${rollback_dir}; then mkdir $mkdirv $mkdirp ${rollback_dir}; fi
	    if test ${3}; then
		if test ! -d ${rollback_dir}/${3}; then mkdir $mkdirv $mkdirp ${rollback_dir}/${3}; fi
		location=${3}/${2}
	    else
    		location=${2}
	    fi
	    mv $mvv $mvf ${1} ${rollback_dir}/${location} && {
	    echo
	    echo "Backup of:"
	    echo " [${1}] can be found at"
	    echo " [${rollback_dir}/${location}]"
	    }
	fi
    else
	debug_msg 1 "[${1}] does not exist"
    fi
}

generate_wrapper()
{

    # This function generates the wrapper script with common plugin paths and correct opera path

    debug_msg 0 "in generate_wrapper()"

    wrapper_opera_plugin_paths="    \"\${HOME}/.opera/plugins\" \\
    ${plugin_dir} \\
    /usr/lib/opera/plugins \\
    /usr/local/lib/opera/plugins \\
    /opt/lib/opera/plugins \\"

    case "${machine}:${os}" in
	i[3456]86:Linux|i[3456]86:FreeBSD|i[3456]86:NetBSD|i[3456]86:OpenBSD)
	    wrapper_plugin_paths="
    /usr/local/Acrobat[45]/Browsers/intellinux \\
    /usr/lib/Acrobat[45]/Browsers/intellinux \\
    /usr/X11R6/lib/Acrobat[45]/Browsers/intellinux \\
    /opt/Acrobat[45]/Browsers/intellinux \\
    /usr/Acrobat[45]/Browsers/intellinux \\
    /usr/j2se/jre/plugin/i386/ns4 \\
    /usr/java/jre1.4.0/plugin/i386/ns4 \\
    /usr/java/jre1.3.1/plugin/i386/ns4 \\
    /usr/lib/j2re1.3/plugin/i386/netscape4 \\
    /usr/local/jdk1.3.1/jre/plugin/i386/ns4 \\
    /usr/local/linux-jdk1.3.1/jre/plugin/i386/ns4 \\"
	    wrapper_ibmjava="
	    IBMJava2-14/jre \\
	    IBMJava2-131/jre \\"
	    wrapper_sunjava_machine="i386"
	;;

	sun*:SunOS*|sparc*:SunOS*)
	    wrapper_plugin_paths="
    /opt/Acrobat[45]/Browsers/sparcsolaris \\
    /usr/local/Acrobat[45]/Browsers/sparcsolaris \\
    /usr/lib/Acrobat[45]/Browsers/sparcsolaris \\
    /usr/Acrobat[45]/Browsers/sparcsolaris \\
    /usr/j2se/jre/plugin/sparc/ns4 \\
    /usr/java/jre1.4.0/plugin/sparc/ns4 \\
    /usr/java/jre1.3.1/plugin/sparc/ns4 \\"
	    wrapper_sunjava_machine="sparc"
	;;

	i[3456]86:SunOS*|i86pc:SunOS*)
	    wrapper_plugin_paths="
    /usr/j2se/jre/plugin/i386/ns4 \\
    /usr/java/jre1.4.0/plugin/i386/ns4 \\
    /usr/java/jre1.3.1/plugin/i386/ns4 \\
    /usr/lib/j2re1.3/plugin/i386/netscape4 \\"
	    wrapper_sunjava_machine="i386"
	;;

	sparc*:Linux)
	    wrapper_plugin_paths=''
	    wrapper_sunjava_machine="sparc"
	;;

	mips*:Linux)
	    wrapper_plugin_paths=''
	;;

	ppc:Linux)
	    wrapper_plugin_paths=''
	    wrapper_ibmjava="
	    IBMJava2-ppc-14/jre \\
	    IBMJava2-ppc-131/jre \\"
	;;


	*:*)
		error 'os'
	;;
    esac
    wrapper_netscape_plugin_paths="
    /usr/lib/RealPlayer8/Plugins \\
    /usr/lib/realplay/plugins \\
    /usr/lib/RealPlayer8 \\
    /usr/lib/realplay \\
    \"\${HOME}/.netscape/plugins\" \\
    /opt/netscape/plugins \\
    /usr/lib/netscape/plugins \\
    /usr/local/netscape/plugins \\
    /usr/local/lib/netscape/plugins \\"

    wrapper_file="${wrapper_dir}/opera"
    
    wrapper_contain="#!/bin/sh

# Location of the Opera binaries
BINARYDIR=${exec_dir}

# Parse commandline parameters
passflags=''
while test \"\$#\" -gt 0; do
    case \$1 in
	--binarydir*|-binarydir*)
	    BINARYDIR=\"\$2\"
	    shift
	    ;;
	--) shift; break;;
	*)
	    passflags=\"\$passflags \$1\"
	    ;;
    esac
    shift
done

if test ! -f \${BINARYDIR}/opera; then
    echo \"The Opera binary is not located at \\\"\${BINARYDIR}/opera\\\".\"
    echo \"Please modify the wrapper script at \\\"\${0}\\\".\"
    exit 1
elif test ! -x \"\${BINARYDIR}/opera\"; then
    echo \"You do not have execute rights on \\\"\${BINARYDIR}/opera\\\", please ask the sysadmin to chmod +x it.\"
    exit 1
fi

# Opera enviroment
if test \"\${OPERA_DIR}\" = '' ; then 
  if test -d ${share_dir} ; then 
    OPERA_DIR=${share_dir}
  else
    echo \"OPERA_DIR unset and not found at expected location (${share_dir})\"
    exit 1
  fi
fi

# Native Java enviroment
if test -f \"\${HOME}/.opera/javapath.txt\"; then
    INIJAVA=\`cat \${HOME}/.opera/javapath.txt\`
    if test -f \"\${INIJAVA}/libjava.so\"; then OPERA_JAVA_DIR=\"\${INIJAVA}\"; fi
fi

if test ! \"\${OPERA_JAVA_DIR}\"; then

    PREFIXES=\"
	/usr
	/usr/java
	/usr/lib
	/usr/local
	/opt\"

    for SUNJAVA in \\
	j2re1.4.1_01 \\
	j2re1.4.1 \\
	j2re1.4.0_01 \\
	j2sdk1.4.0_01/jre \\
	j2re1.4.0 \\
	jre1.4.0 \\
	j2se/1.4/jre \\
	j2se/1.3/jre \\
	j2se/jre \\
	jre1.3.1_02 \\
	jre1.3.1_01 \\
	j2re1.3.1 \\
	jre1.3.1 \\
	j2re1.3 \\
	; do
	for PREFIX in \${PREFIXES}; do
	    if test -f \"\${PREFIX}/\${SUNJAVA}/lib/${wrapper_sunjava_machine}/libjava.so\"; then OPERA_JAVA_DIR=\"\${PREFIX}/\${SUNJAVA}/lib/${wrapper_sunjava_machine}\" && break; fi
	done
	if test \"\${OPERA_JAVA_DIR}\"; then break; fi
    done

    if test ! \"\${OPERA_JAVA_DIR}\"; then
	for IBMJAVA in \\${wrapper_ibmjava}
	    ; do
	    for PREFIX in \${PREFIXES}; do
		if test -f \"\${PREFIX}/\${IBMJAVA}/bin/libjava.so\"; then OPERA_JAVA_DIR=\"\${PREFIX}/\${IBMJAVA}/bin\" && break; fi
	    done
	if test \"\${OPERA_JAVA_DIR}\"; then break; fi
	done
    fi
fi

if test \"\${OPERA_JAVA_DIR}\"; then
    LD_LIBRARY_PATH=\"\${OPERA_JAVA_DIR}:\${OPERA_JAVA_DIR}/native_threads:\${OPERA_JAVA_DIR}/client:\${OPERA_JAVA_DIR}/classic:\${LD_LIBRARY_PATH}\"
    export LD_LIBRARY_PATH
    # Uncomment the next lines to workaround the \"preloaded libXt\" problem.
    #LD_PRELOAD=\"\${OPERA_JAVA_DIR}/libawt.so\"
    #OPERA_FORCE_JAVA_ENABLED=\"1\"
    #export LD_PRELOAD OPERA_FORCE_JAVA_ENABLED
fi

# Opera Plug-in enviroment, Add more plugin search paths here
# If OPERA_PLUGIN_PATH is set NPX_PLUGIN_PATH will be ignored

for DIR in \\
${wrapper_opera_plugin_paths}${wrapper_plugin_paths}${wrapper_netscape_plugin_paths}
    ; do
    if test -d \${DIR} ; then OPERA_PLUGIN_PATH=\"\${OPERA_PLUGIN_PATH}\":\"\${DIR}\"; fi
done

# Opera does not support Mozilla plugins, only Opera and Netscape plugins.
# Since Mozilla supports NS plugins, there might be some in the Mozilla plugin
# directory. Perhaps add below paths to list above
# /usr/X11R6/lib/mozilla/plugins


# Acrobat Reader
for BINDIR in \\
    /usr/local/Acrobat[45]/bin \\
    /usr/lib/Acrobat[45]/bin \\
    /usr/X11R6/lib/Acrobat[45]/bin \\
    /opt/Acrobat[45]/bin \\
    /usr/Acrobat[45]/bin \\
    ; do
    if test -d \${BINDIR} ; then PATH=\${PATH}:\${BINDIR}; fi
done

OPERA_PLUGIN_PATH=\"\${OPERA_PLUGIN_PATH}\":\"\${NPX_PLUGIN_PATH}\"
 
# Exporting the enviroment
export OPERA_DIR OPERA_PLUGIN_PATH PATH

# Running Opera
\"\${BINARYDIR}/opera\" --binarydir \"\${BINARYDIR}\" \$passflags \"\${@}\"
"

echo "${wrapper_contain}" > ${wrapper_file}
}

generate_opera_policy()
{
    policy_contain="
// Standard extensions get all permissions by default

grant codeBase \"file:\${java.home}/lib/ext/*\" {
	permission java.security.AllPermission;
};

// Opera package classes get all permissions
grant codebase \"file://${share_dir}/java//opera.jar\" {
	permission java.security.AllPermission;
};

grant {
	permission java.lang.RuntimePermission \"accessClassInPackage.sun.*\";
};

// default permissions granted to all domains

grant { 
	// Allows any thread to stop itself using the java.lang.Thread.stop()
	// method that takes no argument.
	// Note that this permission is granted by default only to remain
	// backwards compatible.
	// It is strongly recommended that you either remove this permission
	// from this policy file or further restrict it to code sources
	// that you specify, because Thread.stop() is potentially unsafe.
	// See \"http://java.sun.com/notes\" for more information.
	permission java.lang.RuntimePermission \"stopThread\";

	// allows anyone to listen on un-privileged ports
	permission java.net.SocketPermission \"localhost:1024-\", \"listen\";

	// \"standard\" properies that can be read by anyone

	permission java.util.PropertyPermission \"java.version\", \"read\";
	permission java.util.PropertyPermission \"java.vendor\", \"read\";
	permission java.util.PropertyPermission \"java.vendor.url\", \"read\";
	permission java.util.PropertyPermission \"java.class.version\", \"read\";
	permission java.util.PropertyPermission \"os.name\", \"read\";
	permission java.util.PropertyPermission \"os.version\", \"read\";
	permission java.util.PropertyPermission \"os.arch\", \"read\";
	permission java.util.PropertyPermission \"file.separator\", \"read\";
	permission java.util.PropertyPermission \"path.separator\", \"read\";
	permission java.util.PropertyPermission \"line.separator\", \"read\";

	permission java.util.PropertyPermission \"java.specification.version\", \"read\";
	permission java.util.PropertyPermission \"java.specification.vendor\", \"read\";
	permission java.util.PropertyPermission \"java.specification.name\", \"read\";

	permission java.util.PropertyPermission \"java.vm.specification.version\", \"read\";
	permission java.util.PropertyPermission \"java.vm.specification.vendor\", \"read\";
	permission java.util.PropertyPermission \"java.vm.specification.name\", \"read\";
	permission java.util.PropertyPermission \"java.vm.version\", \"read\";
	permission java.util.PropertyPermission \"java.vm.vendor\", \"read\";
	permission java.util.PropertyPermission \"java.vm.name\", \"read\";
};
"
    echo "${policy_contain}" > "${share_dir}/java/opera.policy"
}

run_install()
{
    # This function copies files to selected locations and 

    debug_msg 0 "in run_install()"

    can_write_to ${exec_dir} || error access ${exec_dir}
    can_write_to ${wrapper_dir} || error access ${wrapper_dir}
    can_write_to ${doc_dir} || error access ${doc_dir}
    can_write_to ${share_dir} || error access ${share_dir}
    can_write_to ${plugin_dir} || error access ${plugin_dir}

    backup ${wrapper_dir}/opera opera

    # Executable
	debug_msg 1 "Executable"
	mkdir $mkdirv $mkdirp $exec_dir/
	chmod $chmodv 755 $exec_dir

	cp $cpv $cpf bin/opera $exec_dir/opera
        chmod $chmodv 755 $exec_dir/opera

    # M2
	if test -f lib/m2.so; then
	    cp $cpv $cpf lib/m2.so $exec_dir/m2.so
	    chmod $chmodv 755 $exec_dir/m2.so
	fi

    # Wrapper
	debug_msg 1 "Wrapper"
	mkdir $mkdirv $mkdirp $wrapper_dir/
	chmod $chmodv 755 $wrapper_dir

	#cp $cpv $cpf wrapper.sh $wrapper_dir/opera
	generate_wrapper
        chmod $chmodv 755 $wrapper_dir/opera

    # Documentation
	debug_msg 1 "Documentation"

	mkdir $mkdirv $mkdirp $doc_dir/
	chmod $chmodv 755 $doc_dir
	cp $cpv $cpf LICENSE $doc_dir
	if not "test -f $doc_dir/help"; then ln -s $share_dir/help $doc_dir/help; fi

    # Button sets
	debug_msg 1 "Button sets"

	button_dir=$share_dir/buttons

	mkdir $mkdirv $mkdirp $button_dir/
	chmod $chmodv 755 $button_dir
	
	cp $cpv $cpf $cpR buttons/default buttons/defsmall buttons/examples buttons/standard buttons/readme.txt $button_dir/
	chmod 755 $button_dir/default $button_dir/defsmall $button_dir/examples $button_dir/standard

	ln $lns $lnf default $button_dir/Default
	ln $lns $lnf defsmall $button_dir/DefSmall

    # Shared files
	debug_msg 1 "Shared files"

	mkdir $mkdirv $mkdirp $share_dir/help/
	chmod $chmodv 755 $share_dir/help
	cp $cpv $cpf $cpR help/* $share_dir/help/

	mkdir $mkdirv $mkdirp $share_dir/config/
	chmod $chmodv 755 $share_dir/config
	cp $cpv $cpf $cpR config/* $share_dir/config/
	    
	mkdir $mkdirv $mkdirp $share_dir/images/
	chmod $chmodv 755 $share_dir/images
	cp $cpv $cpf $cpR images/* $share_dir/images/

	mkdir $mkdirv $mkdirp $share_dir/skin/
	chmod $chmodv 755 $share_dir/skin
	cp $cpv $cpf $cpR skin/* $share_dir/skin/

	if test -d ini; then
	    mkdir $mkdirv $mkdirp $share_dir/ini/
	    chmod $chmodv 755 $share_dir/ini
	    cp $cpv $cpf $cpR ini/* $share_dir/ini/
	fi

	if test -d javascript; then
	    cp $cpv $cpf $cpR javascript/* $share_dir/
	fi

	mkdir $mkdirv $mkdirp $share_dir/locale/
	chmod $chmodv 755 $share_dir/locale

	if ls locale/* >/dev/null 2>/dev/null; then
	    cp $cpv $cpf $cpR locale/* $share_dir/locale/
	fi
	
	mkdir $mkdirv $mkdirp $share_dir/styles/
	chmod $chmodv 755 $share_dir/styles
	cp $cpv $cpf $cpR styles/* $share_dir/styles/

	mkdir $mkdirv $mkdirp $share_dir/styles/user/
	chmod $chmodv 755 $share_dir/styles/user
	cp $cpv $cpf $cpR styles/user/* $share_dir/styles/user/

	ln $lns $lnf opf.css $share_dir/styles/OPF.css

	cp $cpv $cpf chartables.bin $share_dir/

	if test -f unicode.dat; then
	    cp $cpv $cpf unicode.dat $share_dir/
	fi

	cp $cpv $cpf opera6.adr $share_dir/
	cp $cpv $cpf *.ini $share_dir/ >/dev/null 2>/dev/null

     # Java
	mkdir $mkdirv $mkdirp $share_dir/java/
	chmod $chmodv 755 $share_dir/java
	
	cp $cpv $cpf java/opera.jar $share_dir/java/
	if test -f java/lc.jar; then cp $cpv $cpf java/lc.jar $share_dir/java/; fi
	generate_opera_policy
	
     # Plug-in files
	debug_msg 1 "Plug-in files"

	mkdir $mkdirv $mkdirp $plugin_dir/
	chmod $chmodv 755 $plugin_dir

	cp $cpv $cpf plugins/operamotifwrapper plugins/operaplugincleaner plugins/libnpp.so $plugin_dir/
        chmod $chmodv 755 $plugin_dir/operamotifwrapper $plugin_dir/operaplugincleaner $plugin_dir/libnpp.so

     # System wide configuration files
	config_dir="/etc"
	if can_write_to "$config_dir"; then
	    echo
	    echo "System wide configuration files:"
	    echo "  $config_dir/opera6rc"
	    echo "  $config_dir/opera6rc.fixed"
	    echo " cannot be prefixed"
	    echo "Do you want to install them [y,n | yes,no]?"
	    while true; do
		read install_config
		case "${install_config}" in

		    ''|y|Y|yes|YES)
			backup $config_dir/opera6rc opera6rc config
			backup $config_dir/opera6rc.fixed opera6rc.fixed config
			cp $cpv $cpf config/opera6rc $config_dir
			cp $cpv $cpf config/opera6rc.fixed $config_dir
			break
			;;

		    n|N|no|NO)
			break
			;;
		    *)
			echo 'Invalid answer, try again:'
			;;
		esac
	    done
	else
	    echo
	    echo "User \"${USERNAME}\" does not have write access to $config_dir"
	    echo " System wide configuration files:"
	    echo "  $config_dir/opera6rc"
	    echo "  $config_dir/opera6rc.fixed"
	    echo " were not installed"
	fi

     # Shorcuts and Icons
	icons
	gnome
	kde 3
	kde 2
	kde1
	mandrake

# Add uninstallation here

	echo
	echo 'Installation completed. Enjoy !'
	if test "$flag_mode" = "--prefix=" -o "$flag_mode" = "--force"; then
	    echo
	    echo "You've completed a forced installation."
#	    echo "To make sure everything works as expected, you need to check a couple things:"
	    echo
#	    echo "Opera looks for some images, stylesheets and localization in the enviroment"
#	    echo "variable OPERA_DIR. Make sure OPERA_DIR is set to \"$share_dir\"."
#	    echo "example: OPERA_DIR=$share_dir; export OPERA_DIR"
#	    echo "     or: setenv OPERA_DIR=$share_dir"
#	    echo
#	    echo "Opera searches for plugins in the enviroment variable OPERA_PLUGIN_PATH"
#	    echo "Please make sure OPERA_PLUGIN_PATH is set to \"$plugin_dir\"."
#	    echo "example: OPERA_PLUGIN_PATH=$plugin_dir; export OPERA_PLUGIN_PATH"
#	    echo "     or: setenv OPERA_PLUGIN_PATH=$plugin_dir"
#	    echo
#	    echo "You may do this in typical startup files (X, WM, sh ...) but the best way is"
#	    echo "by changing the already predefined variables in the wrapper script:"
#	    echo "example: emacs $wrapper_dir/opera"
#	    echo
	fi
	echo "Start with \$PATH/opera, $wrapper_dir/opera or ./opera"
}

generate_wmconfig()
{
    # arg1 = location

    wmconfig_file="${1}/opera"

    wmconfig_contain='opera name "Opera"
opera description "Opera Web Browser"
opera icon "opera.xpm"
opera mini-icon "opera.xpm"
opera exec "opera &"
opera group "Internet"'

    echo "${wmconfig_contain}" > ${wmconfig_file}
    chmod $chmodv 644 ${wmconfig_file}
}

generate_desktop()
{
    # arg1 = location

    desktop_file="${1}/opera.desktop"

    desktop_contain='[Desktop Entry]
Name=Opera
Comment=Web Browser
Exec=opera
Icon=opera.xpm
Terminal=0
Type=Application'

    echo "${desktop_contain}" > ${desktop_file}
    chmod $chmodv 644 ${desktop_file}
}

generate_mdk_menu()
{
    # arg1 = location
    mdk_menu_file="${1}/opera"

    mdk_menu_contain="?package(opera):command=\"${wrapper_dir}/opera\" \\
icon=\"opera.png\" \\
needs=\"X11\" \\
section=\"Networking/WWW\" \\
title=\"Opera\" \\
longtitle=\"Opera Web Browser\" \\
mimetypes=\"text/html,text/plain,text/vnd.csr,text/vnd.wap.wml,text/wml,text/xml,text/xhtml\" \\
accept_url=\"true\" \\
multiple_files=\"false\""

    echo "${mdk_menu_contain}" > ${mdk_menu_file}
    chmod $chmodv 644 ${mdk_menu_file}
}

icons()
{
    # This function searches for common icon paths.

    debug_msg 0 "in icons()"

    if test ! -d /usr/share/icons; then
      if test -w /usr/share; then
        mkdir $mkdirv $mkdirp /usr/share/icons/
	chmod $chmodv 755 /usr/share/icons
	cp $cpv $share_dir/images/opera.xpm /usr/share/icons/opera.xpm
      fi
    elif test -w /usr/share/icons; then cp $cpv $share_dir/images/opera.xpm /usr/share/icons/opera.xpm
    fi
  
    if test ! -d /usr/share/pixmaps; then
      if test -w /usr/share; then
	mkdir $mkdirv $mkdirp /usr/share/pixmaps/
	chmod $chmodv 755 /usr/share/pixmaps
	cp $cpv $share_dir/images/opera.xpm /usr/share/pixmaps/opera.xpm
      fi
    elif test -w /usr/share/pixmaps/; then cp $cpv $share_dir/images/opera.xpm /usr/share/pixmaps/opera.xpm
    fi
  
    if test ! -d /etc/X11/wmconfig/; then
      if test -w /etc/X11; then
	mkdir $mkdirv $mkdirp /etc/X11/wmconfig/
	chmod $chmodv 755 /etc/X11/wmconfig
	generate_wmconfig /etc/X11/wmconfig
      fi
    elif test -w /etc/X11/wmconfig/; then generate_wmconfig /etc/X11/wmconfig
    fi
    
    if test -d /etc/X11/applnk/; then
      if test ! -d /etc/X11/applnk/Internet/; then
	if test -w /etc/X11/applnk; then
	  mkdir $mkdirv $mkdirp /etc/X11/applnk/Internet/
	  chmod $chmodv 755 /etc/X11/applnk/Internet
	  generate_desktop /etc/X11/applnk/Internet
	fi
      elif test -w /etc/X11/applnk/Internet; then generate_desktop /etc/X11/applnk/Internet
      fi
    fi
}

gnome()
{
    # This function searches for common gnome icon paths.

    debug_msg 1 "in gnome()"

    if test -d /opt/gnome/; then

      # /opt/gnome share
      if test -d /opt/gnome/share; then

        # /opt/gnome icon
        if test ! -d /opt/gnome/share/pixmaps/; then
	  if test -w /opt/gnome/share; then
	    mkdir $mkdirv $mkdirp /opt/gnome/share/pixmaps/
	    chmod $chmodv 755 /opt/gnome/share/pixmaps
	    cp $cpv $share_dir/images/opera.xpm /opt/gnome/share/pixmaps/opera.xpm
	  fi
	elif test -w /opt/gnome/share/pixmaps; then cp $cpv $share_dir/images/opera.xpm /opt/gnome/share/pixmaps/opera.xpm
	fi
	# end /opt/gnome icon

	# /opt/gnome link
	if test -d /opt/gnome/share/gnome/apps/; then
          if test -d /opt/gnome/share/gnome/apps/Internet/; then
            if test -w /opt/gnome/share/gnome/apps/Internet; then
              generate_desktop /opt/gnome/share/gnome/apps/Internet
	    fi
          elif test -d /opt/gnome/share/gnome/apps/Networking/WWW/; then
            if test -w /opt/gnome/share/gnome/apps/Networking/WWW; then
	      generate_desktop /opt/gnome/share/gnome/apps/Networking/WWW
	    fi
	  elif test -w /opt/gnome/share/gnome/apps; then
	    mkdir $mkdirv $mkdirp /opt/gnome/share/gnome/apps/Internet/
	    chmod $chmodv 755 /opt/gnome/share/gnome/apps/Internet
            generate_desktop /opt/gnome/share/gnome/apps/Internet
          fi
	fi
	# end /opt/gnome link
      fi
      # end /opt/gnome share

    elif test -d /usr/share/gnome/; then

        # /usr/share/gnome icon
        if test ! -d /usr/share/gnome/pixmaps/; then
	  if test -w /usr/share/gnome; then
	    mkdir $mkdirv $mkdirp /usr/share/gnome/pixmaps/
	    chmod $chmodv 755 /usr/share/gnome/pixmaps
	    cp $cpv $share_dir/images/opera.xpm /usr/share/gnome/pixmaps/opera.xpm
	  fi
	elif test -w /usr/share/gnome/pixmaps; then cp $cpv $share_dir/images/opera.xpm /usr/share/gnome/pixmaps/opera.xpm
	fi
	# end /usr/share/gnome icon

	# /usr/share/gnome link
	if test -d /usr/share/gnome/apps/; then
          if test -d /usr/share/gnome/apps/Internet/; then
            if test -w /usr/share/gnome/apps/Internet; then
              generate_desktop /usr/share/gnome/apps/Internet
	    fi
          elif test -d /usr/share/gnome/apps/Networking/WWW/; then
            if test -w /usr/share/gnome/apps/Networking/WWW; then
	     generate_desktop /usr/share/gnome/apps/Networking/WWW
	    fi
	  elif test -w /usr/share/gnome/apps; then
	    mkdir $mkdirv $mkdirp/usr/share/gnome/apps/Internet/
	    chmod $chmodv 755 /usr/share/gnome/apps/Internet
            generate_desktop /usr/share/gnome/apps/Internet
          fi
	fi
	# end /usr/share/gnome link
   fi
   # Add ximian here
}

kde()
{
    # This function searches for common kde2 and kde 3 icon paths.

    debug_msg 1 "in kde()"

    if test -d /opt/kde${1}/share; then

      DIR_HI=/opt/kde${1}/share/icons/hicolor
      if test -d "$DIR_HI" -a -w "$DIR_HI"; then
        if test -d "$DIR_HI"/48x48/apps -a -w "$DIR_HI"/48x48/apps; then cp $cpv $share_dir/images/opera_48x48.png $DIR_HI/48x48/apps/opera.png; fi
        if test -d "$DIR_HI"/32x32/apps -a -w "$DIR_HI"/32x32/apps; then cp $cpv $share_dir/images/opera_32x32.png $DIR_HI/32x32/apps/opera.png; fi
        if test -d "$DIR_HI"/22x22/apps -a -w "$DIR_HI"/22x22/apps; then cp $cpv $share_dir/images/opera_22x22.png $DIR_HI/22x22/apps/opera.png; fi
      fi

      DIR_LO=/opt/kde${1}/share/icons/locolor
      if test -d $DIR_LO -a -w $DIR_LO; then
        if test -d $DIR_LO/32x32/apps -a -w $DIR_LO/32x32/apps; then cp $cpv $share_dir/images/opera_32x32.png $DIR_LO/32x32/apps/opera.png; fi
        if test -d $DIR_LO/22x22/apps -a -w $DIR_LO/22x22/apps; then cp $cpv $share_dir/images/opera_22x22.png $DIR_LO/22x22/apps/opera.png; fi
        if test -d $DIR_LO/16x16/apps -a -w $DIR_LO/16x16/apps; then cp $cpv $share_dir/images/opera_16x16.png $DIR_LO/16x16/apps/opera.png; fi
      fi

      if test -d /opt/kde${1}/share/applnk/; then
        if test ! -d /opt/kde${1}/share/applnk/Internet/ -a -w /opt/kde${1}/share/applnk; then
	    mkdir $mkdirv $mkdirp /opt/kde${1}/share/applnk/Internet/
	    chmod $chmodv 755 /opt/kde${1}/share/applnk/Internet
	fi
	if test -w /opt/kde${1}/share/applnk/Internet; then generate_desktop /opt/kde${1}/share/applnk/Internet ${1}; fi
      fi

    fi  
    
    if test -d /usr/share/applnk/Networking; then
      if test ! -d /usr/share/applnk/Networking/WWW/ -a -w /usr/share/applnk/Networking; then
	    mkdir $mkdirv $mkdirp /usr/share/applnk/Networking/WWW/
	    chmod $chmodv 755 /usr/share/applnk/Networking/WWW
      fi
      if test -w /usr/share/applnk/Networking/WWW; then generate_desktop /usr/share/applnk/Networking/WWW ${1}; fi
    fi
}

kde1()
{
    # This function searches for common kde1 icon paths.

    debug_msg 1 "in kde1()"

    if test -d /opt/kde/share; then

      if test ! -d /opt/kde/share/icons/ -a -w /opt/kde/share; then
	    mkdir $mkdirv $mkdirp /opt/kde/share/icons/
	    chmod $chmodv 755 /opt/kde/share/icons
      fi
      if test -w /opt/kde/share/icons; then cp $cpv $share_dir/images/opera.xpm /opt/kde/share/icons/opera.xpm;fi

      if test ! -d /opt/kde/share/applnk/Internet/ -a -w /opt/kde/share/applnk; then
	    mkdir $mkdirv $mkdirp /opt/kde/share/applnk/Internet/
	    chmod $chmodv 755 /opt/kde/share/applnk/Internet
      fi
      if test -w /opt/kde/share/applnk/Internet; then generate_desktop /opt/kde/share/applnk/Internet; fi

    elif test -d /usr/share/applnk/Internet/ -a -w /usr/share/applnk/Internet; then
      generate_desktop /usr/share/applnk/Internet
    fi
}

mandrake()
{
    if test -x /usr/bin/update-menus; then

	test ! -d /usr/lib/menu && test -w /usr/lib && mkdir $mkdirv $mkdirp /usr/lib/menu/
	test -d /usr/lib/menu && generate_mdk_menu /usr/lib/menu

	test ! -d /usr/share/icons && test -w /usr/share && mkdir $mkdirv $mkdirp /usr/share/icons/ && chmod $chmodv 755 /usr/share/icons
	test -d /usr/share/icons && cp $cpv $share_dir/images/opera_32x32.png /usr/share/icons/opera.png

	test ! -d /usr/share/icons/mini && test -w /usr/share/icons && mkdir $mkdirv $mkdirp /usr/share/icons/mini/ && chmod $chmodv 755 /usr/share/icons/mini
	test -d /usr/share/icons/mini && cp $cpv $share_dir/images/opera_16x16.png /usr/share/icons/mini/opera.png

	test ! -d /usr/share/icons/large && test -w /usr/share/icons && mkdir $mkdirv $mkdirp /usr/share/icons/large && chmod $chmodv 755 /usr/share/icons/large
	test -d /usr/share/icons/large && cp $cpv $share_dir/images/opera_48x48.png /usr/share/icons/large/opera.png

	/usr/bin/update-menus || true
    fi
}

# These are the only variables the needs to be changed
opera_version='7.11-20030512.3'

plugin_version_major='0'
plugin_version_minor='1'
plugin_version_release='8'

debug_level='0' # From 0 to 2

    echo test | sed -n -e 's/test//'; if test "$?" != '0'; then error 'sed'; fi

    # FreeBSD systems don't have $USERNAME by default
    if test -z "${USERNAME}" && test "${USER}"; then
        USERNAME="${USER}"
        debug_msg 0 "setting USERNAME to ${USER}"
    fi

    check_prefixes $@
    select_installation_type
    set_os_spesific
    run_install
exit 0
