/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Author: Charles Kerr <charles@rebelbase.com>
 *
 * Copyright (C) 2000, 2001  Pan Development Team <pan@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <config.h>

#include <ctype.h>
#include <string.h>

#include <glib.h>

#include <pan/base/pan-i18n.h>
#include <pan/base/pan-glib-extensions.h>
#include <pan/base/util-wrap.h>
#include <pan/base/base-prefs.h>

/****
*****  LINE WRAPPING
****/

typedef struct
{
	gchar * leader;
	gchar * content;
}
Paragraph;

static Paragraph*
paragraph_new (const gchar * leader, const gchar * content)
{
	Paragraph * p = g_new0 (Paragraph, 1);
	p->leader = g_strdup (leader);
	p->content = g_strstrip (g_strdup (content));
	return p;
}

static void
paragraph_free (Paragraph * p)
{
	g_free (p->leader);
	g_free (p->content);
	g_free (p);
}

static GPtrArray*
get_paragraphs (const gchar * body, gint wrap_column)
{
	GPtrArray * lines = g_ptr_array_new ();
	GPtrArray * paragraphs = g_ptr_array_new ();

	/* build an array of lines that are split between leader & content */
	if (1)
	{
		gint line_len = 0;
		const gchar * line_start = NULL;

		while (get_next_token_run (body, '\n', &body, &line_start, &line_len))
		{
			Paragraph * p;
			const gchar * pch = line_start;
			const gchar * end = line_start + line_len;

			while (pch!=end && (is_quote_char((guchar)*pch) || isspace((guchar)*pch)))
				++pch;

			p = g_new (Paragraph, 1);
			p->leader = g_strndup (line_start, pch-line_start);
			p->content = g_strstrip (g_strndup (pch, end-pch));
			g_ptr_array_add (lines, p);
		}

		/* add an empty line to make the paragraph-making loop go smoothly */
		g_ptr_array_add (lines, paragraph_new ("", ""));
	}

	/* merge the lines into paragraphs */
	if (1)
	{
		gint i;
		gint prev_content_len = 0;
		GString * cur_leader = g_string_new (NULL);
		GString * cur_content = g_string_new (NULL);

		for (i=0; i<lines->len; ++i)
		{
			const Paragraph * p = (const Paragraph*) g_ptr_array_index (lines, i);
			gboolean paragraph_end = TRUE;
			gboolean hard_break = FALSE;

			if (!cur_content->len || !pan_strcmp (p->leader, cur_leader->str))
				paragraph_end = FALSE;

			if (!is_nonempty_string(p->content)) {
				hard_break = prev_content_len!=0;
				paragraph_end = TRUE;
			}

			/* we usually don't want to wrap really short lines */
			if (prev_content_len!=0 && prev_content_len<(wrap_column/2))
				paragraph_end = TRUE;

			if (paragraph_end) /* the new line is a new paragraph, so save the old line */
			{
				g_ptr_array_add (paragraphs, paragraph_new (cur_leader->str, cur_content->str));
				g_string_assign (cur_leader, p->leader);
				g_string_assign (cur_content, p->content);

				if (hard_break)
					g_ptr_array_add (paragraphs, paragraph_new (cur_leader->str, ""));
			}
			else /* append to the content */
			{
				if (cur_content->len)
					g_string_append_c (cur_content, ' ');
				g_string_assign (cur_leader, p->leader);
				g_string_append (cur_content, p->content);
			}

			prev_content_len = is_nonempty_string(p->content) ? strlen (p->content) : 0;
		}

		/* cleanup step */
		g_string_free (cur_leader, TRUE);
		g_string_free (cur_content, TRUE);
	}

	/* remember that empty line we added back up at the top?  We remove it now */
	if (paragraphs->len>0)
		paragraph_free ((Paragraph*) g_ptr_array_remove_index (paragraphs, paragraphs->len-1));

	/* cleanup */
	pan_g_ptr_array_foreach (lines, (GFunc)paragraph_free, NULL);
	g_ptr_array_free (lines, TRUE);

	return paragraphs;
}

static gchar*
wrap_long_lines (gchar   * string,
                 gint      maxlen)
{
	gchar * linefeed_here;
	gchar * pch;
	gchar * line_start;


	/* walk through the entire string */
	linefeed_here = NULL;
	for (line_start=pch=string; *pch; )
	{
		/* a linefeed could go here; remember this space */
		if (isspace((guchar)*pch) || *pch=='\n')
			linefeed_here = pch;

		/* line's too long; add a linefeed if we can */
		if (pch-line_start>=maxlen && linefeed_here!=NULL)
		{
			*linefeed_here = '\n';
			pch = line_start = linefeed_here + 1;
			linefeed_here = NULL;
		}
		else ++pch;
	}

	return string;
}

static void
fill_paragraph (const Paragraph   * p,
                GPtrArray         * setme,
                gint                wrap_column)
{
	gchar * pch = g_strdup (p->content);
	const gchar * march;
	const gint wrap_len = wrap_column - strlen(p->leader);
	const gchar * line_begin;
	int line_len;

	wrap_long_lines (pch, wrap_len);
	march = pch;

	if (*march=='\0') /* blank line */
		g_ptr_array_add (setme, g_strdup(p->leader));
	else while (get_next_token_run (march, '\n',  &march, &line_begin, &line_len))
		g_ptr_array_add (setme, g_strdup_printf ("%s%*.*s", p->leader, line_len, line_len, line_begin));

	g_free (pch);
}

gchar*
fill_body (const gchar * body, gint wrap_column)
{
	GPtrArray * array = g_ptr_array_new ();
	GPtrArray * paragraphs = NULL;
	gchar * new_body = NULL;
	gchar * tmp_body = NULL;
	gchar * sig = NULL;
	gint i = 0;

	/* sanity checks */
	g_return_val_if_fail (is_nonempty_string(body), NULL);
	g_return_val_if_fail (wrap_column>0, g_strdup(body));

	/* get a temp copy of the body -- we don't wrap the signature. */
	tmp_body = pan_substitute (body, "\r", "");
	sig = pan_strstr (tmp_body, "\n-- \n");
	if (sig != NULL)
		*sig = '\0';

	/* fill the paragraphs */
       	paragraphs = get_paragraphs (tmp_body, wrap_column);
	for (i=0; i<paragraphs->len; ++i)
		fill_paragraph ((Paragraph*)g_ptr_array_index(paragraphs,i), array, wrap_column);

	/* make a single string of all filled lines */
	g_ptr_array_add (array, NULL);
	if (1) {
		gint i;
		GString * s = g_string_new (NULL);
		for (i=0; g_ptr_array_index(array,i)!=NULL; ++i) {
			g_string_append (s, (const char*)g_ptr_array_index(array,i));
			g_string_append_c (s, '\n');
		}
		if (s->len > 0)
			g_string_truncate (s, s->len-1);
		new_body = s->str;
		g_string_free (s, FALSE);
	}

	/* if we had a sig, put it back in */
	if (sig != NULL) {
		*sig = '\n';
		replace_gstr (&new_body, g_strjoin("\n",new_body, sig, NULL));
	}

	/* cleanup */
	pan_g_ptr_array_foreach (paragraphs, (GFunc)paragraph_free, NULL);
	g_ptr_array_free (paragraphs, TRUE);
	pan_g_ptr_array_foreach (array, (GFunc)g_free, NULL);
	g_ptr_array_free (array, TRUE);
	g_free (tmp_body);

	return new_body;
}

/***
****
***/

gchar*
mute_quoted_text_in_body (const gchar * text)
{
	gint line_len = 0;
	const gchar * line_begin = NULL;
	const gchar * march;
	gboolean last_line_was_quote;
	gchar * retval;
	GString * buf;

	g_return_val_if_fail (text!=NULL, NULL);

	march = text;
	last_line_was_quote = FALSE;
	buf = g_string_sized_new (strlen(text));
	while (get_next_token_run (march, '\n', &march, &line_begin, &line_len))
	{
		const gboolean is_quote = line_len && is_quote_char((guchar)*line_begin);

		if (!is_quote)
		{
			pan_g_string_append_len (buf, line_begin, line_len);
			g_string_append_c (buf, '\n');
		}
		else if (!last_line_was_quote)
		{
			g_string_append (buf, _("> [quoted text muted]"));
			g_string_append_c (buf, '\n');
		}

		last_line_was_quote = is_quote;
	}
	if (buf->len > 0)
		g_string_truncate (buf, buf->len-1); /* trim last \n */

	retval = buf->str;
	g_string_free (buf, FALSE);
	return retval;
}


gchar*
rot13_inplace (gchar * text)
{
	const char * plain = "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ";
	const char * roted = "nopqrstuvwxyzabcdefghijklmNOPQRSTUVWXYZABCDEFGHIJKLM";

	if (text != NULL)
	{
		gchar * march;

		for (march=text; *march!='\0'; ++march)
		{
			const gchar * offset = strchr (plain, *march);
			if (offset != NULL)
			{
				const int index = offset - plain;
				*march = roted[index];
			}
		}
	}

	return text;
}
