/*
 * xlog - GTK+ logging program for amateur radio operators
 * Copyright (C) 2001-2002 Joop Stakenborg <pa4tu@amsat.org>
 *
 * This program is free oftware; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/* callbacks.c
 *
 * This is where the callbacks for all the different dialogs reside. 
 * For callbacks related to the widgets in the main window, see the files
 * called callbacks_<widgetname>.[c|h].
 */

#ifdef HAVE_CONFIG_H
#	include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <libgen.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include "callbacks.h"
#include "interface.h"
#include "support.h"
#include "types.h"
#include "log.h"
#include "utils.h"

#define SEARCHHISTORY 10

#if WANT_HAMLIB
#	include <string.h>
#	include <hamlib-utils.h>
#	include <hamlib/rig.h>
#endif

extern GtkWidget *mainwindow;
extern GtkWidget *mainnotebook;
extern gchar *xlogdir;
extern preferencestype preferences;
extern statetype state;
extern gchar **bandsplit;
extern gchar **modesplit;
extern gchar **qso;
extern GtkWidget *preferencesdialog;
extern GtkWidget *printdialog;
extern gint clocktimer, savetimer;
extern GList *searchhistory;
extern GList *logwindowlist;

#if WANT_HAMLIB
	extern GList *riglist;
	extern gint hamlibtimer;
	extern RIG *myrig;
#endif


/* ABOUT DIALOG */
void on_aboutbutton_clicked (GtkButton *button, gpointer user_data) {
	gtk_widget_destroy(gtk_widget_get_toplevel(GTK_WIDGET(button)));
}


/* PREFERENCES DIALOG */

#if WANT_HAMLIB
/* return value for setting widgets */
static gint whichhamlibwidgets(gboolean frequency, gboolean smeter)
{
	if (frequency && smeter) return(4);
	else if (frequency && (!smeter)) return(2);
	else if ((!frequency) && smeter) return(3);
	else return(1);
}
#endif

/* check autosave value, if larger than 0 start timer */
static void set_autosave(gint value)
{
	if (preferences.autosave == 0) {
		if (value > 0)
			savetimer = gtk_timeout_add(value*60*1000, (GtkFunction)autosave, NULL);
	}
	else {
		if (value == 0) {
			gdk_input_remove(savetimer);
			savetimer = -1;
		}
		else if (value != preferences.autosave) { /* new value, stop and start timer */
			gdk_input_remove(savetimer);
			savetimer = gtk_timeout_add(value*60*1000, (GtkFunction)autosave, NULL);
		}
	}
	preferences.autosave = value;
}

/* set the theme if changed */
static void set_theme(gchar *color, gboolean alsoforlog)
{
	GtkWidget *clocklabel, *qsoframe, *dxccframe;
	gint i, row;
	logtype *logwindow;
	gint logcolor;
	gboolean logcolordone = FALSE;
	GtkStyle *cellstyle;
#if WANT_HAMLIB
	GtkWidget *frequencylabel;
#endif

	logcolor = alsoforlog ? 1: 0;

	if (g_strcasecmp(color, preferences.themecolor))
	{
		clocklabel = lookup_widget(mainwindow, "clocklabel");
		setlabelstyle(clocklabel, color);
#if WANT_HAMLIB
		frequencylabel = lookup_widget(mainwindow, "frequencylabel");
		setlabelstyle(frequencylabel, color);
#endif
		qsoframe = lookup_widget(mainwindow, "qsoframe");
		dxccframe = lookup_widget(mainwindow, "dxccframe");
		setframestyle(qsoframe, color);
		setframestyle(dxccframe, color);
		for (i = 0; i < g_list_length(logwindowlist); i++)
		{
			logwindow = g_list_nth_data(logwindowlist, i);
			setlabelstyle(logwindow->label, color);
		}
		if (alsoforlog) /* apply to the log if needed */
		{
			for (i = 0; i < g_list_length(logwindowlist); i++)
			{
				logwindow = g_list_nth_data(logwindowlist, i);
				gtk_clist_freeze(GTK_CLIST(logwindow->clist));
				for (row = 0 ; row < logwindow->qsos ; row++)
					setcallstyle(logwindow->clist,row, color);
				gtk_clist_thaw(GTK_CLIST(logwindow->clist));
			}
			logcolordone = TRUE;
		}
		preferences.themecolor = color;
	}

	/* does the log need color? */
	if (logcolor != preferences.logcolor)
	{
		if ((logcolor == 1) && (!logcolordone))
		{
			for (i = 0; i < g_list_length(logwindowlist); i++)
			{
				logwindow = g_list_nth_data(logwindowlist, i);
				gtk_clist_freeze(GTK_CLIST(logwindow->clist));
				for (row = 0 ; row < logwindow->qsos ; row++)
					setcallstyle(logwindow->clist,row, preferences.themecolor);
				gtk_clist_thaw(GTK_CLIST(logwindow->clist));
			}
		}
		if (logcolor == 0)
		{
			for (i = 0; i < g_list_length(logwindowlist); i++)
			{
				logwindow = g_list_nth_data(logwindowlist, i);
				gtk_clist_freeze(GTK_CLIST(logwindow->clist));
				cellstyle = gtk_clist_get_cell_style(GTK_CLIST(logwindow->clist), 0, 0);
				for (row = 0 ; row < logwindow->qsos ; row++)
					gtk_clist_set_cell_style(GTK_CLIST(logwindow->clist), row, 3, cellstyle);
				gtk_clist_thaw(GTK_CLIST(logwindow->clist));
			}
		}
		preferences.logcolor = logcolor;
	}
}

/* set savepath if changed */
static void set_path(gchar *pathstr)
{
	gchar *str, *fpathstr;
	GString *path;
	
	if (strlen(pathstr) > 0)
	{
		path = g_string_new(pathstr);
		str = g_strdup(path->str);
		/* remove trailing slash */
		g_strreverse(str);
		if (!g_strncasecmp(str, "/", 1))
		{
			path = g_string_erase(path, (path->len) -1, 1);
		}
		else /* in case a file is selected */
		{
			fpathstr = g_strdup(path->str);
			path = g_string_new(dirname(fpathstr));
			g_free(fpathstr);
		}
		g_free(str);
		if (g_strcasecmp(preferences.savedir, path->str))
			preferences.savedir = g_strdup(path->str);
		g_string_free(path, TRUE);
	}
}

/* set logs to load if changed */
static void set_logstoload(gchar *logs)
{
	gchar **logsplit;

	if (g_strcasecmp(logs, preferences.logstoload) != 0)
	{ 
		if (strlen(logs) == 0)
			logs = g_strdup("*");
		logsplit = g_strsplit(logs, ",", -1); /* delete spaces */
		deletespaces(logsplit);
		preferences.logstoload = g_strjoinv(",", logsplit);
		g_strfreev(logsplit);
	}
}

static void set_clock(gboolean on)
{
	GtkWidget *clocklabel, *clockpixmap, *clockhbox, *clockhandlebox;

	clocklabel = lookup_widget(mainwindow, "clocklabel");
	clockpixmap = lookup_widget(mainwindow, "clockpixmap");
	clockhbox = lookup_widget(mainwindow, "clockhbox");
	clockhandlebox = lookup_widget(mainwindow, "clockhandlebox");
	if (on) { /* yes, we want clock */
		if (preferences.clock == 0) clocktimer = gtk_timeout_add(1000, (GtkFunction)updateclock, NULL);
		gtk_widget_show(clocklabel);
		gtk_widget_show(clockpixmap);
		gtk_widget_show(clockhbox);
		gtk_widget_show(clockhandlebox);
		preferences.clock = 1;
	}
	else { /* no clock */
		 if (preferences.clock > 0) gdk_input_remove(clocktimer);
		 clocktimer = -1;
		 gtk_widget_hide(clocklabel);
		 gtk_widget_hide(clockpixmap);
		 gtk_widget_hide(clockhbox);
		 gtk_widget_hide(clockhandlebox);
		 preferences.clock = 0;
	}
}

/* ok button clicked */
void on_preferencesokbutton_clicked (GtkButton *button, gpointer user_data) {
	GtkWidget *latentry, *longentry, *NScomboentry, *EWcomboentry, *unitscomboentry, 
		*bandsentry, *bandoptionmenu, *modesentry, *modeoptionmenu, *bandsradiobutton1,
		*modesradiobutton1, *clockcheckbutton, *examplelabel, *autosavecomboentry,
		*themecheckbutton, *pathentry, *logsentry;
	GtkWidget *bandentry, *modeentry;
	gchar *temp, *bandcopy;
	gboolean toggletrue, logcoloryesno;
	gint bandindex = 0, modeindex = 0, autosavevalue;
#if WANT_HAMLIB
	GtkWidget *hamlibcheckbutton, *radiocomboentry, *devicecomboentry,
		*frequencycheckbutton, *digitscomboentry, *smetercheckbutton, *drawingarea;
	gchar *radio = NULL, *device = NULL, *digits = NULL;
	gboolean hamlibyes, frequencyyesno, smeteryesno;
	gint rigid, retcode;
	gint hamlibwidgets = 0;
#endif

	/* check autosave value */
	autosavecomboentry = lookup_widget(preferencesdialog, "autosavecomboentry");
	temp = gtk_editable_get_chars(GTK_EDITABLE(autosavecomboentry), 0, -1);
	autosavevalue = atoi(temp);
	set_autosave(autosavevalue);

	/* apply theme color */
	examplelabel = lookup_widget(preferencesdialog, "examplelabel");
	themecheckbutton =lookup_widget(preferencesdialog, "themecheckbutton");
	logcoloryesno = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(themecheckbutton));
	temp = getlabelcolor(examplelabel);
	set_theme(temp, logcoloryesno);

	/* get path for the logs */
	pathentry = lookup_widget(preferencesdialog, "pathentry");
	temp = gtk_editable_get_chars(GTK_EDITABLE(pathentry), 0, -1);
	set_path(temp);

	/* get logs to load */
	logsentry = lookup_widget(preferencesdialog, "logsentry");
	temp = gtk_editable_get_chars(GTK_EDITABLE(logsentry), 0, -1);
	set_logstoload(temp);

	
	/* set latitude, longitude and units */
	latentry = lookup_widget(preferencesdialog, "latentry");
	longentry = lookup_widget(preferencesdialog, "longentry");
	NScomboentry = lookup_widget(preferencesdialog, "NScomboentry");
	EWcomboentry = lookup_widget(preferencesdialog, "EWcomboentry");
	unitscomboentry = lookup_widget(preferencesdialog, "unitscomboentry");
	temp = gtk_editable_get_chars(GTK_EDITABLE(latentry), 0, -1);
	g_strstrip(temp);
	if (strlen(temp) == 0) preferences.latitude = 0.0;
	else preferences.latitude = atof(temp);
	temp = gtk_editable_get_chars(GTK_EDITABLE(longentry), 0, -1);
	g_strstrip(temp);
	if (strlen(temp) == 0) preferences.longitude = 0.0;
	else preferences.longitude = atof(temp);
	temp = gtk_editable_get_chars(GTK_EDITABLE(NScomboentry), 0, 1);
	if (g_strncasecmp(temp, "N", 1) == 0) preferences.NS = 1;
	else preferences.NS = 2;
	temp = gtk_editable_get_chars(GTK_EDITABLE(EWcomboentry), 0, 1);
	if (g_strncasecmp(temp, "E", 1) == 0) preferences.EW = 1;
	else preferences.EW = 2;
	temp = gtk_editable_get_chars(GTK_EDITABLE(unitscomboentry), 0, -1);
	if (g_strncasecmp(temp, "K", 1) == 0) preferences.units = 1;
	else preferences.units = 0;
	state.mylocation = setlocation(preferences.latitude, preferences.NS,
		preferences.longitude, preferences.EW);

	/* get the comma seperated list of modes and assign to the optionmenu */
	modesentry = lookup_widget(preferencesdialog, "modesentry");
	modeoptionmenu = lookup_widget(mainwindow, "modeoptionmenu");
	temp = gtk_editable_get_chars(GTK_EDITABLE(modesentry), 0, -1);
	modesplit = g_strsplit(temp, ",", 0);
	deletespaces(modesplit);
	updateoptionmenu(modeoptionmenu, modesplit);

	/* set the optionmenu for modes */
	modeentry = lookup_widget(mainwindow, "modeentry");
	temp = gtk_editable_get_chars(GTK_EDITABLE(modeentry), 0, -1);
	if (strlen(temp) > 0) {
		for (;;) {
			if (g_strcasecmp(temp, modesplit[modeindex]) == 0) {
				gtk_option_menu_set_history(GTK_OPTION_MENU(modeoptionmenu), modeindex);
				break;
			}
			modeindex++;
		}
	}

	/* get the comma seperated list of bands and assign to the optionmenu */
	bandsentry = lookup_widget(preferencesdialog, "bandsentry");
	bandoptionmenu = lookup_widget(mainwindow, "bandoptionmenu");
	temp = gtk_editable_get_chars(GTK_EDITABLE(bandsentry), 0, -1);
	bandsplit = g_strsplit(temp, ",", 0);
	deletespaces(bandsplit);
	updateoptionmenu(bandoptionmenu, bandsplit);

	/* set the optionmenu for bands in the main window */
	bandentry = lookup_widget(mainwindow, "bandentry");
	temp = gtk_editable_get_chars(GTK_EDITABLE(bandentry), 0, -1);
	if (strlen(temp) > 0) {
		if (strlen(temp) <= 2) bandcopy = g_strdup(temp);
		else bandcopy = finddot(temp);
		if (atoi(bandcopy) < 29) { /* only support HF bands for now */
			for (;;) {
				if (g_strcasecmp(bandcopy, bandsplit[bandindex]) == 0) {
					gtk_option_menu_set_history(GTK_OPTION_MENU(bandoptionmenu), bandindex);
					break;
				}
				bandindex++;
			}
		}
		g_free(bandcopy);
	}

	preferences.modes = g_strjoinv(",", modesplit);
	preferences.bands = g_strjoinv(",", bandsplit);

	/* set visibility of widgets in QSO frame depending on state of toggle buttons */
	bandsradiobutton1 = lookup_widget(preferencesdialog, "bandsradiobutton1");
	toggletrue = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(bandsradiobutton1));
	if (toggletrue) {
		preferences.bandseditbox = 1;
		gtk_widget_hide(bandoptionmenu);
		gtk_widget_show(bandentry);
	}
	else {
		preferences.bandseditbox = 0;
		gtk_widget_hide(bandentry);
		gtk_widget_show(bandoptionmenu);
	}
	modesradiobutton1 = lookup_widget(preferencesdialog, "modesradiobutton1");
	toggletrue = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(modesradiobutton1));
	if (toggletrue) {
		preferences.modeseditbox = 1;
		gtk_widget_hide(modeoptionmenu);
		gtk_widget_show(modeentry);
	}
	else {
		preferences.modeseditbox = 0;
		gtk_widget_hide(modeentry);
		gtk_widget_show(modeoptionmenu);
	}

#if WANT_HAMLIB
	hamlibcheckbutton = lookup_widget(preferencesdialog, "hamlibcheckbutton");
	frequencycheckbutton = lookup_widget(preferencesdialog, "frequencycheckbutton");
	smetercheckbutton = lookup_widget(preferencesdialog, "smetercheckbutton");
	radiocomboentry = lookup_widget(preferencesdialog, "radiocomboentry");
	devicecomboentry = lookup_widget(preferencesdialog, "devicecomboentry");
	digitscomboentry = lookup_widget(preferencesdialog, "digitscomboentry");
	hamlibyes = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(hamlibcheckbutton));
	frequencyyesno = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(frequencycheckbutton));
	smeteryesno = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(smetercheckbutton));
	radio = gtk_editable_get_chars(GTK_EDITABLE(radiocomboentry), 0, -1);
	device = gtk_editable_get_chars(GTK_EDITABLE(devicecomboentry), 0, -1);
	digits = gtk_editable_get_chars(GTK_EDITABLE(digitscomboentry), 0, -1);

	if ((preferences.hamlib == 0) && hamlibyes)
	{
		/* first use of hamlib, open radio */
		rigid = get_rigid(radio);
		if (rigid != -1)
		{
			myrig = rig_init(rigid);
			strncpy(myrig->state.rigport.pathname, device, FILPATHLEN);
			retcode = rig_open(myrig);
			if (retcode != RIG_OK)
			{
				temp = g_strdup_printf ("An error occured while opening port %s: %s",
					myrig->state.rigport.pathname, rigerror(retcode));
				update_statusbar(temp);
			}
			else
			{
				hamlibtimer = gtk_timeout_add(200, (GtkFunction)get_riginfo, NULL);
			}
			preferences.radio = g_strdup(radio);
			preferences.device = g_strdup(device);
			preferences.round = atoi(digits);
			hamlibwidgets = whichhamlibwidgets(frequencyyesno, smeteryesno);
		}
		else
		{
			update_statusbar("Hamlib error: get_rigid failed");
		}
	}
	else if ((preferences.hamlib > 0) && hamlibyes)
	{
		/* see if we have selected another radio or port */
		if ((g_strcasecmp(radio, preferences.radio) != 0)
		|| (g_strcasecmp(device, preferences.device) != 0))
		{
			gdk_input_remove(hamlibtimer);
			rig_close((RIG *)myrig);
			rig_cleanup((RIG *)myrig);
			rigid = get_rigid(radio);
			myrig = rig_init(rigid);
			strncpy(myrig->state.rigport.pathname, device, FILPATHLEN);
			retcode = rig_open(myrig);
			if (retcode != RIG_OK)
			{
				temp = g_strdup_printf ("An error occured while opening port %s: %s",
					myrig->state.rigport.pathname, rigerror(retcode));
				update_statusbar(temp);
			 }
			else
			{
				hamlibtimer = gtk_timeout_add(200, (GtkFunction)get_riginfo, NULL);
			}
		}
		preferences.radio = g_strdup(radio);
		preferences.device = g_strdup(device);
		preferences.round = atoi(digits);
		hamlibwidgets = whichhamlibwidgets(frequencyyesno, smeteryesno);
	}
	else if ((preferences.hamlib > 0) && !hamlibyes)
	{
		/* hamlib disabled, close rig */
		hamlibwidgets = 0;
		gdk_input_remove(hamlibtimer);
		hamlibtimer = -1;
		rig_close((RIG *)myrig);
		rig_cleanup((RIG *)myrig);
		preferences.radio = g_strdup("Dummy");
		preferences.device = g_strdup("/dev/ttyS1");
		preferences.round = 0;
	}
	else if ((preferences.hamlib = 0) && !hamlibyes)
	{
		preferences.radio = g_strdup("Dummy");
		preferences.device = g_strdup("/dev/ttyS1");
		preferences.round = 0;
	}
	
	/* set widgets if they are changed */
	if (hamlibwidgets != preferences.hamlib)
	{
		sethamlibwidgets(hamlibwidgets);
		/* pixmaps need to be loaded if s-meter is enabled and was disabled before */
		if (((hamlibwidgets ==3) || (hamlibwidgets == 4))
			&& ((preferences.hamlib ==0) || (preferences.hamlib == 1) || (preferences.hamlib == 2)))
		{
			drawingarea = lookup_widget(mainwindow, "smeterdrawingarea");
			loadsmeter(drawingarea);
		}
		preferences.hamlib = hamlibwidgets;
	}
	g_free(radio);
	g_free(device);
	g_free(digits);
#endif

	/* clock */
	clockcheckbutton = lookup_widget(preferencesdialog, "clockcheckbutton");
	toggletrue = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(clockcheckbutton));
	set_clock(toggletrue);

	g_free(temp);
	gtk_widget_set_sensitive(mainwindow, 1);
	gtk_widget_destroy(preferencesdialog);
}

void on_hamlibcheckbutton_toggled (GtkToggleButton *togglebutton, gpointer user_data) {
	GtkWidget *hamlibframe;
	gboolean toggletrue;

	hamlibframe = lookup_widget(preferencesdialog, "hamlibframe");
	toggletrue = gtk_toggle_button_get_active(togglebutton);
	if (toggletrue)
		gtk_widget_set_sensitive(hamlibframe, 1);
	else
		gtk_widget_set_sensitive(hamlibframe, 0);
}

void on_preferencescancelbutton_clicked (GtkButton *button, gpointer user_data) {
	gtk_widget_set_sensitive(mainwindow, 1);
	gtk_widget_destroy(gtk_widget_get_toplevel(GTK_WIDGET(button)));
}

gboolean on_preferencesdialog_delete_event (GtkWidget *widget, GdkEvent *event, gpointer user_data) {
	gtk_widget_set_sensitive(mainwindow, 1);
	return FALSE;
}

void on_bandsradiobutton_toggled (GtkToggleButton *togglebutton, gpointer user_data) {
	GtkWidget *bandslabel, *bandsentry;

	bandslabel = lookup_widget(preferencesdialog, "bandslabel");
	bandsentry = lookup_widget(preferencesdialog, "bandsentry");
	switch(atoi(user_data)) {
		case 1:
			gtk_widget_set_sensitive(bandslabel, 0);
			gtk_widget_set_sensitive(bandsentry, 0);
		break;
		case 2:
			gtk_widget_set_sensitive(bandslabel, 1);
			gtk_widget_set_sensitive(bandsentry, 1);
		break;
	}
}

void on_modesradiobutton_toggled (GtkToggleButton *togglebutton, gpointer user_data) {
	GtkWidget *modeslabel, *modesentry;

	modeslabel = lookup_widget(preferencesdialog, "modeslabel");
	modesentry = lookup_widget(preferencesdialog, "modesentry");
	switch(atoi(user_data)) {
		case 1:
			gtk_widget_set_sensitive(modeslabel, 0);
			gtk_widget_set_sensitive(modesentry, 0);
		break;
		case 2:
			gtk_widget_set_sensitive(modeslabel, 1);
			gtk_widget_set_sensitive(modesentry, 1);
		break;
	}

}

/* show color selection dialog and apply the theme color */
void on_themebutton_clicked(GtkButton *button, gpointer user_data)
{
	GtkWidget *colorselectiondialog;
	gdouble rgb[3];
	gchar *themecolor;
	GdkColor color;
	
	colorselectiondialog = create_colorselectiondialog();
	gtk_widget_destroy(GTK_COLOR_SELECTION_DIALOG(colorselectiondialog)->help_button);

	themecolor = color_parse(preferences.themecolor);
	gdk_color_parse(themecolor, &color);
	g_free(themecolor);
	rgb[0] = color.red / 65535.0;
	rgb[1] = color.green / 65535.0;
	rgb[2] = color.blue / 65535.0;
	gtk_color_selection_set_color(GTK_COLOR_SELECTION(GTK_COLOR_SELECTION_DIALOG(colorselectiondialog)->colorsel), rgb);

	gtk_widget_set_sensitive(preferencesdialog, 0);
	gtk_widget_show(colorselectiondialog);
}

/* change path for the logs */
void on_pathbutton_clicked(GtkButton *button, gpointer user_data)
{
	GtkWidget *pathselection;

	pathselection = create_pathselection();
	gtk_entry_set_editable(GTK_ENTRY(GTK_FILE_SELECTION(pathselection)->selection_entry), FALSE);
	gtk_widget_set_sensitive(preferencesdialog, 0);
	gtk_widget_show(pathselection);
}

/* SETUP DIALOG */
void on_setupbutton_clicked (GtkButton *button, gpointer user_data) {
	gtk_widget_destroy(gtk_widget_get_toplevel(GTK_WIDGET(button)));
	gtk_widget_show(mainwindow);
	gdk_window_move_resize(mainwindow->window, preferences.x, preferences.y, preferences.width, preferences.height);
}

gboolean on_setupdialog_delete_event(GtkWidget *widget, GdkEvent *event, gpointer user_data) {
	gtk_widget_show(mainwindow);
	gdk_window_move_resize(mainwindow->window, preferences.x, preferences.y, preferences.width, preferences.height);
	return FALSE;
}

/* SAVE DIALOG */
void on_saveokbutton_clicked (GtkButton *button, gpointer user_data) {
	gtk_exit(0);
}

void on_savecancelbutton_clicked	(GtkButton *button, gpointer user_data) {
	gtk_widget_destroy(gtk_widget_get_toplevel(GTK_WIDGET(button)));
	gtk_widget_set_sensitive(mainwindow, 1);
}

gboolean on_savedialog_delete_event(GtkWidget *widget, GdkEvent *event, gpointer user_data){
	gtk_widget_set_sensitive(mainwindow, 1);
	return FALSE;
}

/* PRINT DIALOG */
void on_printokbutton_clicked (GtkButton *button, gpointer user_data) {
	GtkWidget *printallradiobutton, *printrangeradiobutton, *printlowentry,
		*printhighentry, *sizecomboentry, *pointcomboentry,
		*conteststyleradiobutton, *normalstyleradiobutton, 
		*filedestinationradiobutton, *printerdestinationradiobutton;
	gchar *temp, *texpoint, *texorientation, *texsize, *texfile, *printstring = NULL, **item;
	FILE *fp;
	gint i, j, k = 0, printhigh, printlow;
	gboolean printok = FALSE;
	logtype *logwindow;

	printallradiobutton = lookup_widget(printdialog, "printallradiobutton");
	printrangeradiobutton = lookup_widget(printdialog, "printrangeradiobutton");
	printlowentry = lookup_widget(printdialog, "printlowentry");
	printhighentry = lookup_widget(printdialog, "printhighentry");
	sizecomboentry = lookup_widget(printdialog, "sizecomboentry");
	pointcomboentry = lookup_widget(printdialog, "pointcomboentry");
	filedestinationradiobutton = lookup_widget(printdialog, "filedestinationradiobutton");
	printerdestinationradiobutton = lookup_widget(printdialog, "printerdestinationradiobutton");
	normalstyleradiobutton = lookup_widget(printdialog, "normalstyleradiobutton");
	conteststyleradiobutton = lookup_widget(printdialog, "conteststyleradiobutton");

	/* save settings of the print dialog */
	if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(printallradiobutton)))
		preferences.printall = 0;
	else 
		preferences.printall = 1;
	temp = gtk_editable_get_chars(GTK_EDITABLE(printlowentry), 0, -1);
	printlow = atoi(temp);
	temp = gtk_editable_get_chars(GTK_EDITABLE(printhighentry), 0, -1);
	printhigh = atoi(temp);
	temp = gtk_editable_get_chars(GTK_EDITABLE(sizecomboentry), 0, -1);
	if (g_strcasecmp(temp, "a4") == 0) preferences.printsize = 0;
	else preferences.printsize = 1;
	temp = gtk_editable_get_chars(GTK_EDITABLE(pointcomboentry), 0, -1);
	if (g_strcasecmp(temp, "10") == 0) preferences.printpoint = 0;
	else if (g_strcasecmp(temp, "11") == 0) preferences.printpoint = 1;
	else if (g_strcasecmp(temp, "12") == 0) preferences.printpoint = 2;
	if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(printerdestinationradiobutton)))
		preferences.printdestination = 0;
	else 
		preferences.printdestination = 1;
	if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(normalstyleradiobutton)))
		preferences.printstyle = 0;
	else 
		preferences.printstyle = 1;

	/* tex strings that may differ */
	if (preferences.printsize == 0) texsize = g_strdup("a4paper");
	else texsize = g_strdup("letterpaper");
	if (preferences.printstyle == 0) texorientation = g_strdup("landscape");
	else texorientation = g_strdup("portrait");
	if (preferences.printpoint == 0) texpoint = g_strdup("10pt");
	else if (preferences.printpoint == 1) texpoint = g_strdup("11pt");
	else texpoint = g_strdup("12pt");

	logwindow = g_list_nth_data(logwindowlist,
		gtk_notebook_get_current_page(GTK_NOTEBOOK(mainnotebook)));
	gtk_clist_freeze(GTK_CLIST(logwindow->clist));

	item = g_new0(gchar *, 11);

	texfile = g_strconcat(xlogdir, G_DIR_SEPARATOR_S, "log.tex", NULL);
	fp = fopen(texfile, "w");
	if (fp != NULL)
	{
		fprintf(fp, "\\documentclass[%s,%s,%s]{article}\n", texsize, texorientation, texpoint);
		fprintf(fp, "\\pagestyle{empty}\n");
		fprintf(fp, "\\usepackage[%s,left=15mm,right=15mm,top=10mm,bottom=10mm]{geometry}\n", texsize);
		fprintf(fp, "\\begin{document}\n");
		if (preferences.printstyle == 0)
			fprintf(fp, "\\begin{tabular}{||l|l|l|l|l|l|l|l||l|c|c||}\n");
		else
			fprintf(fp, "\\begin{tabular}{||l|l|l|l|l|l|l|l||l|l||}\n");
		fprintf(fp, "\\hline\n");
		if (preferences.printstyle == 0)
			fprintf(fp, "\\textbf{Nr} &\\textbf{Date} &\\textbf{GMT} &\\textbf{Callsign} &\\textbf{MHz} &\\textbf{Mode} &\\textbf{TX} &\\textbf{RX} &\\textbf{Out} &\\textbf{In} &\\textbf{Remarks}\\\\\n");
		else
			fprintf(fp, "\\textbf{Nr} &\\textbf{Date} &\\textbf{GMT} &\\textbf{Callsign} &\\textbf{MHz} &\\textbf{Mode} &\\textbf{TX} &\\textbf{RX} &\\textbf{Multiplier} &\\textbf{Points}\\\\\n");
		fprintf(fp, "\\hline\n");
		fprintf(fp, "\\hline\n");

		if (preferences.printall == 0)
		{
			printlow = 1;
			printhigh = logwindow->qsos;
		}

		if (GTK_CLIST(logwindow->clist)->rows > 0)
		{
			for(i = GTK_CLIST(logwindow->clist)->rows - printlow; 
				i >= GTK_CLIST(logwindow->clist)->rows - printhigh; i--)
			{
				for (j = 0; j < 11; j++)
					gtk_clist_get_text(GTK_CLIST(logwindow->clist), i, j, &item[j]);
					if (preferences.printstyle == 0)
					fprintf(fp, "%s&%s&%s&\\textbf{%s}&%s&%s&%s&%s&%s&%s&%s\\\\\n",
						item[0], item[1], item[2], item[3], item[4], item[5], item[6], item[7],
							item[8], item[9], item[10]);
					else
					fprintf(fp, "%d&%s&%s&\\textbf{%s}&%s&%s&%s&%s&~&~\\\\\n",
						k+1, item[1], item[2], item[3], item[4], item[5], item[6], item[7]);
				k++;
				if ((k % 30 == 0 && preferences.printstyle == 0 && k != printhigh - printlow + 1) ||
						(k % 50 == 0 && preferences.printstyle == 1 && k != printhigh - printlow + 1)) {
					fprintf(fp, "\\hline\n");
					fprintf(fp, "\\end{tabular}\n");
					fprintf(fp, "\\begin{center}\n");
					fprintf(fp, "\\emph{Printed by xlog version %d.%d}\n", state.majorversion, state.minorversion);
					fprintf(fp, "\\end{center}\n");
					fprintf(fp, "\\newpage\n");
					if (preferences.printstyle == 0)
						fprintf(fp, "\\begin{tabular}{||l|l|l|l|l|l|l|l||l|c|c||}\n");
					else
						fprintf(fp, "\\begin{tabular}{||l|l|l|l|l|l|l|l||l|l||}\n");
					fprintf(fp, "\\hline\n");
					if (preferences.printstyle == 0)
						fprintf(fp, "\\textbf{Nr} &\\textbf{Date} &\\textbf{GMT} &\\textbf{Callsign} &\\textbf{MHz} &\\textbf{Mode} &\\textbf{TX} &\\textbf{RX} &\\textbf{Remarks} &\\textbf{Out} &\\textbf{In}\\\\\n");
					else
						fprintf(fp, "\\textbf{Nr} &\\textbf{Date} &\\textbf{GMT} &\\textbf{Callsign} &\\textbf{MHz} &\\textbf{Mode} &\\textbf{TX} &\\textbf{RX} &\\textbf{Multiplier} &\\textbf{Points}\\\\\n");
					fprintf(fp, "\\hline\n");
					fprintf(fp, "\\hline\n");
				}
			}
		}

		fprintf(fp, "\\hline\n");
		fprintf(fp, "\\end{tabular}\n");
		fprintf(fp, "\\begin{center}\n");
		fprintf(fp, "\\emph{Printed by xlog version %s}\n", VERSION);
		fprintf(fp, "\\end{center}\n");
		fprintf(fp, "\\end{document}\n");
		fclose(fp);
	}

	gtk_clist_thaw(GTK_CLIST(logwindow->clist));
	g_free(item);

	if ((printok = latexavailable()) == FALSE)
	{
		update_statusbar(_("Latex is not available, install the tetex package"));
	}

	if (printok)
	{
		if (preferences.printdestination == 0 && preferences.printstyle == 0)
			printstring = g_strdup_printf("cd %s; latex log.tex; dvips -t landscape log.dvi", xlogdir);
		else if (preferences.printdestination == 1 && preferences.printstyle == 0)
			printstring = g_strdup_printf("cd %s; latex log.tex; dvips -t landscape -o log.ps log.dvi", xlogdir);
		else if (preferences.printdestination == 0 && preferences.printstyle == 1)
			printstring = g_strdup_printf("cd %s; latex log.tex; dvips log.dvi", xlogdir);
		else if (preferences.printdestination == 1 && preferences.printstyle == 1)
			printstring = g_strdup_printf("cd %s; latex log.tex; dvips -o log.ps log.dvi", xlogdir);

		if (preferences.printdestination == 0)
			temp = g_strdup_printf(_("Log send to printer"));
		else
			temp = g_strdup_printf(_("Log printed to %s%slog.ps"), G_DIR_SEPARATOR_S, xlogdir);

		system(printstring);
		update_statusbar(temp);
		g_free(printstring);
	}

	g_free(texsize);
	g_free(texorientation);
	g_free(texpoint);
	g_free(texfile);
	g_free(temp);


	gtk_widget_set_sensitive(mainwindow, 1);
	gtk_widget_destroy(gtk_widget_get_toplevel(GTK_WIDGET(button)));

}

void on_printcancelbutton_clicked (GtkButton *button, gpointer	user_data) {
	gtk_widget_set_sensitive(mainwindow, 1);
	gtk_widget_destroy(gtk_widget_get_toplevel(GTK_WIDGET(button)));
}

gboolean on_printdialog_delete_event (GtkWidget	*widget, GdkEvent	*event, gpointer user_data) {
	gtk_widget_set_sensitive(mainwindow, 1);
	return FALSE;
}

void on_printradiobutton_toggled (GtkToggleButton *togglebutton, gpointer user_data) {
	GtkWidget *printlowentry, *printrangelabel, *printhighentry;

	printlowentry = lookup_widget(printdialog, "printlowentry");
	printhighentry = lookup_widget(printdialog, "printhighentry");
	printrangelabel = lookup_widget(printdialog, "printrangelabel");
	switch(atoi(user_data)) {
		case 1:
			gtk_widget_set_sensitive(printlowentry, 0);
			gtk_widget_set_sensitive(printhighentry, 0);
			gtk_widget_set_sensitive(printrangelabel, 0);
		break;
		case 2:
			gtk_widget_set_sensitive(printlowentry, 1);
			gtk_widget_set_sensitive(printhighentry, 1);
			gtk_widget_set_sensitive(printrangelabel, 1);
		break;
	}
}


/* SEARCH DIALOG */
void on_searchokbutton_clicked (GtkButton *button, gpointer user_data) {
	gchar *searchstr, *upsearchstr, *callsign, *logentry, *labeltext;
	GtkWidget *searchdialog, *searchresultdialog, *searchcomboentry,
		 *searchresultclist, *resultlabel, *searchresultscrolledwindow;
	gint i, row, column;
	gboolean result = FALSE;
	GList *node;
	logtype *logwindow;

	searchdialog = gtk_widget_get_toplevel(GTK_WIDGET(button));
	searchcomboentry = lookup_widget(searchdialog, "searchcomboentry");	
	searchstr = gtk_editable_get_chars(GTK_EDITABLE(searchcomboentry), 0 , -1);

	node = g_list_find_custom(searchhistory, g_strdup(searchstr), (GCompareFunc)strncmp);
	if (!node) searchhistory = g_list_prepend(searchhistory, g_strdup(searchstr));
	if (g_list_length(searchhistory) > SEARCHHISTORY)
		searchhistory = g_list_remove(searchhistory, g_list_last(searchhistory)->data);

	upsearchstr = g_strdup(searchstr);
	g_strup(upsearchstr);
	searchresultdialog = create_searchresultdialog();
	searchresultclist = lookup_widget(searchresultdialog, "searchresultclist");
	for (column = 0; column < 11; column++)
		gtk_clist_set_column_auto_resize(GTK_CLIST(searchresultclist), column, TRUE);
	
	for (i = 0; i < g_list_length(logwindowlist); i++)
	{
		logwindow = g_list_nth_data(logwindowlist, i);
		for (row = 0; row < logwindow->qsos; row++) {
			gtk_clist_get_text(GTK_CLIST(logwindow->clist), row, 3, &callsign);
			if (strstr(callsign, upsearchstr))
			{
				qso[0] = g_strdup(logwindow->logname);
				for (column = 0; column < 11; column++)
				{
					gtk_clist_get_text(GTK_CLIST(logwindow->clist), row, column, &logentry);
					qso[column + 1] = g_strdup(logentry);
				}
			result = TRUE;
			gtk_clist_prepend(GTK_CLIST(searchresultclist), qso); 
			}
		}
	}
	resultlabel = lookup_widget(searchresultdialog, "resultlabel");
	searchresultscrolledwindow = lookup_widget(searchresultdialog, "searchresultscrolledwindow");
	if (result) {
		gtk_widget_show(searchresultclist);
	    gtk_widget_show(searchresultscrolledwindow);
		gtk_widget_hide(resultlabel);
		gtk_window_set_default_size(GTK_WINDOW(searchresultdialog), 750, 300);
	}
	else {
		labeltext = g_strdup_printf(_("\'%s\' was not found"), searchstr);
		gtk_label_set_text(GTK_LABEL(resultlabel), labeltext);
		g_free(labeltext);
		gtk_widget_show(resultlabel);
		gtk_widget_hide(searchresultscrolledwindow);
		gtk_widget_hide(searchresultclist);
		gtk_window_set_default_size(GTK_WINDOW(searchresultdialog), 300, 200);
	}
	g_free(searchstr);
	g_free(upsearchstr);
	gtk_widget_show(searchresultdialog);
	gtk_widget_set_sensitive(mainwindow, 1);
	gtk_widget_destroy(searchdialog);
}


void on_searchcancelbutton_clicked (GtkButton *button, gpointer user_data) {
	gtk_widget_set_sensitive(mainwindow, 1);
	gtk_widget_destroy(gtk_widget_get_toplevel(GTK_WIDGET(button)));
}

gboolean on_searchdialog_delete_event(GtkWidget *widget, GdkEvent *event, gpointer user_data)
{
	gtk_widget_set_sensitive(mainwindow, 1);
	return FALSE;
}

/* SEARCHRESULT DIALOG */
void on_searchresultokbutton_clicked(GtkButton *button, gpointer user_data)
{
	gtk_widget_destroy(gtk_widget_get_toplevel(GTK_WIDGET(button)));
}

/* A row is selected in the search dialog,
 * select the coresponding row in the log */
void on_searchresultclist_select_row(GtkCList *clist, gint row, gint column, GdkEvent *event, gpointer user_data)
{
	gchar *nr, *lognr, *logn;
	gint i, logrow = -1;
	logtype *logwindow = NULL;
	gboolean qsofound = FALSE;

	gtk_clist_get_text(GTK_CLIST(clist), row, 1, &nr);
	gtk_clist_get_text(GTK_CLIST(clist), row, 0, &logn);

	for (i = 0; i < g_list_length(logwindowlist); i++)
	{
		logwindow = g_list_nth_data(logwindowlist, i);
		for (logrow = (logwindow->qsos) - 1; logrow >= 0 ; logrow--) {
			gtk_clist_get_text(GTK_CLIST(logwindow->clist), logrow, 0, &lognr);
			if ((g_strcasecmp(nr, lognr) == 0) && (g_strcasecmp(logn, logwindow->logname) == 0))
			{
				qsofound = TRUE;
				break;
			}
		}
		if (qsofound) break;
	}
	if (qsofound)
	{
		gtk_notebook_set_page(GTK_NOTEBOOK(mainnotebook), logwindow->id);
		gtk_clist_moveto(GTK_CLIST(logwindow->clist), logrow, 0, 0.5, 0);
		gtk_clist_select_row(GTK_CLIST(logwindow->clist), logrow, 0);
	}
}


/* COLOR SELECTION DIALOG */
void on_colorokbutton_clicked(GtkButton *button, gpointer user_data){
	GtkWidget *colorselectiondialog, *examplelabel, *exampleframe;
	gdouble rgb[3];
	gchar *colorred, *colorgreen, *colorblue, *color;
        
	colorselectiondialog = gtk_widget_get_toplevel(GTK_WIDGET(button));
	gtk_color_selection_get_color(GTK_COLOR_SELECTION(GTK_COLOR_SELECTION_DIALOG(colorselectiondialog)->colorsel), rgb);
	colorred = g_strdup_printf("%04X", (gint)(rgb[0] * 65535));
	colorgreen = g_strdup_printf("%04X", (gint)(rgb[1] * 65535));
	colorblue = g_strdup_printf("%04X", (gint)(rgb[2] * 65535));
	color = g_strconcat(colorred, ",", colorgreen, ",", colorblue, NULL);
	exampleframe = lookup_widget(preferencesdialog, "exampleframe");
	setframestyle(exampleframe, color);
	examplelabel = lookup_widget(preferencesdialog, "examplelabel");
	setlabelstyle(examplelabel, color);
	g_free(colorblue);
	g_free(colorgreen);
	g_free(colorred);
	g_free(color);
	gtk_widget_set_sensitive(preferencesdialog, 1);
	gtk_widget_destroy(gtk_widget_get_toplevel(GTK_WIDGET(button)));
}


void on_colorcancelbutton_clicked(GtkButton *button, gpointer user_data)
{
	gtk_widget_set_sensitive(preferencesdialog, 1);
	gtk_widget_destroy(gtk_widget_get_toplevel(GTK_WIDGET(button)));
}

gboolean on_colorselectiondialog_delete_event(GtkWidget *widget, GdkEvent *event, gpointer user_data)
{
	gtk_widget_set_sensitive(preferencesdialog, 1);
	return FALSE;
}

/* NEW LOG DIALOG */
void on_newlogokbutton_clicked(GtkButton *button, gpointer user_data)
{
	GtkWidget *newlogentry, *columnlabel, *newlogdialog;
	gchar *entry, *filename;
	logtype *logwindow;
	gint i, j;
	LOGDB *lp;
	gboolean logopen = FALSE, logexist = FALSE;
	GString *logs;

	newlogdialog = gtk_widget_get_toplevel(GTK_WIDGET(button));
	newlogentry = lookup_widget(newlogdialog, "newlogentry");
	entry = gtk_editable_get_chars(GTK_EDITABLE(newlogentry), 0, -1);

	if (strlen(entry) > 0)
	{
		for (i = 0; i < g_list_length(logwindowlist); i++)
		{	/* see if the log is already open */
			logwindow = g_list_nth_data(logwindowlist, i);
			if (g_strcasecmp(entry, logwindow->logname) == 0)
			{
				gtk_notebook_set_page(GTK_NOTEBOOK(mainnotebook), i);
				logopen = TRUE;
				break;
			}
		}
		if (!logopen)
		{ /* check if the log exists and open it */
			filename = g_strconcat(entry, ".xlog", 0);
			logs = getlogs(preferences.savedir);
			if (((logs->len) > 0) && (strstr(logs->str, filename)))
			{
				logexist = TRUE;
				filename = g_strconcat(preferences.savedir, G_DIR_SEPARATOR_S, entry, ".xlog", NULL);
				lp = log_file_open(filename, TYPE_FLOG);
				if (lp)
				{
					logwindow = new_logwindow(state.logwindows++);
					logwindow->logname = g_strdup(entry);
					logwindow->scrolledwindow = gtk_scrolled_window_new (NULL, NULL);
					gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(logwindow->scrolledwindow),
						GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
					gtk_widget_show(logwindow->scrolledwindow);
					logwindow->clist = gtk_clist_new(lp->column_nr+1);
					gtk_widget_show(logwindow->clist);
					gtk_signal_connect (GTK_OBJECT(logwindow->clist), "select_row",
						GTK_SIGNAL_FUNC (on_log_select_row), NULL);
					gtk_container_add(GTK_CONTAINER(logwindow->scrolledwindow), logwindow->clist);
					for (j = 0; j <= lp->column_nr; j++)
						gtk_clist_set_column_auto_resize(GTK_CLIST(logwindow->clist), j, TRUE);
					gtk_clist_column_titles_show (GTK_CLIST (logwindow->clist));
					columnlabel = gtk_label_new("NR");
					gtk_clist_set_column_widget(GTK_CLIST(logwindow->clist), 0, columnlabel);
					for (j = 0; j <= lp->column_nr; j++)
					{
						columnlabel = gtk_label_new(strfield(lp->column_fields[j]));
						gtk_clist_set_column_widget(GTK_CLIST(logwindow->clist), j + 1, columnlabel);
						gtk_clist_column_title_passive(GTK_CLIST(logwindow->clist), j);
					}
					logwindow->label = gtk_label_new(entry);
					gtk_widget_show(logwindow->label);
					gtk_notebook_append_page(GTK_NOTEBOOK(mainnotebook), logwindow->scrolledwindow, logwindow->label);
					gtk_misc_set_padding(GTK_MISC(logwindow->label), 10, 0);
					setlabelstyle(logwindow->label, preferences.themecolor);
					log_file_qso_foreach(lp, fillin_clist, logwindow);
					log_file_close(lp);
					logwindowlist = g_list_append(logwindowlist, logwindow);
					gtk_notebook_set_page(GTK_NOTEBOOK(mainnotebook), state.logwindows - 1);
				}
				g_free(filename);
				g_string_free(logs, TRUE);
			}
		}
		if ((!logopen) && (!logexist))
		{   /* create a new log */
			filename = g_strconcat(preferences.savedir, G_DIR_SEPARATOR_S, entry, ".xlog", NULL);
			lp = log_file_create(filename, TYPE_FLOG, qso_field_nr, qso_fields, qso_widths);
			if (lp)
			{
				logwindow = new_logwindow(state.logwindows++);
				logwindow->logname = g_strdup(entry);
				logwindow->scrolledwindow = gtk_scrolled_window_new (NULL, NULL);
				gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(logwindow->scrolledwindow),
					GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
				gtk_widget_show(logwindow->scrolledwindow);
				logwindow->clist = gtk_clist_new(lp->column_nr+1);
				gtk_widget_show(logwindow->clist);
				gtk_signal_connect (GTK_OBJECT(logwindow->clist), "select_row",
					GTK_SIGNAL_FUNC (on_log_select_row), NULL);
				gtk_container_add(GTK_CONTAINER(logwindow->scrolledwindow), logwindow->clist);
				for (j = 0; j <= lp->column_nr; j++)
					gtk_clist_set_column_auto_resize(GTK_CLIST(logwindow->clist), j, TRUE);
				gtk_clist_column_titles_show (GTK_CLIST (logwindow->clist));
				columnlabel = gtk_label_new("NR");
				gtk_clist_set_column_widget(GTK_CLIST(logwindow->clist), 0, columnlabel);
				for (j = 0; j <= lp->column_nr; j++)
				{
					columnlabel = gtk_label_new(strfield(lp->column_fields[j]));
					gtk_clist_set_column_widget(GTK_CLIST(logwindow->clist), j + 1, columnlabel);
					gtk_clist_column_title_passive(GTK_CLIST(logwindow->clist), j);
				}
				logwindow->label = gtk_label_new(entry);
				gtk_widget_show(logwindow->label);
				gtk_notebook_append_page(GTK_NOTEBOOK(mainnotebook), logwindow->scrolledwindow, logwindow->label);
				gtk_misc_set_padding(GTK_MISC(logwindow->label), 10, 0);
				setlabelstyle(logwindow->label, preferences.themecolor);
				log_file_close(lp);
				logwindowlist = g_list_append(logwindowlist, logwindow);
				gtk_notebook_set_page(GTK_NOTEBOOK(mainnotebook), state.logwindows - 1);
			}
			g_free(filename);
			}
		g_free(entry);
	}

	gtk_widget_set_sensitive(mainwindow, 1);
	gtk_widget_destroy(newlogdialog);
}


void on_newlogcancelbutton_clicked(GtkButton *button, gpointer user_data)
{
	gtk_widget_set_sensitive(mainwindow, 1);
	gtk_widget_destroy(gtk_widget_get_toplevel(GTK_WIDGET(button)));
}

gboolean on_newlogdialog_delete_event(GtkWidget *widget, GdkEvent *event, gpointer user_data)
{
	gtk_widget_set_sensitive(mainwindow, 1);
	return FALSE;
}

/* FILE SELECTION DIALOG */
/* open a log, if it exist jump to the page in the notebook */
void on_fileselection_ok_button_clicked(GtkButton *button, gpointer user_data)
{
	GtkWidget *fileselection, *columnlabel;
	gchar *logn, *sellog;
	gpointer filen;
	gboolean logexist = FALSE;
	gint i, j;
	logtype *logwindow;
	LOGDB *lp;

	fileselection = gtk_widget_get_toplevel(GTK_WIDGET(button));
	/* don't free sellog */
	sellog = gtk_file_selection_get_filename(GTK_FILE_SELECTION(fileselection));
	filen = basename(sellog);
	logn = logname((gchar *)filen);
	for (i = 0; i < g_list_length(logwindowlist); i++)
	{
		logwindow = g_list_nth_data(logwindowlist, i);
		if (g_strcasecmp(logn, logwindow->logname) == 0)
		{
			gtk_notebook_set_page(GTK_NOTEBOOK(mainnotebook), i);
			logexist = TRUE;
			break;
		}
	}
	if (!logexist)
	{
		logwindow = new_logwindow(state.logwindows++);
		logwindow->logname = g_strdup(logn);
		lp = log_file_open(sellog, TYPE_FLOG);
		if (lp)
		{
			logwindow->scrolledwindow = gtk_scrolled_window_new (NULL, NULL);
			gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(logwindow->scrolledwindow),
				GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
			gtk_widget_show(logwindow->scrolledwindow);
			logwindow->clist = gtk_clist_new(lp->column_nr+1);
			gtk_widget_show(logwindow->clist);
			gtk_signal_connect (GTK_OBJECT(logwindow->clist), "select_row",
				GTK_SIGNAL_FUNC (on_log_select_row), NULL);
			gtk_container_add(GTK_CONTAINER(logwindow->scrolledwindow), logwindow->clist);
			for (j = 0; j <= lp->column_nr; j++)
				gtk_clist_set_column_auto_resize(GTK_CLIST(logwindow->clist), j, TRUE);
			gtk_clist_column_titles_show (GTK_CLIST (logwindow->clist));
			columnlabel = gtk_label_new("NR");
			gtk_clist_set_column_widget(GTK_CLIST(logwindow->clist), 0, columnlabel);
			for (j = 0; j <= lp->column_nr; j++)
			{
				columnlabel = gtk_label_new(strfield(lp->column_fields[j]));
				gtk_clist_set_column_widget(GTK_CLIST(logwindow->clist), j + 1, columnlabel);
				gtk_clist_column_title_passive(GTK_CLIST(logwindow->clist), j);
			}
			logwindow->label = gtk_label_new(logn);
			gtk_widget_show(logwindow->label);
			gtk_notebook_append_page(GTK_NOTEBOOK(mainnotebook), logwindow->scrolledwindow, logwindow->label);
			gtk_misc_set_padding(GTK_MISC(logwindow->label), 10, 0);
			setlabelstyle(logwindow->label, preferences.themecolor);
			log_file_qso_foreach(lp, fillin_clist, logwindow);
			log_file_close(lp);
			logwindowlist = g_list_append(logwindowlist, logwindow);
			gtk_notebook_set_page(GTK_NOTEBOOK(mainnotebook), state.logwindows - 1);
		}
	}
	g_free(logn);
	gtk_widget_destroy(fileselection);
	gtk_widget_set_sensitive(mainwindow, 1);
}


void
on_fileselection_cancel_button_clicked(GtkButton *button, gpointer user_data)
{
	gtk_widget_destroy(gtk_widget_get_toplevel(GTK_WIDGET(button)));
	gtk_widget_set_sensitive(mainwindow, 1);
}

gboolean on_fileselection_delete_event(GtkWidget *widget, GdkEvent *event, gpointer user_data)
{
	gtk_widget_set_sensitive(mainwindow, 1);
	return FALSE;
}

/* PATH SELECTION DIALOG */
gboolean on_pathselection_delete_event(GtkWidget *widget, GdkEvent *event, gpointer user_data)
{
	gtk_widget_set_sensitive(preferencesdialog, 1);
	return FALSE;
}


void on_pathselection_ok_button_clicked(GtkButton *button, gpointer user_data)
{
	GtkWidget *pathselection, *pathentry;
	gchar *selpath;

	pathselection = gtk_widget_get_toplevel(GTK_WIDGET(button));
	selpath = gtk_file_selection_get_filename(GTK_FILE_SELECTION(pathselection));
	pathentry = lookup_widget(preferencesdialog, "pathentry");
	gtk_entry_set_text(GTK_ENTRY(pathentry), selpath);
	gtk_widget_destroy(pathselection);
	gtk_widget_set_sensitive(preferencesdialog, 1);
}


void on_pathselection_cancel_button_clicked (GtkButton *button, gpointer user_data)
{
	gtk_widget_destroy(gtk_widget_get_toplevel(GTK_WIDGET(button)));
	gtk_widget_set_sensitive(preferencesdialog, 1);
}

/* CLOSE DIALOG */
gboolean on_closedialog_delete_event(GtkWidget *widget, GdkEvent *event, gpointer user_data)
{
	gtk_widget_set_sensitive(mainwindow, 1);
	return FALSE;
}


void on_closeokbutton_clicked(GtkButton *button, gpointer user_data)
{
	logtype *logwindow;
	gint page;

	page = gtk_notebook_get_current_page(GTK_NOTEBOOK(mainnotebook));
	logwindow = g_list_nth_data(logwindowlist, page);
	logwindowlist = g_list_remove(logwindowlist, logwindow);
	g_free(logwindow->logname);
	g_free(logwindow);
	gtk_notebook_remove_page(GTK_NOTEBOOK(mainnotebook), page);
	state.logwindows--;
	gtk_widget_destroy(gtk_widget_get_toplevel(GTK_WIDGET(button)));
	gtk_widget_set_sensitive(mainwindow, 1);
}


void on_closecancelbutton_clicked(GtkButton *button, gpointer user_data)
{
	gtk_widget_destroy(gtk_widget_get_toplevel(GTK_WIDGET(button)));
	gtk_widget_set_sensitive(mainwindow, 1);
}


/* end of file */


gboolean
on_logeditorwindow_delete_event           (GtkWidget       *widget,
                                        GdkEvent        *event,
                                        gpointer         user_data)
{
	gtk_widget_set_sensitive(mainwindow, 1);
	return FALSE;
}


void
on_logeditorokbutton_clicked           (GtkButton       *button,
                                        gpointer         user_data)
{
	gtk_widget_destroy(gtk_widget_get_toplevel(GTK_WIDGET(button)));
	gtk_widget_set_sensitive(mainwindow, 1);
}


void
on_logeditorcancelbutton_clicked       (GtkButton       *button,
                                        gpointer         user_data)
{
	gtk_widget_destroy(gtk_widget_get_toplevel(GTK_WIDGET(button)));
	gtk_widget_set_sensitive(mainwindow, 1);
}


void
on_menu_logeditor_activate                 (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
	GtkWidget *logeditor;
	logtype *logwindow;

	logwindow = g_list_nth_data(logwindowlist,
		gtk_notebook_get_current_page(GTK_NOTEBOOK(mainnotebook)));

	if (GTK_CLIST(logwindow->clist)->rows == 0)
	{
		logeditor = create_logeditor();
		gtk_widget_show(logeditor);
	}
}


void
on_menu_import_activate                (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{

}


