/* GNU POP3 - a small, fast, and efficient POP3 daemon
   Copyright (C) 1999 Jakob 'sparky' Kaivo <jkaivo@nodomainname.net>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */

#include "gnu-pop3d.h"

/* Takes a string as input and returns either the remainder of the string
   after the first space, or a zero length string if no space */

char *
pop3_args (const char *cmd)
{
  int space = -1, i = 0, len;
  char *buf;

  len = strlen (cmd) + 1;
  buf = malloc (len * sizeof (char));
  if (buf == NULL)
    pop3_abquit (ERR_NO_MEM);

  while (space < 0 && i < len)
    {
      if (cmd[i] == ' ')
	space = i + 1;
      else if (cmd[i] == '\0' || cmd[i] == '\r' || cmd[i] == '\n')
	len = i;
      i++;
    }

  if (space < 0)
    buf[0] = '\0';
  else
    {
      for (i = space; i < len; i++)
	if (cmd[i] == '\0' || cmd[i] == '\r' || cmd[i] == '\n')
	  buf[i - space] = '\0';
	else
	  buf[i - space] = cmd[i];
    }

  return buf;
}

/* This takes a string and returns the string up to the first space or end of
   the string, whichever occurs first */

char *
pop3_cmd (const char *cmd)
{
  char *buf;
  int i = 0, len;

  len = strlen (cmd) + 1;
  buf = malloc (len * sizeof (char));
  if (buf == NULL)
    pop3_abquit (ERR_NO_MEM);

  for (i = 0; i < len; i++)
    {
      if (cmd[i] == ' ' || cmd[i] == '\0' || cmd[i] == '\r' || cmd[i] == '\n')
	len = i;
      else
	buf[i] = cmd[i];
    }
  buf[i - 1] = '\0';
  return buf;
}

/* Returns OK if a message actually exists and has not been marked by dele(), 
   otherwise ERR_NO_MESG */

int
pop3_mesg_exist (int mesg)
{
  if ((mesg < 0) || (mesg >= num_messages) || (messages[mesg].deleted == 1))
    return ERR_NO_MESG;

  return OK;
}

/* This is called if GNU POP needs to quit without going to the UPDATE stage.
   This is used for conditions such as out of memory, a broken socket, or
   being killed on a signal */

int
pop3_abquit (int reason)
{
  pop3_unlock ();
  if (mbox != NULL)
    fclose (mbox);
  free (messages);

  switch (reason)
    {
    case ERR_NO_MEM:
      fprintf (ofile, "-ERR Out of memory, quitting\r\n");
      syslog (LOG_ERR, "Out of memory");
      break;
    case ERR_DEAD_SOCK:
      fprintf (ofile, "-ERR Socket closed, quitting\r\n");
      syslog (LOG_ERR, "Socket closed");
      break;
    case ERR_SIGNAL:
      fprintf (ofile, "-ERR Quitting on signal\r\n");
      syslog (LOG_ERR, "Quitting on signal");
      break;
    case ERR_TIMEOUT:
      fprintf (ofile, "-ERR Session timed out\r\n");
      if (state == TRANSACTION)
	syslog (LOG_INFO, "Session timed out for user: %s", username);
      else
	syslog (LOG_INFO, "Session timed out for no user");
      break;
    default:
      fprintf (ofile, "-ERR Quitting (reason unknown)\r\n");
      syslog (LOG_ERR, "Unknown quit");
      break;
    }
  fflush (ofile);
  exit (1);
}

/* This locks the mailbox file, works with procmail locking */

int
pop3_lock (void)
{
  struct flock fl;
  fl.l_type = F_RDLCK;
  fl.l_whence = SEEK_CUR;
  fl.l_start = 0;
  fl.l_len = 0;
  lockfile = malloc (strlen (mailbox) + strlen (".lock") + 1);
  if (lockfile == NULL)
    pop3_abquit (ERR_NO_MEM);
  strcpy (lockfile, mailbox);
  strcat (lockfile, ".lock");
  lock = fopen (lockfile, "r");
  if (lock != NULL)
    {
      free (lockfile);
      lockfile = NULL;
      return ERR_MBOX_LOCK;
    }
  lock = fopen (lockfile, "w");
  if (fcntl (fileno (mbox), F_SETLK, &fl) == -1)
    {
      syslog (LOG_ERR, "%s\n", strerror (errno));
      return ERR_FILE;
    }
  return OK;
}

/* Unlocks the mailbox */

int
pop3_unlock (void)
{
  struct flock fl;
  fl.l_type = F_UNLCK;
  fl.l_whence = SEEK_CUR;
  if (lockfile != NULL)
    {
      fclose (lock);
      if (unlink (lockfile) == -1)
	{
	  syslog (LOG_ERR, "%s\n", strerror (errno));
	  return ERR_FILE;
	}
      free (lockfile);
    }
  if (mbox != NULL)
    fcntl (fileno (mbox), F_SETLK, &fl);
  return OK;
}

/* This parses the mailbox to get the number of messages, the size of each
   message, and the read offset of the first character of each message. This
   is where the real speed magic is done */

int
pop3_getsizes (void)
{
  char buf[80];
  unsigned int max_count = 0;
  num_messages = 0;

  mbox = freopen (mailbox, "r+", mbox);
  if (mbox == NULL)
    return ERR_FILE;

  if (messages == NULL)
    {
      messages = malloc (sizeof (message) * 10);
      if (messages == NULL)
	pop3_abquit (ERR_NO_MEM);
      max_count = 10;
    }

  while (fgets (buf, 80, mbox))
    {
      if (!strncmp (buf, "From ", 5))
	{
	  while (strchr (buf, '\n') == NULL)
	    fgets (buf, 80, mbox);

	  num_messages++;

	  if (num_messages > max_count)
	    {
	      max_count = num_messages * 2;
	      messages = realloc (messages, 2 * max_count * sizeof (message));
	      if (messages == NULL)
		pop3_abquit (ERR_NO_MEM);
	    }

	  fgetpos (mbox, &(messages[num_messages - 1].header));
	  messages[num_messages - 1].size = 0;
	  messages[num_messages - 1].deleted = 0;
	}
      else
	messages[num_messages - 1].size += strlen (buf) + 1;
    }

  return OK;
}

/* Prints out usage information and exits the program */

void
pop3_usage (char *argv0)
{
  printf ("Usage: %s [OPTIONS]\n", argv0);
  printf ("Runs the GNU POP3 daemon.\n\n");
  printf ("  -d, --daemon=MAXCHILDREN runs in daemon mode with a maximum\n");
  printf ("                           of MAXCHILDREN child processes\n");
  printf ("  -h, --help               display this help and exit\n");
  printf ("  -i, --inetd              runs in inetd mode (default)\n");
  printf ("  -p, --port=PORT          specifies port to listen on, implies -d\n");
  printf ("                           defaults to 110, which need not be specified\n");
  printf ("  -t, --timeout=TIMEOUT    sets idle timeout to TIMEOUT seconds\n");
  printf ("                           TIMEOUT must be at least 600 (10 minutes) or\n");
  printf ("                           it will be disabled\n");
  printf ("  -v, --version            display version information and exit\n");
  printf ("\nReport bugs to gnu-pop-list@nodomainname.net\n");
  exit (0);
}

/* Default signal handler to call the pop3_abquit() function */

void
pop3_signal (int signal)
{
  if (signal == SIGCHLD)
    {
      int stat;
      while (waitpid (-1, &stat, WNOHANG) > 0);
      return;
    }
  else
    pop3_abquit (ERR_SIGNAL);
}

/* Gets a line of input from the client */

char *
pop3_readline (int fd)
{
  fd_set rfds;
  struct timeval tv;
  char buf[1024], *ret = NULL;
  int available;

  FD_ZERO (&rfds);
  FD_SET (fd, &rfds);
  tv.tv_sec = timeout;
  tv.tv_usec = 0;
  memset (buf, '\0', 1024);

  while (strchr (buf, '\n') == NULL)
    {
      if (timeout > 0)
	{
	  available = select (fd + 1, &rfds, NULL, NULL, &tv);
	  if (!available)
	    pop3_abquit (ERR_TIMEOUT);
	}

      if (read (fd, buf, 1024) < 1)
	pop3_abquit (ERR_DEAD_SOCK);

      if (ret == NULL)
	{
	  ret = malloc ((strlen (buf) + 1) * sizeof (char));
	  strcpy (ret, buf);
	}
      else
	{
	  ret = realloc (ret, (strlen (ret) + strlen (buf) + 1) * sizeof (char));
	  strcat (ret, buf);
	}
    }
  return ret;
}
