/* ****************************************************************************
  This file is part of KBabel

  Copyright (C) 1999-2000 by Matthias Kiefer
                            <matthias.kiefer@gmx.de>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

**************************************************************************** */
#include "catalogitem.h"
#include "editcmd.h"
#include "resources.h"
#include "tagextractor.h"

#include <qtextstream.h>

CatalogItem::CatalogItem()
    : _tagList(0)
{
   clear();
}

CatalogItem::CatalogItem(QString itemStr)
    : _tagList(0)
{
   QTextStream stream(&itemStr,IO_ReadOnly);

   read(stream);
}

CatalogItem::~CatalogItem()
{
    if(_tagList)
        delete _tagList;
}

CatalogItem::CatalogItem(const char* msgid, const char* msgstr,
			 const char* comment)
    : _tagList(0)
{
    _comment=comment;
    _msgid=msgid;
    _msgstr=msgstr;
    _valid=true;

	_error = None;
}


QStringList CatalogItem::msgidAsList() const
{
   QStringList list=QStringList::split("\n",msgid());

   if(msgid().left(1)=="\n")
      list.prepend("");

   if(list.isEmpty())
      list.append("");

   return list;
}


QStringList CatalogItem::msgstrAsList() const
{
   QStringList list=QStringList::split("\n",msgstr());

   if(msgstr().left(1)=="\n")
      list.prepend("");

   if(list.isEmpty())
      list.append("");

   return list;
}

QStringList CatalogItem::commentAsList() const
{
   QStringList list=QStringList::split("\n",comment());

   return list;
}


const QStringList& CatalogItem::tagList()
{
    if(!_tagList)
    {
        TagExtractor te(msgid());
        _tagList = new QStringList(te.tags());
    }

    return *_tagList;
}


bool CatalogItem::isFuzzy() const
{
   bool flag=false;
   if( _comment.contains(", fuzzy") )
     flag=true;

   return flag;
}

bool CatalogItem::isCformat() const
{
   bool flag=false;
   if(_comment.contains(", c-format"))
     flag=true;

   return flag;
}

bool CatalogItem::isUntranslated() const
{
   return _msgstr.isEmpty();
}

int CatalogItem::totalLines() const
{
   int lines=0;
   if(!_comment.isEmpty())
   {
      lines = _comment.contains("\n")+1;
   }
   int msgidLines=_msgid.contains("\n")+1;
   int msgstrLines=_msgstr.contains("\n")+1;
   if(msgidLines>1)
      msgidLines++;
   if(msgstrLines>1)
      msgstrLines++;

   lines+=( msgidLines+msgstrLines );

   return lines;
}


void CatalogItem::setSyntaxError(bool on)
{
	if(on)
		_error = _error | Syntax;
	else
		_error = _error & ~Syntax;
}

QList<EditCommand> CatalogItem::removeFuzzy(bool doIt)
{
   QList<EditCommand> editList;
   editList.setAutoDelete(false);

   QString comment=_comment;

   if(isFuzzy())
   {
       EditCommand *cmd;
       QString fuzzyStr(", fuzzy");

       int offset;
       offset=comment.find(fuzzyStr);
       while(offset>=0)
       {
          cmd = new DelTextCmd(offset,fuzzyStr);
          cmd->setPart(EditCommand::Comment);
          editList.append(cmd);

          comment.remove(offset,fuzzyStr.length());

          offset=comment.find(fuzzyStr,offset+1);
       }

       // remove empty comment lines
       if( comment.contains( QRegExp("^#\\s*$") ))
       {
          cmd = new DelTextCmd(0,comment);
          cmd->setPart(EditCommand::Comment);
          editList.append(cmd);

          comment="";
       }
       if( comment.contains( QRegExp("\n#\\s*$") ))
       {
          offset=comment.find( QRegExp("\n#\\s*$") );
          while(offset>=0)
          {
             cmd = new DelTextCmd(offset,comment.mid(offset));
             cmd->setPart(EditCommand::Comment);
             editList.append(cmd);

             comment.remove(offset,comment.length()-offset);

             offset=comment.find( QRegExp("\n#\\s*$"), offset+1 );
          }
       }
       if( comment.contains( QRegExp("\n#\\s*\n") ))
       {
          offset=comment.find( QRegExp("\n#\\s*\n") );
          while(offset>=0)
          {
             int endIndex=comment.find("\n",offset+1)+1;

             cmd = new DelTextCmd(offset,comment.mid(offset,endIndex));
             cmd->setPart(EditCommand::Comment);
             editList.append(cmd);

             comment.remove(offset,endIndex-offset);

             offset=comment.find( QRegExp("\n#\\s*\n"), offset+1 );
          }
       }

       if(doIt)
          _comment=comment;

   }

   return editList;
}



QList<EditCommand> CatalogItem::addFuzzy(bool doIt)
{
   QList<EditCommand> editList;
   editList.setAutoDelete(false);


   if(!isFuzzy())
   {
       EditCommand *cmd;
       int offset=_comment.length();

       QString addStr;
       if(offset > 0 && _comment[offset-1] != '\n')
       {
           addStr='\n';
       }
       addStr+="#, fuzzy";

       cmd = new InsTextCmd(offset,addStr);
       cmd->setPart(EditCommand::Comment);
       editList.append(cmd);


       if(doIt)
          _comment+=addStr;
   }

   return editList;
}


void CatalogItem::processCommand(EditCommand* cmd, bool undo)
{
    if(cmd->terminator()!=0)
       return;

    DelTextCmd* delcmd = (DelTextCmd*) cmd;

    bool ins =  true;
    if (delcmd->type() == EditCommand::Delete )
       ins = undo;
    else if (delcmd->type() == EditCommand::Insert )
       ins = !undo;
    else
    {
       kdDebug() << "what kind of edit command is this?" << endl;
       return;
    }

    if ( ins )
    {
       if(delcmd->part()==EditCommand::Msgstr)
       {
          _msgstr.insert(delcmd->offset,delcmd->str);
       }
       else if(delcmd->part()==EditCommand::Comment)
       {
          _comment.insert(delcmd->offset,delcmd->str);
       }
    }
    else
    { // del
       if(delcmd->part()==EditCommand::Msgstr)
       {
          _msgstr.remove(delcmd->offset,delcmd->str.length());
       }
       else if(delcmd->part()==EditCommand::Comment)
       {
          _comment.remove(delcmd->offset,delcmd->str.length());
       }
    }
}


void CatalogItem::clear()
{
   _comment="";
   _msgid="";
   _msgstr="";
   _valid=true;
   _error=None;

   if(_tagList)
   {
       delete _tagList;
       _tagList=0;
   }
}


bool CatalogItem::checkArgs() 
{
	bool hasError = false;

	if(!isUntranslated())
	{
		QString formatChars="dioxXucsfeEgGp%";

		QString line=_msgid.simplifyWhiteSpace();
		int index=line.find(QRegExp("%."));
		
		QStringList argList;
		while(index>=0)
		{
			int endIndex=line.find(QRegExp("[^\\d]"),index+1);
			if(endIndex<0)
			{
				endIndex=line.length();
			}
			else if( formatChars.contains(line[endIndex]) )
			{
					endIndex++;
			}
			
			if(endIndex - index > 1 )
				argList.append(line.mid(index,endIndex-index));
	
			index=line.find(QRegExp("%."),endIndex);
		}
	
		line=_msgstr.simplifyWhiteSpace();
		index=line.find(QRegExp("%."));

		while(index>=0)
		{
			int endIndex=line.find(QRegExp("[^\\d]"),index+1);
			if(endIndex<0)
			{
				endIndex=line.length();
			}
			else if( formatChars.contains(line[endIndex]) )
			{
					endIndex++;
			}
	
			if(endIndex - index > 1 )
			{
				QStringList::Iterator it = argList.find(line.mid(index,endIndex-index));
				if( it!= argList.end())
				{
					argList.remove(it);
				}
				else
				{
					hasError = true;
				}
			}

			index=line.find(QRegExp("%."),endIndex);
		}

		if(!argList.isEmpty())
		{
			hasError = true;
		}
	}
	
	if(hasError)
	{
		_error = _error | Args;
	}
	else
	{
		_error = _error & ~Args;
	}
			
	return !hasError;
}


bool CatalogItem::checkAccelerator(QChar accelMarker)
{
	bool hasError = false;
	if(!isUntranslated())
	{
		QString regStr(accelMarker);
		regStr+="[^\\s]";
		QRegExp reg(regStr);
		int n = _msgid.contains(reg);

		hasError = (_msgstr.contains(reg) != n);
	}

	if(hasError)
	{
		_error = _error | Accel;
	}
	else
	{
		_error = _error & ~Accel;
	}
			
	return !hasError;
}


bool CatalogItem::checkEquation() 
{
	bool error = false;
	
	if(!isUntranslated() && !_msgid.contains('\n') 
					&& _msgid.contains(QRegExp("[a-zA-Z0-9]+=.+")))
	{
		int index = _msgid.find('=');
		QString left = _msgid.left(index);
		index = _msgstr.find('=');
		if(left != _msgstr.left(index))
			error = true;
	}
	
	if(error)
	{
		_error = _error | Equation;
	}
	else
	{
		_error = _error & ~Equation;
	}

	return !error;
}


bool CatalogItem::checkForContext(const QRegExp& reg)
{
	bool error = false;

	if(!isUntranslated() && _msgid.contains(reg) && _msgstr.contains(reg))
	{
		error = true;
	}
	
	if(error)
	{
		_error = _error | Context;
	}
	else
	{
		_error = _error & ~Context;
	}

	return !error;
}


int CatalogItem::checkErrors(QChar accelMarker,const QRegExp& contextInfo)
{
	_error=None;

	checkArgs();
	checkAccelerator(accelMarker);
	checkEquation();
	checkForContext(contextInfo);

	return _error;
}

QString CatalogItem::asString()
{
   QString temp;
   QTextStream stream(&temp,IO_WriteOnly);

   write(stream);
   return temp;
}


void CatalogItem::write(QTextStream& stream) const
{
   if(!comment().isEmpty())
   {
      stream << comment() << "\n";
   }

   QStringList list=msgidAsList();
   QValueList<QString>::ConstIterator lit;

   // if the msgid has more than one line
   if(list.count() > 1)
      list.prepend("");

   stream << "msgid ";
   for( lit = list.begin(); lit != list.end(); ++lit )
   {
      stream << "\"" << (*lit) << "\"\n";
   }

   list=msgstrAsList();
   // if the msgstr has more than one line
   if(list.count() > 1)
      list.prepend("");

   stream << "msgstr ";
   for( lit = list.begin(); lit != list.end(); ++lit )
   {
      stream << "\"" << (*lit) << "\"\n";
   }
}


CatalogItem::IOStatus CatalogItem::read(QTextStream& stream)
{
   enum {Begin,Comment,Msgid,Msgstr} part=Begin;

   clear();

   _valid=false;

   QString line;
   bool cancelLoop=false;
   bool error=false;
   bool recoverableError=false;

   while(!stream.eof() && !cancelLoop)
   {
       int pos=stream.device()->at();

       line=stream.readLine();
       if (line.isNull()) // file end
		   break;

       // remove whitespaces from beginning and end of line
       line=line.stripWhiteSpace();

       if(part==Begin)
       {
           // ignore trailing newlines
           if(line.isEmpty())
              continue;

           if(line.contains(QRegExp("^#~")))
           {
              return Obsolete;
           }
           else if(line.contains(QRegExp("^#")))
           {
               part=Comment;
               _comment=line;
           }
           else if(line.contains(QRegExp("^msgid\\s*\".*\"$")))
           {
               part=Msgid;

               // remove quotes at beginning and the end of the lines
               line.replace(QRegExp("^msgid\\s*\""),"");
               line.replace(QRegExp("\"$"),"");

               _msgid=line;
           }
		     // one of the quotation marks is missing
           else if(line.contains(QRegExp("^msgid\\s*\"?.*\"?$")))
           {
               part=Msgid;

               // remove quotes at beginning and the end of the lines
               line.replace(QRegExp("^msgid\\s*\"?"),"");
               line.replace(QRegExp("\"$"),"");

               _msgid=line;

			      if(!line.isEmpty())
				      recoverableError=true;
           }
           else
           {
              kdDebug(KBABEL) << "no comment or msgid found after a comment" << endl;

               cancelLoop=true;
               error=true;
           }
       }
       else if(part==Comment)
       {
            if(line.isEmpty())
               continue;
            else if(line.contains(QRegExp("^#~")))
            {
               return Obsolete;
            }
            else if(line.contains(QRegExp("^#")))
            {
               _comment+=("\n"+line);
            }
            else if(line.contains(QRegExp("^msgid\\s*\".*\"$")))
            {
               part=Msgid;

               // remove quotes at beginning and the end of the lines
               line.replace(QRegExp("^msgid\\s*\""),"");
               line.replace(QRegExp("\"$"),"");

               _msgid=line;
            }
			   // one of the quotation marks is missing
			   else if(line.contains(QRegExp("^msgid\\s*\"?.*\"?$")))
            {
               part=Msgid;

               // remove quotes at beginning and the end of the lines
               line.replace(QRegExp("^msgid\\s*\"?"),"");
               line.replace(QRegExp("\"$"),"");

               _msgid=line;
			   
			      if(!line.isEmpty())
				      recoverableError=true;
            }
            else
            {
               kdDebug(KBABEL) << "no comment or msgid found after a comment while parsing: " << _comment << endl;

               error=true;
               cancelLoop=true;
            }
        }
        else if(part==Msgid)
        {
            if(line.isEmpty())
               continue;
            else if(line.contains(QRegExp("^\".*\\n?\"$")))
            {
               // remove quotes at beginning and the end of the lines
               line.replace(QRegExp("^\""),"");
               line.replace(QRegExp("\"$"),"");

               // add Msgid line to item
               if(_msgid.isEmpty())
                  _msgid=line;
               else
                  _msgid+=("\n"+line);
            }
            else if(line.contains(QRegExp("^msgstr\\s*\".*\\n?\"$")))
            {
               part=Msgstr;

               // remove quotes at beginning and the end of the lines
               line.replace(QRegExp("^msgstr\\s*\"?"),"");
               line.replace(QRegExp("\"$"),"");

               _msgstr=line;
            }
			   else if(line.contains(QRegExp("^msgstr\\s*\"?.*\\n?\"?$")))
            {
               part=Msgstr;

               // remove quotes at beginning and the end of the lines
               line.replace(QRegExp("^msgstr\\s*\"?"),"");
               line.replace(QRegExp("\"$"),"");

               _msgstr=line;

			      if(!line.isEmpty())
				      recoverableError=true;
            }
            else if(line.contains(QRegExp("^#")) || line.contains(QRegExp("^msgid")))
            {
               kdDebug(KBABEL) << "no msgstr or msgid found after a msgid while parsing: " << _msgid << endl;

               cancelLoop=true;
               error=true;
            }
            // a line of the msgid with a missing quotation mark
            else if(line.contains(QRegExp("^\"?.+\\n?\"?$")))
            {
               recoverableError=true;

               // remove quotes at beginning and the end of the lines
               line.replace(QRegExp("^\""),"");
               line.replace(QRegExp("\"$"),"");

               // add Msgid line to item
               if(_msgid.isEmpty())
                  _msgid=line;
               else
                  _msgid+=("\n"+line);
            }
            else
            {
               kdDebug(KBABEL) << "no msgstr or msgid found after a msgid while parsing: " << _msgid << endl;

               cancelLoop=true;
               error=true;
            }
        }
        else if(part==Msgstr)
        {
            if(line.isEmpty())
               continue;
            // another line of the msgstr
            else if(line.contains(QRegExp("^\".*\\n?\"$")))
            {
               // remove quotes at beginning and the end of the lines
               line.replace(QRegExp("^\""),"");
               line.replace(QRegExp("\"$"),"");

               if(_msgstr.isEmpty())
                  _msgstr=line;
               else
                  _msgstr+=("\n"+line);
            }
            else if(line.contains(QRegExp("^\\s*msgid")) || line.contains(QRegExp("^\\s*#")))
            {
               cancelLoop=true;
               stream.device()->at(pos);// reset position in stream to beginning of this line
               break;
            }
            else if(line.contains(QRegExp("^msgstr")))
            {
               kdDebug(KBABEL) << "no msgid or comment found after a msgstr while parsing: " << _msgstr << endl;

               cancelLoop=true;
               error=true;
            }
            // another line of the msgstr with a missing quotation mark
            else if(line.contains(QRegExp("^\"?.+\\n?\"?$")))
            {
               recoverableError=true;

               // remove quotes at beginning and the end of the lines
               line.replace(QRegExp("^\""),"");
               line.replace(QRegExp("\"$"),"");

               if(_msgstr.isEmpty())
                  _msgstr=line;
               else
                  _msgstr+=("\n"+line);
            }
            else
            {
               kdDebug(KBABEL) << "no msgid or comment found after a msgstr while parsing: " << _msgstr << endl;

               cancelLoop=true;
               error=true;
            }
        }
    }

    if(error)
       return ParseError;
	else if(recoverableError)
		return RecoveredParseError;
    else
    {
      _valid=true;
      return Ok;
    }
}

