/*
 *  This file is part of the KDE System Control Tool,
 *  Copyright (C)1999 Thorsten Westheider <twesthei@physik.uni-bielefeld.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 ****************************************************************************/

#include <qdir.h>

#include "dmaport.h"
#include "interrupt.h"
#include "ioaddress.h"
#include "parportscanner.h"

#include "kdebug.h"

ParallelPortScanner::ParallelPortScanner() : _numports(0),
					     _port(0)
{
  QDir  pardir("/proc/parport");

  if (pardir.exists()) {
    pardir.setFilter(QDir::Dirs); 
    _numports = pardir.count()-2;		// . and .. do not count
  } else {
	pardir = "/proc/sys/dev/parport";
	if (pardir.exists())
  	{
    	pardir.setFilter(QDir::Dirs);
    	_numports = pardir.count()-3;       // . and .. do not count 
	}
  }
}


/*
 * Public methods
 *****************/
 
Device  *ParallelPortScanner::firstDevice()
{
  if (_numports)
  {
    _port = 0;
    return device();
  }
  
  return 0L;
}

 
Device  *ParallelPortScanner::nextDevice()
{

  if ((_port+1) < _numports)
  {
    _port++;    
    return device();
  }
  return 0L;
}

 
/* 
 * Private methods
 ******************/
 
ParallelPort  *ParallelPortScanner::device()
{
  FILE          *parfhd;
  char          path[128], buffer[256];
  QString       scanline;
  uint          base = 0, irq = 0, dma = 0; //, pos;
  bool          ecp = false;
  ParallelPort  *parport = 0L;
  
  sprintf(path, "/proc/parport/%u/hardware", _port);
  
  if ((parfhd = fopen(path, "r"))) 
  {
    while (fgets(buffer, sizeof(buffer), parfhd))
    {
      scanline = buffer;
      
      if (scanline.left(5) == "base:")
      {
        scanline = scanline.right(scanline.length()-5).simplifyWhiteSpace();
	
	if (scanline.left(2) == "0x") sscanf(scanline.ascii()+2, "%x", &base);
      }
      else if (scanline.left(4) == "irq:")
      {
        scanline = scanline.right(scanline.length()-4).simplifyWhiteSpace();	
        if (scanline != "none") sscanf(scanline.ascii(), "%u", &irq);
      }
      else if (scanline.left(4) == "dma:")
      {
        scanline = scanline.right(scanline.length()-4).simplifyWhiteSpace();	
        if (scanline != "none") sscanf(scanline.ascii(), "%u", &dma);
      }
      else if (scanline.left(6) == "modes:")
      {
        /* Matthias ran into trouble with this code
        scanline = scanline.right(scanline.length()-6).simplifyWhiteSpace();
	
        while (!ecp && (pos = scanline.find(',')) >= 0)
	{
	  if (scanline.left(pos) == "ECP") ecp = true;
          else scanline = scanline.right(scanline.length()-pos-1).simplifyWhiteSpace();
	}
	*/
	
	ecp = (scanline.find("ECP") >= 0);
      }
    }
  
    fclose(parfhd);
    
    parport = new ParallelPort(_port, (ecp) ? ParallelPort::ParPortClassECP
                                            : ParallelPort::ParPortClassDefault);
					    

   if (base) parport->addResource(new IOAddress(base, ioRange(base)));
   if (irq)  parport->addResource(new Interrupt(irq));
   if (dma)  parport->addResource(new DMAPort(dma, 8));

    return parport;
  } else {

	sprintf(path, "/proc/sys/dev/parport/parport%u/base-addr", _port);
  	if ((parfhd = fopen(path, "r"))) {
    	fgets(buffer, sizeof(buffer), parfhd);
		scanline = buffer;
		sscanf(scanline.ascii(), "%i ", &base);
        fclose(parfhd);

		sprintf(path, "/proc/sys/dev/parport/parport%u/irq", _port);
		if ((parfhd = fopen(path, "r"))) {
			fgets(buffer, sizeof(buffer), parfhd);
        	scanline = buffer;
			if (scanline.left(2) != "-1") sscanf(scanline.ascii(), "%u", &irq);
		}
        fclose(parfhd);
		sprintf(path, "/proc/sys/dev/parport/parport%u/irq", _port);
		if ((parfhd = fopen(path, "r"))) {
			fgets(buffer, sizeof(buffer), parfhd);
        	scanline = buffer;
        	if (scanline.left(2) != "-1") sscanf(scanline.ascii(), "%u", &dma);
      	}
		fclose(parfhd);
		sprintf(path, "/proc/sys/dev/parport/parport%u/modes", _port);
		
		if ((parfhd = fopen(path, "r"))) {
			fgets(buffer, sizeof(buffer), parfhd);
        	scanline = buffer;
			ecp = (scanline.find("ECP") >= 0);
      	}
		fclose(parfhd);

    	parport = new ParallelPort(_port, (ecp) ? ParallelPort::ParPortClassECP
                                            : ParallelPort::ParPortClassDefault); 
    	if (base) parport->addResource(new IOAddress(base, ioRange(base)));
    	if (irq)  parport->addResource(new Interrupt(irq));
    	if (dma)  parport->addResource(new DMAPort(dma, 8));
		return parport;		
	}		
 }
  
  return 0L;
}
 
 
 
