/********************************
 TCP peer to peer functions
 (c) 1999 Jeremy Wise
 GnomeICU
*********************************/

/*** GnomeICU header files ***/
#include "common.h"
#include "dirbrowser.h"

/*** Toplevel header files ***/
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <netinet/in.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <sys/time.h>
#include <fcntl.h>
#include <time.h>
#include <stdarg.h>
#include <string.h>
#include <ctype.h>
#include <assert.h>
#include <sys/wait.h>
#include <signal.h>
#ifdef HAVE_SOCKS5
#define SOCKS
#include <socks.h>
#endif

GtkWidget *filesel;

typedef struct
{
	struct sokandlb *data;
	int i;
} dataandint;

/* 127.0.0.1 is localhost */
const DWORD LOCALHOST = 0x7F000001;

/*** Local function declarations ***/
static int TCPInitChannel( GIOChannel *source, GIOCondition cond, gpointer data );
static int TCPAckPacket( int sock, GSList *contact, WORD cmd, int seq );
static void TCPProcessPacket( BYTE *packet, int packet_length, int sock );

static int TCPConnectFile( struct xfer_struct *xfer );
static int TCPFileHandshake( GIOChannel *source, GIOCondition condition, struct xfer_struct *xfer );
static int TCPFileReadServer( GIOChannel *source, GIOCondition condition, struct xfer_struct *xfer );
static int TCPFileReadClient( GIOChannel *source, GIOCondition condition, struct xfer_struct *xfer );
static void TCPSendFileInfo( struct xfer_struct *xfer );
static int TCPSendPartFile( GIOChannel *source, GIOCondition condition, struct xfer_struct *xfer );

static void file_place( struct xfer_struct *xfer );
static void set_dir(gchar *dir, struct xfer_struct *xfer);
static void save_incoming_file( GtkWidget *widget, gpointer data );

static int next_file( struct xfer_struct *xfer );

static int TCPFinishConnection( GIOChannel *source, GIOCondition cond, gpointer data );
static int TCPReadPacket( GIOChannel *source, GIOCondition cond, gpointer data );
static int TCPTimeout( tcp_message *m );

static void timeout_continue( GtkWidget *widget, tcp_message *m );
static void timeout_server( GtkWidget *widget, tcp_message *m );
static void timeout_cancel( GtkWidget *widget, tcp_message *m );

GSList *pending_xfers = 0;

/*** Global functions ***/

/***************************************************************
 * This function will put a message on the stack of a contact  *
 * contact - the destination contact                           *
 * data    - the actual packet to be sent                      *
 * text    - the text of the message, if applicable            *
 * seq     - sequence of the TCP packet                        *
 * prepend - if true, prepend the packet instead of appending  *
 ***************************************************************/
tcp_message *tcp_prepare_message( CONTACT_PTR contact, BYTE *data,
                                  gchar *text, gint seq,
                                  gboolean prepend )
{
	tcp_message *m;

	m = g_new0( tcp_message, 1 );
	if( prepend == FALSE )
		m->seq = seq;
	else
		m->seq = 0;
	m->text = text;
	m->data = data;
	m->uin = contact->uin;

	if( prepend == TRUE )
	{
		contact->tcp_msg_queue = g_slist_prepend( contact->tcp_msg_queue, m );
/*		contact->tcp_text_queue = g_slist_prepend( contact->tcp_text_queue, text );*/
	}
	else
	{
		contact->tcp_msg_queue = g_slist_append( contact->tcp_msg_queue, m );
/*		contact->tcp_text_queue = g_slist_append( contact->tcp_text_queue, text );*/
	}

	return m;
}

int TCPGainConnection( DWORD ip, WORD port, GSList *contact )
{
	typedef struct
 	{
		BYTE size[2];
		BYTE command;
		BYTE version[4];
		BYTE szero[4];
		BYTE uin[4];
		BYTE ip[4];
		BYTE real_ip[4];
		BYTE four;
		BYTE port[4];
		gchar str[13];
	} hello_packet_struct;

	hello_packet_struct *hello_packet;
	struct sockaddr_in local, remote;
	int sizeofSockaddr = sizeof( struct sockaddr );
	int sock;

#ifdef TRACE_FUNCTION
	g_print( "TCPGainConnection\n" );
#endif

	if( kontakt->have_tcp_connection == FALSE )
		kontakt->sok = 0;

	if( kontakt->sok > 0 )
		return kontakt->sok;

	if( kontakt->force != 0x04 )
		return -1;

	hello_packet = g_new0( hello_packet_struct, 1 );

	Word_2_Chars( hello_packet->size, 0x001A );
	hello_packet->command = 0xFF;
	DW_2_Chars( hello_packet->version, 0x00000003 );
	DW_2_Chars( hello_packet->szero, 0x00000000 );
	DW_2_Chars( hello_packet->uin, our_info->uin );
	DW_2_IP( hello_packet->ip, LOCALHOST );
	DW_2_IP( hello_packet->real_ip, LOCALHOST );
	hello_packet->four = 0x04;
	DW_2_Chars( hello_packet->port, (DWORD)our_port );
	strcpy( hello_packet->str, "HELLO PACKET" );

	if( ip == 0 )
		return -1;

	sock = socket( AF_INET, SOCK_STREAM, 0 );
	if( sock == -1 )
		return -1;

	fcntl( sock, F_SETFL, O_NONBLOCK );

	memset( &local.sin_zero, 0x00, 8 );
	memset( &remote.sin_zero, 0x00, 8 );
	
	local.sin_family = AF_INET;
	remote.sin_family = AF_INET;
	local.sin_port = g_htons( 0 );
	local.sin_addr.s_addr = g_htonl( INADDR_ANY );
	if( ( bind( sock, (struct sockaddr*)&local, sizeof( struct sockaddr ) ) )== -1 )
		return -1;

	getsockname( sock, (struct sockaddr*)&local, &sizeofSockaddr );

	remote.sin_port = g_htons( port );
	remote.sin_addr.s_addr = g_htonl( ip );

	connect( sock, (struct sockaddr *)&remote, sizeofSockaddr );

	kontakt->sok = sock;
	kontakt->gioc = g_io_channel_unix_new( sock );

	tcp_prepare_message( kontakt, (BYTE*)(hello_packet), NULL,
	                     kontakt->tcp_seq--, TRUE );

	g_io_add_watch( kontakt->gioc,
	                G_IO_IN | G_IO_OUT | G_IO_ERR | G_IO_HUP,
	                TCPFinishConnection, NULL );

	return -2;
}

int TCPClearQueue( GSList *contact )
{
	WORD psize;
	GSList *packet;

	tcp_message *m;

#ifdef TRACE_FUNCTION
	g_print( "TCPClearQueue\n" );
#endif

	if( contact == NULL )
		return FALSE;

	packet = kontakt->tcp_msg_queue;
	while( packet != NULL )
	{
		m = (tcp_message*)packet->data;
		if( m->sent == FALSE )
		{
			psize = Chars_2_Word( m->data );

			write( kontakt->sok, m->data, psize + 2 );
			packet_print( m->data, psize + 2,
			              PACKET_TYPE_TCP | PACKET_DIRECTION_SEND,
			              (gchar*)((BYTE*)(m->data)+2+psize) );
			m->sent = TRUE;
		}
		if( m->seq == 0 )
		{
			g_free( m->data );
			g_free( m );
			kontakt->tcp_msg_queue = g_slist_remove( kontakt->tcp_msg_queue, m );
			packet = kontakt->tcp_msg_queue;
		}
		else
			packet = packet->next;
	}

	return FALSE;
}

int TCPSendMessage( UIN_T uin, gchar *msg )
{
	int sock;
	unsigned short intsize;
	BYTE *buffer;
	tcp_message *m;

	typedef struct
	{
		BYTE uin_a[4];
		BYTE version[2];
		BYTE cmd[2];
		BYTE zero[2];
		BYTE uin_b[4];
		BYTE command[2];
		BYTE msg_length[2];
	} tcp_head;

	typedef struct
	{
		BYTE ip[4];
		BYTE real_ip[4];
		BYTE port[4];
		BYTE four;
		BYTE zero[4];
		BYTE seq[4];
	} tcp_tail;

	struct
	{
		tcp_head head;
		const gchar *body;
		tcp_tail tail;
	} packet;

	GSList *contact;

#ifdef TRACE_FUNCTION
	g_print( "TCPSendMessage\n" );
#endif

	DW_2_Chars( packet.head.uin_a, our_info->uin );
	Word_2_Chars( packet.head.version, 0x0003 );
	Word_2_Chars( packet.head.cmd, ICQ_CMDxTCP_START );
	Word_2_Chars( packet.head.zero, 0x0000 );
	DW_2_Chars( packet.head.uin_b, our_info->uin );
	Word_2_Chars( packet.head.command, ICQ_CMDxTCP_MSG );
	Word_2_Chars( packet.head.msg_length, ( strlen( msg ) + 1 ) );

	packet.body = msg;

	contact = Find_User( uin );
	if( contact == NULL || kontakt->force_override )
		return FALSE;

	DW_2_IP( packet.tail.ip, our_ip );
	DW_2_IP( packet.tail.real_ip, our_ip );
	DW_2_Chars( packet.tail.port, our_port );
	packet.tail.four = 0x04;
	if( kontakt->status == STATUS_DND || kontakt->status == STATUS_OCCUPIED )
		DW_2_Chars( packet.tail.zero, 0x00200000 );
	else
		DW_2_Chars( packet.tail.zero, 0x00100000 );
	DW_2_Chars( packet.tail.seq, kontakt->tcp_seq );

	/* If the user is offline, we use UDP to avoid hanging */
	if( kontakt->status == STATUS_OFFLINE &&
	    kontakt->sok == 0 )
		return 0;

/*** Construct packet ***/
	intsize = sizeof( tcp_head ) + sizeof( tcp_tail ) + strlen( msg ) + 1;
	buffer = (BYTE*)g_malloc0( strlen( "SEND MESSAGE" ) + 1 + intsize + 2 );

	Word_2_Chars (buffer, intsize);
	memcpy( buffer+2, &packet.head, sizeof( packet.head ) );
	memcpy( buffer+2 + sizeof( packet.head ), packet.body,
	        strlen( packet.body ) + 1 );
	memcpy( buffer+2 + sizeof( packet.head ) +
	        strlen( packet.body ) + 1,
	        &packet.tail, sizeof( packet.tail ) );
	strcpy( buffer + 2 + intsize, "SEND MESSAGE" );
/************************/

	m = tcp_prepare_message( kontakt, buffer, g_strdup( msg ),
	                         kontakt->tcp_seq--, FALSE );

	sock = TCPGainConnection( kontakt->current_ip, kontakt->port, contact );
	if( sock != -1 )
		m->timeout = gtk_timeout_add( 30000, (GtkFunction)TCPTimeout, m );

	if( sock == -2 ) /* We're waiting for a connection */
	{
		animate_on();
		return 2;
	}

	if( sock != -1 )
		TCPClearQueue( contact );
	else
		return 0;

	animate_on();
	return 1;
}

int TCPSendURL( UIN_T uin, gchar *msg )
{
	int sock;
	unsigned short intsize;
	BYTE *buffer;
	tcp_message *m;

	typedef struct
	{
		BYTE uin_a[4];
		BYTE version[2];
		BYTE cmd[2];
		BYTE zero[2];
		BYTE uin_b[4];
		BYTE command[2];
		BYTE msg_length[2];
	} tcp_head;

	typedef struct
	{
		BYTE ip[4];
		BYTE real_ip[4];
		BYTE port[4];
		BYTE four;
		BYTE zero[4];
		BYTE seq[4];
	} tcp_tail;

	struct
	{
		tcp_head head;
		const gchar *body;
		tcp_tail tail;
	} packet;

	GSList *contact;

#ifdef TRACE_FUNCTION
	g_print( "TCPSendURL\n" );
#endif

	DW_2_Chars( packet.head.uin_a, our_info->uin );
	Word_2_Chars( packet.head.version, 0x0003 );
	Word_2_Chars( packet.head.cmd, ICQ_CMDxTCP_START );
	Word_2_Chars( packet.head.zero, 0x0000 );
	DW_2_Chars( packet.head.uin_b, our_info->uin );
	Word_2_Chars( packet.head.command, ICQ_CMDxTCP_URL );
	Word_2_Chars( packet.head.msg_length, ( strlen( msg ) + 1 ) );

	packet.body = g_strdup(msg);

	DW_2_IP( packet.tail.ip, our_ip );
	DW_2_IP( packet.tail.real_ip, our_ip );
	DW_2_Chars( packet.tail.port, our_port );
	packet.tail.four = 0x04;
	DW_2_Chars( packet.tail.zero, 0x00100000 );

	contact = Find_User( uin );
	if( contact == NULL )
		return 0;

	DW_2_Chars( packet.tail.seq, kontakt->tcp_seq );

/*** Construct packet ***/
	intsize = sizeof( tcp_head ) + sizeof( tcp_tail ) + strlen( msg ) + 1;
	buffer = (BYTE*)g_malloc0( strlen( "SEND URL" ) + 1 + intsize + 2 );

	Word_2_Chars (buffer, intsize);
	memcpy( buffer+2, &packet.head, sizeof( packet.head ) );
	memcpy( buffer+2 + sizeof( packet.head ), packet.body,
	        strlen( packet.body ) + 1 );
	memcpy( buffer+2 + sizeof( packet.head ) + strlen( packet.body ) + 1,
	        &packet.tail, sizeof( packet.tail ) );
	strcpy( buffer + 2 + intsize, "SEND URL" );
/************************/

	m = tcp_prepare_message( kontakt, buffer, g_strdup( msg ),
	                         kontakt->tcp_seq--, FALSE );

	sock = TCPGainConnection( kontakt->current_ip, kontakt->port, contact );

	if( sock != -1 )
		m->timeout = gtk_timeout_add( 30000, (GtkFunction)TCPTimeout, m );

	if( sock == -2 ) /* We're waiting for a connection */
		return TRUE;

	if( sock != -1 )
		TCPClearQueue( contact );
	else
		return 0;

	animate_on();

	return 1;
}

int TCPAcceptIncoming( gpointer data, int sock, GdkInputCondition cond )
{
	struct sockaddr_in addr;
	int size = sizeof( struct sockaddr_in );
	int new_sock;
	GIOChannel *gioc;

#ifdef TRACE_FUNCTION
	g_print( "TCPAcceptIncoming\n" );
#endif
	
	new_sock = accept( sock, (struct sockaddr *)&addr, &size );
	if( new_sock == -1 )
		return 0;

	if( new_sock == -1 )
		return FALSE;

	gioc = g_io_channel_unix_new( new_sock );

	g_io_add_watch( gioc, G_IO_IN | G_IO_ERR | G_IO_HUP,
	                TCPInitChannel, NULL );

	return 1;
}

int TCPRetrieveAwayMessage( GSList *contact, gpointer data )
{
	int sock;
	unsigned short intsize;
	BYTE *buffer;
	int status_type;
	tcp_message *m;

	typedef struct
	{
		BYTE uin_a[4];
		BYTE version[2];
		BYTE cmd[2];
		BYTE zero[2];
		BYTE uin_b[4];
		BYTE command[2];
		BYTE msg_length[2];
	} tcp_head;

	typedef struct
	{
		BYTE ip[4];
		BYTE real_ip[4];
		BYTE port[4];
		BYTE four;
		BYTE zero[4];
		BYTE seq[4];
	} tcp_tail;

	struct
	{
		tcp_head head;
		char *body;
		tcp_tail tail;
	} packet;

#ifdef TRACE_FUNCTION
	g_print( "TCPRetrieveAwayMessage\n" );
#endif

	switch( kontakt->status & 0xffff )
	{
		case STATUS_AWAY:
			status_type = ICQ_CMDxTCP_READxAWAYxMSG;
			break;
		case STATUS_NA:
			status_type = ICQ_CMDxTCP_READxNAxMSG;
			break;
		case STATUS_OCCUPIED:
			status_type = ICQ_CMDxTCP_READxOCCxMSG;
			break;
		case STATUS_DND:
			status_type = ICQ_CMDxTCP_READxDNDxMSG;
			break;
		default:
			status_type = ICQ_CMDxTCP_READxAWAYxMSG;
			break;
	}

	DW_2_Chars( packet.head.uin_a, our_info->uin );
	Word_2_Chars( packet.head.version, 0x0003 );
	Word_2_Chars( packet.head.cmd, ICQ_CMDxTCP_START );
	Word_2_Chars( packet.head.zero, 0x0000 );
	DW_2_Chars( packet.head.uin_b, our_info->uin );
	Word_2_Chars( packet.head.command, status_type );
	Word_2_Chars( packet.head.msg_length, 0x0001 );

	packet.body = "";

	DW_2_IP( packet.tail.ip, our_ip );
	DW_2_IP( packet.tail.real_ip, LOCALHOST );
	DW_2_Chars( packet.tail.port, our_port );
	packet.tail.four = 0x04;
	DW_2_Chars( packet.tail.zero, 0x00001000 );
	DW_2_Chars( packet.tail.seq, kontakt->tcp_seq );

/*** Create packet ***/
	intsize = sizeof( tcp_head ) + sizeof( tcp_tail ) + 1;
	buffer = (BYTE*)g_malloc0( strlen( "REQUEST AWAY MSG" ) + 1 + intsize + 2 );

	Word_2_Chars (buffer, intsize);
	memcpy( buffer+2, &packet.head, sizeof( packet.head ) );
	memcpy( buffer+2 + sizeof( packet.head ), packet.body,
	        strlen( packet.body ) + 1 );
	memcpy( buffer+2 + sizeof( packet.head ) + strlen( packet.body ) + 1,
	        &packet.tail, sizeof( packet.tail ) );
	strcpy( buffer + 2 + intsize, "REQUEST AWAY MSG" );
/*********************/

	m = tcp_prepare_message( kontakt, buffer, NULL,
	                         kontakt->tcp_seq--, FALSE );

	sock = TCPGainConnection( kontakt->current_ip, kontakt->port, contact );

	if( sock == -2 ) /* We're waiting for a connection */
		return TRUE;

	if( sock != -1 )
		TCPClearQueue( contact );
	else
		return FALSE;

	return 1;
}

int TCPRetrieveVersion( GSList *contact, gpointer data )
{
	int sock;
	unsigned short intsize;
	BYTE *buffer;
	tcp_message *m;

	typedef struct
	{
		BYTE uin_a[4];
		BYTE version[2];
		BYTE cmd[2];
		BYTE zero[2];
		BYTE uin_b[4];
		BYTE command[2];
		BYTE msg_length[2];
	} tcp_head;

	typedef struct
	{
		BYTE ip[4];
		BYTE real_ip[4];
		BYTE port[4];
		BYTE four;
		BYTE zero[4];
		BYTE seq[4];
	} tcp_tail;

	struct
	{
		tcp_head head;
		char *body;
		tcp_tail tail;
	} packet;

#ifdef TRACE_FUNCTION
	g_print( "TCPRetrieveVersion\n" );
#endif

	DW_2_Chars( packet.head.uin_a, our_info->uin );
	Word_2_Chars( packet.head.version, 0x0003 );
	Word_2_Chars( packet.head.cmd, ICQ_CMDxTCP_START );
	Word_2_Chars( packet.head.zero, 0x0000 );
	DW_2_Chars( packet.head.uin_b, our_info->uin );
	Word_2_Chars( packet.head.command, ICQ_CMDxTCP_VERSION );
	Word_2_Chars( packet.head.msg_length, 0x0001 );

	packet.body = VERSION;

	DW_2_IP( packet.tail.ip, our_ip );
	DW_2_IP( packet.tail.real_ip, LOCALHOST );
	DW_2_Chars( packet.tail.port, our_port );
	packet.tail.four = 0x04;
	DW_2_Chars( packet.tail.zero, 0x00001000 );
	DW_2_Chars( packet.tail.seq, kontakt->tcp_seq );

	/*** Create packet ***/
	intsize = sizeof( tcp_head ) + sizeof( tcp_tail ) + 1;
	buffer = (BYTE*)g_malloc0( strlen( "REQUEST VERSION" ) + 1 + intsize + 2 );

	Word_2_Chars (buffer, intsize);
	memcpy( buffer+2, &packet.head, sizeof( packet.head ) );
	memcpy( buffer+2 + sizeof( packet.head ), packet.body,
	        strlen( packet.body ) + 1 );
	memcpy( buffer+2 + sizeof( packet.head ) + strlen( packet.body ) + 1,
	        &packet.tail, sizeof( packet.tail ) );
	strcpy( buffer + 2 + intsize, "REQUEST VERSION" );
/*********************/

	m = tcp_prepare_message( kontakt, buffer, NULL,
	                         kontakt->tcp_seq--, FALSE );

	sock = TCPGainConnection( kontakt->current_ip, kontakt->port, contact );
        
	if( sock == -2 ) /* We're waiting for a connection */
		return TRUE;

	if( sock != -1 )
		TCPClearQueue( contact );
	else
		return FALSE;

	return 1;
}

int TCPAcceptFile( struct xfer_struct *xfer )
{
	BYTE *buffer;
	unsigned short intsize;
	unsigned short backport, tempport;
	gint length = sizeof( struct sockaddr );

	typedef struct
	{
		BYTE uin1[4];
		BYTE version[2];
		BYTE command[2];
		BYTE zero[2];
		BYTE uin2[4];
		BYTE cmd[2];
		BYTE message_length[2];
	} tcp_head;

	typedef struct
	{
		BYTE ip[4];
		BYTE ip_real[4];
		BYTE porta[4];
		BYTE junk;
		BYTE status[4];
		BYTE back_port[4];
		BYTE one[2];
		BYTE zero;
		BYTE bigzero[4];
		BYTE portb[4];
		BYTE seq[4];
	} tcp_tail;

	tcp_head pack_head;
	tcp_tail pack_tail;

	struct sockaddr_in my_addr;

#ifdef TRACE_FUNCTION
	g_print( "TCPAcceptFile\n" );
#endif

	pending_xfers = g_slist_remove(pending_xfers, xfer);
	xfer->status = XFER_STATUS_CONNECTING;

	xfer->sock_fd = socket( PF_INET, SOCK_STREAM, 0 );

	if( xfer->sock_fd <= 0 ) {
		perror("socket");
		return FALSE;
	}

	fcntl( xfer->sock_fd, O_NONBLOCK );

	my_addr.sin_family = AF_INET;
	my_addr.sin_port = 0;
	my_addr.sin_addr.s_addr = g_htonl (INADDR_ANY);
	
	if( bind( xfer->sock_fd, (struct sockaddr *)&my_addr, sizeof(struct sockaddr ) ) == -1 ) {
		perror("bind");
		return FALSE;
	}

	listen( xfer->sock_fd, 1 );

	xfer->gioc = g_io_channel_unix_new(xfer->sock_fd);

        g_io_add_watch_full( xfer->gioc, G_PRIORITY_DEFAULT_IDLE,
				G_IO_IN | G_IO_ERR | G_IO_HUP | G_IO_NVAL,
                               (GIOFunc) TCPFileHandshake,
                               xfer, NULL );

	getsockname( xfer->sock_fd, ( struct sockaddr * ) &my_addr, &length );

	xfer->port = g_ntohs( my_addr.sin_port );

	tempport = xfer->port;

	backport = ( tempport >> 8 ) + ( tempport << 8 );

	DW_2_Chars( pack_head.uin1, our_info->uin );
	Word_2_Chars( pack_head.version, 0x0003 );
	Word_2_Chars( pack_head.command, ICQ_CMDxTCP_ACK );
	Word_2_Chars( pack_head.zero, 0x0000 );
	DW_2_Chars( pack_head.uin2, our_info->uin );
	DW_2_Chars( pack_head.cmd, ICQ_CMDxTCP_FILE );
	DW_2_Chars( pack_head.message_length, 1 );

	DW_2_IP( pack_tail.ip, our_ip );
	DW_2_IP( pack_tail.ip_real, LOCALHOST );
	DW_2_Chars( pack_tail.porta, our_port );
	pack_tail.junk = 0x04;
	DW_2_Chars( pack_tail.status, ICQ_ACKxTCP_ONLINE );
	DW_2_Chars( pack_tail.portb, tempport );
	DW_2_Chars( pack_tail.one, 0x0001 );
	pack_tail.zero = 0x00;
	DW_2_Chars( pack_tail.bigzero, backport );
	DW_2_Chars( pack_tail.back_port, backport );
	DW_2_Chars( pack_tail.seq, xfer->seq );

	if( xfer->remote_contact->sok != -1 )
	{
		intsize = sizeof( tcp_head ) + sizeof( tcp_tail ) + 1;
		buffer = (BYTE*)g_malloc( intsize + 2 );

		Word_2_Chars (buffer, intsize);
		memcpy( buffer+2, &pack_head, sizeof( pack_head ) );
		buffer[2 + sizeof( pack_head )] = 0x00;
		memcpy( buffer+2 + sizeof( pack_head ) + 1,
		        &pack_tail, sizeof( pack_tail ) );
		write( xfer->remote_contact->sok, buffer, intsize + 2 );
		packet_print( buffer + 2, intsize,
		              PACKET_TYPE_TCP | PACKET_DIRECTION_SEND, "ACCEPT FILE" );

		g_free(buffer);
	}
	else
		return -1;

	return 1;
}

int TCPRefuseFile( struct xfer_struct *xfer )
{
	BYTE *buffer;
	unsigned short intsize;

	typedef struct
	{
		BYTE uin1[4];
		BYTE version[2];
		BYTE command[2];
		BYTE zero[2];
		BYTE uin2[4];
		BYTE cmd[2];
		BYTE message_length[2];
	} tcp_head;

        typedef struct
        {
                BYTE ip[4];
                BYTE ip_real[4];
                BYTE porta[4];
                BYTE junk;
                BYTE status[4];
                BYTE back_port[4];
                BYTE one[2];
                BYTE zero;
                BYTE bigzero[4];  
                BYTE portb[4];   
                BYTE seq[4]; 
        } tcp_tail;

	tcp_head pack_head;
	tcp_tail pack_tail;


#ifdef TRACE_FUNCTION
	g_print( "TCPRefuseFile\n" );
#endif

	pending_xfers = g_slist_remove(pending_xfers, xfer);

	DW_2_Chars( pack_head.uin1, our_info->uin );
	Word_2_Chars( pack_head.version, 0x0003 );
	Word_2_Chars( pack_head.command, ICQ_CMDxTCP_CANCEL );
	Word_2_Chars( pack_head.zero, 0x0000 );
	DW_2_Chars( pack_head.uin2, our_info->uin );
	DW_2_Chars( pack_head.cmd, ICQ_CMDxTCP_FILE );
	DW_2_Chars( pack_head.message_length, 1 );
	
        DW_2_IP( pack_tail.ip, our_ip );
        DW_2_IP( pack_tail.ip_real, LOCALHOST );
        DW_2_Chars( pack_tail.porta, our_port );
        pack_tail.junk = 0x04;
        DW_2_Chars( pack_tail.status, ICQ_ACKxTCP_REFUSE );
        DW_2_Chars( pack_tail.portb, 0 );  
        DW_2_Chars( pack_tail.one, 0x0001 );
        pack_tail.zero = 0x00;
        DW_2_Chars( pack_tail.bigzero, 0 );
        DW_2_Chars( pack_tail.back_port, 0 );
        DW_2_Chars( pack_tail.seq, xfer->seq );

        if( xfer->remote_contact->sok != -1 )
        {
                intsize = sizeof( tcp_head ) + sizeof( tcp_tail ) + 1;
                buffer = (BYTE*)g_malloc( intsize + 2 );

                Word_2_Chars (buffer, intsize);
                memcpy( buffer+2, &pack_head, sizeof( pack_head ) );
                buffer[2 + sizeof( pack_head )] = 0x00;
                memcpy( buffer+2 + sizeof( pack_head ) + 1,
                        &pack_tail, sizeof( pack_tail ) );
                write( xfer->remote_contact->sok, buffer, intsize + 2 );
                packet_print( buffer + 2, intsize,
                              PACKET_TYPE_TCP | PACKET_DIRECTION_SEND, "REFUSE FILE" );

		g_free(buffer);
        }
        else
                return -1;
	return 1;
}

int TCPSendFileRequest( UIN_T uin, const gchar *msg, GSList *files )
{
	int cx;
	int sock;
	unsigned short intsize;
	BYTE *buffer;
	int size;
	tcp_message *m;

	typedef struct
	{
		BYTE uin_a[4]; /* Sender's UIN */
		BYTE version[2]; /* 0x0003 */
		BYTE cmd[2]; /* 0x07ee */
		BYTE zero[2]; /* 0x0000 (duh) */
		BYTE uin_b[4]; /* Sender's UIN */
		BYTE type[2]; /* 0x0003 */
		BYTE msg_length[2];
	} tcp_head;

	typedef struct
	{
		BYTE ip[4];
		BYTE real_ip[4];
		BYTE port[4];
		BYTE four; /* 0x04 */
		BYTE status[2];
		BYTE type[2]; /* Always 0x0010 */
		BYTE revport[2];
		BYTE zero[2]; /* Always 0x0000 */
		BYTE filename_len[2];
	} tcp_mid;

	typedef struct
	{
		BYTE flen[4]; /* Length of file */
		BYTE port[4]; /* Always 0x00000000 */
		BYTE seq[4];
	} tcp_tail;

	struct
	{
		tcp_head head;
		const gchar *body;
		tcp_mid mid;
		const gchar *filename;
		tcp_tail tail;
	} packet;

	struct stat file_stat;
	gchar *nopathfile = 0;

	struct file_struct *fileinfo;
	struct xfer_struct *xferinfo;

	GSList *file;
	GSList *contact;

#ifdef TRACE_FUNCTION
	g_print( "TCPSendFileRequest\n" );
#endif
	
	contact = Find_User( uin );
	if( contact == NULL )
		return FALSE;

	xferinfo = g_malloc0(sizeof(struct xfer_struct));

	xferinfo->seq = kontakt->tcp_seq;
	xferinfo->remote_contact = kontakt;

	xferinfo->direction = XFER_DIRECTION_SENDING;
	xferinfo->status = XFER_STATUS_PENDING;

	for (file = files;file;file = file->next) {
		nopathfile = ((gchar*)file->data);

		for( cx = strlen((gchar*)file->data); cx; cx-- )
			if( ((gchar*)file->data)[ cx ] == '/' ) {
				nopathfile = &((gchar*)file->data)[ cx + 1 ];
				break;
			}

		if( strlen( nopathfile ) == 0 )
			return FALSE;

		if( stat( (gchar*)file->data, &file_stat ) == -1 )
			return FALSE;

		fileinfo = g_malloc0(sizeof(struct file_struct));

		fileinfo->full_filename = g_strdup(((gchar*)file->data));
		fileinfo->short_filename = g_strdup(nopathfile);

		fileinfo->total_bytes = file_stat.st_size;
		xferinfo->total_bytes += file_stat.st_size;

		xferinfo->file_queue = g_slist_append(xferinfo->file_queue, fileinfo);
	}

	if (g_slist_length(xferinfo->file_queue) == 1)
		xferinfo->filename = g_strdup(nopathfile);
	else {
		GString *buf = g_string_new("");
		g_string_sprintf(buf, "%u files", g_slist_length(xferinfo->file_queue));
		xferinfo->filename = g_strdup(buf->str);
		g_string_free(buf, TRUE);
	}

	pending_xfers = g_slist_append(pending_xfers, xferinfo);

	DW_2_Chars( packet.head.uin_a, our_info->uin );
	Word_2_Chars( packet.head.version, 0x0003 );
	Word_2_Chars( packet.head.cmd, ICQ_CMDxTCP_START );
	Word_2_Chars( packet.head.zero, 0x0000 );
	DW_2_Chars( packet.head.uin_b, our_info->uin );
	Word_2_Chars( packet.head.type, ICQ_CMDxTCP_FILE );
	Word_2_Chars( packet.head.msg_length, ( strlen( msg ) + 1 ) );

	packet.body = msg;

	DW_2_IP( packet.mid.ip, our_ip );
	DW_2_IP( packet.mid.real_ip, LOCALHOST );
	DW_2_Chars( packet.mid.port, our_port );
	packet.mid.four = 0x04;

	switch( Current_Status )
	{
		case STATUS_ONLINE:
			Word_2_Chars( packet.mid.status, ICQ_ACKxTCP_ONLINE );
			break;
		case STATUS_AWAY:
			Word_2_Chars( packet.mid.status, ICQ_ACKxTCP_AWAY );
			break;
		case STATUS_DND:
			Word_2_Chars( packet.mid.status, ICQ_ACKxTCP_DND );
			break;
		case STATUS_OCCUPIED:
			Word_2_Chars( packet.mid.status, ICQ_ACKxTCP_OCC );
			break;
		case STATUS_NA:
			Word_2_Chars( packet.mid.status, ICQ_ACKxTCP_NA );
			break;
		case STATUS_INVISIBLE:
			Word_2_Chars( packet.mid.status, ICQ_ACKxTCP_ONLINE );
			break;
	}

	Word_2_Chars( packet.mid.type, 0x0010 );
	Word_2_Chars( packet.mid.revport, 0x0000 );
	Word_2_Chars( packet.mid.zero, 0x0000 );
	Word_2_Chars( packet.mid.filename_len, strlen( xferinfo->filename ) + 1 );

	packet.filename = xferinfo->filename;

	DW_2_Chars( packet.tail.flen, xferinfo->total_bytes );
	DW_2_Chars( packet.tail.port, 0x00000000 );
	DW_2_Chars( packet.tail.seq, kontakt->tcp_seq );

/*** Create packet ***/
	intsize = sizeof( tcp_head ) + sizeof( tcp_mid ) + sizeof( tcp_tail ) + strlen( msg ) + strlen( packet.filename ) + 2;
	buffer = (BYTE*)g_malloc( strlen( "REQUEST FILE" ) + 1 + intsize + 2 );

	size = 0;

	Word_2_Chars (buffer + size, intsize);
	size += 2;
	memcpy( buffer + size, &packet.head, sizeof( packet.head ) );
	size += sizeof( packet.head );
	memcpy( buffer + size, packet.body,
	        strlen( packet.body ) + 1 );
	size += strlen( packet.body ) + 1;
	memcpy( buffer + size, &packet.mid, sizeof( packet.mid ) );
	size += sizeof( packet.mid );
	memcpy( buffer + size, packet.filename, strlen( packet.filename ) + 1 );
	size += strlen( packet.filename ) + 1;
	memcpy( buffer + size,
	        &packet.tail, sizeof( packet.tail ) );
	size += sizeof( packet.tail );
	strcpy( buffer + 2 + intsize, "REQUEST FILE" );
/*********************/

	m = tcp_prepare_message( kontakt, buffer, NULL,
	                         kontakt->tcp_seq--, FALSE );

	sock = TCPGainConnection( kontakt->current_ip, kontakt->port, contact );

	if( sock == -2 ) /* We're waiting for a connection */
		return TRUE;

	if( sock != -1 )
		TCPClearQueue( contact );
	else {
		cancel_transfer(0, xferinfo);
		return -1;
	}

	return 1;
}

void TCPSendContactList( GtkWidget *widget, gpointer data )
{
	GtkCList *clist = GTK_CLIST( data );
	gchar *nick;
	int cx;
	tcp_message *m;

	GSList *contact;

	int sock;
	unsigned short intsize;
	guchar *msg;
	BYTE *buffer;
	UIN_T uin = GPOINTER_TO_INT( gtk_object_get_data( GTK_OBJECT( widget ), "uin" ) );

	typedef struct
	{
		BYTE uin_a[4];
		BYTE version[2];
		BYTE cmd[2];
		BYTE zero[2];
		BYTE uin_b[4];
		BYTE command[2];
		BYTE msg_length[2];
	} tcp_head;

	typedef struct
	{
		BYTE ip[4];
		BYTE real_ip[4];
		BYTE port[4];
		BYTE four;
		BYTE zero[4];
		BYTE seq[4];
	} tcp_tail;

	struct
	{
		tcp_head head;
		const gchar *body;
		tcp_tail tail;
	} packet;

#ifdef TRACE_FUNCTION
	g_print( "TCPSendContactList\n" );
#endif

	msg = g_strdup_printf( "%d\xFE", clist->rows );

	for( cx = 0; cx < clist->rows; cx ++ )
	{
		gtk_clist_get_text( clist, cx, 0, &nick );
		msg = g_strconcat( msg, nick, "\xFE", NULL );
		gtk_clist_get_text( clist, cx, 1, &nick );
		msg = g_strconcat( msg, nick, "\xFE", NULL );
	}

	DW_2_Chars( packet.head.uin_a, our_info->uin );
	Word_2_Chars( packet.head.version, 0x0003 );
	Word_2_Chars( packet.head.cmd, ICQ_CMDxTCP_START );
	Word_2_Chars( packet.head.zero, 0x0000 );
	DW_2_Chars( packet.head.uin_b, our_info->uin );
	Word_2_Chars( packet.head.command, ICQ_CMDxTCP_CONT_LIST );
	Word_2_Chars( packet.head.msg_length, ( strlen( msg ) + 1 ) );

	packet.body = msg;

	contact = Find_User( uin );
	if( contact == NULL )
		return;

	DW_2_IP( packet.tail.ip, our_ip );
	DW_2_IP( packet.tail.real_ip, our_ip );
	DW_2_Chars( packet.tail.port, our_port );
	packet.tail.four = 0x04;
	if( kontakt->status == STATUS_DND || kontakt->status == STATUS_OCCUPIED )
		DW_2_Chars( packet.tail.zero, 0x00200000 );
	else
		DW_2_Chars( packet.tail.zero, 0x00100000 );
	DW_2_Chars( packet.tail.seq, kontakt->tcp_seq );

	/* If the user is offline, we use UDP to avoid hanging */
	if( kontakt->status == STATUS_OFFLINE )
		return;

/*** Construct packet ***/
	intsize = sizeof( tcp_head ) + sizeof( tcp_tail ) + strlen( msg ) + 1;
	buffer = (BYTE*)g_malloc0( strlen( "SEND CONTACT LIST" ) + 1 + intsize + 2 );

	Word_2_Chars (buffer, intsize);
	memcpy( buffer+2, &packet.head, sizeof( packet.head ) );
	memcpy( buffer+2 + sizeof( packet.head ), packet.body,
	        strlen( packet.body ) + 1 );
	memcpy( buffer+2 + sizeof( packet.head ) +
	        strlen( packet.body ) + 1,
	        &packet.tail, sizeof( packet.tail ) );
	strcpy( buffer + 2 + intsize, "SEND CONTACT LIST" );
/************************/

	m = tcp_prepare_message( kontakt, buffer, g_strdup( msg ),
	                         kontakt->tcp_seq--, FALSE );

	sock = TCPGainConnection( kontakt->current_ip, kontakt->port, contact );
	if( sock != -1 )
		m->timeout = gtk_timeout_add( 30000, (GtkFunction)TCPTimeout, m );

	if( sock == -2 ) /* We're waiting for a connection */
	{
		animate_on();
		return;
	}

	if( sock != -1 )
		TCPClearQueue( contact );
	else
		return;

	animate_on();
	return;
}

/*** Local functions ***/
int TCPInitChannel( GIOChannel *source, GIOCondition cond, gpointer data )
{
	UIN_T uin;
	guint16 packet_size;
	BYTE *packet;
	int sock;

	GSList *contact;
	
#ifdef TRACE_FUNCTION
	g_print( "TCPInitChannel(" );
	switch( cond )
	{
		case G_IO_IN: g_print( "G_IO_IN)\n" ); break;
		case G_IO_OUT: g_print( "G_IO_OUT)\n" ); break;
		case G_IO_ERR: g_print( "G_IO_ERR)\n" ); break;
		case G_IO_HUP: g_print( "G_IO_HUP)\n" ); break;
		default: g_print( "default)\n" );
	}
#endif

	sock = g_io_channel_unix_get_fd( source );

	read( sock, (char*)(&packet_size), 1 );
	if( ( read( sock, (char*)(&packet_size) + 1, 1 ) ) <= 0 )
	{
		close( sock );
		return FALSE;
	}
	packet_size = GUINT16_FROM_LE (packet_size);

	packet = (BYTE *)g_malloc0( packet_size );
	read( sock, packet, packet_size );

	uin = Chars_2_DW (packet + 9);

	contact = Find_User( uin );
	if( contact == NULL )
		contact = Add_User( uin, NULL, FALSE );

	kontakt->sok = sock;

	TCPProcessPacket( packet, packet_size, sock );

	fcntl( sock, F_SETFL, O_NONBLOCK );

	kontakt->have_tcp_connection = TRUE;
	kontakt->gioc = g_io_channel_unix_new( sock );

	kontakt->giocw =
	g_io_add_watch( kontakt->gioc,
	                G_IO_IN | G_IO_ERR | G_IO_HUP,
	                TCPReadPacket, NULL );

	g_free( data );
	g_free( packet );
	return FALSE;
}

int TCPAckPacket( int sock, GSList *contact, WORD cmd, int seq )
{
	BYTE *buffer;
	unsigned short intsize;
	char *sent_message;

	typedef struct
	{
		BYTE uin1[4];
		BYTE version[2];
		BYTE command[2];
		BYTE zero[2];
		BYTE uin2[4];
		BYTE cmd[2];
		BYTE message_length[2];
	} tcp_head;

	typedef struct
	{
		BYTE ip[4];
		BYTE ip_real[4];
		BYTE port[4];
		BYTE junk;
		BYTE status[4];
		BYTE seq[4];
	} tcp_tail;

	tcp_head pack_head;
	tcp_tail pack_tail;

#ifdef TRACE_FUNCTION
	g_print( "TCPAckPacket\n" );
#endif

	sent_message = "";
	if( (Current_Status&0xffff) != STATUS_ONLINE && (Current_Status&0xffff) != STATUS_FREE_CHAT &&
	    (Current_Status&0xffff) != STATUS_INVISIBLE )
		sent_message = kanji_conv_auto(Away_Message, KANJI_SJIS);
	if( cmd == ICQ_CMDxTCP_VERSION )
		sent_message = VERSION;
	
	DW_2_Chars( pack_head.uin1, our_info->uin );
	Word_2_Chars( pack_head.version, 0x0003 );
	Word_2_Chars( pack_head.command, ICQ_CMDxTCP_ACK );
	Word_2_Chars( pack_head.zero, 0x0000 );
	DW_2_Chars( pack_head.uin2, our_info->uin );
	DW_2_Chars( pack_head.cmd, cmd );
	DW_2_Chars( pack_head.message_length, strlen( sent_message ) + 1 );
	
	DW_2_IP( pack_tail.ip, our_ip );
	DW_2_IP( pack_tail.ip_real, LOCALHOST );
	DW_2_Chars( pack_tail.port, our_port );
	pack_tail.junk = 0x04;
	DW_2_Chars( pack_tail.seq, seq );

	switch( Current_Status & 0xffff )
	{
		case STATUS_ONLINE:
			DW_2_Chars( pack_tail.status, ICQ_ACKxTCP_ONLINE );
			break;
		case STATUS_AWAY:
			DW_2_Chars( pack_tail.status, ICQ_ACKxTCP_AWAY );
			break;
		case STATUS_DND:
			DW_2_Chars( pack_tail.status, ICQ_ACKxTCP_DND );
			break;
		case STATUS_OCCUPIED:
			/* Don't ask me why this is - I just follow precedent */
			DW_2_Chars( pack_tail.status, ICQ_ACKxTCP_ONLINE );
			break;
		case STATUS_NA:
			DW_2_Chars( pack_tail.status, ICQ_ACKxTCP_NA );
			break;
		case STATUS_INVISIBLE:
			DW_2_Chars( pack_tail.status, ICQ_ACKxTCP_ONLINE );
			break;
	}

	if( sock != -1 )
	{
		intsize = sizeof( tcp_head ) + sizeof( tcp_tail ) + strlen( sent_message ) + 1;
		buffer = (BYTE*)g_malloc0( intsize + 2 );

		Word_2_Chars (buffer, intsize);
		memcpy( buffer+2, &pack_head, sizeof( pack_head ) );
		memcpy( buffer+2 + sizeof( pack_head ), sent_message,
		        strlen( sent_message ) + 1 );
		memcpy( buffer+2 + sizeof( pack_head ) + strlen( sent_message ) + 1,
		        &pack_tail, sizeof( pack_tail ) );
		write( sock, buffer, intsize + 2 );
		packet_print( buffer+2, intsize,
		              PACKET_TYPE_TCP | PACKET_DIRECTION_SEND, "ACK PACKET" );
		g_free( buffer );
	}
	else
		return -1;

	return 1;
}

void TCPProcessPacket( BYTE *packet, int packet_length, int sock )
{
	BYTE *size;
	GSList *xfer;

	typedef struct
	{
		UIN_T uin1;
		WORD version;
		WORD command;
		WORD zero;
		UIN_T uin2;
		WORD cmd;
		WORD message_length;
	} tcp_head;

	typedef struct
	{
		DWORD ip_sender;
		DWORD ip_local;
		DWORD port;
		BYTE junk;
		WORD status;
		WORD msgcommand;

		DWORD seq;
	} tcp_tail;

	typedef struct
	{
		WORD name_len;
		char *name;
		WORD revport;
		WORD zero;
		DWORD port;
	} tcp_chat;

	typedef struct
	{
		WORD revport;
		WORD zero;
		WORD name_len;
		char *name;
		DWORD size;
		WORD port;
	} tcp_file;

	gchar *message;
	gchar *away_message;
	tcp_head pack_head;
	tcp_tail pack_tail;
	tcp_file pack_file;
	tcp_chat pack_chat;

	GSList *contact;
	GSList *messages;
	tcp_message *m = NULL;
	ChatContact *ccontact;

	DWORD i;

#ifdef TRACE_FUNCTION
	g_print( "TCPProcessPacket\n" );
#endif

	if( packet[0] == 0xFF ) /* 0xFF means it's just a "Hello" packet; ignore */
	{
		packet_print( packet, packet_length,
		              PACKET_TYPE_TCP | PACKET_DIRECTION_RECEIVE, "HELLO PACKET" );
		return;
	}

	size = packet;
	pack_head.uin1 = Chars_2_DW (size);
	size += 4;
	pack_head.version = Chars_2_Word (size);
	size += 2;
	pack_head.command = Chars_2_Word (size);
	size += 2;
	pack_head.zero = Chars_2_Word (size);
	size += 2;
	pack_head.uin2 = Chars_2_DW (size);
	size += 4;
	pack_head.cmd = Chars_2_Word (size);
	size += 2;
	pack_head.message_length = Chars_2_Word (size);
	size += 2;

	message = (char *)g_malloc0( pack_head.message_length );
	memcpy( message, size,
	        pack_head.message_length );

	size += pack_head.message_length;
	pack_tail.ip_sender = Chars_2_DW (size);
	size += 4;
	pack_tail.ip_local = Chars_2_DW (size);
	size += 4;
	pack_tail.port = Chars_2_DW (size);
	size += 4;
	memcpy( &pack_tail.junk, size, 1 );
	size ++;
	pack_tail.status = Chars_2_Word (size);
	size += 2;
	pack_tail.msgcommand = Chars_2_Word( size );
	size += 2;
//	if( pack_head.command == ICQ_CMDxTCP_ACK && pack_head.cmd == ICQ_CMDxTCP_CHAT )
	if( pack_head.cmd == ICQ_CMDxTCP_CHAT )
	{
		pack_chat.name_len = Chars_2_Word( size );
		size += 2;
		if (pack_chat.name_len  > 0) {
		     pack_chat.name = g_malloc0( pack_chat.name_len );
		     memcpy( pack_chat.name, size, pack_chat.name_len );
		} else 
		     size += 1;
	      
		size += pack_chat.name_len;
		pack_chat.revport = Chars_2_Word( size );
		size += 2;
		size += 2; /* Chat_X2 */
		pack_chat.port = Chars_2_DW( size );
		size += 4;
	}
 
	if ( pack_head.cmd == ICQ_CMDxTCP_FILE )
	{
		pack_file.revport = Chars_2_Word( size );
		size += 2;
		size += 2; /* File_X3 */
		pack_file.name_len = Chars_2_Word( size );
		size += 2;
		pack_file.name = g_malloc( pack_file.name_len );
		memcpy( pack_file.name, size, pack_file.name_len );
		size += pack_file.name_len;
		pack_file.size = Chars_2_DW( size );
		size += 4;
		pack_file.port = Chars_2_Word( size ); 
		size += 4; 
	}

	pack_tail.seq = Chars_2_DW (size);
	size += 4;

	/* What are these. Not used anyway... */
	i = pack_tail.ip_sender;
	pack_tail.ip_sender = ( ( i << 24 ) | ( ( i & 0xff00 ) << 8 ) | ( ( i & 0xff0000 ) >> 8 ) | ( i >> 24 ) );
	i = pack_tail.ip_local;
	pack_tail.ip_local = ( ( i << 24 ) | ( ( i & 0xff00 ) << 8 ) | ( ( i & 0xff0000 ) >> 8 ) | ( i >> 24 ) );

    if( toggles->no_new_users == TRUE){
        TCPAckPacket( sock, NULL, pack_head.cmd, pack_tail.seq );
        return;
    }

	switch( pack_head.command )
	{
		case ICQ_CMDxTCP_START:
			switch( pack_head.cmd )
			{
				case ICQ_CMDxTCP_MSG:
					packet_print( packet, packet_length,
					              PACKET_TYPE_TCP | PACKET_DIRECTION_RECEIVE, "TCP MESSAGE(MSG)" );
					contact = Do_Msg( time( NULL ), 0, message, pack_head.uin1, MESSAGE_TEXT );
					log_window_add( _("Received Message (TCP)"), pack_head.uin1 );
					TCPAckPacket( sock, contact, pack_head.cmd, pack_tail.seq );
					break;
				case ICQ_CMDxTCP_VERSION:
					packet_print( packet, packet_length,
					              PACKET_TYPE_TCP | PACKET_DIRECTION_RECEIVE, "VERSION REQUEST" );

					contact = Find_User( pack_head.uin1 );
					if( contact == NULL )
						break;

					TCPAckPacket( sock, contact, ICQ_CMDxTCP_VERSION, pack_tail.seq );
					break;

				case ICQ_CMDxTCP_READxAWAYxMSG:
				case ICQ_CMDxTCP_READxOCCxMSG:
				case ICQ_CMDxTCP_READxDNDxMSG:
				case ICQ_CMDxTCP_READxNAxMSG:
					packet_print( packet, packet_length,
					              PACKET_TYPE_TCP | PACKET_DIRECTION_RECEIVE, "AWAY MSG REQUEST" );
					contact = Find_User( pack_head.uin1 );
					if( contact == NULL )
						break;

					TCPAckPacket( sock, contact, ICQ_CMDxTCP_READxAWAYxMSG,
					              pack_tail.seq );
					break;

				case ICQ_CMDxTCP_URL:  /* url sent */
					packet_print( packet, packet_length,
					              PACKET_TYPE_TCP | PACKET_DIRECTION_RECEIVE, "TCP MESSAGE(URL)" );
					contact = Do_Msg( time( NULL ), 0, message, pack_head.uin1, MESSAGE_URL );
					TCPAckPacket( sock, contact, pack_head.cmd, pack_tail.seq );
					break;

				case ICQ_CMDxTCP_CONT_LIST:
					packet_print( packet, packet_length,
					              PACKET_TYPE_TCP | PACKET_DIRECTION_RECEIVE, "TCP MESSAGE(CONT_LIST)" );
					contact = Do_Msg( time( NULL ), 0, message, pack_head.uin1, MESSAGE_CONT_LIST );
					TCPAckPacket( sock, contact, pack_head.cmd, pack_tail.seq );
					break;

				case ICQ_CMDxTCP_CHAT:
				     packet_print( packet, packet_length,
						   PACKET_TYPE_TCP | PACKET_DIRECTION_RECEIVE, "CHAT REQUEST" );
				     ccontact = Do_Chat( pack_chat.name_len, message, pack_head.uin1, pack_tail.seq, pack_chat.name, pack_chat.port );
				     if( Current_Status == STATUS_FREE_CHAT )
					  TCPAcceptChat( sock, ccontact);
				     break;
				     
				case ICQ_CMDxTCP_FILE:
					packet_print( packet, packet_length,
					              PACKET_TYPE_TCP | PACKET_DIRECTION_RECEIVE, "FILE REQUEST" );
			
					contact = Do_File( message, pack_head.uin1, pack_tail.seq, pack_file.name, pack_file.size );
					break;      

				default:
					break;
			}
			break;

		case ICQ_CMDxTCP_ACK:
			switch ( pack_head.cmd )
			{
				case ICQ_CMDxTCP_VERSION:
					contact = Find_User( pack_head.uin1 );
					if( contact == NULL )
						break;

					packet_print( packet, packet_length,
					              PACKET_TYPE_TCP | PACKET_DIRECTION_RECEIVE, "RESPONSE VERSION" );

					away_message = g_strdup_printf( "%s is running\nGnomeICU version:",
					                                kontakt->nick );

					AMessage_Box( away_message, message, kontakt->uin );
					g_free( away_message );

					break;
				case ICQ_CMDxTCP_MSG:
				case ICQ_CMDxTCP_CONT_LIST:
			      	case ICQ_CMDxTCP_READxAWAYxMSG:
			      	case ICQ_CMDxTCP_READxOCCxMSG:
			      	case ICQ_CMDxTCP_READxDNDxMSG:
			      	case ICQ_CMDxTCP_READxNAxMSG:
				case ICQ_CMDxTCP_URL:
					contact = Find_User( pack_head.uin1 );

					if( contact != NULL && ( pack_head.cmd == ICQ_CMDxTCP_MSG ||
					                         pack_head.cmd == ICQ_CMDxTCP_URL ||
					                         pack_head.cmd == ICQ_CMDxTCP_CONT_LIST ) )
					{
						packet_print( packet, packet_length,
						              PACKET_TYPE_TCP | PACKET_DIRECTION_RECEIVE, "MSG/URL/CONT LIST ACK" );
						animate_off();

						messages = kontakt->tcp_msg_queue;
						while( messages != NULL )
						{
							m = messages->data;
							if( m->seq == pack_tail.seq )
								break;
							messages = messages->next;
						}

						if( m != NULL )
						{
							g_free( m->data );
							if( m->text != NULL )
								g_free( m->text );
							if( m->timeout )
								gtk_timeout_remove( m->timeout );
							g_free( m );
							kontakt->tcp_msg_queue = g_slist_remove( kontakt->tcp_msg_queue, m );
						}
					}
					else
						packet_print( packet, packet_length,
						              PACKET_TYPE_TCP | PACKET_DIRECTION_RECEIVE, "RESPONSE AWAY MSG" );


					if( (pack_tail.status & 0xffff) == ICQ_ACKxTCP_AWAY ||
					    (pack_tail.status & 0xffff) == ICQ_ACKxTCP_NA   ||
					    (pack_tail.status & 0xffff) == ICQ_ACKxTCP_DND  ||
					    (pack_tail.status & 0xffff) == ICQ_ACKxTCP_OCC )
					{
						switch( pack_tail.status )
						{
							case ICQ_ACKxTCP_AWAY:
								away_message = g_strdup_printf( _("User %s is Away:"), kontakt->nick );
								break;
							case ICQ_ACKxTCP_NA:
								away_message = g_strdup_printf( _("User %s is Not Available:"), kontakt->nick );
								break;
							case ICQ_ACKxTCP_DND:
								away_message = g_strdup_printf( _("User %s cannot be disturbed:"), kontakt->nick );
								break;
							case ICQ_ACKxTCP_OCC:
								away_message = g_strdup_printf( _("User %s is Occupied:"), kontakt->nick );
								break;
							default:
								away_message = g_strdup( _("User is in an unknown mode:") );
						}
	
						AMessage_Box( away_message, message, kontakt->uin );
						g_free( away_message );
					}
					break;

				case ICQ_CMDxTCP_CHAT:
					packet_print( packet, packet_length,
					              PACKET_TYPE_TCP | PACKET_DIRECTION_RECEIVE, "ACK CHAT REQUEST" );
					if( pack_chat.port > 0 )
					     TCPConnectChat( pack_chat.port, pack_head.uin1, FALSE );
					break;

				case ICQ_CMDxTCP_FILE:
					packet_print( packet, packet_length,
					              PACKET_TYPE_TCP | PACKET_DIRECTION_RECEIVE, "ACK FILE REQUEST" );
					
					for(xfer = pending_xfers;xfer;xfer = xfer->next)
						if ( ( ((struct xfer_struct*)xfer->data)->remote_contact->uin == pack_head.uin1) && 
							( ((struct xfer_struct*)xfer->data)->seq == pack_tail.seq) &&
							( ((struct xfer_struct*)xfer->data)->direction == XFER_DIRECTION_SENDING)) 
							break;

					if (xfer) {
						struct xfer_struct *transfer = xfer->data;

						pending_xfers = g_slist_remove(pending_xfers, transfer);

						if( pack_file.port > 0 ) {
							transfer->port = pack_file.port;

							if (TCPConnectFile( transfer ) == -1)
								cancel_transfer(0, transfer);
						}
					}

					break;

				default:
					break;
			}
			break;
      
		case ICQ_CMDxTCP_CANCEL:
			switch ( pack_head.cmd )
			{

			     ChatSession *csession;

				case ICQ_CMDxTCP_CHAT:
					packet_print( packet, packet_length,
					              PACKET_TYPE_TCP | PACKET_DIRECTION_RECEIVE, "CHAT REQ CANCEL" );
					gnome_ok_dialog( _("Chat Request Refused") );
					contact = Find_User( pack_head.uin1 );
					if (contact == NULL)
					     break;
					
					if (g_list_first(kontakt->chat_requests))
					{
					     csession = (ChatSession *) g_list_first(kontakt->chat_requests)->data;
					     kontakt->chat_requests = g_list_remove(kontakt->chat_requests, csession);

					     if (csession->members == NULL && csession->chatwindow == NULL) {
						  Sessions = g_slist_remove (Sessions, csession);
						  g_free(csession);
					     }
					}
					else
					{
					     GSList *temp_storedmsg;
					     
					     temp_storedmsg = kontakt->stored_messages;
					     while (temp_storedmsg != NULL)
					     {
						  if (((STORED_MESSAGE_PTR) temp_storedmsg->data)->type == MESSAGE_CHAT_REQ)
						       break;
						  temp_storedmsg = temp_storedmsg->next;
					     }

					     if (temp_storedmsg != NULL)
					     {
						  
						  g_free(((STORED_MESSAGE_PTR) temp_storedmsg->data)->message);
						  g_free(((STORED_MESSAGE_PTR) temp_storedmsg->data)->chatsessionname);
						  g_free(((STORED_MESSAGE_PTR) temp_storedmsg->data)->chatcontact);
						  g_free(temp_storedmsg->data);
						  kontakt->stored_messages = g_slist_remove( kontakt->stored_messages, temp_storedmsg->data);
						  
						  applet_update( Current_Status, NULL );
						  if( g_slist_length( kontakt->stored_messages ) == 0 )
						  {
						       kontakt->icon_p = GetIcon_p( kontakt->status );
						       kontakt->icon_b = GetIcon_b( kontakt->status );
						       kontakt->need_update = 1;
						  }
						  
						  Show_Quick_Status();
					     }
					}
					

					/* This should be reimplemented with a "More Info" button */
					/* OK_Box( _("Chat Request Refused"), message ); */
					break;
      
				case ICQ_CMDxTCP_FILE:
					packet_print( packet, packet_length,
					              PACKET_TYPE_TCP | PACKET_DIRECTION_RECEIVE, "FILE REQ CANCEL" );

					for(xfer = pending_xfers;xfer;xfer = xfer->next) {
						if ( ( ((struct xfer_struct*)xfer->data)->remote_contact->uin == pack_head.uin1) && 
							( ((struct xfer_struct*)xfer->data)->seq == pack_tail.seq))
							break;
					}

					if (xfer) 
						cancel_transfer(0, xfer->data);

					break;
      
				default:
					break;
			}
			break;
   
		default:
			break;
	}

	g_free( message );
} 

int TCPConnectFile( struct xfer_struct *xfer )
{
	DWORD localport;
	struct sockaddr_in local, remote;
	int sizeofSockaddr = sizeof( struct sockaddr );
	BYTE *buffer;
	int index, psize;

	typedef struct
	{
		BYTE id; /* 0xff */
		BYTE version[4]; /* 0x00000003 */
		BYTE port[4]; /* TCP listen port */
		BYTE uin[4]; /* UIN */
		BYTE ipa[4]; /* IP A */
		BYTE ipb[4]; /* IP B */
		BYTE four; /* 0x04 */
		BYTE listenport[4]; /* File listen port */
	} handshake;

	handshake hs;
	xfer0 pak;

#ifdef TRACE_FUNCTION
	g_print( "TCPConnectFile\n" );
#endif

	xfer->status = XFER_STATUS_CONNECTING;

	if( xfer->sock_fd > 0 )
		return xfer->sock_fd;

	if (!xfer->remote_contact->current_ip)
		return -1;

	xfer->sock_fd = socket( PF_INET, SOCK_STREAM, 0 );
	if( xfer->sock_fd == -1 )
		return -1;

	fcntl( xfer->sock_fd, O_NONBLOCK );

	local.sin_family = AF_INET;
	remote.sin_family = AF_INET;
	local.sin_port = g_htons( 0 );
	local.sin_addr.s_addr = g_htonl( INADDR_ANY );

	remote.sin_port = g_htons( xfer->port );

	remote.sin_addr.s_addr = g_htonl( xfer->remote_contact->current_ip );

	if( connect( xfer->sock_fd, (struct sockaddr *)&remote, sizeofSockaddr ) < 0 )
	{
		perror("connect");
		return -1;
	}

	getsockname( xfer->sock_fd, (struct sockaddr*)&local, &sizeofSockaddr );
	localport = g_ntohs( local.sin_port );

	hs.id = 0xff;
	DW_2_Chars( hs.version, 0x00000003 );
	DW_2_Chars( hs.port, our_port );
	DW_2_Chars( hs.uin, our_info->uin );
	DW_2_IP( hs.ipa, our_ip );
	DW_2_IP( hs.ipb, LOCALHOST );
	hs.four = 0x04;
	DW_2_Chars( hs.listenport, localport );

	Word_2_Chars( pak.psize, sizeof( xfer0 ) - 2 + strlen( our_info->nick ) + 1 );
	pak.id = 0x00;
	DW_2_Chars( pak.zero, 0x00000000 );
	DW_2_Chars( pak.n_files, g_slist_length(xfer->file_queue) );
	DW_2_Chars( pak.n_bytes, xfer->total_bytes );
	DW_2_Chars( pak.speed, 0x00000064 );
	Word_2_Chars( pak.nick_len, strlen( our_info->nick ) + 1 );

	fcntl( xfer->sock_fd, O_NONBLOCK );

	xfer->gioc = g_io_channel_unix_new(xfer->sock_fd);

	g_io_add_watch_full( xfer->gioc, G_PRIORITY_DEFAULT_IDLE,
				G_IO_IN | G_IO_ERR | G_IO_HUP | G_IO_NVAL,
				(GIOFunc) TCPFileReadServer,
				xfer, NULL );

	index = 0;
	psize = sizeof( handshake );
	buffer = (BYTE*)g_malloc( psize + 2 );

	Word_2_Chars (buffer, psize);
	index += 2;
	memcpy( buffer + index, &hs, sizeof( handshake ) );

	packet_print( buffer, psize+2,
	              PACKET_TYPE_TCP | PACKET_DIRECTION_SEND, "FILE(INIT)" );
	write( xfer->sock_fd, buffer, psize + 2 );
	g_free( buffer );

	index = 0;
	psize = sizeof( xfer0 ) + strlen( our_info->nick ) + 1;
	buffer = (BYTE*)g_malloc( psize );
	
	memcpy( buffer, &pak, sizeof( xfer0 ) );
	index += sizeof( xfer0 );
	memcpy( buffer + index, our_info->nick, strlen( our_info->nick ) + 1 );

	packet_print( buffer, psize,
	              PACKET_TYPE_TCP | PACKET_DIRECTION_SEND, "FILE(INIT)" );
	write( xfer->sock_fd, buffer, psize );
	g_free( buffer );

	return xfer->sock_fd;
}

int TCPFileHandshake( GIOChannel *source, GIOCondition condition, struct xfer_struct *xfer )
{
	int new_sock;
	int size = sizeof( struct sockaddr );
	struct sockaddr_in their_addr;

#ifdef TRACE_FUNCTION
	g_print( "TCPFileHandshake\n" );
#endif	

	new_sock = accept( xfer->sock_fd, ( struct sockaddr * )&their_addr, &size );

        if (xfer->gioc)
                g_io_channel_close(xfer->gioc);

	xfer->sock_fd = new_sock;
	xfer->gioc = g_io_channel_unix_new(new_sock);

	xfer->port = g_ntohs( their_addr.sin_port );

	fcntl( new_sock, O_NONBLOCK );

	g_io_add_watch_full( xfer->gioc, G_PRIORITY_DEFAULT_IDLE,
					G_IO_IN | G_IO_ERR | G_IO_HUP,
					(GIOFunc) TCPFileReadClient,
					xfer, NULL );

	return FALSE;
}

int TCPFileReadServer(GIOChannel *source, GIOCondition condition, struct xfer_struct *xfer)
{
	unsigned short packet_size;
	BYTE *packet;

#ifdef TRACE_FUNCTION
	g_print( "TCPFileReadServer\n" );
#endif

       /* We closed the socket, so the transfer's already been cancelled  */
       if (condition & G_IO_NVAL)
               return FALSE;

       if (read( xfer->sock_fd, (char*)(&packet_size), 2 ) != 2) {
		cancel_transfer(0, xfer);
		return FALSE;
	}
        
	packet_size = GUINT16_FROM_LE (packet_size);

	packet = (BYTE *)g_malloc( packet_size );
	read( xfer->sock_fd, packet, packet_size );
	packet_print( packet, packet_size,
	              PACKET_TYPE_TCP | PACKET_DIRECTION_RECEIVE, "FILE(INIT(READ))" );

	switch( packet[ 0 ] )
	{
	case 0x01: /* We need to send XFER2 now */
		TCPSendFileInfo( xfer );
		break;

	case 0x03: /* Start sending the file :) */
	/* The best way to do this is to set up an IO watch to send it so we don't
	   freeze the program, especially on large files */
		((struct file_struct*)xfer->file_queue->data)->completed_bytes = Chars_2_DW (&packet[1]);

		xfer->file_fd = open( ((struct file_struct*)xfer->file_queue->data)->full_filename, O_RDONLY, S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH );
		if( xfer->file_fd == -1 )
		{
 			g_free(packet);
			return next_file(xfer);
		}

		lseek( xfer->file_fd, ((struct file_struct*)xfer->file_queue->data)->completed_bytes, SEEK_SET );

		if (!xfer->dialog)
			create_file_xfer_dialog( xfer );

		g_io_add_watch_full( xfer->gioc, G_PRIORITY_DEFAULT_IDLE,
			G_IO_OUT | G_IO_ERR | G_IO_HUP | G_IO_NVAL,
			(GIOFunc) TCPSendPartFile,
			xfer, NULL );

 		((struct file_struct *)xfer->file_queue->data)->start_time = time( NULL );
 
 		if( xfer->start_time == 0 )
 			xfer->start_time = time( NULL );  

		xfer->status = XFER_STATUS_TRANSFERING;

		break;
 	case 0x04:
 		skip_file(0, xfer);
 		break;
	}

	g_free( packet );

	return TRUE;
}

int TCPFileReadClient( GIOChannel *source, GIOCondition condition, struct xfer_struct *xfer )
{
	int file_count, bytes_read, ret = TRUE;

	xfer1 pak1;
	BYTE *buf;

#ifdef TRACE_FUNCTION
	g_print( "TCPFileReadClient\n" );
#endif

        /* We closed the socket, so the transfer's already been cancelled */
        if (condition & G_IO_NVAL)
                return FALSE;

	if (!xfer->packet_size) {
		/* We have to turn off blocking for just a bit, or else
		file transfers can get a bit flakey */

		fcntl( xfer->sock_fd, F_SETFL, 0 );

		if (read( xfer->sock_fd, (char*)(&xfer->packet_size), 2 ) != 2) {
			cancel_transfer(0, xfer);
			return FALSE;
		}

		/* Alright, don't block anymore */
		fcntl( xfer->sock_fd, F_SETFL, 0 );

		xfer->packet_size = GUINT16_FROM_LE (xfer->packet_size);

		xfer->packet_offset = 0;
		xfer->packet = (BYTE *)g_malloc( xfer->packet_size );
	}


	bytes_read = read( xfer->sock_fd, xfer->packet + xfer->packet_offset, xfer->packet_size - xfer->packet_offset );

	if (bytes_read < 1) {
		cancel_transfer(0, xfer);
		return FALSE;
	}

	xfer->packet_offset += bytes_read;

	if (xfer->packet_size != xfer->packet_offset)
		return TRUE;

	packet_print( xfer->packet, xfer->packet_size,
	              PACKET_TYPE_TCP | PACKET_DIRECTION_RECEIVE, "FILE PACKET" );

	switch( xfer->packet[ 0 ] )
	{
		case 0x00:
			file_count = Chars_2_DW( &xfer->packet[ 5 ] );
			xfer->total_bytes = Chars_2_DW( &xfer->packet[ 9 ] );

			for(;file_count;file_count--)
				xfer->file_queue = g_slist_append(xfer->file_queue, g_malloc0(sizeof(struct file_struct)));

			pak1.id = 0x01;
			DW_2_Chars( pak1.speed, 0x00000064 );
			Word_2_Chars( pak1.nick_len, strlen( our_info->nick ) + 1 );
			buf = (BYTE*)g_malloc( sizeof( xfer1 ) + strlen( our_info->nick ) + 1 );
			Word_2_Chars( pak1.psize, sizeof( xfer1 ) - 1 + strlen( our_info->nick ) );
			memcpy( buf, &pak1, sizeof( xfer1 ) );
			memcpy( &buf[ sizeof( xfer1 ) ], our_info->nick, strlen( our_info->nick ) + 1 );
			write( xfer->sock_fd, buf, sizeof( xfer1 ) + strlen( our_info->nick ) + 1 );
			packet_print( &buf[2], sizeof( xfer1 ) + strlen( our_info->nick ) - 1,
			              PACKET_TYPE_TCP | PACKET_DIRECTION_SEND, "FILE(01)" );
			
			break;

		case 0x02:
			if (xfer->status == XFER_STATUS_TRANSFERING)
				if (!next_file(xfer))
					return FALSE;

			((struct file_struct *)xfer->file_queue->data)->short_filename = g_strdup(&xfer->packet[ 4 ]);
			((struct file_struct *)xfer->file_queue->data)->total_bytes = Chars_2_DW(&xfer->packet[ 8 + strlen( &xfer->packet[4]) ]);

			file_place( xfer );			

			xfer->status = XFER_STATUS_TRANSFERING;

			break;

		case 0x06:
			if( xfer->file_fd <= 0 )
				break;

			write( xfer->file_fd, &xfer->packet[ 1 ], xfer->packet_size - 1);

			((struct file_struct *)xfer->file_queue->data)->completed_bytes += xfer->packet_size - 1;
			xfer->completed_bytes += xfer->packet_size - 1;

		break;
	}

	g_free(xfer->packet);
	xfer->packet = 0;
	xfer->packet_size = 0;

	return ret;
}

void TCPSendFileInfo( struct xfer_struct *xfer ) {
	BYTE *buffer;
	unsigned short packet_size;
	int index;

	xfer2a pak2a;
	xfer2b pak2b;

	pak2a.id = 0x02;
	pak2a.zero = 0x00;
	Word_2_Chars( pak2a.fn_len, strlen( ((struct file_struct*)xfer->file_queue->data)->short_filename ) + 1 );
	Word_2_Chars( pak2b.one, 0x0001 );
	pak2b.zero = 0;
	DW_2_Chars( pak2b.file_size, ((struct file_struct*)xfer->file_queue->data)->total_bytes );
	DW_2_Chars( pak2b.longzero, 0x00000000 );
	DW_2_Chars( pak2b.speed, 0x00000064 );
		
	packet_size = sizeof( xfer2a ) + sizeof( xfer2b ) + strlen( ((struct file_struct*)xfer->file_queue->data)->short_filename ) + 1;
	index = 0;
	buffer = g_malloc( packet_size + 2 );
	Word_2_Chars (buffer, packet_size);
	index += 2;
	memcpy( buffer + index, &pak2a, sizeof( xfer2a ) );
	index += sizeof( xfer2a );
	memcpy( buffer + index, ((struct file_struct*)xfer->file_queue->data)->short_filename, strlen( ((struct file_struct*)xfer->file_queue->data)->short_filename ) + 1 );
	index += strlen( ((struct file_struct*)xfer->file_queue->data)->short_filename ) + 1;
	memcpy( buffer + index, &pak2b, sizeof( xfer2b ) );
	index += sizeof( xfer2b );

	packet_print( buffer+2, packet_size,
	              PACKET_TYPE_TCP | PACKET_DIRECTION_SEND, "FILE(INIT(2))" );
	write( xfer->sock_fd, buffer, packet_size + 2 );
	g_free( buffer );
}

int TCPSendPartFile( GIOChannel *source, GIOCondition condition, struct xfer_struct *xfer )
{
	int bytes;
	xfer6 pak;	
	
#ifdef TRACE_FUNCTION
	g_print("TCPSendPartFile\n");
#endif	

        /* They hung up on us... <sniff> */
        if ( condition & (G_IO_HUP | G_IO_NVAL | G_IO_ERR) ) 
                return FALSE;                     

	/* The file we're sending was skipped by someone, commit suicide */
	if (xfer->status != XFER_STATUS_TRANSFERING)
		return FALSE;

	pak.id = 0x06;
		
	bytes = read( xfer->file_fd, pak.data, 2048 );

	Word_2_Chars( pak.psize, 1 + bytes );
		
	write( xfer->sock_fd, &pak, 3 + bytes );
	packet_print( &pak.id, bytes + 1, PACKET_TYPE_TCP | PACKET_DIRECTION_SEND, "FILE PACKET" );
			
	((struct file_struct *)xfer->file_queue->data)->completed_bytes += bytes;
	xfer->completed_bytes += bytes;

	if ((((struct file_struct *)xfer->file_queue->data)->completed_bytes >= ((struct file_struct *)xfer->file_queue->data)->total_bytes) || (bytes < 2048) ) {
		next_file(xfer);
		return FALSE;
	}

	return TRUE;
}

void file_place( struct xfer_struct *xfer )
{
	GtkWidget *filesel;
#ifdef TRACE_FUNCTION
	g_print( "file_place\n" );
#endif

	if (xfer->auto_save_path) {
            /* We have been given a directory to dump this stuff in to; don't prompt the user */
            ((struct file_struct*)xfer->file_queue->data)->full_filename = g_strdup_printf("%s%s", xfer->auto_save_path, ((struct file_struct*)xfer->file_queue->data)->short_filename);
            save_incoming_file(0, xfer);
            return;
	}

	if (g_slist_length(xfer->file_queue) > 1) {
		filesel = create_dir_browser(_("Select destination directory"), (void (*)(gchar*, gpointer))&set_dir, (gpointer)xfer);
		gtk_signal_connect(GTK_OBJECT(filesel), "destroy", GTK_SIGNAL_FUNC(gtk_widget_destroyed), &filesel);
	}
	else {
		filesel = gtk_file_selection_new( _("GnomeICU: File Transfer") );

		gtk_file_selection_set_filename( GTK_FILE_SELECTION( filesel ), ((struct file_struct*)xfer->file_queue->data)->short_filename );
		gtk_signal_connect(GTK_OBJECT(filesel), "destroy", GTK_SIGNAL_FUNC(gtk_widget_destroy), NULL);
		gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(filesel)->ok_button),
		                   "clicked", GTK_SIGNAL_FUNC( save_incoming_file ), xfer );
		gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(filesel)->cancel_button),
		                   "clicked", GTK_SIGNAL_FUNC( cancel_transfer ), xfer );
		gtk_signal_connect_object(GTK_OBJECT (GTK_FILE_SELECTION
	 	                  (filesel)->cancel_button), "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy),
	  	                 GTK_OBJECT( filesel ) );
	}

	gtk_widget_show(filesel);
}

void set_dir(gchar *dir, struct xfer_struct *xfer) {
	xfer->auto_save_path = g_strdup(dir);
	file_place(xfer);
}

void save_incoming_file( GtkWidget *widget, gpointer data )
{
	struct xfer_struct *xfer = data;
	struct stat file_stat;
	xfer3 pak;

	if (widget)
		((struct file_struct*)xfer->file_queue->data)->full_filename = g_strdup( gtk_file_selection_get_filename( GTK_FILE_SELECTION( widget->parent->parent->parent ) ) );

	if (!xfer->dialog)
		create_file_xfer_dialog( xfer );

	gtk_entry_set_text( GTK_ENTRY( xfer->dialog->local_filename ), ((struct file_struct*)xfer->file_queue->data)->full_filename );

	if (widget)
		gtk_widget_destroy( GTK_WIDGET( widget->parent->parent->parent ) );

	xfer->file_fd = open( ((struct file_struct*)xfer->file_queue->data)->full_filename, O_WRONLY | O_CREAT | O_APPEND, S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP | S_IROTH | S_IWOTH );

	if( xfer->file_fd > 0 )
	{
		Word_2_Chars( pak.psize, 0x0d );
		pak.id = 0x03;
		if( stat( ((struct file_struct*)xfer->file_queue->data)->full_filename, &file_stat ) == -1 )
			((struct file_struct *)xfer->file_queue->data)->completed_bytes = 0;
		else
			((struct file_struct *)xfer->file_queue->data)->completed_bytes = file_stat.st_size;

		xfer->completed_bytes += ((struct file_struct *)xfer->file_queue->data)->completed_bytes;

		DW_2_Chars( pak.resume_loc, ((struct file_struct *)xfer->file_queue->data)->completed_bytes );

		DW_2_Chars( pak.longzero, 0x00000000 );
		DW_2_Chars( pak.speed, 0x00000064 );
		write( xfer->sock_fd, &pak, sizeof( pak ) );
	}

 	((struct file_struct *)xfer->file_queue->data)->start_time = time( NULL );
 
 	if( xfer->start_time == 0 )
 		xfer->start_time = time( NULL );

	return;
}

void cancel_transfer( GtkWidget *widget, struct xfer_struct *xfer )
{ 
	gchar *msg;
        time_t timedate;

	GSList *file_queue;

	if( xfer->gioc )
		g_io_channel_close(xfer->gioc);

	destroy_file_xfer_dialog(xfer);

	if (xfer->total_bytes == xfer->completed_bytes)
	{
            if( xfer->direction == XFER_DIRECTION_RECEIVING )
            {
		/* <kludge> */
		/*  Grrr, or else the last file won't be counted */
		struct file_struct *completed_file = (struct file_struct *)xfer->file_queue->data;
        
		xfer->file_queue = g_slist_remove( xfer->file_queue, completed_file );
		xfer->completed_files = g_slist_append( xfer->completed_files, completed_file );
		/* </kludge> */

                msg = g_strdup_printf( _("File transfer:\nCompleted transfer of %s from %s."), xfer->filename, xfer->remote_contact->nick );
                time( &timedate );
                add_incoming_to_history( xfer->remote_contact->uin, msg, &timedate );
                request_file_action(xfer->completed_files);
                g_free( msg );
            }
            else
            {
                msg = g_strdup_printf( _("File transfer:\nCompleted transfer of %s to %s."), xfer->filename, xfer->remote_contact->nick );
                add_outgoing_to_history( xfer->remote_contact->uin, msg );
                g_free( msg );
                
                msg = g_strdup_printf( _("File transfer completed: %s"), xfer->filename );
                gnome_ok_dialog( msg );
                g_free( msg );
            }
	}
	else
	{
		if (xfer->status == XFER_STATUS_PENDING) {
			pending_xfers = g_slist_remove(pending_xfers, xfer);
			msg = g_strdup_printf( _("File transfer refused: %s"), xfer->filename );
			gnome_ok_dialog( msg );
			g_free( msg );
		}
		else {
			msg = g_strdup_printf( _("File transfer terminated: %s"), xfer->filename );
			gnome_error_dialog( msg );
			g_free( msg );
		}
	}

	close( xfer->sock_fd );

	g_free( xfer->filename );
	g_free( xfer->auto_save_path );

	g_free( xfer->packet );
	xfer->packet = 0;

	if( xfer->file_fd )
		close( xfer->file_fd );

	for(file_queue = xfer->file_queue;file_queue;file_queue = file_queue->next) {
		g_free( ((struct file_struct*)file_queue->data)->short_filename);
		g_free( ((struct file_struct*)file_queue->data)->full_filename);
		g_free( (struct file_struct*)file_queue->data );
	}

	g_slist_free(xfer->file_queue);

        for(file_queue = xfer->completed_files;file_queue;file_queue = file_queue->next) {            
            g_free( ((struct file_struct*)file_queue->data)->short_filename);
            g_free( ((struct file_struct*)file_queue->data)->full_filename);
            g_free( (struct file_struct*)file_queue->data );
        }
                
        g_slist_free(xfer->completed_files);

	g_free(xfer); 

	return;
}

void skip_file( GtkWidget *widget, struct xfer_struct *xfer) {
	xfer4 pak;

	if (xfer->direction == XFER_DIRECTION_SENDING)
		next_file( xfer );
	else {
		pak.id = 0x04;
		DW_2_Chars(pak.data, g_slist_length(xfer->completed_files) + 1);

		write(xfer->sock_fd, &pak, sizeof(xfer4));
	}
}

int next_file( struct xfer_struct *xfer ) {
	struct file_struct *completed_file = (struct file_struct *)xfer->file_queue->data;

	xfer->file_queue = g_slist_remove( xfer->file_queue, completed_file );
	xfer->completed_files = g_slist_append( xfer->completed_files, completed_file );

	close(xfer->file_fd);

	if (!xfer->file_queue) {
		cancel_transfer(0, xfer);
		return FALSE;
	}
	else
		xfer->status = XFER_STATUS_CONNECTING;

	xfer->completed_bytes += (completed_file->total_bytes - completed_file->completed_bytes);

	if (xfer->direction == XFER_DIRECTION_SENDING)
		TCPSendFileInfo( xfer );

	return TRUE;
}

int TCPFinishConnection( GIOChannel *source, GIOCondition cond, gpointer data )
{
	GSList *contact;
	int sock;

#ifdef TRACE_FUNCTION
	g_print( "TCPFinishConnection(%d)\n", cond );
#endif

	if( cond & G_IO_ERR || cond & G_IO_HUP )
		return FALSE;

	contact = Contacts;

	while( contact != NULL )
	{
		if( kontakt->gioc == source )
			break;
		contact = contact->next;
	}

	if( contact == NULL )
		return FALSE;

	sock = g_io_channel_unix_get_fd( source );

	TCPClearQueue( contact );

	kontakt->have_tcp_connection = TRUE;

	kontakt->giocw =
	g_io_add_watch( kontakt->gioc,
	                G_IO_IN | G_IO_ERR | G_IO_HUP,
	                TCPReadPacket, NULL );

	return FALSE;
}

int TCPReadPacket( GIOChannel *source, GIOCondition cond, gpointer data )
{
	UIN_T uin;
	BYTE *packet;
	guint bytes_read;
	int sock;

	GSList *contact;

#ifdef TRACE_FUNCTION
	g_print( "TCPReadPacket(" );
	switch( cond )
	{
		case G_IO_IN:
			g_print( "G_IO_IN)\n" );
			break;
		case G_IO_HUP:
			g_print( "G_IO_HUP)\n" );
			break;
		case G_IO_ERR:
			g_print( "G_IO_ERR)\n" );
			break;
	default: 
			g_print( "UNKNOWN)\n" );
	}
#endif

	contact = Contacts;

	while( contact != NULL )
	{
		if( kontakt->gioc == source )
			break;
		contact = contact->next;
	}

	if( contact == NULL )
		return FALSE;

	if( kontakt->sok == 0 )
		return FALSE;

	if( cond != G_IO_IN )
		return FALSE; /* FIXME - This will be called if they close connection */

	sock = g_io_channel_unix_get_fd( source );

	if( kontakt->tcp_buf_len == 0 )
	{
		g_io_channel_read( source, (gchar*)&kontakt->tcp_buf_len,
		                   2, &bytes_read );

		if( !bytes_read )
		{
			kontakt->have_tcp_connection = FALSE;
			kontakt->sok = 0;
			close( sock );
			g_io_channel_close( source );
			kontakt->gioc = 0;
			return FALSE;
		}

		if( bytes_read == 1 )
		{
			bytes_read = 0;
			while( !bytes_read )
				g_io_channel_read( source, (gchar*)&kontakt->tcp_buf_len + 1,
				                   1, &bytes_read );
		}

		kontakt->tcp_buf_len =
		GINT16_FROM_LE
		  (kontakt->tcp_buf_len);
		kontakt->tcp_buf = (BYTE*)g_malloc0( kontakt->tcp_buf_len );

		return TRUE;
	}

	packet = kontakt->tcp_buf;

	if( packet == NULL )
		return TRUE;

	g_io_channel_read( source, packet + kontakt->tcp_buf_read,
	                   kontakt->tcp_buf_len - kontakt->tcp_buf_read,
	                   &bytes_read );

	kontakt->tcp_buf_read += bytes_read;
	if( kontakt->tcp_buf_read != kontakt->tcp_buf_len )
		return FALSE;

	uin = Chars_2_DW (packet);

	contact = Find_User( uin );
	if( contact == NULL )
		return FALSE;

	kontakt->sok = sock;
	TCPProcessPacket( packet, kontakt->tcp_buf_len, sock );

	kontakt->tcp_buf_read = 0;
	kontakt->tcp_buf_len = 0;
	g_free( packet );
	kontakt->tcp_buf = NULL;

	return TRUE;
}

static int TCPTimeout( tcp_message *m )
{
	GtkWidget *label;
        GString *SendError;
	static int first = TRUE;

	GSList *contact;

	if( first )
	{
		first = FALSE;
		return TRUE;
	}

	contact = Find_User( m->uin );
	if( contact == NULL )
		return TRUE;

	animate_off();

	if( kontakt->force_override == TRUE )
	{
		if( m->text != NULL )
			icq_sendmsg( m->uin, m->text, TRUE );

		g_free( m->data );
		if( m->text != NULL )
			g_free( m->text );

		g_free( m );
		kontakt->tcp_msg_queue = g_slist_remove( kontakt->tcp_msg_queue, m );

		return TRUE;
	}

        SendError = g_string_new("");
        g_string_sprintf(SendError, "Unable to send a message directly to %s.", (gchar*)kontakt->nick);
	label = gtk_label_new( (gchar*)SendError->str );
        g_string_free(SendError, FALSE);
	m->dialog = gnome_dialog_new( "GnomeICU: Question", "Retry", "Send Through Server", "Cancel", NULL );
 	gtk_window_set_wmclass (GTK_WINDOW (m->dialog), "Question", "GnomeICU");
	gtk_box_pack_start( GTK_BOX( GNOME_DIALOG( m->dialog )->vbox ), label,
	                    FALSE, FALSE, GNOME_PAD );

	gnome_dialog_set_close( GNOME_DIALOG( m->dialog ), TRUE );

	gnome_dialog_button_connect( GNOME_DIALOG(m->dialog), 0, timeout_continue, m );
	gnome_dialog_button_connect( GNOME_DIALOG(m->dialog), 1, timeout_server, m );
	gnome_dialog_button_connect( GNOME_DIALOG(m->dialog), 2, timeout_cancel, m );
	gtk_widget_show( label );
	gtk_widget_show( m->dialog );

	return FALSE;
}

void timeout_continue( GtkWidget *widget, tcp_message *m )
{
	GSList *contact;

	/* We do this in case the user was removed before the message
	   was sent.. */
	contact = Find_User( m->uin );
	if( contact == NULL )
		return;

	m->timeout = gtk_timeout_add( 10000, (GtkFunction)TCPTimeout, m );
	animate_on();
}

void timeout_server( GtkWidget *widget, tcp_message *m )
{
	GSList *contact;

	/* We do this in case the user was removed before the message
	   was sent.. */
	contact = Find_User( m->uin );
	if( contact == NULL )
		return;

	if( m->text != NULL )
		icq_sendmsg( m->uin, m->text, TRUE );

	g_free( m->data );
	g_free( m->text );
	g_free( m );

	kontakt->tcp_msg_queue = g_slist_remove( kontakt->tcp_msg_queue, m );
	kontakt->force_override = TRUE;
}

void timeout_cancel( GtkWidget *widget, tcp_message *m )
{
	GSList *contact;

	/* We do this in case the user was removed before the message
	   was sent.. */
	contact = Find_User( m->uin );
	if( contact == NULL )
		return;

	g_free( m->data );
	g_free( m->text );
	g_free( m );

	kontakt->tcp_msg_queue = g_slist_remove( kontakt->tcp_msg_queue, m );
}
