/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 */

#include "galeon.h"
#include "window.h"
#include "misc.h"
#include "dialog.h"

#include <gtk/gtkpaned.h>
#include <gtk/gtkwindow.h>
#include <gtk/gtkclist.h>
#include <gtk/gtkentry.h>
#include <libgnome/gnome-config.h>
#include <libgnome/gnome-util.h>
#include <libgnome/gnome-i18n.h>
#include <libgnomeui/gnome-dialog-util.h>
#include <libgnomeui/gnome-file-entry.h>

static void
dialog_ask_file_name_ok_cb (GtkButton *button, GtkWidget *dialog);
static void 
dialog_ask_file_name_cancel_cb (GtkButton *button, GtkWidget *dialog); 
static void
dialog_ask_question_cb (gint reply, gpointer data);

static void
dialog_ask_file_name_ok_cb (GtkButton *button, GtkWidget *dialog) 
{
	FileCallbackFunc f = gtk_object_get_data (GTK_OBJECT (dialog), "callback_ok");
	GtkWidget *fentry = gtk_object_get_data (GTK_OBJECT (dialog), "fentry");
	gchar *fname = gnome_file_entry_get_full_path (GNOME_FILE_ENTRY (fentry), FALSE);
	gpointer data = gtk_object_get_data (GTK_OBJECT (dialog), "user_data");
	gboolean must_exist = GPOINTER_TO_INT 
		(gtk_object_get_data (GTK_OBJECT (dialog), "must_exist"));
	gboolean warn_overwrite = GPOINTER_TO_INT 
		(gtk_object_get_data (GTK_OBJECT (dialog), "warn_overwrite"));
	gboolean proceed = TRUE;
	if (must_exist && !g_file_exists (fname)) {
		GtkWidget *err_dialog = gnome_error_dialog_parented 
			(_("The file must exist.\n"
			   "Please select an existing file or chose \"Cancel\""),
			 GTK_WINDOW (dialog));
		proceed = FALSE;
	        window_set_layer (err_dialog);
	        gnome_dialog_run (GNOME_DIALOG (err_dialog));
	}
	if (warn_overwrite && g_file_exists (fname)) {
		proceed = dialog_ask_question
			(_("The file already exists.\n"
			   "Do you want to overwrite it?"),
			 GTK_WINDOW (dialog));
	}
	if (proceed) {
		if (f)
			f (fname, data);
		gtk_widget_destroy (dialog);
	}
	g_free (fname);
}

static void
dialog_ask_file_name_cancel_cb (GtkButton *button, GtkWidget *dialog) 
{
	FileCallbackFunc f = gtk_object_get_data (GTK_OBJECT (dialog), 
						  "callback_cancel");
	GtkWidget *fentry = gtk_object_get_data (GTK_OBJECT (dialog), "fentry");
	gchar *fname = gnome_file_entry_get_full_path (GNOME_FILE_ENTRY (fentry), FALSE);
	gpointer data = gtk_object_get_data (GTK_OBJECT (dialog), "user_data");
	if (f)
		f (fname, data);
	g_free (fname);
	gtk_widget_destroy (dialog);
}

/**
 * Show a dialog to ask for a file. When the user selects ok/cancel the given 
 * callbacks will be called.
 * TODO: extend to handle file overwriting, inexistent files...
 */
void
dialog_ask_file_name (const gchar *title, const gchar *label_title, 
		      const gchar *def,
		      FileCallbackFunc callback_ok, 
		      FileCallbackFunc callback_cancel, 
		      gboolean must_exist, gboolean warn_overwrite,
		      gpointer user_data)
{
	GladeXML *gxml;
	GtkWidget *dialog;
	GtkWidget *ok_button;
	GtkWidget *cancel_button;
	GtkWidget *fentry;
	GtkWidget *fgtkentry;
	GtkWidget *label;
	WidgetLookup lookup_table[] =
	{
		{ "file_selection_dialog_ok_button",        &ok_button     },
		{ "file_selection_dialog_cancel_button",    &cancel_button },
		{ "file_selection_dialog_fileentry", &fentry        },
		{ "file_selection_dialog_label",            &label         },
		{ NULL, NULL } /* terminator, must be last */
	};

	/* make the dialog */
	gxml = glade_widget_new ("galeon.glade", "file_selection_dialog",
				 &dialog, NULL);

        /* look them all up */
        lookup_widgets (gxml, lookup_table);

	/* get the gtk entry */
	fgtkentry = gnome_file_entry_gtk_entry (GNOME_FILE_ENTRY (fentry));

	/* no longer need XML */
	gtk_object_unref (GTK_OBJECT (gxml));

	/* set data in objects (FIXME, use a structure) */
	gtk_object_set_data (GTK_OBJECT (dialog), "callback_ok", callback_ok);
	gtk_object_set_data (GTK_OBJECT (dialog), "callback_cancel", 
			     callback_cancel);
	gtk_object_set_data (GTK_OBJECT (dialog), "must_exist", 
			     GINT_TO_POINTER (must_exist));
	gtk_object_set_data (GTK_OBJECT (dialog), "warn_overwrite", 
			     GINT_TO_POINTER (warn_overwrite));
	gtk_object_set_data (GTK_OBJECT (dialog), "user_data", user_data);
	gtk_object_set_data (GTK_OBJECT (dialog), "fentry", fentry);

	/* setup window according to parameters */
	gtk_window_set_title (GTK_WINDOW (dialog), title);
	gtk_label_set_text (GTK_LABEL (label), label_title);
	gnome_file_entry_set_title (GNOME_FILE_ENTRY (fentry), title);
	gtk_entry_set_text (GTK_ENTRY (gnome_file_entry_gtk_entry
				       (GNOME_FILE_ENTRY (fentry))), def);

	/* connect signals */
	gtk_signal_connect (GTK_OBJECT (ok_button), "clicked", 
			    dialog_ask_file_name_ok_cb, dialog);
	gtk_signal_connect (GTK_OBJECT (cancel_button), "clicked", 
			    dialog_ask_file_name_cancel_cb, dialog);
	gtk_signal_connect (GTK_OBJECT (fgtkentry), "activate", 
			    dialog_ask_file_name_ok_cb, dialog);

	/* grab the focus in the entry */
	gtk_widget_grab_focus (fgtkentry);

	/* raise to visibility */
	window_set_layer (GTK_WIDGET (dialog));
}

/** Shared var for the next two functions */
static gboolean misc_ask_question_ret = FALSE;

/**
 * Asks a yes/no question and returns a boolean
 */
gboolean 
dialog_ask_question (const gchar *question, GtkWindow *parent)
{
	GtkWidget *dialog = gnome_question_dialog_parented
		(question, dialog_ask_question_cb, NULL, parent);
	window_set_layer (dialog);
	gnome_dialog_run (GNOME_DIALOG (dialog));
	return misc_ask_question_ret;
}

static void
dialog_ask_question_cb (gint reply, gpointer data) {
	misc_ask_question_ret = !reply;
}


