/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"
#include "bookmarks.h"
#include "embed.h"
#include "window.h"
#include "misc.h"
#include "find.h"

#include <string.h>
#include <libgnome/gnome-config.h>
#include <libgnome/gnome-i18n.h>
#include <libgnomeui/gnome-preferences.h>
#include <libgnomeui/gnome-popup-menu.h>
#include <libgnomeui/gnome-stock.h>
#include <gtk/gtkcheckmenuitem.h>
#include <gtk/gtktoolbar.h>
#include <gtk/gtkbox.h>

extern const GtkTargetEntry bookmarks_dnd_targets[];

/* local function prototypes */
static void bookmarks_toolbars_create_from_root (GaleonWindow *window, 
						 BookmarkItem *bi);
static void  bookmarks_toolbar_create_from_category (GaleonWindow *window,
						     BookmarkItem *bi);
static void bookmarks_toolbar_create_from_bookmark (GaleonWindow *window, 
						    GtkToolbar *tb,
						    BookmarkItem *b);
static gint bookmarks_folder_show_menu_cb (GtkWidget *menu);
static void  bookmarks_nick_entry_activate_cb (GtkEntry *entry, 
					       BookmarkItem *bi);
static gboolean bookmarks_toolbar_item_button_release_event_cb 
                                       (GtkWidget *item, 
					GdkEventButton *event,
					BookmarkItem *bm);
static gboolean bookmarks_toolbar_button_release_event_cb 
                                       (GtkWidget *item, 
					GdkEventButton *event,
					BookmarkItem *bm);
static void bookmarks_toolbar_drag_data_received_cb 
                                       (GtkWidget *widget, 
					GdkDragContext *drag_context,
					gint x, gint y,
					GtkSelectionData *selection_data,
					guint info, guint time,
					BookmarkItem *bm);

/* this magic number is used to hack around a problem with
 * gnome-libs (I think) which causes multiple toolbars with
 * the same number not to display properly -- MattA
 * This number will also be appended to toolbar names to make sure no two
 * toolbars have the same name.  This keeps the gnome layout routines from
 * getting confused when restoring toolbar positions  --Josh */
static gint magic_number;

/**
 * bookmarks_toolbars_create: create personal toolbars
 */
void 
bookmarks_toolbars_create (GaleonWindow *window)
{
	gboolean visible;

	/* if no toolbars, skip this */
	if (!(window->toolBarOn))
	{
		return;
	}

	/* hack, we need probably to unify not created / not visible case */
	/* FIXME: shouldnt need menu bar on to know visiblity of toolbar */
	if (window->menuBarOn && window->view_toolbar)
	{
		visible = GTK_CHECK_MENU_ITEM (window->view_toolbar)->active;
	}
	else
	{
		visible = TRUE;
	}

	/* reset the magic number */
	magic_number = 99;

	/* build all the toolbars */
	bookmarks_toolbars_create_from_root (window, bookmarks_root);

	/* setup visibility of them all */
	bookmarks_toolbars_set_visibility (window, visible);
}

/**
 * bookmarks_toolbars_recreate: recreate personal toolbars
 */
void 
bookmarks_toolbars_recreate (GaleonWindow *window)
{
	GList *l;

	/* destroy all the toolbars */
	for (l = window->bookmarks_toolbars; l != NULL; l = g_list_next (l))
	{
		GtkToolbar *toolbar = GTK_TOOLBAR (l->data);
		gtk_widget_destroy (GTK_WIDGET (toolbar)->parent);
	}

	/* list is now empty */
	g_list_free (window->bookmarks_toolbars);
	window->bookmarks_toolbars = NULL;

	/* now create toolbars for all of these */
	bookmarks_toolbars_create (window);
}

/**
 * bookmarks_toolbars_create_from_root: create toolbars from a given
 * root bookmark
 */
static void
bookmarks_toolbars_create_from_root (GaleonWindow *window, BookmarkItem *bi)
{
	GList *l;

	switch (bi->type)
	{
	case BM_SITE:
	case BM_SEPARATOR:
		/* can't create a toolbar from these */
		return;

	case BM_CATEGORY:
	case BM_AUTOBOOKMARKS:
		/* continue... */
		break;
	}

	/* if we're meant to create a toolbar... */
	if (bi->create_toolbar)
	{
		/* do so */
		bookmarks_toolbar_create_from_category (window, bi);
	}
	
	/* now iterate to find any marked subfolders, 
	 * unless this is an alias */
	if (!(bi->alias_of))
	{
		for (l = bi->list; l != NULL; l = g_list_next (l))
		{
			bookmarks_toolbars_create_from_root (window, l->data);
		}
	}
}

/**
 * bookmarks_toolbar_create_from_category: create a toolbar from a folder
 */
static void 
bookmarks_toolbar_create_from_category (GaleonWindow *window, BookmarkItem *bi)
{
	gboolean toolbar_relief;
	gboolean toolbar_detachable;
	GnomeDockItemBehavior props;
	GtkWidget *toolbar;
	gchar *unique_name;
	GList *l;

	/* read gnome prefs */
	toolbar_relief = gnome_preferences_get_toolbar_relief ();
	toolbar_detachable = gnome_preferences_get_toolbar_detachable ();

	/* make the toolbar */
	toolbar = gtk_toolbar_new (GTK_ORIENTATION_HORIZONTAL, 
				   GTK_TOOLBAR_TEXT);

	/* check whether it should be detachable or not */
	props = (GNOME_DOCK_ITEM_BEH_EXCLUSIVE |
		 (toolbar_detachable ? 0 : GNOME_DOCK_ITEM_BEH_LOCKED));

	/* make sure each toolbar has a unique name */
	unique_name = g_strdup_printf("%ld", bi->id);

	/* add the toolbar it to the main window */
	gnome_app_add_docked (GNOME_APP (window->WMain), toolbar, unique_name,
			      props, GNOME_DOCK_TOP, magic_number++, 0, 0);
	gtk_container_set_border_width (GTK_CONTAINER (toolbar->parent),
					toolbar_relief ? 2 : 0);
	gnome_dock_item_set_shadow_type (GNOME_DOCK_ITEM (toolbar->parent), 
					 toolbar_relief ? 
					 GTK_SHADOW_OUT : GTK_SHADOW_NONE);
	g_free (unique_name);

	/* override global setting */
	gtk_toolbar_set_space_style (GTK_TOOLBAR (toolbar),
				     GTK_TOOLBAR_SPACE_EMPTY);
	gtk_toolbar_set_space_size (GTK_TOOLBAR (toolbar), 6);

	/* iterate over the contents of the folder */
	for (l = bi->list; l != NULL; l = g_list_next (l))
	{
		BookmarkItem *bm = l->data;

		/* build the appropriate element */
		if (bm->type != BM_SEPARATOR)
		{
			bookmarks_toolbar_create_from_bookmark
				(window, GTK_TOOLBAR (toolbar), bm);
		} 
		else 
		{
			gtk_toolbar_append_space (GTK_TOOLBAR (toolbar));
		}
	}

	gtk_signal_connect (GTK_OBJECT (toolbar), "drag_data_received",
			    GTK_SIGNAL_FUNC 
			    (bookmarks_toolbar_drag_data_received_cb), bi);
	gtk_drag_dest_set (toolbar,
			   GTK_DEST_DEFAULT_DROP,
			   bookmarks_dnd_targets,
			   bookmarks_dnd_targets_num_items,
			   GDK_ACTION_COPY | GDK_ACTION_MOVE |
			   GDK_ACTION_LINK);

	/* the click handler */
	gtk_signal_connect (GTK_OBJECT (toolbar->parent), 
			    "button-release-event",
			    GTK_SIGNAL_FUNC 
			    (bookmarks_toolbar_button_release_event_cb), bi);

	/* show it */
	gtk_widget_show_all (GTK_WIDGET (toolbar));

	/* add it to the per-window list of toolbars */
	window->bookmarks_toolbars = 
		g_list_append (window->bookmarks_toolbars,
			       GTK_TOOLBAR (toolbar));
}

/**
 * bookmarks_toolbar_create_from_bookmark:
 */
static void
bookmarks_toolbar_create_from_bookmark (GaleonWindow *window, GtkToolbar *tb,
					BookmarkItem *b)
{
        GtkWidget *button, *hbox, *pixmap, *entry;
	const PixmapData *bm_icon;
	GtkWidget *dnd_dest;
	gchar *name;
	
	/* build a box, and pack the pixmap and label inside */
	hbox = gtk_hbox_new (FALSE, 2);
	if (b->pixmap_data == NULL)
	{
		if (b->type == BM_CATEGORY || b->type == BM_AUTOBOOKMARKS)
		{
			pixmap = gtk_pixmap_new (folder_pixmap_data->pixmap,
						 folder_pixmap_data->mask);
		}
		else
		{
			bm_icon = bookmarks_get_siteicon (b->url);
			pixmap = gtk_pixmap_new (bm_icon->pixmap, 
						 bm_icon->mask);
		}
		name = strip_uline_accel (b->name);
		gtk_box_pack_start (GTK_BOX (hbox), pixmap, FALSE, FALSE, 0);
		gtk_box_pack_start (GTK_BOX (hbox), gtk_label_new (name),
				    TRUE, TRUE, 0);
		g_free (name);
	}
	else
	{
		pixmap = gtk_pixmap_new (b->pixmap_data->pixmap, 
					 b->pixmap_data->mask);
		gtk_box_pack_start (GTK_BOX (hbox), GTK_WIDGET (pixmap),
				    TRUE, FALSE, 0);
	}

	/* is this a normal (i.e. non-smart) bookmark */
	if (b->url == NULL || strstr(b->url, "%s") == NULL)
	{
		/* yes, build a button and pack it */
		dnd_dest = button = gtk_button_new ();
		gtk_container_add(GTK_CONTAINER(button), hbox);
		gtk_toolbar_append_widget(tb, button, b->url, NULL);
		gtk_button_set_relief (GTK_BUTTON (button), GTK_RELIEF_NONE);
		GTK_WIDGET_UNSET_FLAGS (button, GTK_CAN_FOCUS);
	}
	else
	{
		GtkWidget *button = gtk_button_new ();
		gint entry_width;

		dnd_dest = button;
		entry_width = gnome_config_get_int
			("/galeon/Appearance/smart_bm_entry_width=100");
		gtk_container_add (GTK_CONTAINER (button), 
				   GTK_WIDGET (hbox));
		gtk_button_set_relief (GTK_BUTTON (button), GTK_RELIEF_NONE);
		GTK_WIDGET_UNSET_FLAGS (button, GTK_CAN_FOCUS);
		gtk_container_set_border_width (GTK_CONTAINER (hbox), 0);
		/* otherwise build an entry field and pack it */
		entry = gtk_entry_new ();
		gtk_widget_set_usize(entry, entry_width, -2);
		gtk_toolbar_append_widget (tb, button, 
					   _("Search for entered phrase"),
					   NULL);
		gtk_toolbar_append_widget (tb, entry, b->url, NULL);
		gtk_toolbar_append_space (tb);
		gtk_object_set_data (GTK_OBJECT (button), "entry", entry);
		gtk_object_set_data (GTK_OBJECT (entry), "GaleonWindow", 
				     window);
		gtk_signal_connect (GTK_OBJECT (entry), "activate",
				    bookmarks_nick_entry_activate_cb, b);
	}

	gtk_signal_connect (GTK_OBJECT (dnd_dest), "drag_data_received",
			    GTK_SIGNAL_FUNC 
			    (bookmarks_toolbar_drag_data_received_cb), b);
	gtk_drag_dest_set (dnd_dest,
			   GTK_DEST_DEFAULT_ALL,
			   bookmarks_dnd_targets,
			   bookmarks_dnd_targets_num_items,
			   GDK_ACTION_COPY | GDK_ACTION_MOVE |
			   GDK_ACTION_LINK);
			   
	/* the click handler */
	gtk_signal_connect (GTK_OBJECT (dnd_dest), "button-release-event",
			    GTK_SIGNAL_FUNC 
			    (bookmarks_toolbar_item_button_release_event_cb),
			    b);				
}

#if 0
/**
 * bookmarks_toolbars_update_entries_global: search the bookmarks toolbars for
 *	smart bookmark entry widgets and sync their sizes with the
 *	preferences setting
 */
void
bookmarks_toolbars_update_entries_global (void)
{
	gint entry_width;
	GList *toolbars;
	GList *l;

	entry_width = gnome_config_get_int
		("/galeon/Appearance/smart_bm_entry_width=100");

	for (l = all_windows; l != NULL; l = g_list_next (l))
	{
		GaleonWindow *window = (GaleonWindow *)(l->data);

		for (toolbars = window->bookmarks_toolbars; toolbars != NULL;
		     toolbars = g_list_next (toolbars))
		{
			GtkToolbar *toolbar = GTK_TOOLBAR (toolbars->data);
			GList *children = toolbar->children;

			while (children)
			{
				GtkToolbarChild *child = children->data;

				if (GTK_IS_ENTRY (child->widget))
				{
					gtk_widget_set_usize (child->widget,
							      entry_width, -2);
				}
				children = children->next;
			}
		}
	}			
}
#endif

void 
bookmarks_toolbars_set_visibility (GaleonWindow *window, gboolean visible)
{
	GList *l;

	for (l = window->bookmarks_toolbars; l != NULL; l = g_list_next (l))
	{
		GtkToolbar *toolbar = GTK_TOOLBAR (l->data);

		if (GTK_IS_WIDGET (GTK_WIDGET (toolbar)->parent))
		{
			GtkWidget *w = GTK_WIDGET (toolbar)->parent;

			if (visible)
			{
				gtk_widget_show (w);
			}
			else
			{
				gtk_widget_hide (w);
			}
		}
	}
}

#define TOOLBAR_ITEM_CONTEXT_MENU_EDIT_POS 0
#define TOOLBAR_ITEM_CONTEXT_MENU_REMOVE_POS 1
#define TOOLBAR_ITEM_CONTEXT_MENU_HIDE_TOOLBAR_POS 3

/** 
 * bookmarks_toolbar_item_button_release_event_cb:
 * Handles clicks on bookmarks toolbar items.
 *
 * Button 1: Load bookmark in current window
 * Button 2: Load bookmark in new tab/window
 * Button 3: Show context menu
 */
static gboolean
bookmarks_toolbar_item_button_release_event_cb (GtkWidget *item, 
						GdkEventButton *event,
						BookmarkItem *bi)
{
	static GnomeUIInfo toolbar_item_context_menu_uiinfo[] =
	{
		GNOMEUIINFO_ITEM_STOCK (N_("Edit..."), NULL, NULL,
				        GNOME_STOCK_MENU_PROP),
		GNOMEUIINFO_ITEM_STOCK (N_("Remove bookmark"), NULL, NULL,
				        GNOME_STOCK_MENU_CLOSE),
		GNOMEUIINFO_SEPARATOR,
		GNOMEUIINFO_ITEM_STOCK (N_("Hide toolbar"), NULL, NULL,
					GNOME_STOCK_MENU_BLANK),
		GNOMEUIINFO_END
	};
	GtkWidget *WMain = window_lookup_widget (GTK_WIDGET(item), "WMain");
	GaleonWindow *window = gtk_object_get_data(GTK_OBJECT(WMain),
						   "GaleonWindow");
	GtkWidget *entry;
	GtkWidget *popup;
	gint item_action;
	gboolean autobookmarks_deleted = FALSE;
	gchar *text;
	BookmarksEditorControls *controls;
	
	if (item->window != event->window)
		return FALSE;

	if (event->button == 1)
	{
		/* Load the bookmark or create the folder menu */
		if ((bi->type == BM_SITE) && !GTK_IS_EVENT_BOX(item))
		{
			if (strstr (bi->url, "%s") == NULL)
			{
				/* normal bookmark */
				window_load_url(window, bi->url);
			}
			else
			{
				/* smart bookmark, do search for phrase */
				entry = gtk_object_get_data (GTK_OBJECT (item),
							     "entry");
				text = gtk_entry_get_text (GTK_ENTRY (entry));
				if (text != NULL && strlen (text) > 0)
				{
					find_next (window->active_embed, text);
				}
				else
				{
					BookmarkItem *b = 
						bookmarks_find_by_nick (
								bi->nick,
								bookmarks_root,
								FALSE);
					if (b != NULL && b->url != NULL)
					{
						window_load_url (window, 
								 b->url);
					}
				}
			}
		}
		else if ((bi->type == BM_CATEGORY) || (bi->type == BM_AUTOBOOKMARKS))
		{
			GtkWidget *menu = gtk_menu_new();
			GtkTooltips *tooltips = gtk_tooltips_new ();

			bookmarks_menu_create_recursively
				(bi, GTK_MENU (menu), NULL, tooltips, 
				 TRUE, TRUE);

			/* attach "WMain" to the menu so it can be looked up */
			gtk_object_set_data (GTK_OBJECT (menu), "WMain", WMain);
			gtk_object_set_data (GTK_OBJECT (menu), "tooltips",
					     tooltips);
			gtk_object_set_data (GTK_OBJECT (menu), "widget", item);

			/* show the menu in a timeout function so GTK+ will
			   clean up any pointer grabs before displaying the
			   popup menu */
			gtk_timeout_add (50,
				(GtkFunction) bookmarks_folder_show_menu_cb,
				menu);
		}
	}
	else if (event->button == 2)
	{
		gboolean tabbed_mode;

		tabbed_mode = gnome_config_get_bool (CONF_APPEARANCE_TABBED);
		if (event->state & GDK_SHIFT_MASK)
			tabbed_mode = !tabbed_mode;

		/* Load the bookmark in a new tab/window */
		if ((bi->type == BM_SITE) && !GTK_IS_EVENT_BOX(item))
		{
			embed_create_from_url (window->active_embed, bi->url,
					       FALSE, !tabbed_mode);

		}
		else if (bi->type == BM_AUTOBOOKMARKS || bi->type == BM_CATEGORY)
		{
			bookmarks_folder_open_all_items (window->active_embed,
							 bi, !tabbed_mode,
							 FALSE);
		}
	}
	else if (event->button == 3)
	{
		popup = gnome_popup_menu_new(toolbar_item_context_menu_uiinfo);
		item_action = gnome_popup_menu_do_popup_modal(popup, NULL,
							      NULL, NULL, NULL);
		gtk_widget_destroy(popup);
		
		switch (item_action) 
		{
		case TOOLBAR_ITEM_CONTEXT_MENU_EDIT_POS:
			controls = bookmarks_editor_show_dialog 
				(window, STANDARD_BOOKMARKS_EDITOR);
			bookmarks_editor_select_bookmark (controls, bi);
			break;
		case TOOLBAR_ITEM_CONTEXT_MENU_REMOVE_POS:
			if (bi->type == BM_AUTOBOOKMARKS)
			{
				autobookmarks_deleted = TRUE;
			}			
			bookmarks_editor_remove_tree_items (bi);
			bookmarks_remove_recursively (bi);
			if (autobookmarks_deleted)
			{
				autobookmarks_root = NULL;
			}
			bookmarks_updated ();
			break;
		case TOOLBAR_ITEM_CONTEXT_MENU_HIDE_TOOLBAR_POS:
			bi->parent->create_toolbar = FALSE;
			bookmarks_updated ();
			break;
		}
		return FALSE;
	}
	return TRUE;
}

#define TOOLBAR_CONTEXT_MENU_EDIT_POS 0
#define TOOLBAR_CONTEXT_MENU_HIDE_TOOLBAR_POS 2

static gboolean
bookmarks_toolbar_button_release_event_cb (GtkWidget *item, 
					   GdkEventButton *event,
					   BookmarkItem *bm)
{
	static GnomeUIInfo toolbar_item_context_menu_uiinfo[] =
	{
		GNOMEUIINFO_ITEM_STOCK (N_("Edit..."), NULL, NULL,
				        GNOME_STOCK_MENU_PROP),
		GNOMEUIINFO_SEPARATOR,
		GNOMEUIINFO_ITEM_STOCK (N_("Hide toolbar"), NULL, NULL,
					GNOME_STOCK_MENU_BLANK),
		GNOMEUIINFO_END
	};
	GtkWidget *popup;
	gint item_action;
	BookmarksEditorControls *controls;
	GtkWidget *WMain = window_lookup_widget (GTK_WIDGET(item), "WMain");
	GaleonWindow *window = gtk_object_get_data(GTK_OBJECT(WMain),
						   "GaleonWindow");

	if (event->button == 3)
	{
		popup = gnome_popup_menu_new(toolbar_item_context_menu_uiinfo);
		item_action = gnome_popup_menu_do_popup_modal(popup, NULL,
							      NULL, NULL, NULL);
		gtk_widget_destroy(popup);
		
		switch (item_action) {
		case TOOLBAR_CONTEXT_MENU_EDIT_POS:
			controls = bookmarks_editor_show_dialog 
				(window, STANDARD_BOOKMARKS_EDITOR);
			bookmarks_editor_select_bookmark (controls, bm);
			break;
		case TOOLBAR_CONTEXT_MENU_HIDE_TOOLBAR_POS:
			bm->create_toolbar = FALSE;
			bookmarks_updated ();
			break;
		}
		return FALSE;
	}
	return TRUE;
}

static int
bookmarks_folder_show_menu_cb (GtkWidget *menu)
{
	GtkTooltips *tooltips = gtk_object_get_data (GTK_OBJECT (menu),
						     "tooltips");
	GtkWidget *widget = gtk_object_get_data (GTK_OBJECT (menu), "widget");

	g_return_val_if_fail (menu != NULL, FALSE);

	gnome_popup_menu_do_popup_modal (menu, menu_position_under_widget,
					 widget, NULL, NULL);
	gtk_widget_destroy (menu);
	if (tooltips)
	{
		gtk_object_destroy (GTK_OBJECT (tooltips));
	}

	return FALSE;
}

/**
 * bookmarks_nick_entry_activate_cb: called when the user hits return
 * on an entry field on the toolbar (i.e. one created by a bookmark
 * which has %s in the url)
 */
static void 
bookmarks_nick_entry_activate_cb (GtkEntry *entry, BookmarkItem *bi)
{
	gchar *text, *url, *translated_text;
	GaleonWindow *window;

	/* find the window */
	window = gtk_object_get_data (GTK_OBJECT (entry), "GaleonWindow");

	/* get the entry text: DON'T free the returned text! */
	text = gtk_entry_get_text (entry);

	/* translate non-alphanumeric characters into %{} format */
	translated_text = bookmarks_translate_string (text);

	/* get a completed url */
	url = bookmarks_substitute_argument (bi, translated_text);

	/* load it in this window */
	window_load_url (window, url);

	/* free all allocated strings */
	g_free (url);
	g_free (translated_text);
}

/** 
 * bookmarks_toolbar_drag_data_received_cb:
 */
static void
bookmarks_toolbar_drag_data_received_cb (GtkWidget *widget, 
					 GdkDragContext *drag_context,
					 gint x, gint y,
					 GtkSelectionData *selection_data,
					 guint info, guint time,
					 BookmarkItem *drag_bm)
{
	BookmarkItem *b = NULL; 
	gchar *mem = selection_data->data;
	
	g_return_if_fail (drag_bm != NULL);

	switch (info)
	{
	case DND_TARGET_GALEON_BOOKMARK:
		b = bookmarks_item_from_string (mem);
		break;
	case DND_TARGET_STRING:
	case DND_TARGET_NETSCAPE_URL:
	case DND_TARGET_GALEON_URL:
		b = bookmarks_new_bookmark (BM_SITE, TRUE, NULL, mem, 
					    NULL, NULL, NULL);
		break;
	default:
		g_warning ("Unknown DND type");
		break;
	}
	if (b)
	{
		bookmarks_insert_bookmark (b, drag_bm, GTK_CLIST_DRAG_INTO);
		bookmarks_editor_place_tree_items (b);
		bookmarks_updated ();
	 }
}

