#ifndef PLAYER_H
#define PLAYER_H

#include <qobject.h>
#include <qtimer.h>
#include <kurl.h>
#include <noatunplaylist.h>

class Engine;
class PlayList;
class KLibrary;

/**
 * This class has slots for all the common media player buttons
 * The slots are called, and it queries the PlayList for the appropriate
 * file.
 **/
class Player : public QObject
{
Q_OBJECT
friend class Effects;
friend class PlayListItem;

public:
	enum LoopType { None=0, Song, Playlist };
	
public:
	Player(QObject *parent=0);
	~Player();

	/**
	 * returns a string with the time that can
	 * be used in the UI:
	 * CC:CC/LL:LL (mm:ss)
	 **/
	QString lengthString(int _position=-1);
	/**
	 * returns the LoopType enum
	 **/
	int loopStyle() const { return mLoopStyle; }
	/**
	 * the volume from 0-100
	 * use volume() instead
	 **/
	int volume() const;
	/**
	 * the position in mille
	 **/
	int getTime() const { return position; }
	
	/**
	 * true if we're playing
	 **/
	bool isPlaying();
	/**
	 * true if paused
	 **/
	bool isPaused();
	/**
	 * true if stopped
	 **/
	bool isStopped();

	/**
	 * get the current playlist
	 * this may be null
	 * And this may not be necessarily an item allocated
	 * by playlist()
	 **/
	PlayListItem *current() const { return mCurrent;}

	/**
	 * load the @param file, clear the playlist if @param purg is set
	 * and if autoplay is set, play the item.
	 **/
	void openFile(const KURL &file, bool purge=true, bool autoplay=false);
	
	Engine *engine() const { return mEngine; }
	
public slots:
	/**
	 * show or hide the playlist
	 **/
	void toggleListView();
	/**
	 * force the playing/paused/stopped/playlist shown signals to
	 * be sent out
	 **/
	void handleButtons();
	/**
	 * remove current from playlist
	 **/
	void removeCurrent();
	
	/**
	 * go back a track
	 **/
	void back();
	/**
	 * stop playing
	 **/
	void stop();
	/**
	 * start playing
	 **/
	void play();
	/**
	 * start playing, or pause if we're currently playing
	 **/
	void playpause();
	/**
	 * go forward a track
	 **/
	void fastForward();
	/**
	 * skip to the position, proportional to the length
	 * of the current file, from 0-1000
	 **/
	void skipTo(int perMille);
	void loop();
	void loop(int i);
	
	void newCurrent();
	void playCurrent();
	void setVolume(int);

	void toggleInterfaces();
	void showInterfaces();
	void hideInterfaces();
	
private slots:
	void posTimeout();

signals:
	/**
	 * Tells you to update the seekbar and volume
	 **/
	void timeout();

	/**
	 * Tells you to hide
	 */
	void hideYourself();

	/**
	 * Tells you to show again
	 */
	void showYourself();
	
	void stopped();
	void playing();
	void paused();
	void loopTypeChange(int t);
	void playlistShown();
	void playlistHidden();
	void newSongLen(int mins, int sec);
	void newSong();

private:
	Engine *mEngine;
	QTimer filePos;
	int position;
	int mLoopStyle;
	bool firstTimeout;
	PlayListItem *mCurrent;
	bool mDeleteMe;
	bool showingInterfaces;
};

#endif

