/*  XMPS - X MPEG Player System
 *  Copyright (C) 1999 Damien Chavarria
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public Licensse as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
 * main : main program for xmps.
 *
 */

/*********************************************************************
 *                             INCLUDES                              *
 *********************************************************************/

/* 
 * includes : local
 *
 */

#include "libxmps/libxmps.h"
#include "libxmpsutils/libxmpsutils.h"

/*
 * includes : system
 *
 */

#include <stdio.h>

/*********************************************************************
 *                            VARIABLES                              *
 *********************************************************************/

xmps_session_t     *session;
xmps_playlist_t    *playlist;
xmps_gui_data_t     gui_data;
xmps_gui_plugin_t  *gui;
xmps_menu_t        *shared_menu;
xmps_config_file_t *cfgfile;
char               *gui_plugin_name;


/*********************************************************************
 *                            FUNCTIONS                              *
 *********************************************************************/

/*
 * xmps_shared_menu_callback
 *
 * - the main callback for the
 *   shared menu.
 *
 */

void xmps_shared_menu_callback(unsigned int id, 
			       unsigned int on)
{
  xmps_gui_plugin_t *wanted;

  if(session == NULL) {
    return;
  }

  if(on) {
    
    xmps_plugin_types_t type;
    
    type = xmps_get_plugin_type_from_id(session, id);
		
    XMPS_DEBUG("got type : %d", type);

    switch(type)
      {
      case XMPS_PLUGIN_GUI:
	
	wanted = (xmps_gui_plugin_t *) xmps_get_plugin_from_id(session, id);
	
	XMPS_DEBUG("want to switch to %s", wanted->name);

	if(wanted != NULL && 
	   wanted->id != gui->id)
	  {
	    XMPS_DEBUG("switching gui to %s", 
		       wanted->name);
	    
	    xmps_playback_close(session);
	    
	    gui->close(gui);

	    gui = wanted;
	    
	    gui_plugin_name = gui->name;
	    
	    gui->init(gui, &gui_data);
	    gui->main(gui);
	  }
	break;
      default:
	break;
      }
  }
}

/*
 * xmps_create_shared_menu
 *
 * - creates the main menu 
 *   based on current plugins
 *
 */

xmps_menu_t *xmps_create_shared_menu()
{
	xmps_menu_t                  *menu;
	GList                        *gui_plugins_list, *item;
	xmps_gui_plugin_t            *gui_plugin;
	GList                        *renderer_plugins_list;
	xmps_video_renderer_plugin_t *renderer_plugin;
	GList                        *audio_renderer_plugins_list;
	xmps_audio_renderer_plugin_t *audio_renderer_plugin;

	if(gui_plugin_name == NULL) {
		
	  XMPS_DEBUG("current gui plugin is null!");
	  
	  return NULL;
	}
	
	menu = xmps_menu_new(xmps_shared_menu_callback);
	
	gui_plugins_list = session->plugin_center->gui_plugins_list;

	for(item = gui_plugins_list; item != NULL; item = g_list_next(item)) {
		
		if(item->data != NULL) {
			
			unsigned int active;
						
			gui_plugin         = (xmps_gui_plugin_t *) item->data;
						
			if(gui == NULL) {
				
			  XMPS_DEBUG("current gui plugin is null!");
			  
			  return NULL;
			}
			
			if(strstr(gui_plugin->name, gui->name) != NULL)
			  active = 1;
			else
			  active = 0;

			/*
			 * we add the item for the GUI
			 *
			 */

			xmps_menu_add(menu, 
				      xmps_menu_item_new(gui_plugin->id, 
							 gui_plugin->name, 
							 XMPS_MENU_ITEM_TYPE_RADIO, 
							 1, 
							 active));
		}
	}
  
	/*
	 * we add a separator
	 *
	 */

	xmps_menu_add(menu, 
		      xmps_menu_item_new(0, 
					 "-", 
					 XMPS_MENU_ITEM_TYPE_ITEM, 
					 0, 
					 0));

	/*
	 * and proceed renderers (video)
	 *
	 */
	
	renderer_plugins_list = session->plugin_center->video_renderer_plugins_list;
  
	for(item = renderer_plugins_list; item != NULL; item = g_list_next(item))
	{
		if(item->data != NULL)
		{
		  renderer_plugin = (xmps_video_renderer_plugin_t *) item->data;
		  xmps_menu_add(menu, 
				xmps_menu_item_new(renderer_plugin->id, 
						   g_strconcat(renderer_plugin->name, " Renderer", NULL), 
						   XMPS_MENU_ITEM_TYPE_RADIO, 
						   2, 
						   0));
		}
	}
	
	/*
	 * we add a separator
	 *
	 */

	xmps_menu_add(menu, 
		      xmps_menu_item_new(0, 
					 "-", 
					 XMPS_MENU_ITEM_TYPE_ITEM, 
					 0, 
					 0));

	/*
	 * and proceed renderers (audio)
	 *
	 */
	
	audio_renderer_plugins_list = session->plugin_center->audio_renderer_plugins_list;
  
	for(item = audio_renderer_plugins_list; item != NULL; item = g_list_next(item))
	{
		if(item->data != NULL)
		{
			audio_renderer_plugin = (xmps_audio_renderer_plugin_t *) item->data;
			xmps_menu_add(menu, 
				      xmps_menu_item_new(audio_renderer_plugin->id, 
							 g_strconcat(audio_renderer_plugin->name, " Renderer", NULL), 
							 XMPS_MENU_ITEM_TYPE_RADIO, 
							 3, 
							 0));
		}
	}

	return menu;
}

void xmps_main_load_options()
{
	/*
	 * put a default value (GNOME
	 *
	 * 
	 * TODO : needs #ifdefs
	 *
	 */

	gui_plugin_name = (char *) malloc(15);
	gui_plugin_name = "GNOME";

	if(cfgfile) {
		xmps_config_read_string(cfgfile, "main", "gui", &gui_plugin_name);
	} 
}

void xmps_main_save_options()
{
	char *filename;

	/*
	 * build a default file name.
	 *
	 */

	filename = g_strconcat(g_get_home_dir(), 
			       "/.xmps/config", 
			       NULL);

	if (!cfgfile)
		cfgfile = xmps_config_file_new();

	xmps_config_write_string(cfgfile, "main", "gui", gui_plugin_name);
	xmps_config_write_file(cfgfile, filename);
}

/*
 * main
 *
 * - main function for xmps
 *   creates playback, playlist, ...
 *   and initialize GUIs
 *
 */

int main(int argc, char **argv)
{
	int                 index;
	char               *filename;
	struct stat         buf;
	gchar              *xmps_dir;
	GList              *node;
	
	/* 
	 * check for first time launch 
	 *
	 */

	xmps_dir = g_strconcat(g_get_home_dir(), 
			       "/.xmps", 
			       NULL);
  
	if (stat(xmps_dir, &buf) != 0) {

	  XMPS_DEBUG("first time running xmps! Welcome.");
      
	  xmps_make_config_dir();
	}
	
	/* 
	 * config file 
	 *
	 */

	filename = g_strconcat(g_get_home_dir(), 
			       "/.xmps/config", 
			       NULL);
	
	cfgfile  = xmps_config_open_file(filename);

	xmps_main_load_options();

	/* 
	 * objects creation
	 *
	 */
 
	playlist       = xmps_playlist_new();

	/*
	 * lib init
	 */

	session = xmps_init(argc, argv, cfgfile);

	/*
	 * scan for available plugins.
	 *
	 */

	xmps_load_all(session);
	xmps_playback_init(session);
	
	/* 
	 * proceed command-line options 
	 *
	 */

	XMPS_DEBUG("number of args %d", 
		   argc);

	if(argc > 1)
	{
		/* 
		 * try and see if this files 
		 * 
		 */
		
		xmps_item_t *item;

		for(index = 1; index < argc; index++)
		{
			item = xmps_item_new(XMPS_MEDIA_FILE, 
					     argv[index]);
			
			XMPS_DEBUG("adding %s to playlist", 
				   item->full_path);
				
			xmps_playlist_add(playlist, 
					  item);
		}
	}

	/*
	 * set selected GUI.
	 *
	 */

	gui = NULL;


	for(node = session->plugin_center->gui_plugins_list; node != NULL; node = g_list_next(node)) {
	  
	  xmps_gui_plugin_t *the_gui;

	  the_gui = (xmps_gui_plugin_t *) node->data;

	  if(the_gui != NULL) {

	    if(strstr(the_gui->name, gui_plugin_name) != NULL) {
	      gui = the_gui;
	    }
	  }
	}
	
	if(gui == NULL) {
	  
	  if(session->plugin_center->gui_plugins_list == NULL) {
	    
	    XMPS_DEBUG("no gui plugin could be found, check your installation!");
	    exit(1);
	  }
	  
	  gui = (xmps_gui_plugin_t *) session->plugin_center->gui_plugins_list->data;
	  
	  if(gui == NULL) {
	    
	    XMPS_DEBUG("no gui plugin could be found, check your installation!");
	    exit(1);
	  }
	  
	}
	
	/* 
	 * the shared menu 
	 *
	 */

	shared_menu = xmps_create_shared_menu();

	if(shared_menu == NULL)
	{
	  XMPS_DEBUG("Could not create shared menu!");
	  exit(1);
	}

	/* 
	 * main session info
	 *
	 */
  
	gui_data.argc        = argc;
	gui_data.argv        = argv;
	gui_data.session     = session;
	gui_data.playlist    = playlist;
	gui_data.shared_menu = shared_menu;
	gui_data.cfgfile     = cfgfile;

	XMPS_DEBUG("%s init",
		   gui->name);
	
	gui->init(gui, &gui_data);

	XMPS_DEBUG("%s main()",
		   gui->name);

	gui->main(gui);

	/* 
	 * object cleanup 
	 *
	 */
	
	xmps_playlist_destroy(playlist);
	
	/* 
	 * save all options 
	 *
	 */

	xmps_main_save_options();

	XMPS_DEBUG("exiting...");

	return 0;
}




