/*  XMPS - X MPEG Player System
 *  Copyright (C) 1999 Damien Chavarria
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public Licensse as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/**
 *
 * sdl_audio_renderer.c
 *
 * SDL Audio Renderer.
 * @Author Chavarria Damien.
 * 	   Copyright 1999-2000
 */

/*********************************************************************
 *                             INCLUDES                              *
 *********************************************************************/

#include "sdl_audio_renderer.h"

/*********************************************************************
 *                             VARIABLES                             *
 *********************************************************************/

typedef struct {

  SDL_AudioSpec               *wanted;
  SDL_AudioSpec               *obtained;
  xmps_audio_callback          audio_callback;
  void                        *user_data;
  int                          our_stream_id;
  int                          volume;
  Uint8                       *empty;

} sdl_audio_t;

/*
 * get_audio_renderer_info : MANDATORY
 *
 * Used by scanplugins to get a pointer to the renderer.
 *
 */

xmps_audio_renderer_plugin_t *get_audio_renderer_info(void)
{
  xmps_audio_renderer_plugin_t *audio_renderer;
  
  audio_renderer = (xmps_audio_renderer_plugin_t *) malloc(sizeof(xmps_audio_renderer_plugin_t));

  audio_renderer->name = "SDL Audio";
  audio_renderer->data = malloc(sizeof(sdl_audio_t));

  audio_renderer->open  = sdl_audio_open;
  audio_renderer->get   = sdl_audio_get;
  audio_renderer->set   = sdl_audio_set;
  audio_renderer->start = sdl_audio_start;
  audio_renderer->close = sdl_audio_close;

  ((sdl_audio_t *) audio_renderer->data)->wanted         = NULL;
  ((sdl_audio_t *) audio_renderer->data)->obtained       = NULL;
  ((sdl_audio_t *) audio_renderer->data)->audio_callback = NULL; 
  ((sdl_audio_t *) audio_renderer->data)->user_data      = NULL;
  ((sdl_audio_t *) audio_renderer->data)->our_stream_id  = 0;
  ((sdl_audio_t *) audio_renderer->data)->volume         = 100;
  ((sdl_audio_t *) audio_renderer->data)->empty          = NULL;

  return audio_renderer;
}

/*********************************************************************
 *                            FUNCTIONS                              *
 *********************************************************************/

/*
 * INTERNAL
 * 
 *
 */

void sdl_audio_callback(void *userdata, Uint8 *stream, int len)
{
  int                           i;
  sdl_audio_t                  *data;


  data = (sdl_audio_t *) userdata;

  if(data == NULL) {
    return;
  }
  
  /*
   * this function is called when 
   * the audio device needs data
   *
   */

  if(data->audio_callback == NULL) {

    XMPS_DEBUG("null callback!");

    return;
  }

  if(data->user_data == NULL) {
    XMPS_DEBUG("null user data");
  }

  data->audio_callback(data->user_data, (void *) stream, len);

  /*
   * proceedeing to volume adjustment.
   *
   */

  //XMPS_DEBUG("volume pass");
  
  if(data->volume != 100)
    {
      for(i = 0; i < len/2; i++)
	{
	  *((short *) stream + i) /= 100;
	  *((short *) stream + i) *= data->volume;
	}
    }
  
  //XMPS_DEBUG("sound done!");
}

unsigned int sdl_audio_open(xmps_audio_renderer_plugin_t *audio_renderer)
{
  if(audio_renderer == NULL) {
    return 0;
  }

  return 1;
}

unsigned int sdl_audio_init(xmps_audio_renderer_plugin_t *audio_renderer, xmps_audio_info_t *info)
{
  sdl_audio_t *data;

  if(audio_renderer == NULL) {
    return 0;
  }

  data = (sdl_audio_t *) audio_renderer->data; 

  if(data->wanted != NULL) {
    
    XMPS_DEBUG("already initialized!");

    return 0;
  }

  if(info == NULL)
    return 0;

  SDL_InitSubSystem(SDL_INIT_AUDIO);
  
  data->wanted   = (SDL_AudioSpec *) malloc(sizeof(SDL_AudioSpec));
  data->obtained = (SDL_AudioSpec *) malloc(sizeof(SDL_AudioSpec));

  /*
   * we fill-in the desired spec
   *
   */

  data->wanted->freq = info->frequency;

  switch(info->format.type)
    {
    case XMPS_AUDIO_FORMAT_U8:
      data->wanted->format = AUDIO_U8;
      break;
    case XMPS_AUDIO_FORMAT_S8:
      data->wanted->format = AUDIO_S8;
      break;
    case XMPS_AUDIO_FORMAT_U16:
      data->wanted->format = AUDIO_U16;
      break;
    case XMPS_AUDIO_FORMAT_S16:
      data->wanted->format = AUDIO_S16;
      break;
    default:
      data->wanted->format = AUDIO_U16;
      break;
    }

  data->wanted->channels = info->channels;
  data->wanted->samples  = 16384; //1024;
  data->wanted->callback = sdl_audio_callback;
  data->wanted->userdata = (void *) data;
  
  XMPS_DEBUG("setting parameters %db, %dch, %dHz",
	     16, data->wanted->channels, data->wanted->freq);

  if( SDL_OpenAudio(data->wanted, data->obtained) != -1) {

    XMPS_DEBUG("intitialization ok!");
    
    return 1;
    
  }

  return 0;
}

void* sdl_audio_get(xmps_audio_renderer_plugin_t *audio_renderer, unsigned int flag, void* arg)
{
  sdl_audio_t *data;

  if(audio_renderer == NULL) {
    return 0;
  }

  data = (sdl_audio_t *) audio_renderer->data; 

  switch(flag)
    {
    case XMPS_FLAG_XML_CONFIG:
      {
	GList *list = NULL;

	list = g_list_prepend(list, xmps_xml_config_entry_string_new("buffer_size", "1600", "entry", "Audio Buffer Size (in samples)"));
	
	return list;
      }
      break;

    case XMPS_FLAG_AUDIO_FORMAT_LIST:
      {
	GList *list = NULL;
	xmps_audio_format_t *f;
	
	f = (xmps_audio_format_t *) malloc(sizeof(xmps_audio_format_t));

	f->type      = XMPS_AUDIO_FORMAT_U8;
	f->bit_count = 8;

	list = g_list_prepend(list, (void *) f);

	f = (xmps_audio_format_t *) malloc(sizeof(xmps_audio_format_t));

	f->type      = XMPS_AUDIO_FORMAT_S8;
	f->bit_count = 8;

	list = g_list_prepend(list, (void *) f);

	f = (xmps_audio_format_t *) malloc(sizeof(xmps_audio_format_t));

	f->type      = XMPS_AUDIO_FORMAT_U16;
	f->bit_count = 16;

	list = g_list_prepend(list, (void *) f);

	f = (xmps_audio_format_t *) malloc(sizeof(xmps_audio_format_t));

	f->type      = XMPS_AUDIO_FORMAT_S16;
	f->bit_count = 16;

	list = g_list_prepend(list, (void *) f);

	return list;
      }
      break;

    case XMPS_FLAG_AUDIO_INFO:
      {
	xmps_audio_info_t *info;

	if(data->wanted == NULL)
	  return NULL;
	
	info = (xmps_audio_info_t *) malloc(sizeof(xmps_audio_info_t));

	info->channels  = data->wanted->channels;
	info->frequency = data->wanted->freq;

	return info;
      }
      break;
      
    default:
      break;
    }

  return NULL;
}

unsigned int sdl_audio_set(xmps_audio_renderer_plugin_t *audio_renderer, unsigned int flag, void *arg)
{
  sdl_audio_t *data;
  
  if(audio_renderer == NULL) {
    return 0;
  }

  data = (sdl_audio_t *) audio_renderer->data; 

  switch(flag)
    {
    case XMPS_FLAG_USER_DATA:
      
      XMPS_DEBUG("setting audio user data!");
      
      data->user_data = arg;
      break;
    case XMPS_FLAG_AUDIO_INFO:

      return sdl_audio_init(audio_renderer, (xmps_audio_info_t *) arg);
      break;

    case XMPS_FLAG_AUDIO_CALLBACK:
      
      XMPS_DEBUG("setting audio callback");

      data->audio_callback = ((xmps_audio_callback) arg);

      return 1;
      break;
      
    case XMPS_FLAG_AUDIO_VOLUME:
      
      data->volume = (unsigned int *) arg != NULL ? *((unsigned int *) arg) : 100;
      
      XMPS_DEBUG("setting audio volume to %d",
		 data->volume);

      break;

    default:
      break;
    }

  return 0;
}

unsigned int sdl_audio_start(xmps_audio_renderer_plugin_t *audio_renderer)
{
  sdl_audio_t *data;

  if(audio_renderer == NULL) {
    return 0;
  }

  data = (sdl_audio_t *) audio_renderer->data; 
  
  if(data->wanted == NULL)
    return 0;

  XMPS_DEBUG("starting asking for data");

  SDL_PauseAudio(0);

  return 1;
}

unsigned int sdl_audio_close(xmps_audio_renderer_plugin_t *audio_renderer)
{
  sdl_audio_t *data;

  if(audio_renderer == NULL) {
    return 0;
  }

  data = (sdl_audio_t *) audio_renderer->data; 

  data->wanted   = NULL;
  data->obtained = NULL;
    
  SDL_QuitSubSystem(SDL_INIT_AUDIO);

  return 0;
}








