/*  XMPS - X MPEG Player System
 *  Copyright (C) 1999 Damien Chavarria
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public Licensse as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
 * main : skinned window main file.
 *
 */

/* includes */

#include <skinned_window.h> 

/* menus */
GtkItemFactory *main_menu;

void main_menu_callback(gpointer cb_data, guint action, GtkWidget * w);

/* main menu */

enum
{
  MAIN_ABOUT, 
  MAIN_SKIN, 
  MAIN_SAVE, 
  MAIN_PREFS,
  MAIN_LOOP,
  MAIN_OPTIONS,
  MAIN_PLAYLIST,
  MAIN_QUIT
};

#define MAIN_MENU_ENTRIES 13

GtkItemFactoryEntry main_menu_entries[MAIN_MENU_ENTRIES] =
{
  { "/About XMPS", "<alt>A", main_menu_callback, MAIN_ABOUT, "<Item>" },
  { "/-", NULL, NULL, 0, "<Separator>" },
  { "/Playlist", "P", main_menu_callback, MAIN_PLAYLIST, "<ToggleItem>" },
  { "/-", NULL, NULL, 0, "<Separator>" },
  { "/Loop", "L", main_menu_callback, MAIN_LOOP, "<ToggleItem>" },
  { "/Active Plugins", NULL, main_menu_callback, MAIN_OPTIONS, "<Item>" },
  { "/-", NULL, NULL, 0, "<Separator>" },
  { "/Preferences...", "<control>G", main_menu_callback, MAIN_PREFS, "<Item>" },
  { "/Save Options Now", "<control>S",  main_menu_callback, MAIN_SAVE, "<Item>" },
  { "/-", NULL, NULL, 0, "<Separator>" },
  { "/Skin browser...", "<alt>S", main_menu_callback, MAIN_SKIN, "<Item>" },
  { "/-", NULL, NULL, 0, "<Separator>" },
  { "/Quit...", "<control>Q", main_menu_callback, MAIN_QUIT, "<Item>" }
};

/* 
 * functions 
 *
 */

void skinned_start_playing()
{
  xmps_time_t *t_info;

  if(xmps_playlist_get_nbr_items(xmps_config.playlist) > 0)
    {
      xmps_video_info_t *info;

      if(xmps_config.session == NULL)
	{
	  XMPS_DEBUG("null session");
	  return;
	}

      xmps_playback_setup_video(xmps_config.session, 1, 9, 29, ((GdkWindowPrivate *)(xmps_config.skinned_window->window))->xwindow, "SDL");
      
      if(xmps_playback_open(xmps_config.session, ((xmps_item_t *) xmps_playlist_get_item(xmps_config.playlist))->full_path))
	{
	  info = (xmps_video_info_t *) xmps_playback_get_video_info(xmps_config.session);
	  
	  if(info != NULL && xmps_playback_has_video(xmps_config.session))
	    {
	      XMPS_DEBUG("setting image size...");

	      xmps_set_image_size(xmps_config.skinned_window, info->width, info->height);
	      
	      if (xmps_config.is_playlist_toggled)
		{
		   xmps_config.playlist_size_x = 275 + (info->width - IMAGE_DEFAULT_SIZE_X) - 4;
      
		   gdk_window_resize(playlist_window->window, xmps_config.playlist_size_x,  105 + 15*(xmps_config.playlist_nbr_lines-3));
		   gtk_widget_set_usize(playlist_window, xmps_config.playlist_size_x,  105 + 15*(xmps_config.playlist_nbr_lines-3));

		   xmps_config.playlist_pos_x = xmps_config.window_pos_x;
		   xmps_config.playlist_pos_y = xmps_config.window_pos_y + xmps_config.window_size_y;
		   gtk_widget_set_uposition(playlist_window, xmps_config.playlist_pos_x, 
					    xmps_config.playlist_pos_y);
		}
	     
	      gdk_flush();
	    }
	    else {

		xmps_set_image_size(xmps_config.skinned_window, IMAGE_DEFAULT_SIZE_X, IMAGE_DEFAULT_SIZE_Y); 
		
	    }

	  /*
	  if(xmps_config.is_loop)
	    xmps_playback_set_flags(xmps_config.playback, XMPS_LOOP_SET, 1);
	  */

	  /* let's rock */
	  
	  xmps_config.has_to_play = 0;
	  xmps_playback_play(xmps_config.session);
	  
	  // set the total time
	  
	  
	  t_info = xmps_playback_get_time_info(xmps_config.session);
	  
	  if(t_info != NULL)
	    xmps_config.total_time = t_info->total_time;
	  else
	    xmps_config.total_time = 0;
	  
	  draw_current_skin(xmps_config.skinned_window, 1);
	  gdk_flush();
	  //	  draw_current_skin(xmps_config.skinned_window, 1);
	  //      gdk_flush();
	}
    }
}

gint skinned_gui_idle(gpointer data)
{
  gint event;

  if(xmps_config.has_to_play)
    {
      xmps_playlist_reset(xmps_config.playlist);
      xmps_config.has_to_play = 0;
      skinned_start_playing();
    }

  if(xmps_playback_is_playing(xmps_config.session))
    {
      xmps_time_t *t_info = NULL;
      
      /* 
       * get current playback time 
       */
      
      t_info = xmps_playback_get_time_info(xmps_config.session);
      
      if(t_info != NULL)
	{	  
	  draw_time(t_info->current_time, xmps_config.skinned_window);

	  if(xmps_config.current_action != PROGRESS_BAR_MOVING && t_info->total_time != 0)
	    
	    if( t_info->current_time * 100 / t_info->total_time != xmps_config.progress_bar.value)
	      {
		set_progress_bar_value(PROGRESS_BAR, t_info->current_time*100/t_info->total_time);
		draw_progression_bar(xmps_config.skinned_window);
	      }
	}
    }

  /*
   * getting events
   */

  do
    {
      event = pop_xmps_event();
      
      switch(event)
	{
	case XMPS_EVENT_BACK_FROM_FULLSCREEN:
	  
	  XMPS_DEBUG("back from fullscreen");
	  
	  xmps_config.fullscreen = 0; 
	  xmps_playback_video_fullscreen(xmps_config.session, 0);
	  break;
	default:
	  break;
	}
    }
  while(event != XMPS_EVENT_NO_EVENT);

  return TRUE;
}


void skinned_file_ok (GtkWidget *w, GtkFileSelection *fs)
{
  GList *file_node;
  char  *filename;
  gint   index;
  gchar *dir = NULL;
  struct stat buf;
  xmps_item_t *item;

  gtk_widget_hide(xmps_config.file_selection);
  
  filename = gtk_file_selection_get_filename(GTK_FILE_SELECTION(xmps_config.file_selection));

  /* remove the blanks */
  g_strstrip(filename);

  /* remember the directory into config */

  /* check if it's a directory or a file */
  if (stat(filename, &buf) == 0)
    {
      if (S_ISDIR(buf.st_mode))
	{
	  g_free(xmps_config.filesel_dir);
	  xmps_config.filesel_dir = (char*) malloc(strlen(filename)+1);
	  strcpy(xmps_config.filesel_dir, filename);
	  
	  /* set the dir for next time */
	  gtk_file_selection_set_filename(GTK_FILE_SELECTION(xmps_config.file_selection),  
					  g_strdup_printf("%s/", filename));
	  return;
	}
      else
	/* remember the dir in case of multiple sel */
	{
	  index = strlen(filename);
	  while (filename[index - 1] != '/')
	    {
	      filename[index - 1] = '\0';
	      index = strlen(filename);
	    }
	  
	  dir = filename;
	
	  xmps_config.filesel_dir = (char*) malloc(strlen(dir)+1);
	  strcpy(xmps_config.filesel_dir, dir);
	}
    }

  /* process the selected files */

  xmps_playback_stop(xmps_config.session);
  xmps_playback_close(xmps_config.session);
  xmps_playlist_clear(xmps_config.playlist);
  
  xmps_config.playlist_current_display_position = 1;

  for(file_node = GTK_CLIST(GTK_FILE_SELECTION(xmps_config.file_selection)->file_list)->selection; file_node != NULL; file_node = g_list_next(file_node))
    {
      gtk_clist_get_text(GTK_CLIST(GTK_FILE_SELECTION(xmps_config.file_selection)->file_list), (gint) file_node->data, 0, &filename);
      
      XMPS_DEBUG("trying to open %s", filename);

      item = (xmps_item_t *) xmps_item_new(XMPS_MEDIA_FILE, (char *) g_strconcat(dir, filename, NULL));

      xmps_playlist_add(xmps_config.playlist, item);
    }

  xmps_playlist_reset(xmps_config.playlist);
  skinned_start_playing();
}

void skinned_file_cancel (GtkWidget *widget, gpointer data)
{
  gtk_widget_hide(xmps_config.file_selection);
}

void skinned_file_destroy (GtkWidget *widget, gpointer data)
{
  xmps_config.file_selection = NULL;
}

void skinned_create_file_selection()
{
  xmps_config.file_selection = gtk_file_selection_new("Open Video files...");
  
  gtk_clist_set_selection_mode(GTK_CLIST(GTK_FILE_SELECTION(xmps_config.file_selection)->file_list), 
			       GTK_SELECTION_EXTENDED);
  
  gtk_signal_connect (GTK_OBJECT (GTK_FILE_SELECTION (xmps_config.file_selection)->ok_button),
		      "clicked", (GtkSignalFunc) skinned_file_ok, xmps_config.file_selection );
  gtk_signal_connect_object (GTK_OBJECT (GTK_FILE_SELECTION (xmps_config.file_selection)->cancel_button),
			     "clicked", (GtkSignalFunc) skinned_file_cancel, GTK_OBJECT (xmps_config.file_selection));
  gtk_signal_connect_object (GTK_OBJECT (GTK_FILE_SELECTION (xmps_config.file_selection)->cancel_button),
			     "destroy", (GtkSignalFunc) skinned_file_destroy, GTK_OBJECT (xmps_config.file_selection));
  
  gtk_file_selection_set_filename (GTK_FILE_SELECTION(xmps_config.file_selection), xmps_config.filesel_dir);
}

void skinned_initialize_xmps(xmps_config_file_t *cfgfile)
 {
   /* DEFAULT VALUES */

   XMPS_DEBUG("reading default valuse");

   xmps_config.has_focus = FALSE;
   xmps_config.playlist_has_focus = FALSE;

   xmps_config.window_pos_x = 250;
   xmps_config.window_pos_y = 100;

   xmps_config.window_size_x = 275;
   xmps_config.window_size_y = 150;
   
   xmps_config.allways_glued = FALSE;
   xmps_config.allow_multiples_instances = TRUE;
   xmps_config.reverse_file_order = FALSE;
   xmps_config.scroll_title = TRUE;
   xmps_config.confirm_fullscreen = FALSE;
   xmps_config.save_on_exit = TRUE;

   xmps_config.save_playlist_size = TRUE;
   xmps_config.save_playlist_content = TRUE;
   xmps_config.save_playlist_pos = FALSE;
   xmps_config.playlist_multiple_selection = TRUE;
   
   xmps_config.current_action = NONE;
   xmps_config.current_time   = NO_TIME;
   
   xmps_config.current_skin = (xmps_skin *) malloc(sizeof(xmps_skin));
   xmps_config.current_skin->name = (char *) malloc(9);
   xmps_config.current_skin->name = "(Default)";
   
   xmps_config.current_skin->directory = (char *) malloc(9);
   xmps_config.current_skin->directory = "(Default)";
   
   xmps_config.current_skin->playlist_bgcolor.red   = 179;
   xmps_config.current_skin->playlist_bgcolor.green = 197;
   xmps_config.current_skin->playlist_bgcolor.blue  = 198;
   xmps_config.current_skin->playlist_bgcolor.pixel = 1;
   
   xmps_config.current_skin->filename_bgcolor.red   = 179;
   xmps_config.current_skin->filename_bgcolor.green = 197;
   xmps_config.current_skin->filename_bgcolor.blue  = 198;
   xmps_config.current_skin->filename_bgcolor.pixel = 1;
   
   xmps_config.current_skin->playlist_selected.red   = 115;
   xmps_config.current_skin->playlist_selected.green = 165;
   xmps_config.current_skin->playlist_selected.blue  = 189;
   xmps_config.current_skin->playlist_selected.pixel = 1;

   xmps_config.current_skin->filename_font_color.red   = 0;
   xmps_config.current_skin->filename_font_color.green = 0;
   xmps_config.current_skin->filename_font_color.blue  = 0;
   xmps_config.current_skin->filename_font_color.pixel = 1;
   
   xmps_config.current_skin->playlist_font_color.red   = 0;
   xmps_config.current_skin->playlist_font_color.green = 0;
   xmps_config.current_skin->playlist_font_color.blue  = 0;
   xmps_config.current_skin->playlist_font_color.pixel = 1;
   
   xmps_config.is_playlist_toggled = FALSE;
   xmps_config.is_playlist_glued   = TRUE;
   
   xmps_config.is_invert_time = FALSE;
   xmps_config.is_loop        = FALSE;
   
   xmps_config.save_window_pos = TRUE;
   xmps_config.save_skin = TRUE;

   xmps_config.showing_prop = 0;
   xmps_config.fullscreen = 0;
   xmps_config.filesel_dir = g_strconcat(g_get_home_dir(), "/", NULL);
   
   if(cfgfile)
     {
       XMPS_DEBUG("reading from cfgfile");


       xmps_config_read_int(cfgfile, "skinned_gui", "window_pos_x", &xmps_config.window_pos_x);
       xmps_config_read_int(cfgfile, "skinned_gui", "window_pos_y", &xmps_config.window_pos_y);

       xmps_config_read_boolean(cfgfile, "skinned_gui", "playlist_toggled", &xmps_config.is_playlist_toggled);
       xmps_config_read_boolean(cfgfile, "skinned_gui", "playlist_glued", &xmps_config.is_playlist_glued);
       xmps_config_read_boolean(cfgfile, "skinned_gui", "invert_time", &xmps_config.is_invert_time);
       xmps_config_read_boolean(cfgfile, "skinned_gui", "loop", &xmps_config.is_loop);
       xmps_config_read_boolean(cfgfile, "skinned_gui", "fullscreen", &xmps_config.fullscreen);
       xmps_config_read_boolean(cfgfile, "skinned_gui", "allways_glued", &xmps_config.allways_glued);
       xmps_config_read_boolean(cfgfile, "skinned_gui", "allow_multiples_instances", &xmps_config.allow_multiples_instances);
       xmps_config_read_boolean(cfgfile, "skinned_gui", "reverse_file_order", &xmps_config.reverse_file_order);
       xmps_config_read_boolean(cfgfile, "skinned_gui", "scroll_title", &xmps_config.scroll_title);
       xmps_config_read_boolean(cfgfile, "skinned_gui", "confirm_fullscreen", &xmps_config.confirm_fullscreen);
       xmps_config_read_boolean(cfgfile, "skinned_gui", "save_on_exit", &xmps_config.save_on_exit);
       
       xmps_config_read_boolean(cfgfile, "skinned_gui", "save_playlist_size", &xmps_config.save_playlist_size);
       xmps_config_read_boolean(cfgfile, "skinned_gui", "save_playlist_pos", &xmps_config.save_playlist_pos);
       xmps_config_read_boolean(cfgfile, "skinned_gui", "save_playlist_content", &xmps_config.save_playlist_content);
       xmps_config_read_boolean(cfgfile, "skinned_gui", "playlist_multiple_selection", &xmps_config.playlist_multiple_selection);
       xmps_config_read_boolean(cfgfile, "skinned_gui", "save_window_pos", &xmps_config.save_window_pos);
       xmps_config_read_boolean(cfgfile, "skinned_gui", "save_skin", &xmps_config.save_skin);
       xmps_config_read_boolean(cfgfile, "skinned_gui", "is_playlist_toggled", &xmps_config.is_playlist_toggled);
       
       
       xmps_config_read_string(cfgfile, "skinned_gui", "filesel_dir", &xmps_config.filesel_dir);
       xmps_config_read_string(cfgfile, "skinned_gui", "skin_path", &xmps_config.current_skin->directory);
       xmps_config_read_string(cfgfile, "skinned_gui", "skin_name", &xmps_config.current_skin->name);
     }
}

/*************************** save_config() ******************************
 *									*
 ************************************************************************/

void skinned_save_config(xmps_config_file_t *cfgfile)
{
  char *filename;

  filename = g_strconcat(g_get_home_dir(), "/.xmps/config", NULL);

  if (!cfgfile)
    cfgfile = xmps_config_file_new();

  xmps_config_write_int(cfgfile, "skinned_gui", "window_pos_x", xmps_config.window_pos_x);
  xmps_config_write_int(cfgfile, "skinned_gui", "window_pos_y", xmps_config.window_pos_y);
  
  xmps_config_write_boolean(cfgfile, "skinned_gui", "playlist_toggled", xmps_config.is_playlist_toggled);
  xmps_config_write_boolean(cfgfile, "skinned_gui", "playlist_glued", xmps_config.is_playlist_glued);
  xmps_config_write_boolean(cfgfile, "skinned_gui", "invert_time", xmps_config.is_invert_time);
  xmps_config_write_boolean(cfgfile, "skinned_gui", "loop", xmps_config.is_loop);
  xmps_config_write_boolean(cfgfile, "skinned_gui", "fullscreen", xmps_config.fullscreen);
  xmps_config_write_boolean(cfgfile, "skinned_gui", "allways_glued", xmps_config.allways_glued);
  xmps_config_write_boolean(cfgfile, "skinned_gui", "allow_multiples_instances", xmps_config.allow_multiples_instances);
  xmps_config_write_boolean(cfgfile, "skinned_gui", "reverse_file_order", xmps_config.reverse_file_order);
  xmps_config_write_boolean(cfgfile, "skinned_gui", "scroll_title", xmps_config.scroll_title);
  xmps_config_write_boolean(cfgfile, "skinned_gui", "confirm_fullscreen", xmps_config.confirm_fullscreen);
  xmps_config_write_boolean(cfgfile, "skinned_gui", "save_on_exit", xmps_config.save_on_exit);

  xmps_config_write_boolean(cfgfile, "skinned_gui", "save_playlist_size", xmps_config.save_playlist_size);
  xmps_config_write_boolean(cfgfile, "skinned_gui", "save_playlist_pos", xmps_config.save_playlist_pos);
  xmps_config_write_boolean(cfgfile, "skinned_gui", "save_playlist_content", xmps_config.save_playlist_content);
  xmps_config_write_boolean(cfgfile, "skinned_gui", "playlist_multiple_selection", xmps_config.playlist_multiple_selection);

  xmps_config_write_boolean(cfgfile, "skinned_gui", "save_window_pos", xmps_config.save_window_pos);
  xmps_config_write_boolean(cfgfile, "skinned_gui", "save_skin", xmps_config.save_skin);
  xmps_config_write_boolean(cfgfile, "skinned_gui", "is_playlist_toggled", xmps_config.is_playlist_toggled);

  xmps_config_write_string(cfgfile, "skinned_gui", "filesel_dir", xmps_config.filesel_dir);
  xmps_config_write_string(cfgfile, "skinned_gui", "skin_path", xmps_config.current_skin->directory);
  xmps_config_write_string(cfgfile, "skinned_gui", "skin_name", xmps_config.current_skin->name);

}

void skinned_quit()
{ 
  XMPS_DEBUG("exiting...");

  skinned_save_config(xmps_config.cfgfile);
  xmps_playback_close(xmps_config.session);
  gtk_main_quit();
}

static gint expose_event (GtkWidget *widget, GdkEventExpose *event)
 {
  draw_current_skin(widget,xmps_playback_is_playing(xmps_config.session));
  gdk_flush();
  return FALSE;
 }

/************************** focus_in_event() ****************************
 *     									*
 * - Main window gains focus.              				*
 *     									*
 ************************************************************************/

void focus_in_event(GtkWidget * widget, GdkEvent * event, gpointer callback_data)
{
  xmps_config.has_focus = TRUE;
  draw_titlebar(ACTIVE, widget);
}

/************************** focus_out_event() ***************************
 *     									*
 * - Main window looses focus.              				*
 *     									*
 ************************************************************************/

void focus_out_event(GtkWidget * widget, GdkEventButton * event, gpointer callback_data)
{
  xmps_config.has_focus = FALSE;
  draw_titlebar(INACTIVE, widget);
}

/************************** key_press_event() ***************************
 *									*
 * Called when a key is pressed                    			*
 ************************************************************************/

gint key_press_event (GtkWidget * w, GdkEventKey * event, gpointer data)
{
  switch(event->keyval)
    {
    case GDK_f:
    case GDK_F:
      XMPS_DEBUG("switching to fullscreen ");
      
      xmps_config.fullscreen = 1; 
      xmps_playback_video_fullscreen(xmps_config.session, 1);
      break;
    case GDK_Left:
      xmps_playback_close(xmps_config.session);
      xmps_playlist_prev(xmps_config.playlist);
      skinned_start_playing();
      break;
    case GDK_Right:
      xmps_playback_close(xmps_config.session);
      xmps_playlist_next(xmps_config.playlist);
      skinned_start_playing();
      break;
    case GDK_p:
    case GDK_P:
      XMPS_DEBUG("toggle_pause");
      xmps_playback_pause(xmps_config.session);
      break;
    default:
      break;      
    } 

  return FALSE;
}

/************************ button_press_event() **************************
 *									*
 * Called when the drawing area is clicked				*
 ************************************************************************/
    
static gint button_press_event (GtkWidget *widget, GdkEventButton *event)
     {
       GdkModifierType state;

       gdk_window_raise(widget->window);

       if( xmps_config.is_playlist_toggled )
       	 gdk_window_raise(playlist_window->window);
       
      if (event->button == 1)
	switch (get_selected_zone(event->x, event->y))
	  {
	  case ABOUT_MENU 	  :
	    /* popup menu for about and help */
	    xmps_config.current_action = ABOUT_MENU_CLICKED;
	    break;
	  case TITLE 		  :
	    /* Clic on the title */
	    xmps_config.current_action = WINDOW_MOVING;
	    gdk_window_get_pointer (event->window, &xmps_config.move_x, &xmps_config.move_y, &state);
	    break;
	  case CLOSE 	 	  :
	    /* Close the application */
	    xmps_config.current_action = WINDOW_CLOSING;
	    break;
	  case INVERT_TIME 	  :
	    /* toggle the time display forward or backward */
	    xmps_config.current_action = INVERT_TIME_DIGIT_CLICKED;
	    break;
	  case LOOP 		  :
	    /* toggle loop playing */
	    xmps_config.current_action = LOOP_DIGIT_CLICKED;
	    break;
	  case DIGITS 		  :
	    /* also toggle time display */
	    break;
	  case SMALL_DIGITS 	  :
	    /* nothing */
	    break;
	  case FULL_SCREEN_BUTTON  :
	    /* go to full screen mode */
	    draw_full_screen_button(PUSHED, widget);
	    xmps_config.current_action = FULL_SCREEN_BUTTON_CLICKED;
	    break;
	  case LOOP_BUTTON 	  :
	    /* toggle loop playing */
	    draw_loop_button(PUSHED, widget);
	    xmps_config.current_action = LOOP_BUTTON_CLICKED;
	    break;
	  case BALANCE_BAR 	  :
	    /* for changing the balance */
	    set_progress_bar_value_x(BALANCE_BAR, event->x);
	    draw_progress_bar(xmps_config.balance_bar, widget);
	    break;
	  case VOLUME_BAR 	  :
	    /* for changing the volume */
	    set_progress_bar_value_x(VOLUME_BAR, event->x);
	    xmps_playback_set_volume(xmps_config.session, xmps_config.volume_bar.value);
	    draw_progress_bar(xmps_config.volume_bar, widget);
	    break;
	  case PROGRESS_BAR 	  :
	    /* for changing the balance */
	    set_progress_bar_value_x(PROGRESS_BAR, event->x);
	    draw_progression_bar(widget);
	    xmps_playback_seek(xmps_config.session, xmps_config.progress_bar.value);
	    break;
	  case PLAYLIST_BUTTON 	  :
	    /* toggle playlist */
	    if (xmps_config.is_playlist_toggled)
	      draw_playlist_button(PUSHED_TOGGLED, widget);
	    else
	      draw_playlist_button(PUSHED, widget);
	    xmps_config.current_action = PLAYLIST_BUTTON_CLICKED;
	    break;
	  case REWIND_BUTTON 	  :
	    /* rewind movie */
	    draw_rewind_button(PUSHED, widget);
	    xmps_config.current_action = REWIND_BUTTON_CLICKED;
	    break;
	  case STOP_BUTTON 	  :
	    /* stop playing */
	    draw_stop_button(PUSHED, widget);
	    xmps_config.current_action = STOP_BUTTON_CLICKED;
	    break;
	  case PLAY_BUTTON 	  :
	    /* start playing */
	    draw_play_button(PUSHED, widget);
	    xmps_config.current_action = PLAY_BUTTON_CLICKED;
	    break;
	  case PAUSE_BUTTON 	  :
	    /* pause and step by step */
	    xmps_config.current_action = PAUSE_BUTTON_CLICKED;
	    draw_pause_button(PUSHED, widget);
	    break;
	  case FAST_FORWARD_BUTTON :
	    /* fast forward control */
	    xmps_config.current_action = FAST_FORWARD_BUTTON_CLICKED;
	    draw_fast_forward_button(PUSHED, widget);
	    break;
	  case LOAD_BUTTON 	  :
	    /* open files */
	    xmps_config.current_action = LOAD_BUTTON_CLICKED;
	    draw_load_button(PUSHED, widget);
	    break;
	  case PROGRESS_BAR_CURSOR :
	    //	    xmps_playback_pause(xmps_config.playback);
	    xmps_config.current_action = PROGRESS_BAR_MOVING;
	    break;
	  case BALANCE_BAR_CURSOR :  	  
	    /*  */
	    xmps_config.current_action = BALANCE_BAR_MOVING;
	    break;
	  case VOLUME_BAR_CURSOR :  	  
	    /*  */
	    xmps_config.current_action = VOLUME_BAR_MOVING;
	    break;
	  case RESIZE 		  :
	    /* resize the window */
	    xmps_config.resize_x = event->x;
	    xmps_config.resize_y = event->y;
	    xmps_config.current_action = WINDOW_RESIZING;
	    break;
	  default:
	    /* Other zone ? -> move the application */
	    xmps_config.current_action = WINDOW_MOVING;
	    gdk_window_get_pointer (event->window, &xmps_config.move_x, &xmps_config.move_y, &state);
	    break;
	 }
      else
	{
	  gtk_menu_popup(GTK_MENU(main_menu->widget), NULL, NULL, 
			 NULL, NULL, event->button, event->time);
	}
      return TRUE;
     }   
     
/*********************** button_release_event() *************************
 *									*
 * Called when a clic on the drawing area is released			*
 ************************************************************************/
     
static gint button_release_event (GtkWidget *widget, GdkEventButton *event)
     {
       if (event->button == 1)
        {
	 switch (xmps_config.current_action)
	  {
	  case ABOUT_MENU_CLICKED :
	    gtk_menu_popup(GTK_MENU(main_menu->widget), NULL, NULL, 
			   NULL, NULL, event->button, event->time);
	    break;
	  case INVERT_TIME_DIGIT_CLICKED :
	    if (xmps_config.is_invert_time)
	      {
		draw_minus_digit(INACTIVE, widget);
		xmps_config.is_invert_time = FALSE;
	      }
	    else
	      {
		draw_minus_digit(ACTIVE, widget);
		xmps_config.is_invert_time = TRUE;
	      }
	    break;
	  case LOOP_DIGIT_CLICKED  :
	    if (xmps_config.is_loop)
	      {
		draw_loop_digit(INACTIVE, widget);
		xmps_config.is_loop = FALSE;
	      }
	    else
	      {
		draw_loop_digit(ACTIVE, widget);
		xmps_config.is_loop = TRUE;
	      }
	    break;
	  case FULL_SCREEN_BUTTON_CLICKED :
	    draw_full_screen_button(NORMAL, widget);
	    xmps_config.fullscreen = 1; 
	    
	    xmps_playback_video_fullscreen(xmps_config.session, 1);
	    break;
	  case LOOP_BUTTON_CLICKED :
	    draw_loop_button(NORMAL, widget);
	    if (xmps_config.is_loop)
	      {
		draw_loop_digit(INACTIVE, widget);
		xmps_config.is_loop = FALSE;
	      }
	    else
	      {
		draw_loop_digit(ACTIVE, widget);
		xmps_config.is_loop = TRUE;
	      }
	    break;
	  case PLAYLIST_BUTTON_CLICKED :
	    if (xmps_config.is_playlist_toggled)
	      {
		draw_playlist_button(NORMAL, widget);
		xmps_config.is_playlist_toggled = FALSE;
		GTK_CHECK_MENU_ITEM(gtk_item_factory_get_widget(main_menu, "/Playlist"))->active = FALSE;
		playlist_window_hide();
	      }
	    else
	      {
		draw_playlist_button(NORMAL_TOGGLED, widget);
		xmps_config.is_playlist_toggled = TRUE;
		GTK_CHECK_MENU_ITEM(gtk_item_factory_get_widget(main_menu, "/Playlist"))->active = TRUE;
		
		xmps_config.playlist_size_x =  xmps_config.window_size_x;
		
		gdk_window_resize(playlist_window->window, xmps_config.playlist_size_x,  105 + 15*(xmps_config.playlist_nbr_lines-3));
		gtk_widget_set_usize(playlist_window, xmps_config.playlist_size_x,  105 + 15*(xmps_config.playlist_nbr_lines-3));

		playlist_window_show();

		xmps_config.playlist_pos_x = xmps_config.window_pos_x;
		xmps_config.playlist_pos_y = xmps_config.window_pos_y + xmps_config.window_size_y;
		gtk_widget_set_uposition(playlist_window, xmps_config.playlist_pos_x, 
					 xmps_config.playlist_pos_y);
		
	      }
	    break;
	    
	  case REWIND_BUTTON_CLICKED :
	    draw_rewind_button(NORMAL, widget);
	    xmps_playback_close(xmps_config.session);
	    xmps_playlist_prev(xmps_config.playlist);
	    skinned_start_playing();
	    break;
	    
	  case STOP_BUTTON_CLICKED :
	    draw_stop_button(NORMAL, widget);
	    xmps_playback_stop(xmps_config.session);
	    break;
	    
	  case PLAY_BUTTON_CLICKED :
	    draw_play_button(NORMAL, widget);
	    
	    if(xmps_config.playlist->nbr_items > 0) {
	      xmps_playback_play(xmps_config.session);
	    }
	    else {
	       skinned_create_file_selection();
	       if(xmps_config.file_selection != NULL)
		 gtk_widget_show(xmps_config.file_selection);
	    }
	    break;
	    
	  case PAUSE_BUTTON_CLICKED :
	    draw_pause_button(NORMAL, widget);
	    xmps_playback_pause(xmps_config.session);
	    break;
	    
	  case FAST_FORWARD_BUTTON_CLICKED :
	    draw_fast_forward_button(NORMAL, widget);
	    xmps_playback_close(xmps_config.session);
	    xmps_playlist_next(xmps_config.playlist);
	    skinned_start_playing();
	    break;
	    
	  case LOAD_BUTTON_CLICKED :
	    draw_load_button(NORMAL, widget);
	    skinned_create_file_selection();
	    if(xmps_config.file_selection != NULL)
	      gtk_widget_show(xmps_config.file_selection);

	    break;
	    
	  case WINDOW_CLOSING: 
	    skinned_quit();
	    break;
	    
	  case PROGRESS_BAR_MOVING:
	    xmps_playback_seek(xmps_config.session, xmps_config.progress_bar.value);
	    break;
	    
	  case NONE 		    : 
	    break;
	  default:
	    break;
	  }
	}
	xmps_config.current_action = NONE;

	/* after a clic the action is NONE */

	return TRUE;
     }

/************************ motion_notify_event() *************************
 *									*
 * Called when the mouse moves over the drawing area			*
 ************************************************************************/

static gint motion_notify_event (GtkWidget *widget, GdkEventMotion *event)
 {
  int x, y;
  int dx, dy;
  GdkModifierType state;
  XEvent ev;
  gint i = 0;
  
  XSync(GDK_DISPLAY(), 0);
  
  while (XCheckTypedEvent(GDK_DISPLAY(), MotionNotify, &ev))
    {
      event->x = ev.xmotion.x;
      event->y = ev.xmotion.y;
      i++;
    }
  
  gdk_window_get_pointer (event->window, &x, &y, &state);
         
  switch(xmps_config.current_action)
   {
     case WINDOW_MOVING : 

       dx = x - xmps_config.move_x;
       dy = y - xmps_config.move_y;
       xmps_config.window_pos_x += dx;
       xmps_config.window_pos_y += dy;
       
       gtk_widget_set_uposition(widget, xmps_config.window_pos_x, xmps_config.window_pos_y);
       
       if (xmps_config.is_playlist_toggled)
	 {
	   xmps_config.playlist_pos_x = xmps_config.window_pos_x;
	   xmps_config.playlist_pos_y = xmps_config.window_pos_y + xmps_config.window_size_y;
	   gtk_widget_set_uposition(playlist_window, xmps_config.playlist_pos_x, 
				    xmps_config.playlist_pos_y);
	 }
       
       gdk_flush();
       break;

     case PROGRESS_BAR_MOVING : 
	 		   	set_progress_bar_value_x(PROGRESS_BAR, x);
				draw_progression_bar(widget);
     				break;
     case BALANCE_BAR_MOVING : 
	 		   	set_progress_bar_value_x(BALANCE_BAR, x);
  			        draw_progress_bar(xmps_config.balance_bar, widget);
     				break;
     case VOLUME_BAR_MOVING : 
	 		   	set_progress_bar_value_x(VOLUME_BAR, x);
				xmps_playback_set_volume(xmps_config.session, xmps_config.volume_bar.value);
				draw_progress_bar(xmps_config.volume_bar, widget);
     				break;
   default:
     break;
   }       
       return 1;
 }

void skinned_shared_menu_callback(gpointer cb_data, guint action, GtkWidget * w)
{
  
  if( ((xmps_menu_item_t *)cb_data)->active == 0 )
    ((xmps_menu_item_t *)cb_data)->active = 1;
  else 
    ((xmps_menu_item_t *)cb_data)->active = 0;

  xmps_config.menu_callback(((xmps_menu_item_t *)cb_data)->id, ((xmps_menu_item_t *)cb_data)->active);
}

void create_menu(xmps_menu_t *menu)
{
  GtkWidget *shared_menu, *gtk_item;
  xmps_menu_item_t *menu_item;
  GSList *items_group = NULL;
  int group = 0;
  int first;

  first = 1;
  
  main_menu = gtk_item_factory_new(GTK_TYPE_MENU, "<Main>", xmps_config.global_accel);

  //gtk_item_factory_set_translate_func(main_menu, menu_translate, "<Main>", NULL);
  gtk_item_factory_create_items(main_menu, MAIN_MENU_ENTRIES, main_menu_entries, NULL);
  
  xmps_config.menu_callback = menu->callback;

  XMPS_DEBUG("creating shaed menu root");

  shared_menu = gtk_menu_new ();
  xmps_menu_reset(menu);

  do
    {
      menu_item = (xmps_menu_item_t *) xmps_menu_get_item(menu);
      if(menu_item != NULL)
	{
	  XMPS_DEBUG("processing menu item");

	  if( strstr(menu_item->label, "-") )
	    {
	      gtk_item = gtk_menu_item_new();
	      gtk_menu_append (GTK_MENU (shared_menu), gtk_item);
	      gtk_widget_show (gtk_item);
	    }
	  else
	    {
	      if(menu_item->type == XMPS_MENU_ITEM_TYPE_RADIO)
		{
		  if(first)
		    {
		      //temp
		      menu_item->active = 1;
		      
		      gtk_item = gtk_radio_menu_item_new_with_label(items_group, menu_item->label);
		      items_group = gtk_radio_menu_item_group (GTK_RADIO_MENU_ITEM (gtk_item));
		      first = 0;
		      group = menu_item->group; 
		    }
		  else
		    if(menu_item->group == group)
		      {
			gtk_item = gtk_radio_menu_item_new_with_label(items_group, menu_item->label);
			items_group = gtk_radio_menu_item_group (GTK_RADIO_MENU_ITEM (gtk_item));
		      }
		    else
		      {
			gtk_item = gtk_radio_menu_item_new_with_label(NULL, menu_item->label);
			items_group = gtk_radio_menu_item_group (GTK_RADIO_MENU_ITEM (gtk_item));
			group = menu_item->group; 
		      }
		}
	      else
		gtk_item = gtk_menu_item_new_with_label( menu_item->label );
	      
	      gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(gtk_item), menu_item->active);
	      gtk_menu_append (GTK_MENU (shared_menu), gtk_item);
	      
	      gtk_signal_connect_object (GTK_OBJECT (gtk_item), "activate",
					 GTK_SIGNAL_FUNC (skinned_shared_menu_callback),
					 (gpointer) menu_item );
	      gtk_widget_show (gtk_item);
	    }
	}
    }
  while(xmps_menu_next(menu));

  gtk_menu_item_set_submenu(GTK_MENU_ITEM(gtk_item_factory_get_widget(main_menu, "/Active Plugins")), shared_menu);

  GTK_CHECK_MENU_ITEM(gtk_item_factory_get_widget(main_menu, "/Playlist"))->active = xmps_config.is_playlist_toggled;
  GTK_CHECK_MENU_ITEM(gtk_item_factory_get_widget(main_menu, "/Loop"))->active = xmps_config.is_loop;
    
}

void main_menu_callback(gpointer cb_data, guint action, GtkWidget * w)
{
  switch (action)
    {
    case MAIN_ABOUT :
      xmps_about_create();
      xmps_about_show();
      break;
      
    case MAIN_SAVE:
      skinned_save_config(NULL);
      break;
      
    case MAIN_LOOP:
      if(xmps_config.is_loop)
	xmps_config.is_loop = 0;
      else
	xmps_config.is_loop = 1;
      break;

    case MAIN_PREFS:
      
      xmps_gtk_config_create(xmps_config.session);
      xmps_gtk_config_show();
      break;

    case MAIN_SKIN:
      xmps_show_skin_browser();
      break;

    case MAIN_PLAYLIST: 
      if (xmps_config.is_playlist_toggled)
	{
	  draw_playlist_button(NORMAL, xmps_config.skinned_window);
	  xmps_config.is_playlist_toggled = FALSE;
	  playlist_window_hide();
	}
      else
	{
	  draw_playlist_button(NORMAL_TOGGLED, xmps_config.skinned_window);

	  xmps_config.is_playlist_toggled = TRUE;

	  xmps_config.playlist_pos_x = xmps_config.window_pos_x;
	  xmps_config.playlist_pos_y = xmps_config.window_pos_y + xmps_config.window_size_y;

	  gtk_widget_set_uposition(playlist_window, xmps_config.playlist_pos_x, xmps_config.playlist_pos_y);

	  playlist_window_show();
	}
      break;
      
    case MAIN_QUIT : 
      skinned_quit();
      break;
    default:
      break;
    }  
}

GtkWidget *skinned_window_new(xmps_config_file_t *cfgfile, xmps_menu_t *menu)
{
  GtkWidget *window;

  /* load GUI options */

  XMPS_DEBUG("intitalizing stuff");

  skinned_initialize_xmps(cfgfile);
  initialize_playlist();
  initialize_skins();

  XMPS_DEBUG("done");

 /* Create the main window */

  window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  gtk_widget_set_app_paintable(window, 1);
  gtk_window_set_policy(GTK_WINDOW(window), FALSE, FALSE, TRUE);
        
 /* default size */
  
  gtk_widget_set_usize(GTK_WIDGET(window), xmps_config.window_size_x, xmps_config.window_size_y);
  gtk_widget_set_name (window, "X Movie Player System");
        
 /* position loaded from options */

  XMPS_DEBUG("placing at %d, %d", xmps_config.window_pos_x, xmps_config.window_pos_y);
  
  gtk_widget_set_uposition(GTK_WIDGET(window), xmps_config.window_pos_x, xmps_config.window_pos_y);

 /* we never know */
  gtk_signal_connect (GTK_OBJECT (window), "destroy",
                      GTK_SIGNAL_FUNC (skinned_quit), NULL);
       
 /* connect to all the signals we need */

  //gtk_drag_dest_set (window, GTK_DEST_DEFAULT_ALL,
  //		     dnd_target_table, 1, GDK_ACTION_COPY);
  /*  gtk_signal_connect (GTK_OBJECT (window), "drag_data_received",
      GTK_SIGNAL_FUNC (dnd_data_received),NULL);*/

  gtk_signal_connect (GTK_OBJECT (window), "expose_event",
		      (GtkSignalFunc) expose_event, NULL);

  gtk_signal_connect (GTK_OBJECT (window), "button_press_event",
		      (GtkSignalFunc) button_press_event, NULL);
  gtk_signal_connect (GTK_OBJECT (window), "button_release_event",
		      (GtkSignalFunc) button_release_event, NULL);
  gtk_signal_connect (GTK_OBJECT (window), "motion_notify_event",
		      (GtkSignalFunc) motion_notify_event, NULL);
  gtk_signal_connect (GTK_OBJECT (window), "key-press-event",
		      (GtkSignalFunc) key_press_event, NULL);
  gtk_signal_connect(GTK_OBJECT(window), "focus_in_event", 
		     GTK_SIGNAL_FUNC(focus_in_event), NULL);
  gtk_signal_connect(GTK_OBJECT(window), "focus_out_event", 
		     GTK_SIGNAL_FUNC(focus_out_event), NULL);

 /* define what we want to be noticed for */      
  gtk_widget_set_events (window, GDK_EXPOSURE_MASK
			 | GDK_LEAVE_NOTIFY_MASK
			 | GDK_BUTTON_PRESS_MASK
			 | GDK_BUTTON_RELEASE_MASK
			 | GDK_POINTER_MOTION_MASK
			 | GDK_POINTER_MOTION_HINT_MASK);
			      
  gtk_widget_realize(window);

  /* window without decorations */ 
  gdk_window_set_decorations(window->window, (GdkWMDecoration) 0);

  style = gtk_widget_get_style( window );

  if( strstr(xmps_config.current_skin->directory,"(Default)" ) )
    load_default_skin(window->window);
  else
    load_skin(xmps_config.current_skin->directory, window->window);

  xmps_config.global_accel = gtk_accel_group_new();
  gtk_window_add_accel_group(GTK_WINDOW(window), xmps_config.global_accel);

  XMPS_DEBUG("creating menu");

  create_menu(menu);

  XMPS_DEBUG("creating playlist");

  playlist_window_create();

  xmps_config.skinned_window = window;

  XMPS_DEBUG("creating file selection");

  skinned_create_file_selection();
  
  XMPS_DEBUG("creating skin browser");
  
  xmps_skin_browser_create();
  
  gtk_timeout_add(50, skinned_gui_idle, NULL);

  gtk_widget_show(window);

  return window;
}
