/* dumpspk.c
   setting display utility for the speakup screen review package
   written by: Andy Berdan.

    Copyright (C) 1998  Andy Berdan.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Kirk Reiser <kirk@braille.uwo.ca>
    261 Trott dr. London, Ontario, Canada. N6G 1B6

    Andy Berdan <ed@facade.dhs.org>
    3-337 Wharncliffe Rd. N  London, Ontario, Canada N6G 1E4
    */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <unistd.h>
#include <stdarg.h>
#include <sys/ioctl.h>
#include "symbols.h"
#include "console_fd.h"

#define VER "dumpspk v" VERSION
/* VERSION passed in from Makefile */
#define HEADER VER

#define toctrl(x) ( ( ((x) >= 'A') && ((x) <='Z')) ? ((x) - 'A') : \
			(((x) >= 'a') && ((x) <= 'z')) ? ((x) - 'a') : 0 )  

int synth = -1;
int verbose = 0;
char *names[] = NAMES;

struct spk_variable spkup_vars[] = SPKUP_VARS;

struct spk_variable synth_static[] = ALL_STATIC;
struct spk_variable synth_xtend[] = ALL_XTEND;
struct spk_variable synth_alias[] = ALL_ALIAS;
static int num_static[] = ALL_NUM_STATIC;
static int num_xtend[] = ALL_NUM_XTEND;
static int num_alias[] = ALL_NUM_ALIAS;

void inline getoffsets(	int *pre,
			int *post,
			struct spk_variable *cmd) {

	char* start = cmd->build;
	char* end = cmd->build + strlen(cmd->build);
	char* line = strchr(cmd->build,'_');

/*
	    v----- line
	test_foo
	^      ^-- end
	start
*/

	if ((cmd->id)[0] != '\0') {
		*pre = line - start;
		*post = end - line;
	}
}

static int inline synth_offset(int *num_cnt) {
	int i=0,sum=0;
	for (; i<synth; ++i) sum += num_cnt[i];
	return sum;
}

struct spk_variable *seperate(int index) {
	struct spk_variable *parser;

	if (index < num_static[synth]) {
		parser = &(synth_static[ synth_offset(num_static) ]);
		parser += (index);
	} else if (index < (num_static[synth] + num_xtend[synth])) {
		parser = &(synth_xtend[ synth_offset(num_xtend) ]);
		parser += (index-num_static[synth]);
	} else {
		parser = spkup_vars;
		parser += (index-num_static[synth]-num_xtend[synth]);
	}

	return parser;
}

int alias(char *buf, char *input, int *adjust, int *inadjust) {

	int var=0;
	char match[65];
	char param[3];
	struct spk_variable *alias = &(synth_alias[ synth?num_alias[synth-1]:0 ]);

	/* each alias */
	for (var=0; var<num_alias[synth]; ++var) {
		int i=0, in=0, final=0;
		int pcnt=0;

		/* spin 1st part, add to match */
		while (alias->build[i] == input[in]) {
			match[in] = input[in];
			in++;
			i++;
		}

		if (alias->build[i++] != '_') continue; /* not a match */

		/* check if anything is after '_' */
		if (alias->build[i] != '\0') {
			final=strlen(alias->build+i);
		} else
			final=strlen(input)-in-1;

		while (in < strlen(input)-final)
			param[pcnt++] = input[in++];
		param[pcnt] = '\0';

		noisy("Param = [%s]\n",param);

		/* spin 2nd part (if exists), add to match */
		while ((final--) && (alias->build[i] != '\0')) {
			if (input[in] == alias->build[i++])
				match[in]=input[in++];
			else
				return 0;
		}

		/* translate to mnemonic */
		sprintf(buf,"%s(%s) %n",alias->id,param,adjust);

		*inadjust = in-1;
		return 1;
	}
	return 0;
}

void trans(char *buf, char *input, int index, int *adjust) {

	char *orig = input;
	char *origbuf = buf;
	int pre=0, post=0;
	char *end;
	int move, inmove;
	struct spk_variable *parser;

	parser = seperate(index);
	getoffsets(&pre,&post,parser);

	end = input + strlen(input) - post + 1;
	input += pre;

	noisy("pre %d, post %d\n",pre,post);

	while (input < end) {
		if (*input < 31) {
			if (alias(buf,input,&move,&inmove)) {
				input += inmove;
			} else
				sprintf(buf,"\\x%02x%n",*input,&move);
		} else if (*input == '\\') {
			/* quote the quote */
			move = 2;
			*buf = *(buf+1) = '\\';
		} else {
			move = 1;
			*buf = *input;
		}
		buf += move;
		input++;
	}

	*buf = 0;
	*adjust = strlen(orig);
	if (parser->id[0] != '\0')
		printf("%-20s = %s\n",parser->id, origbuf);
}

/* prints usage and bails with errcode */
void bail_out(int errcode) {
printf("\
Usage: dumpspk [OPTIONS] [file]\n\
Dumps configuration from speakup into [file].  If [file] is not specified,\n\
config. info is written to stdout.\n\
OPTIONS:\n\
\t-h    this help listing\n\
\t-v    verbose output\n");
exit(errcode);
}

int main(int argc, char *argv[]) {
	int fd=0, i, count, adjust;
	char spk_struct[1000];
	char buf[100];
	char *spk_ptr = spk_struct;

	while ((i=getopt(argc,argv, "hv")) != EOF) {
                switch (i) {
                case 'h':       /* help */
                        bail_out(0);
                        break;
                case 'v':       /* (really) verbose */
                        verbose=1;
                        break;
                }
        }
         /*  command line leftovers argv[optind++] .. argv[argc] */
        argc -= optind;
        argv += optind;

	if (argc == 1) {
		if (NULL == freopen(argv[0],"w",stdout)) {
			perror("Error opening file");
			exit(1);
		}
        } else if ( argc ) { /* more than one file */
		/* bzzzt... no soup for you. */
		bail_out(1);
	}

	fprintf(stderr, HEADER);

	for (i = 0; i<1000; i++) {
		spk_struct[i] = '\0';
	}

	fd = getfd();
	if (fd) {
		unsigned char tmp[]="\xff";

		ioctl(fd,SPKDUMPCFG,spk_ptr);
		ioctl(fd,SPKHARDWARE,tmp);
		if (tmp[0] == 0xff) {
			printf("\nError: Synth not identified!\n");
			return -1;
		} else synth = tmp[0];

		fprintf(stderr,":  %s found.\n",names[synth]);

		for (count = 0; count < num_static[synth]+num_xtend[synth]+NUM_SPKUP_VARS; count++) {
			trans(buf,spk_ptr,count,&adjust);
			spk_ptr += adjust+1;
		}
		return 0;
	}
	fprintf(stderr,"\nError: Couldn't get console file descriptor!\n");
	return -1; /* couldn't get fd */
}
