/********************************
 TCP peer to peer functions
 (c) 1999 Jeremy Wise
 GnomeICU
*********************************/

/*** GnomeICU header files ***/
#include "common.h"

/*** Toplevel header files ***/
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <netinet/in.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <sys/time.h>
#include <fcntl.h>
#include <time.h>
#include <stdarg.h>
#include <string.h>
#include <ctype.h>
#include <assert.h>
#include <sys/wait.h>
#include <signal.h>
#ifdef HAVE_SOCKS5
#define SOCKS
#include <socks.h>
#endif

GtkWidget *filesel;

typedef struct
{
	struct sokandlb *data;
	int i;
} dataandint;

// 127.0.0.1 is localhost
const DWORD LOCALHOST = 0x7F000001;

/*** Local function declarations ***/
static int TCPInitChannel( GIOChannel *source, GIOCondition cond, gpointer data );
static int TCPAckPacket( int sock, GSList *contact, WORD cmd, int seq );
static void TCPProcessPacket( BYTE *packet, int packet_length, int sock );

static int TCPConnectFile( DWORD port, UIN_T uin );
static int TCPFileHandshake( GSList *contact, int sock, GdkInputCondition cond );
static int TCPFileReadServer( GtkWidget *widget, int sock, GdkInputCondition cond );
static int TCPFileReadClient( GtkWidget *widget, int sock, GdkInputCondition cond );
static int TCPSendPartFile( gpointer data );

static void file_place( char *prename, UIN_T uin );
static void save_incoming_file( GtkWidget *widget, gpointer data );
static void cancel_transfer( GtkWidget *widget, gpointer data );

static int TCPFinishConnection( GIOChannel *source, GIOCondition cond, gpointer data );

static int TCPReadPacket( GIOChannel *source, GIOCondition cond, gpointer data );

static int TCPTimeout( CONTACT_PTR contact );
static void timeout_continue( GtkWidget *widget, CONTACT_PTR contact );
static void timeout_server( GtkWidget *widget, CONTACT_PTR contact );
static void timeout_cancel( GtkWidget *widget, CONTACT_PTR contact );

/*** Global functions ***/
int TCPGainConnection( DWORD ip, WORD port, GSList *contact )
{
	typedef struct
	{
		BYTE size[2];
		BYTE command;
		BYTE version[4];
		BYTE szero[4];
		BYTE uin[4];
		BYTE ip[4];
		BYTE real_ip[4];
		BYTE four;
		BYTE port[4];
		gchar str[13];
	} hello_packet_struct;

	hello_packet_struct *hello_packet;
	struct sockaddr_in local, remote;
	int sizeofSockaddr = sizeof( struct sockaddr );
	int sock;

#ifdef TRACE_FUNCTION
	g_print( "TCPGainConnection\n" );
#endif

	if( ((CONTACT_PTR)contact->data)->have_tcp_connection == FALSE )
		((CONTACT_PTR)contact->data)->sok = 0;

	if( ((CONTACT_PTR)contact->data)->sok > 0 )
		return ((CONTACT_PTR)contact->data)->sok;

	if( ((CONTACT_PTR)contact->data)->force != 0x04 )
		return -1;

	hello_packet = g_new0( hello_packet_struct, 1 );

	Word_2_Chars( hello_packet->size, 0x001A );
	hello_packet->command = 0xFF;
	DW_2_Chars( hello_packet->version, 0x00000003 );
	DW_2_Chars( hello_packet->szero, 0x00000000 );
	DW_2_Chars( hello_packet->uin, our_info->uin );
	DW_2_IP( hello_packet->ip, LOCALHOST );
	DW_2_IP( hello_packet->real_ip, LOCALHOST );
	hello_packet->four = 0x04;
	DW_2_Chars( hello_packet->port, (DWORD)our_port );
	strcpy( hello_packet->str, "HELLO PACKET" );

	if( ip == 0 )
		return -1;

	sock = socket( AF_INET, SOCK_STREAM, 0 );
	if( sock == -1 )
		return -1;

	fcntl( sock, F_SETFL, O_NONBLOCK );

	memset( &local.sin_zero, 0x00, 8 );
	memset( &remote.sin_zero, 0x00, 8 );
	
	local.sin_family = AF_INET;
	remote.sin_family = AF_INET;
	local.sin_port = g_htons( 0 );
	local.sin_addr.s_addr = g_htonl( INADDR_ANY );
	if( ( bind( sock, (struct sockaddr*)&local, sizeof( struct sockaddr ) ) )== -1 )
		return -1;

	getsockname( sock, (struct sockaddr*)&local, &sizeofSockaddr );

	remote.sin_port = g_htons( port );
	remote.sin_addr.s_addr = g_htonl( ip );
/*
 *	localport = g_ntohs( local.sin_port );
 *	localip = g_ntohl( local.sin_addr.s_addr );
 */
	connect( sock, (struct sockaddr *)&remote, sizeofSockaddr );

	((CONTACT_PTR)contact->data)->sok = sock;
	((CONTACT_PTR)contact->data)->gioc = g_io_channel_unix_new( sock );
	((CONTACT_PTR)contact->data)->tcp_msg_queue = g_slist_prepend( ((CONTACT_PTR)contact->data)->tcp_msg_queue, hello_packet );

	/* For alignment purposes */
	((CONTACT_PTR)contact->data)->tcp_text_queue = g_slist_prepend( ((CONTACT_PTR)contact->data)->tcp_text_queue, NULL );

	g_io_add_watch( ((CONTACT_PTR)contact->data)->gioc,
	                G_IO_IN | G_IO_OUT | G_IO_ERR | G_IO_HUP,
	                TCPFinishConnection, NULL );

	return -2;
}

int TCPClearQueue( GSList *contact )
{
	WORD psize;
	GSList *packet;

#ifdef TRACE_FUNCTION
	g_print( "TCPClearQueue\n" );
#endif

	if( contact == NULL )
		return FALSE;

	packet = ((CONTACT_PTR)contact->data)->tcp_msg_queue;
	while( packet != NULL )
	{
		psize = Chars_2_Word (packet->data);

		write( ((CONTACT_PTR)contact->data)->sok, packet->data, psize + 2 );
		packet_print( (BYTE*)packet->data, psize + 2,
		              PACKET_TYPE_TCP | PACKET_DIRECTION_SEND,
		              (gchar*)((BYTE*)(packet->data)+2+psize) );
	
		g_free( packet->data );

		packet = packet->next;
	}

	g_slist_free( packet );

	((CONTACT_PTR)contact->data)->tcp_msg_queue = NULL;

	return FALSE;
}

int TCPSendMessage( UIN_T uin, gchar *msg )
{
	int sock;
	unsigned short intsize;
	BYTE *buffer;

	typedef struct
	{
		BYTE uin_a[4];
		BYTE version[2];
		BYTE cmd[2];
		BYTE zero[2];
		BYTE uin_b[4];
		BYTE command[2];
		BYTE msg_length[2];
	} tcp_head;

	typedef struct
	{
		BYTE ip[4];
		BYTE real_ip[4];
		BYTE port[4];
		BYTE four;
		BYTE zero[4];
		BYTE seq[4];
	} tcp_tail;

	struct
	{
		tcp_head head;
		const gchar *body;
		tcp_tail tail;
	} packet;

	GSList *contact;

#ifdef TRACE_FUNCTION
	g_print( "TCPSendMessage\n" );
#endif

	DW_2_Chars( packet.head.uin_a, our_info->uin );
	Word_2_Chars( packet.head.version, 0x0003 );
	Word_2_Chars( packet.head.cmd, ICQ_CMDxTCP_START );
	Word_2_Chars( packet.head.zero, 0x0000 );
	DW_2_Chars( packet.head.uin_b, our_info->uin );
	Word_2_Chars( packet.head.command, ICQ_CMDxTCP_MSG );
	Word_2_Chars( packet.head.msg_length, ( strlen( msg ) + 1 ) );

	packet.body = msg;

	contact = Contacts;

	while( contact != NULL )
	{
		if( ((CONTACT_PTR)contact->data)->uin == uin )
			break;
		contact = contact->next;
	}

	if( contact == NULL )
		return FALSE;

	if( ((CONTACT_PTR)contact->data)->force_override )
		return FALSE;

	DW_2_IP( packet.tail.ip, our_ip );
	DW_2_IP( packet.tail.real_ip, our_ip );
	DW_2_Chars( packet.tail.port, our_port );
	packet.tail.four = 0x04;
	if( ((CONTACT_PTR)contact->data)->status == STATUS_DND || ((CONTACT_PTR)contact->data)->status == STATUS_OCCUPIED )
		DW_2_Chars( packet.tail.zero, 0x00200000 );
	else
		DW_2_Chars( packet.tail.zero, 0x00100000 );
	DW_2_Chars( packet.tail.seq, seq_num ++ );

	contact = Contacts;

	while( contact != NULL )
	{
		if( ((CONTACT_PTR)contact->data)->uin == uin )
			break;
		contact = contact->next;
	}

	if( contact == NULL )
		return 0;

	/* If the user is offline, we use UDP to avoid hanging */
	if( ((CONTACT_PTR)contact->data)->status == STATUS_OFFLINE &&
	    ((CONTACT_PTR)contact->data)->sok == 0 )
		return 0;

/*** Construct packet ***/
	intsize = sizeof( tcp_head ) + sizeof( tcp_tail ) + strlen( msg ) + 1;
	buffer = (BYTE*)g_malloc0( strlen( "SEND MESSAGE" ) + 1 + intsize + 2 );

	Word_2_Chars (buffer, intsize);
	memcpy( buffer+2, &packet.head, sizeof( packet.head ) );
	memcpy( buffer+2 + sizeof( packet.head ), packet.body,
	        strlen( packet.body ) + 1 );
	memcpy( buffer+2 + sizeof( packet.head ) +
	        strlen( packet.body ) + 1,
	        &packet.tail, sizeof( packet.tail ) );
	strcpy( buffer + 2 + intsize, "SEND MESSAGE" );
/************************/

	((CONTACT_PTR)contact->data)->tcp_msg_queue = g_slist_append( ((CONTACT_PTR)contact->data)->tcp_msg_queue, buffer );
	((CONTACT_PTR)contact->data)->tcp_text_queue = g_slist_append( ((CONTACT_PTR)contact->data)->tcp_text_queue, g_strdup( msg ) );

	sock = TCPGainConnection( ((CONTACT_PTR)contact->data)->current_ip, ((CONTACT_PTR)contact->data)->port, contact );
	if( sock != -1 )
		((CONTACT_PTR)contact->data)->tcp_timeout = gtk_timeout_add( 30000, (GtkFunction)TCPTimeout, (CONTACT_PTR)contact->data );

	if( sock == -2 ) /* We're waiting for a connection */
	{
		animate_on();
		return 2;
	}

	if( sock != -1 )
		TCPClearQueue( contact );
	else
		return 0;

	animate_on();
	return 1;
}

int TCPSendURL( UIN_T uin, gchar *msg )
{
	int sock;
	unsigned short intsize;
	BYTE *buffer;

	typedef struct
	{
		BYTE uin_a[4];
		BYTE version[2];
		BYTE cmd[2];
		BYTE zero[2];
		BYTE uin_b[4];
		BYTE command[2];
		BYTE msg_length[2];
	} tcp_head;

	typedef struct
	{
		BYTE ip[4];
		BYTE real_ip[4];
		BYTE port[4];
		BYTE four;
		BYTE zero[4];
		BYTE seq[4];
	} tcp_tail;

	struct
	{
		tcp_head head;
		const gchar *body;
		tcp_tail tail;
	} packet;

	GSList *contact;

#ifdef TRACE_FUNCTION
	g_print( "TCPSendURL\n" );
#endif

	DW_2_Chars( packet.head.uin_a, our_info->uin );
	Word_2_Chars( packet.head.version, 0x0003 );
	Word_2_Chars( packet.head.cmd, ICQ_CMDxTCP_START );
	Word_2_Chars( packet.head.zero, 0x0000 );
	DW_2_Chars( packet.head.uin_b, our_info->uin );
	Word_2_Chars( packet.head.command, ICQ_CMDxTCP_URL );
	Word_2_Chars( packet.head.msg_length, ( strlen( msg ) + 1 ) );

	packet.body = g_strdup(msg);

	DW_2_IP( packet.tail.ip, our_ip );
	DW_2_IP( packet.tail.real_ip, our_ip );
	DW_2_Chars( packet.tail.port, our_port );
	packet.tail.four = 0x04;
	DW_2_Chars( packet.tail.zero, 0x00100000 );
	DW_2_Chars( packet.tail.seq, seq_num ++ );

	contact = Contacts;

	while( contact != NULL )
	{
		if( ((CONTACT_PTR)contact->data)->uin == uin )
			break;
		contact = contact->next;
	}

	if( contact == NULL )
		return 0;

/*** Construct packet ***/
	intsize = sizeof( tcp_head ) + sizeof( tcp_tail ) + strlen( msg ) + 1;
	buffer = (BYTE*)g_malloc0( strlen( "SEND URL" ) + 1 + intsize + 2 );

	Word_2_Chars (buffer, intsize);
	memcpy( buffer+2, &packet.head, sizeof( packet.head ) );
	memcpy( buffer+2 + sizeof( packet.head ), packet.body,
	        strlen( packet.body ) + 1 );
	memcpy( buffer+2 + sizeof( packet.head ) + strlen( packet.body ) + 1,
	        &packet.tail, sizeof( packet.tail ) );
	strcpy( buffer + 2 + intsize, "SEND URL" );
/************************/

	((CONTACT_PTR)contact->data)->tcp_msg_queue = g_slist_append( ((CONTACT_PTR)contact->data)->tcp_msg_queue, buffer );
	((CONTACT_PTR)contact->data)->tcp_text_queue = g_slist_append( ((CONTACT_PTR)contact->data)->tcp_text_queue, g_strdup( msg ) );

	sock = TCPGainConnection( ((CONTACT_PTR)contact->data)->current_ip, ((CONTACT_PTR)contact->data)->port, contact );

	if( sock == -2 ) /* We're waiting for a connection */
		return TRUE;

	if( sock != -1 )
		TCPClearQueue( contact );
	else
		return 0;

	animate_on();

	return 1;
}

int TCPAcceptIncoming( gpointer data, int sock, GdkInputCondition cond )
{
	struct sockaddr_in addr;
	int size = sizeof( struct sockaddr_in );
	int new_sock;
	GIOChannel *gioc;

#ifdef TRACE_FUNCTION
	g_print( "TCPAcceptIncoming\n" );
#endif
	
	new_sock = accept( sock, (struct sockaddr *)&addr, &size );
	if( new_sock == -1 )
		return 0;

	if( new_sock == -1 )
		return FALSE;

	gioc = g_io_channel_unix_new( new_sock );

	g_io_add_watch( gioc, G_IO_IN | G_IO_ERR | G_IO_HUP,
	                TCPInitChannel, NULL );

	return 1;
}

int TCPRetrieveAwayMessage( GSList *contact, gpointer data )
{
	int sock;
	unsigned short intsize;
	BYTE *buffer;
	int status_type;

	typedef struct
	{
		BYTE uin_a[4];
		BYTE version[2];
		BYTE cmd[2];
		BYTE zero[2];
		BYTE uin_b[4];
		BYTE command[2];
		BYTE msg_length[2];
	} tcp_head;

	typedef struct
	{
		BYTE ip[4];
		BYTE real_ip[4];
		BYTE port[4];
		BYTE four;
		BYTE zero[4];
		BYTE seq[4];
	} tcp_tail;

	struct
	{
		tcp_head head;
		char *body;
		tcp_tail tail;
	} packet;

#ifdef TRACE_FUNCTION
	g_print( "TCPRetrieveAwayMessage\n" );
#endif

	switch( ((CONTACT_PTR)contact->data)->status & 0xffff )
	{
		case STATUS_AWAY:
			status_type = ICQ_CMDxTCP_READxAWAYxMSG;
			break;
		case STATUS_NA:
			status_type = ICQ_CMDxTCP_READxNAxMSG;
			break;
		case STATUS_OCCUPIED:
			status_type = ICQ_CMDxTCP_READxOCCxMSG;
			break;
		case STATUS_DND:
			status_type = ICQ_CMDxTCP_READxDNDxMSG;
			break;
		default:
			status_type = ICQ_CMDxTCP_READxAWAYxMSG;
			break;
	}

	DW_2_Chars( packet.head.uin_a, our_info->uin );
	Word_2_Chars( packet.head.version, 0x0003 );
	Word_2_Chars( packet.head.cmd, ICQ_CMDxTCP_START );
	Word_2_Chars( packet.head.zero, 0x0000 );
	DW_2_Chars( packet.head.uin_b, our_info->uin );
	Word_2_Chars( packet.head.command, status_type );
	Word_2_Chars( packet.head.msg_length, 0x0001 );

	packet.body = "";

	DW_2_IP( packet.tail.ip, our_ip );
	DW_2_IP( packet.tail.real_ip, LOCALHOST );
	DW_2_Chars( packet.tail.port, our_port );
	packet.tail.four = 0x04;
	DW_2_Chars( packet.tail.zero, 0x00001000 );
	DW_2_Chars( packet.tail.seq, seq_num ++ );

/*** Create packet ***/
	intsize = sizeof( tcp_head ) + sizeof( tcp_tail ) + 1;
	buffer = (BYTE*)g_malloc0( strlen( "REQUEST AWAY MSG" ) + 1 + intsize + 2 );

	Word_2_Chars (buffer, intsize);
	memcpy( buffer+2, &packet.head, sizeof( packet.head ) );
	memcpy( buffer+2 + sizeof( packet.head ), packet.body,
	        strlen( packet.body ) + 1 );
	memcpy( buffer+2 + sizeof( packet.head ) + strlen( packet.body ) + 1,
	        &packet.tail, sizeof( packet.tail ) );
	strcpy( buffer + 2 + intsize, "REQUEST AWAY MSG" );
/*********************/

	((CONTACT_PTR)contact->data)->tcp_msg_queue = g_slist_append( ((CONTACT_PTR)contact->data)->tcp_msg_queue, buffer );

	/* We'll do this for alignment purposes */
	((CONTACT_PTR)contact->data)->tcp_text_queue = g_slist_append( ((CONTACT_PTR)contact->data)->tcp_text_queue, NULL );

	sock = TCPGainConnection( ((CONTACT_PTR)contact->data)->current_ip, ((CONTACT_PTR)contact->data)->port, contact );

	if( sock == -2 ) /* We're waiting for a connection */
		return TRUE;

	if( sock != -1 )
		TCPClearQueue( contact );
	else
		return FALSE;

	return 1;
}

int TCPRetrieveVersion( GSList *contact, gpointer data )
{
	int sock;
	unsigned short intsize;
	BYTE *buffer;

	typedef struct
	{
		BYTE uin_a[4];
		BYTE version[2];
		BYTE cmd[2];
		BYTE zero[2];
		BYTE uin_b[4];
		BYTE command[2];
		BYTE msg_length[2];
	} tcp_head;

	typedef struct
	{
		BYTE ip[4];
		BYTE real_ip[4];
		BYTE port[4];
		BYTE four;
		BYTE zero[4];
		BYTE seq[4];
	} tcp_tail;

	struct
	{
		tcp_head head;
		char *body;
		tcp_tail tail;
	} packet;

#ifdef TRACE_FUNCTION
	g_print( "TCPRetrieveVersion\n" );
#endif

	DW_2_Chars( packet.head.uin_a, our_info->uin );
	Word_2_Chars( packet.head.version, 0x0003 );
	Word_2_Chars( packet.head.cmd, ICQ_CMDxTCP_START );
	Word_2_Chars( packet.head.zero, 0x0000 );
	DW_2_Chars( packet.head.uin_b, our_info->uin );
	Word_2_Chars( packet.head.command, ICQ_CMDxTCP_VERSION );
	Word_2_Chars( packet.head.msg_length, 0x0001 );

	packet.body = VERSION;

	DW_2_IP( packet.tail.ip, our_ip );
	DW_2_IP( packet.tail.real_ip, LOCALHOST );
	DW_2_Chars( packet.tail.port, our_port );
	packet.tail.four = 0x04;
	DW_2_Chars( packet.tail.zero, 0x00001000 );
	DW_2_Chars( packet.tail.seq, seq_num ++ );

/*** Create packet ***/
	intsize = sizeof( tcp_head ) + sizeof( tcp_tail ) + 1;
	buffer = (BYTE*)g_malloc0( strlen( "REQUEST VERSION" ) + 1 + intsize + 2 );

	Word_2_Chars (buffer, intsize);
	memcpy( buffer+2, &packet.head, sizeof( packet.head ) );
	memcpy( buffer+2 + sizeof( packet.head ), packet.body,
	        strlen( packet.body ) + 1 );
	memcpy( buffer+2 + sizeof( packet.head ) + strlen( packet.body ) + 1,
	        &packet.tail, sizeof( packet.tail ) );
	strcpy( buffer + 2 + intsize, "REQUEST VERSION" );
/*********************/

	((CONTACT_PTR)contact->data)->tcp_msg_queue = g_slist_append( ((CONTACT_PTR)contact->data)->tcp_msg_queue, buffer );

	/* We'll do this for alignment purposes */
	((CONTACT_PTR)contact->data)->tcp_text_queue = g_slist_append( ((CONTACT_PTR)contact->data)->tcp_text_queue, NULL );

	sock = TCPGainConnection( ((CONTACT_PTR)contact->data)->current_ip, ((CONTACT_PTR)contact->data)->port, contact );

	if( sock == -2 ) /* We're waiting for a connection */
		return TRUE;

	if( sock != -1 )
		TCPClearQueue( contact );
	else
		return FALSE;

	return 1;
}

int TCPAcceptFile( int sock, GSList *contact, DWORD seq )
{
	BYTE *buffer;
	unsigned short intsize;
	gint file_sock;
	unsigned short file_port, backport, tempport;
	gint length = sizeof( struct sockaddr );

	typedef struct
	{
		BYTE uin1[4];
		BYTE version[2];
		BYTE command[2];
		BYTE zero[2];
		BYTE uin2[4];
		BYTE cmd[2];
		BYTE message_length[2];
	} tcp_head;

	typedef struct
	{
		BYTE ip[4];
		BYTE ip_real[4];
		BYTE porta[4];
		BYTE junk;
		BYTE status[4];
		BYTE back_port[4];
		BYTE one[2];
		BYTE zero;
		BYTE bigzero[4];
		BYTE portb[4];
		BYTE seq[4];
	} tcp_tail;

	tcp_head pack_head;
	tcp_tail pack_tail;

	struct sockaddr_in my_addr;

#ifdef TRACE_FUNCTION
	g_print( "TCPAcceptFile\n" );
#endif

	file_sock = socket( AF_INET, SOCK_STREAM, 0 );
	if( file_sock <= 0 )
		return FALSE;

	fcntl( file_sock, O_NONBLOCK );

	my_addr.sin_family = AF_INET;
	my_addr.sin_port = 0;
	my_addr.sin_addr.s_addr = g_htonl (INADDR_ANY);
	memset(&(my_addr.sin_zero), 0x00, 8);
	
	if( bind( file_sock, (struct sockaddr *)&my_addr, sizeof( struct sockaddr ) ) == -1 )
		return FALSE;

	listen( file_sock, 1 );
	((CONTACT_PTR)contact->data)->file_gdk_input =
		gdk_input_add( file_sock, GDK_INPUT_READ,
			       (GdkInputFunction) TCPFileHandshake,
			       contact );

	getsockname( file_sock, ( struct sockaddr * ) &my_addr, &length );

	file_port = g_ntohs( my_addr.sin_port );

	tempport = file_port;
	backport = ( tempport >> 8 ) + ( tempport << 8 );

	DW_2_Chars( pack_head.uin1, our_info->uin );
	Word_2_Chars( pack_head.version, 0x0003 );
	Word_2_Chars( pack_head.command, ICQ_CMDxTCP_ACK );
	Word_2_Chars( pack_head.zero, 0x0000 );
	DW_2_Chars( pack_head.uin2, our_info->uin );
	DW_2_Chars( pack_head.cmd, ICQ_CMDxTCP_FILE );
	DW_2_Chars( pack_head.message_length, 1 );

	DW_2_IP( pack_tail.ip, our_ip );
	DW_2_IP( pack_tail.ip_real, LOCALHOST );
	DW_2_Chars( pack_tail.porta, our_port );
	pack_tail.junk = 0x04;
	DW_2_Chars( pack_tail.status, ICQ_ACKxTCP_ONLINE );
	DW_2_Chars( pack_tail.portb, tempport );
	DW_2_Chars( pack_tail.one, 0x0001 );
	pack_tail.zero = 0x00;
	DW_2_Chars( pack_tail.bigzero, backport );
	DW_2_Chars( pack_tail.back_port, backport );
	DW_2_Chars( pack_tail.seq, seq );
	if( sock != -1 )
	{
		intsize = sizeof( tcp_head ) + sizeof( tcp_tail ) + 1;
		buffer = (BYTE*)g_malloc( intsize + 2 );

		Word_2_Chars (buffer, intsize);
		memcpy( buffer+2, &pack_head, sizeof( pack_head ) );
		buffer[2 + sizeof( pack_head )] = 0x00;
		memcpy( buffer+2 + sizeof( pack_head ) + 1,
		        &pack_tail, sizeof( pack_tail ) );
		write( sock, buffer, intsize + 2 );
		packet_print( buffer + 2, intsize,
		              PACKET_TYPE_TCP | PACKET_DIRECTION_SEND, "ACCEPT FILE" );
	}
	else
		return -1;

	return 1;
}

int TCPRefuseFile( int sock, DWORD seq )
{
	BYTE *buffer;
	unsigned short intsize;

	typedef struct
	{
		BYTE uin1[4];
		BYTE version[2];
		BYTE command[2];
		BYTE zero[2];
		BYTE uin2[4];
		BYTE cmd[2];
		BYTE message_length[2];
	} tcp_head;

	typedef struct
	{
		BYTE ip[4];
		BYTE ip_real[4];
		BYTE porta[4];
		BYTE junk;
		BYTE status[4];
		BYTE zeroa[4];
		BYTE zerob[4];
		BYTE zeroc[2];
		BYTE zerod;
		BYTE seq[4];
	} tcp_tail;

	tcp_head pack_head;
	tcp_tail pack_tail;

#ifdef TRACE_FUNCTION
	g_print( "TCPRefuseFile\n" );
#endif

	DW_2_Chars( pack_head.uin1, our_info->uin );
	Word_2_Chars( pack_head.version, 0x0003 );
	Word_2_Chars( pack_head.command, ICQ_CMDxTCP_CANCEL );
	Word_2_Chars( pack_head.zero, 0x0000 );
	DW_2_Chars( pack_head.uin2, our_info->uin );
	DW_2_Chars( pack_head.cmd, ICQ_CMDxTCP_FILE );
	DW_2_Chars( pack_head.message_length, 1 );
	
	DW_2_IP( pack_tail.ip, our_ip );
	DW_2_IP( pack_tail.ip_real, LOCALHOST );
	DW_2_Chars( pack_tail.porta, our_port );
	pack_tail.junk = 0x04;
	DW_2_Chars( pack_tail.status, ICQ_ACKxTCP_REFUSE );

	DW_2_Chars( pack_tail.zeroa, 0x00000001 );
	DW_2_Chars( pack_tail.zerob, 0x00000000 );
	DW_2_Chars( pack_tail.zeroc, 0x00000000 );
	pack_tail.zerod = 0x00;
	DW_2_Chars( pack_tail.seq, seq );

	if( sock != -1 )
	{
		intsize = sizeof( tcp_head ) + sizeof( tcp_tail ) + 1;
		buffer = (BYTE*)g_malloc0( intsize + 2 );

		Word_2_Chars (buffer, intsize);
		memcpy( buffer+2, &pack_head, sizeof( pack_head ) );
		buffer[2 + sizeof( pack_head )] = 0x00;
		memcpy( buffer+2 + sizeof( pack_head ) + 1,
		        &pack_tail, sizeof( pack_tail ) );
		write( sock, buffer, intsize + 2 );
		packet_print( buffer + 2, intsize,
		              PACKET_TYPE_TCP | PACKET_DIRECTION_SEND, "REFUSE FILE" );
		g_free( buffer );
	}
	else
		return -1;

	return 1;
}

int TCPSendFileRequest( UIN_T uin, const gchar *msg, const gchar *file )
{
	int cx;
	int sock;
	unsigned short intsize;
	BYTE *buffer;
	int size;

	typedef struct
	{
		BYTE uin_a[4]; /* Sender's UIN */
		BYTE version[2]; /* 0x0003 */
		BYTE cmd[2]; /* 0x07ee */
		BYTE zero[2]; /* 0x0000 (duh) */
		BYTE uin_b[4]; /* Sender's UIN */
		BYTE type[2]; /* 0x0003 */
		BYTE msg_length[2];
	} tcp_head;

	typedef struct
	{
		BYTE ip[4];
		BYTE real_ip[4];
		BYTE port[4];
		BYTE four; /* 0x04 */
		BYTE status[2];
		BYTE type[2]; /* Always 0x0010 */
		BYTE zero[4]; /* Always 0x00000000 */
		BYTE filename_len[2];
	} tcp_mid;

	typedef struct
	{
		BYTE flen[4]; /* Length of file */
		BYTE zero[4]; /* Always 0x00000000 */
		BYTE seq[4];
	} tcp_tail;

	struct
	{
		tcp_head head;
		const gchar *body;
		tcp_mid mid;
		const gchar *filename;
		tcp_tail tail;
	} packet;

	struct stat file_stat;
	const gchar *nopathfile = file;

	GSList *contact, *contact2;

#ifdef TRACE_FUNCTION
	g_print( "TCPSendFileRequest\n" );
#endif

	contact = Contacts;

	while( contact != NULL )
	{
		if( ((CONTACT_PTR)contact->data)->uin == uin )
			break;
		contact = contact->next;
	}
	
	if( contact == NULL )
		return FALSE;

	contact2 = contact;

	((CONTACT_PTR)contact->data)->file_name = g_strdup( file );

	for( cx = 0; cx < strlen( file ); cx ++ )
	{
		if( file[ cx ] == '/' )
			nopathfile = &file[ cx + 1 ];
	}

	((CONTACT_PTR)contact->data)->file_short_name = (char*)g_malloc0( strlen( nopathfile ) + 1 );
	strcpy( ((CONTACT_PTR)contact->data)->file_short_name, nopathfile );

	if( strlen( nopathfile ) == 0 )
		return FALSE;

	if( stat( file, &file_stat ) == -1 )
		return FALSE;

	DW_2_Chars( packet.head.uin_a, our_info->uin );
	Word_2_Chars( packet.head.version, 0x0003 );
	Word_2_Chars( packet.head.cmd, ICQ_CMDxTCP_START );
	Word_2_Chars( packet.head.zero, 0x0000 );
	DW_2_Chars( packet.head.uin_b, our_info->uin );
	Word_2_Chars( packet.head.type, ICQ_CMDxTCP_FILE );
	Word_2_Chars( packet.head.msg_length, ( strlen( msg ) + 1 ) );

	packet.body = msg;

	DW_2_IP( packet.mid.ip, our_ip );
	DW_2_IP( packet.mid.real_ip, LOCALHOST );
	DW_2_Chars( packet.mid.port, our_port );
	packet.mid.four = 0x04;

	switch( Current_Status )
	{
		case STATUS_ONLINE:
			Word_2_Chars( packet.mid.status, ICQ_ACKxTCP_ONLINE );
			break;
		case STATUS_AWAY:
			Word_2_Chars( packet.mid.status, ICQ_ACKxTCP_AWAY );
			break;
		case STATUS_DND:
			Word_2_Chars( packet.mid.status, ICQ_ACKxTCP_DND );
			break;
		case STATUS_OCCUPIED:
			Word_2_Chars( packet.mid.status, ICQ_ACKxTCP_OCC );
			break;
		case STATUS_NA:
			Word_2_Chars( packet.mid.status, ICQ_ACKxTCP_NA );
			break;
		case STATUS_INVISIBLE:
			Word_2_Chars( packet.mid.status, ICQ_ACKxTCP_ONLINE );
			break;
		}

	Word_2_Chars( packet.mid.type, 0x0010 );
	DW_2_Chars( packet.mid.zero, 0x00000000 );
	Word_2_Chars( packet.mid.filename_len, strlen( nopathfile ) + 1 );

	packet.filename = nopathfile;

	DW_2_Chars( packet.tail.flen, file_stat.st_size );
	DW_2_Chars( packet.tail.zero, 0x00000000 );
	DW_2_Chars( packet.tail.seq, seq_num ++ );

	contact = Contacts;

	while( contact != NULL )
	{
		if( ((CONTACT_PTR)contact->data)->uin == uin )
			break;
		contact = contact->next;
	}

	if( contact == NULL )
		return 0;

/*** Create packet ***/
	intsize = sizeof( tcp_head ) + sizeof( tcp_mid ) + sizeof( tcp_tail ) + strlen( msg ) + strlen( packet.filename ) + 2;
	buffer = (BYTE*)g_malloc0( strlen( "REQUEST FILE" ) + 1 + intsize + 2 );

	size = 0;

	Word_2_Chars (buffer + size, intsize);
	size += 2;
	memcpy( buffer + size, &packet.head, sizeof( packet.head ) );
	size += sizeof( packet.head );
	memcpy( buffer + size, packet.body,
	        strlen( packet.body ) + 1 );
	size += strlen( packet.body ) + 1;
	memcpy( buffer + size, &packet.mid, sizeof( packet.mid ) );
	size += sizeof( packet.mid );
	memcpy( buffer + size, packet.filename, strlen( packet.filename ) + 1 );
	size += strlen( packet.filename ) + 1;
	memcpy( buffer + size,
	        &packet.tail, sizeof( packet.tail ) );
	size += sizeof( packet.tail );
	strcpy( buffer + 2 + intsize, "REQUEST FILE" );
/*********************/

	((CONTACT_PTR)contact->data)->tcp_msg_queue = g_slist_append( ((CONTACT_PTR)contact->data)->tcp_msg_queue, buffer );

	/* We'll do this for alignment purposes */
	((CONTACT_PTR)contact->data)->tcp_text_queue = g_slist_append( ((CONTACT_PTR)contact->data)->tcp_text_queue, NULL );

	sock = TCPGainConnection( ((CONTACT_PTR)contact->data)->current_ip, ((CONTACT_PTR)contact->data)->port, contact );

	if( sock == -2 ) /* We're waiting for a connection */
		return TRUE;

	if( sock != -1 )
		TCPClearQueue( contact );
	else
		return -1;

	return 1;
}

void TCPSendContactList( GtkWidget *widget, gpointer data )
{
	GtkCList *clist = GTK_CLIST( data );
	gchar *nick;
	int cx;

	GSList *contact;

	int sock;
	unsigned short intsize;
	guchar *msg;
	BYTE *buffer;
	UIN_T uin = GPOINTER_TO_INT( gtk_object_get_data( GTK_OBJECT( widget ), "uin" ) );

	typedef struct
	{
		BYTE uin_a[4];
		BYTE version[2];
		BYTE cmd[2];
		BYTE zero[2];
		BYTE uin_b[4];
		BYTE command[2];
		BYTE msg_length[2];
	} tcp_head;

	typedef struct
	{
		BYTE ip[4];
		BYTE real_ip[4];
		BYTE port[4];
		BYTE four;
		BYTE zero[4];
		BYTE seq[4];
	} tcp_tail;

	struct
	{
		tcp_head head;
		const gchar *body;
		tcp_tail tail;
	} packet;


#ifdef TRACE_FUNCTION
	g_print( "TCPSendContactList\n" );
#endif

	msg = g_strdup_printf( "%d\xFE", clist->rows );

	for( cx = 0; cx < clist->rows; cx ++ )
	{
		gtk_clist_get_text( clist, cx, 0, &nick );
		msg = g_strconcat( msg, nick, "\xFE", NULL );
		gtk_clist_get_text( clist, cx, 1, &nick );
		msg = g_strconcat( msg, nick, "\xFE", NULL );
	}

	DW_2_Chars( packet.head.uin_a, our_info->uin );
	Word_2_Chars( packet.head.version, 0x0003 );
	Word_2_Chars( packet.head.cmd, ICQ_CMDxTCP_START );
	Word_2_Chars( packet.head.zero, 0x0000 );
	DW_2_Chars( packet.head.uin_b, our_info->uin );
	Word_2_Chars( packet.head.command, ICQ_CMDxTCP_CONT_LIST );
	Word_2_Chars( packet.head.msg_length, ( strlen( msg ) + 1 ) );

	packet.body = msg;

	contact = Contacts;

	while( contact != NULL )
	{
		if( ((CONTACT_PTR)contact->data)->uin == uin )
			break;
		contact = contact->next;
	}

	if( contact == NULL )
		return;

	DW_2_IP( packet.tail.ip, our_ip );
	DW_2_IP( packet.tail.real_ip, our_ip );
	DW_2_Chars( packet.tail.port, our_port );
	packet.tail.four = 0x04;
	if( ((CONTACT_PTR)contact->data)->status == STATUS_DND || ((CONTACT_PTR)contact->data)->status == STATUS_OCCUPIED )
		DW_2_Chars( packet.tail.zero, 0x00200000 );
	else
		DW_2_Chars( packet.tail.zero, 0x00100000 );
	DW_2_Chars( packet.tail.seq, seq_num ++ );

	contact = Contacts;

	while( contact != NULL )
	{
		if( ((CONTACT_PTR)contact->data)->uin == uin )
			break;
		contact = contact->next;
	}

	if( contact == NULL )
		return;

	/* If the user is offline, we use UDP to avoid hanging */
	if( ((CONTACT_PTR)contact->data)->status == STATUS_OFFLINE )
		return;

/*** Construct packet ***/
	intsize = sizeof( tcp_head ) + sizeof( tcp_tail ) + strlen( msg ) + 1;
	buffer = (BYTE*)g_malloc0( strlen( "SEND CONTACT LIST" ) + 1 + intsize + 2 );

	Word_2_Chars (buffer, intsize);
	memcpy( buffer+2, &packet.head, sizeof( packet.head ) );
	memcpy( buffer+2 + sizeof( packet.head ), packet.body,
	        strlen( packet.body ) + 1 );
	memcpy( buffer+2 + sizeof( packet.head ) +
	        strlen( packet.body ) + 1,
	        &packet.tail, sizeof( packet.tail ) );
	strcpy( buffer + 2 + intsize, "SEND CONTACT LIST" );
/************************/

	((CONTACT_PTR)contact->data)->tcp_msg_queue = g_slist_append( ((CONTACT_PTR)contact->data)->tcp_msg_queue, buffer );
	((CONTACT_PTR)contact->data)->tcp_text_queue = g_slist_append( ((CONTACT_PTR)contact->data)->tcp_text_queue, g_strdup( msg ) );

	sock = TCPGainConnection( ((CONTACT_PTR)contact->data)->current_ip, ((CONTACT_PTR)contact->data)->port, contact );
	if( sock != -1 )
		((CONTACT_PTR)contact->data)->tcp_timeout = gtk_timeout_add( 10000, (GtkFunction)TCPTimeout, (CONTACT_PTR)contact->data );

	if( sock == -2 ) /* We're waiting for a connection */
	{
		animate_on();
		return;
	}

	if( sock != -1 )
		TCPClearQueue( contact );
	else
		return;

	animate_on();
	return;
}

/*** Local functions ***/
int TCPInitChannel( GIOChannel *source, GIOCondition cond, gpointer data )
{
	UIN_T uin;
	guint16 packet_size;
	BYTE *packet;
	int sock;

	gchar *newnick;

	GSList *contact;
	
	contact = Contacts;

#ifdef TRACE_FUNCTION
	g_print( "TCPInitChannel(" );
	switch( cond )
	{
		case G_IO_IN: g_print( "G_IO_IN)\n" ); break;
		case G_IO_OUT: g_print( "G_IO_OUT)\n" ); break;
		case G_IO_ERR: g_print( "G_IO_ERR)\n" ); break;
		case G_IO_HUP: g_print( "G_IO_HUP)\n" ); break;
		default: g_print( "default)\n" );
	}
#endif

	sock = g_io_channel_unix_get_fd( source );

	read( sock, (char*)(&packet_size), 1 );
	if( ( read( sock, (char*)(&packet_size) + 1, 1 ) ) <= 0 )
	{
		close( sock );
		return FALSE;
	}
	packet_size = GUINT16_FROM_LE (packet_size);

	packet = (BYTE *)g_malloc0( packet_size );
	read( sock, packet, packet_size );

	uin = Chars_2_DW (packet + 9);

	contact = Contacts;
	
	while( contact != NULL )
	{
		if( ((CONTACT_PTR)contact->data)->uin == uin )
		{
			((CONTACT_PTR)contact->data)->sok = sock;
			break;
		}
		contact = contact->next;
	}

	if( contact == NULL && toggles->no_new_users == FALSE )
	{
		newnick = g_strdup_printf( "%d", uin );
		Add_User( uin, newnick, FALSE );
		g_free( newnick );
	}

	contact = Contacts;
	
	while( contact != NULL )
	{
		if( ((CONTACT_PTR)contact->data)->uin == uin )
		{
			((CONTACT_PTR)contact->data)->sok = sock;
			break;
		}
		contact = contact->next;
	}

	TCPProcessPacket( packet, packet_size, sock );

	fcntl( sock, F_SETFL, O_NONBLOCK );

	((CONTACT_PTR)contact->data)->have_tcp_connection = TRUE;
	((CONTACT_PTR)contact->data)->gioc = g_io_channel_unix_new( sock );

	((CONTACT_PTR)contact->data)->giocw =
	g_io_add_watch( ((CONTACT_PTR)contact->data)->gioc,
	                G_IO_IN | G_IO_ERR | G_IO_HUP,
	                TCPReadPacket, NULL );

	g_free( data );
	g_free( packet );
	return FALSE;
}

int TCPAckPacket( int sock, GSList *contact, WORD cmd, int seq )
{
	BYTE *buffer;
	unsigned short intsize;
	char *sent_message;

	typedef struct
	{
		BYTE uin1[4];
		BYTE version[2];
		BYTE command[2];
		BYTE zero[2];
		BYTE uin2[4];
		BYTE cmd[2];
		BYTE message_length[2];
	} tcp_head;

	typedef struct
	{
		BYTE ip[4];
		BYTE ip_real[4];
		BYTE port[4];
		BYTE junk;
		BYTE status[4];
		BYTE seq[4];
	} tcp_tail;

	tcp_head pack_head;
	tcp_tail pack_tail;

#ifdef TRACE_FUNCTION
	g_print( "TCPAckPacket\n" );
#endif

	sent_message = "";
	if( (Current_Status&0xffff) != STATUS_ONLINE && (Current_Status&0xffff) != STATUS_FREE_CHAT &&
	    (Current_Status&0xffff) != STATUS_INVISIBLE )
		sent_message = Away_Message;
	if( cmd == ICQ_CMDxTCP_VERSION )
		sent_message = VERSION;
	
	DW_2_Chars( pack_head.uin1, our_info->uin );
	Word_2_Chars( pack_head.version, 0x0003 );
	Word_2_Chars( pack_head.command, ICQ_CMDxTCP_ACK );
	Word_2_Chars( pack_head.zero, 0x0000 );
	DW_2_Chars( pack_head.uin2, our_info->uin );
	DW_2_Chars( pack_head.cmd, cmd );
	DW_2_Chars( pack_head.message_length, strlen( sent_message ) + 1 );
	
	DW_2_IP( pack_tail.ip, our_ip );
	DW_2_IP( pack_tail.ip_real, LOCALHOST );
	DW_2_Chars( pack_tail.port, our_port );
	pack_tail.junk = 0x04;
	DW_2_Chars( pack_tail.seq, seq );

	switch( Current_Status )
	{
		case STATUS_ONLINE:
			DW_2_Chars( pack_tail.status, ICQ_ACKxTCP_ONLINE );
			break;
		case STATUS_AWAY:
			DW_2_Chars( pack_tail.status, ICQ_ACKxTCP_AWAY );
			break;
		case STATUS_DND:
			DW_2_Chars( pack_tail.status, ICQ_ACKxTCP_DND );
			break;
		case STATUS_OCCUPIED:
			/* Don't ask me why this is - I just follow precedent */
			DW_2_Chars( pack_tail.status, ICQ_ACKxTCP_ONLINE );
			break;
		case STATUS_NA:
			DW_2_Chars( pack_tail.status, ICQ_ACKxTCP_NA );
			break;
		case STATUS_INVISIBLE:
			DW_2_Chars( pack_tail.status, ICQ_ACKxTCP_ONLINE );
			break;
		}

	if( sock != -1 )
	{
		intsize = sizeof( tcp_head ) + sizeof( tcp_tail ) + strlen( sent_message ) + 1;
		buffer = (BYTE*)g_malloc0( intsize + 2 );

		Word_2_Chars (buffer, intsize);
		memcpy( buffer+2, &pack_head, sizeof( pack_head ) );
		memcpy( buffer+2 + sizeof( pack_head ), sent_message,
		        strlen( sent_message ) + 1 );
		memcpy( buffer+2 + sizeof( pack_head ) + strlen( sent_message ) + 1,
		        &pack_tail, sizeof( pack_tail ) );
		write( sock, buffer, intsize + 2 );
		packet_print( buffer+2, intsize,
		              PACKET_TYPE_TCP | PACKET_DIRECTION_SEND, "ACK PACKET" );
		g_free( buffer );
	}
	else
		return -1;

	return 1;
}

void TCPProcessPacket( BYTE *packet, int packet_length, int sock )
{
	int chat_sock;
	int file_sock;
	BYTE *size;

	typedef struct
	{
		UIN_T uin1;
		WORD version;
		WORD command;
		WORD zero;
		UIN_T uin2;
		WORD cmd;
		WORD message_length;
	} tcp_head;

	typedef struct
	{
		DWORD ip_sender;
		DWORD ip_local;
		DWORD port;
		BYTE junk;
		DWORD status;
		DWORD chat_port;
		DWORD seq;
	} tcp_tail;

	typedef struct
	{
		WORD name_len;
		char *name;
		DWORD size;
		DWORD zero;
		DWORD seq;
	} tcp_file;

	char *message;
	char *away_message;
	tcp_head pack_head;
	tcp_tail pack_tail;
	tcp_file pack_file;

	GSList *contact;

	DWORD i;

#ifdef TRACE_FUNCTION
	g_print( "TCPProcessPacket\n" );
#endif

	if( packet[0] == 0xFF ) /* 0xFF means it's just a "Hello" packet; ignore */
	{
		packet_print( packet, packet_length,
		              PACKET_TYPE_TCP | PACKET_DIRECTION_RECEIVE, "HELLO PACKET" );
		return;
	}

	size = packet;
	pack_head.uin1 = Chars_2_DW (size);
	size += 4;
	pack_head.version = Chars_2_Word (size);
	size += 2;
	pack_head.command = Chars_2_Word (size);
	size += 2;
	pack_head.zero = Chars_2_Word (size);
	size += 2;
	pack_head.uin2 = Chars_2_DW (size);
	size += 4;
	pack_head.cmd = Chars_2_Word (size);
	size += 2;
	pack_head.message_length = Chars_2_Word (size);
	size += 2;

	message = (char *)g_malloc0( pack_head.message_length );
	memcpy( message, size,
	        pack_head.message_length );
	size += pack_head.message_length;
	pack_tail.ip_sender = Chars_2_DW (size);
	size += 4;
	pack_tail.ip_local = Chars_2_DW (size);
	size += 4;
	pack_tail.port = Chars_2_DW (size);
	size += 4;
	memcpy( &pack_tail.junk, size, 1 );
	size ++;
	pack_tail.status = Chars_2_DW (size);
	size += 4;
	if( pack_head.command == ICQ_CMDxTCP_ACK && pack_head.cmd == ICQ_CMDxTCP_FILE )
	{
		size += 11;
		pack_tail.chat_port = Chars_2_DW (size);
		size += 4;
	}
	if( pack_head.cmd == ICQ_CMDxTCP_CHAT )
	{
		size += 7; /* What is this supposed to be? */
		pack_tail.chat_port = Chars_2_DW (size);
		size += 4;
	}
	pack_tail.seq = Chars_2_DW (size);
	size += 4;

	// What are these. Not used anyway...
	i = pack_tail.ip_sender;
	pack_tail.ip_sender = ( ( i << 24 ) | ( ( i & 0xff00 ) << 8 ) | ( ( i & 0xff0000 ) >> 8 ) | ( i >> 24 ) );
	i = pack_tail.ip_local;
	pack_tail.ip_local = ( ( i << 24 ) | ( ( i & 0xff00 ) << 8 ) | ( ( i & 0xff0000 ) >> 8 ) | ( i >> 24 ) );

   switch( pack_head.command )
   {
   case ICQ_CMDxTCP_START:
      switch( pack_head.cmd )
      {
      case ICQ_CMDxTCP_MSG:
			packet_print( packet, packet_length,
			              PACKET_TYPE_TCP | PACKET_DIRECTION_RECEIVE, "TCP MESSAGE(MSG)" );
			contact = Do_Msg( time( NULL ), 0, message, pack_head.uin1, MESSAGE_TEXT );
			log_window_add( _("Received Message (TCP)"), pack_head.uin1 );
			TCPAckPacket( sock, contact, pack_head.cmd, pack_tail.seq );
         break;
		case ICQ_CMDxTCP_VERSION:
			packet_print( packet, packet_length,
			              PACKET_TYPE_TCP | PACKET_DIRECTION_RECEIVE, "VERSION REQUEST" );
			contact = Contacts;

			while( contact != NULL )
			{
				if( ((CONTACT_PTR)contact->data)->uin == pack_head.uin1 )
					break;
				contact = contact->next;
			}

			if( contact == NULL )
				break;

			TCPAckPacket( sock, contact, ICQ_CMDxTCP_VERSION, pack_tail.seq );
			break;
      case ICQ_CMDxTCP_READxAWAYxMSG:
      case ICQ_CMDxTCP_READxOCCxMSG:
      case ICQ_CMDxTCP_READxDNDxMSG:
      case ICQ_CMDxTCP_READxNAxMSG:
			packet_print( packet, packet_length,
			              PACKET_TYPE_TCP | PACKET_DIRECTION_RECEIVE, "AWAY MSG REQUEST" );
			contact = Contacts;

			while( contact != NULL )
			{
				if( ((CONTACT_PTR)contact->data)->uin == pack_head.uin1 )
					break;
				contact = contact->next;
			}

			if( contact == NULL )
				break;

			TCPAckPacket( sock, contact, ICQ_CMDxTCP_READxAWAYxMSG,
			              pack_tail.seq );
         break;

      case ICQ_CMDxTCP_URL:  /* url sent */
			packet_print( packet, packet_length,
			              PACKET_TYPE_TCP | PACKET_DIRECTION_RECEIVE, "TCP MESSAGE(URL)" );
			contact = Do_Msg( time( NULL ), 0, message, pack_head.uin1, MESSAGE_URL );
			TCPAckPacket( sock, contact, pack_head.cmd, pack_tail.seq );
         break;

		case ICQ_CMDxTCP_CONT_LIST:
			packet_print( packet, packet_length,
			              PACKET_TYPE_TCP | PACKET_DIRECTION_RECEIVE, "TCP MESSAGE(CONT_LIST)" );
			contact = Do_Msg( time( NULL ), 0, message, pack_head.uin1, MESSAGE_CONT_LIST );
			TCPAckPacket( sock, contact, pack_head.cmd, pack_tail.seq );
			break;

      case ICQ_CMDxTCP_CHAT:
			packet_print( packet, packet_length,
			              PACKET_TYPE_TCP | PACKET_DIRECTION_RECEIVE, "CHAT REQUEST" );
			contact = Do_Chat( 0, message, pack_head.uin1, pack_tail.seq );
			if( Current_Status == STATUS_FREE_CHAT )
				TCPAcceptChat( sock, contact, pack_tail.seq );
			break;

      case ICQ_CMDxTCP_FILE:
			packet_print( packet, packet_length,
			              PACKET_TYPE_TCP | PACKET_DIRECTION_RECEIVE, "FILE REQUEST" );
			pack_file.name_len = Chars_2_Word (size);
			size += 2;
			pack_file.name = (char*)g_malloc0( pack_file.name_len );
			memcpy( pack_file.name, size, pack_file.name_len );
			size += pack_file.name_len;
			pack_file.size = Chars_2_DW (size);
			size += 4;
			pack_file.zero = Chars_2_DW (size);
			size += 4;
			pack_file.seq = Chars_2_DW (size);
			
			contact = Do_File( 0, message, pack_head.uin1, pack_file.seq );
			((CONTACT_PTR)contact->data)->file_short_name = (char*)g_malloc0( strlen( pack_file.name ) + 1 );
			strcpy( ((CONTACT_PTR)contact->data)->file_short_name, pack_file.name );
			((CONTACT_PTR)contact->data)->file_size = pack_file.size;
			gnomeicu_event( EV_FILEREQ , ((CONTACT_PTR)contact->data)->uin );

         break;      

      default:
         break;
      }
      break;

	case ICQ_CMDxTCP_ACK:
		switch ( pack_head.cmd )
      {
			case ICQ_CMDxTCP_VERSION:
				contact = Contacts;

				while( contact != NULL )
				{
					if( ((CONTACT_PTR)contact->data)->uin == pack_head.uin1 )
						break;
					contact = contact->next;
				}

				packet_print( packet, packet_length,
				              PACKET_TYPE_TCP | PACKET_DIRECTION_RECEIVE, "RESPONSE VERSION" );

				away_message = g_strdup_printf( "%s is running\nGnomeICU version:",
				                                ((CONTACT_PTR)contact->data)->nick );

				AMessage_Box( away_message, message, ((CONTACT_PTR)contact->data)->uin );
				g_free( away_message );

				break;
			case ICQ_CMDxTCP_MSG:
			case ICQ_CMDxTCP_CONT_LIST:
      	case ICQ_CMDxTCP_READxAWAYxMSG:
      	case ICQ_CMDxTCP_READxOCCxMSG:
      	case ICQ_CMDxTCP_READxDNDxMSG:
      	case ICQ_CMDxTCP_READxNAxMSG:
			case ICQ_CMDxTCP_URL:
				contact = Contacts;

				while( contact != NULL )
				{
					if( ((CONTACT_PTR)contact->data)->uin == pack_head.uin1 )
						break;
					contact = contact->next;
				}

				if( contact != NULL && ( pack_head.cmd == ICQ_CMDxTCP_MSG ||
				                         pack_head.cmd == ICQ_CMDxTCP_URL ||
				                         pack_head.cmd == ICQ_CMDxTCP_CONT_LIST ) )
				{
					packet_print( packet, packet_length,
					              PACKET_TYPE_TCP | PACKET_DIRECTION_RECEIVE, "MSG/URL/CONT LIST ACK" );
					animate_off();

					if( ((CONTACT_PTR)contact->data)->tcp_text_queue )
					{
						g_free( ((CONTACT_PTR)contact->data)->tcp_text_queue->data );
						((CONTACT_PTR)contact->data)->tcp_text_queue = g_slist_remove( ((CONTACT_PTR)contact->data)->tcp_text_queue, ((CONTACT_PTR)contact->data)->tcp_text_queue->data );
					}
					if( ((CONTACT_PTR)contact->data)->tcp_timeout )
						gtk_timeout_remove( ((CONTACT_PTR)contact->data)->tcp_timeout );
				}
				else
					packet_print( packet, packet_length,
					              PACKET_TYPE_TCP | PACKET_DIRECTION_RECEIVE, "RESPONSE AWAY MSG" );


				if( pack_tail.status == ICQ_ACKxTCP_AWAY ||
				    pack_tail.status == ICQ_ACKxTCP_NA   ||
				    pack_tail.status == ICQ_ACKxTCP_DND  ||
				    pack_tail.status == ICQ_ACKxTCP_OCC )
				{
					switch( pack_tail.status )
					{
						case ICQ_ACKxTCP_AWAY:
							away_message = g_strdup_printf( _("User %s is Away:"), ((CONTACT_PTR)contact->data)->nick );
							break;
						case ICQ_ACKxTCP_NA:
							away_message = g_strdup_printf( _("User %s is Not Available:"), ((CONTACT_PTR)contact->data)->nick );
							break;
						case ICQ_ACKxTCP_DND:
							away_message = g_strdup_printf( _("User %s cannot be disturbed:"), ((CONTACT_PTR)contact->data)->nick );
							break;
						case ICQ_ACKxTCP_OCC:
							away_message = g_strdup_printf( _("User %s is Occupied:"), ((CONTACT_PTR)contact->data)->nick );
							break;
						default:
							away_message = g_strdup( _("User is in an unknown mode:") );
					}
	
					AMessage_Box( away_message, message, ((CONTACT_PTR)contact->data)->uin );
					g_free( away_message );
				}
				break;

			case ICQ_CMDxTCP_CHAT:
				packet_print( packet, packet_length,
				              PACKET_TYPE_TCP | PACKET_DIRECTION_RECEIVE, "ACK CHAT REQUEST" );
				if( pack_tail.chat_port > 0 )
					chat_sock = TCPConnectChat( pack_tail.chat_port, pack_head.uin1 );
      	   break;

			case ICQ_CMDxTCP_FILE:
				packet_print( packet, packet_length,
				              PACKET_TYPE_TCP | PACKET_DIRECTION_RECEIVE, "ACK FILE REQUEST" );
				if( pack_tail.chat_port > 0 )
					file_sock = TCPConnectFile( pack_tail.chat_port, pack_head.uin1 );
      	   break;

			default:
				break;
		}
		break;
      
	case ICQ_CMDxTCP_CANCEL:
      switch ( pack_head.cmd )
      {
      case ICQ_CMDxTCP_CHAT:
			packet_print( packet, packet_length,
			              PACKET_TYPE_TCP | PACKET_DIRECTION_RECEIVE, "CHAT REQ CANCEL" );
			OK_Box( _("Chat Request Refused"), message );
         break;
      
      case ICQ_CMDxTCP_FILE:
			packet_print( packet, packet_length,
			              PACKET_TYPE_TCP | PACKET_DIRECTION_RECEIVE, "FILE REQ CANCEL" );
			OK_Box( _("File Transfer Refused"), message );
         break;
      
      default:
         break;
      }
      break;
   
   default:
      break;
   }

	g_free( message );
} 

int TCPConnectFile( DWORD port, UIN_T uin )
{
	DWORD localport;
	struct sockaddr_in local, remote;
	int sizeofSockaddr = sizeof( struct sockaddr );
	int sock;
	DWORD ip;
	BYTE *buffer;
	int index, psize;

	struct stat file_stat;

	typedef struct
	{
		BYTE id; /* 0xff */
		BYTE version[4]; /* 0x00000003 */
		BYTE port[4]; /* TCP listen port */
		BYTE uin[4]; /* UIN */
		BYTE ipa[4]; /* IP A */
		BYTE ipb[4]; /* IP B */
		BYTE four; /* 0x04 */
		BYTE listenport[4]; /* File listen port */
	} handshake;

	handshake hs;
	xfer0 pak;

	GSList *contact;

#ifdef TRACE_FUNCTION
	g_print( "TCPConnectFile\n" );
#endif

	contact = Contacts;

	while( contact != NULL )
	{
		if( ((CONTACT_PTR)contact->data)->uin == uin )
			break;
		contact = contact->next;
	}

	if( contact == NULL )
		return -1;

	if( ((CONTACT_PTR)contact->data)->file_sok > 0 )
		return ((CONTACT_PTR)contact->data)->file_sok;

	ip = ((CONTACT_PTR)contact->data)->current_ip;

	if( ip == 0 )
		return -1;

	sock = socket( AF_INET, SOCK_STREAM, 0 );
	if( sock == -1 )
		return -1;

	fcntl( sock, O_NONBLOCK );

	memset( &local.sin_zero, 0x00, 8 );
	memset( &remote.sin_zero, 0x00, 8 );
	
	local.sin_family = AF_INET;
	remote.sin_family = AF_INET;
	local.sin_port = g_htons( 0 );
	local.sin_addr.s_addr = g_htonl( INADDR_ANY );

	remote.sin_port = g_htons( port );
	remote.sin_addr.s_addr = g_htonl( ip );

	if( connect( sock, (struct sockaddr *)&remote, sizeofSockaddr ) < 0 )
	{
		return -1;
	}

	getsockname( sock, (struct sockaddr*)&local, &sizeofSockaddr );
	localport = g_ntohs( local.sin_port );

	if( stat( ((CONTACT_PTR)contact->data)->file_name, &file_stat ) == -1 )
		return FALSE;

	((CONTACT_PTR)contact->data)->file_size = file_stat.st_size;

	hs.id = 0xff;
	DW_2_Chars( hs.version, 0x00000003 );
	DW_2_Chars( hs.port, our_port );
	DW_2_Chars( hs.uin, our_info->uin );
	DW_2_IP( hs.ipa, our_ip );
	DW_2_IP( hs.ipb, LOCALHOST );
	hs.four = 0x04;
	DW_2_Chars( hs.listenport, localport );

	Word_2_Chars( pak.psize, sizeof( xfer0 ) - 2 + strlen( our_info->nick ) + 1 );
	pak.id = 0x00;
	DW_2_Chars( pak.zero, 0x00000000 );
	DW_2_Chars( pak.n_files, 0x00000001 );
	DW_2_Chars( pak.n_bytes, file_stat.st_size );
	DW_2_Chars( pak.speed, 0x00000064 );
	Word_2_Chars( pak.nick_len, strlen( our_info->nick ) + 1 );

	fcntl( sock, O_NONBLOCK );
	((CONTACT_PTR)contact->data)->file_gdk_input = gdk_input_add( sock, GDK_INPUT_READ, (GdkInputFunction) TCPFileReadServer, NULL );

	((CONTACT_PTR)contact->data)->file_sok = sock;

	index = 0;
	psize = sizeof( handshake );
	buffer = (BYTE*)g_malloc0( psize + 2 );

	Word_2_Chars (buffer, psize);
	index += 2;
	memcpy( buffer + index, &hs, sizeof( handshake ) );

	packet_print( buffer, psize+2,
	              PACKET_TYPE_TCP | PACKET_DIRECTION_SEND, "FILE(INIT)" );
	write( sock, buffer, psize + 2 );
	g_free( buffer );

	index = 0;
	psize = sizeof( xfer0 ) + strlen( our_info->nick ) + 1;
	buffer = (BYTE*)g_malloc0( psize );
	
	memcpy( buffer, &pak, sizeof( xfer0 ) );
	index += sizeof( xfer0 );
	memcpy( buffer + index, our_info->nick, strlen( our_info->nick ) + 1 );

	packet_print( buffer, psize,
	              PACKET_TYPE_TCP | PACKET_DIRECTION_SEND, "FILE(INIT)" );
	write( sock, buffer, psize );
	g_free( buffer );

	return sock;
}

int TCPFileHandshake( GSList *contact, int sock, GdkInputCondition cond )
{
	int new_sock;
	int size = sizeof( struct sockaddr );
	struct sockaddr_in their_addr;

#ifdef TRACE_FUNCTION
	g_print( "TCPFileHandshake\n" );
#endif
	
	new_sock = accept( sock, ( struct sockaddr * )&their_addr, &size );
	((CONTACT_PTR)contact->data)->file_sok = new_sock;
	((CONTACT_PTR)contact->data)->file_port = g_ntohs( their_addr.sin_port );

	if( ((CONTACT_PTR)contact->data)->file_gdk_input )
		gtk_input_remove( ((CONTACT_PTR)contact->data)->file_gdk_input );

	fcntl( new_sock, O_NONBLOCK );
	((CONTACT_PTR)contact->data)->file_gdk_input = gdk_input_add( new_sock, GDK_INPUT_READ, (GdkInputFunction) TCPFileReadClient, NULL );

	return TRUE;
}

int TCPFileReadServer( GtkWidget *widget, int sock, GdkInputCondition cond )
{
	unsigned short packet_size;
	BYTE *packet;
	BYTE *buffer;

	int index;

	char message[256];

	xfer2a pak2a;
	xfer2b pak2b;

	GSList *contact;

#ifdef TRACE_FUNCTION
	g_print( "TCPFileReadServer\n" );
#endif

	contact = Contacts;

	while( contact != NULL )
	{
		if( ((CONTACT_PTR)contact->data)->file_sok == sock )
		break;
		contact = contact->next;
	}

	if( contact == NULL )
		return FALSE;

	if( ( read( sock, (char*)(&packet_size), 1 ) ) <= 0 )
	{
		if( ((CONTACT_PTR)contact->data)->file_gdk_input )
		{
			gdk_input_remove( ((CONTACT_PTR)contact->data)->file_gdk_input );
			((CONTACT_PTR)contact->data)->file_gdk_input = 0;
		}
		sprintf( message, _("File transfer terminated:\n%s"), ((CONTACT_PTR)contact->data)->nick );
		OK_Box( message, ((CONTACT_PTR)contact->data)->file_short_name );
		close( sock );

		((CONTACT_PTR)contact->data)->file_sok = 0;
		((CONTACT_PTR)contact->data)->file_port = 0;
		((CONTACT_PTR)contact->data)->file_seq = 0;
		g_free( ((CONTACT_PTR)contact->data)->file_name );
		g_free( ((CONTACT_PTR)contact->data)->file_short_name );
		((CONTACT_PTR)contact->data)->file_name = NULL;
		((CONTACT_PTR)contact->data)->file_short_name = NULL;
		if( ((CONTACT_PTR)contact->data)->file_fd > 0 )
			close( ((CONTACT_PTR)contact->data)->file_fd );
		((CONTACT_PTR)contact->data)->file_fd = 0;
		((CONTACT_PTR)contact->data)->file_offset = 0;
		((CONTACT_PTR)contact->data)->file_size = 0;

		if( ((CONTACT_PTR)contact->data)->file_info != NULL &&
		    ((CONTACT_PTR)contact->data)->file_info->window != NULL )
			gtk_widget_destroy( ((CONTACT_PTR)contact->data)->file_info->window );
		g_free( ((CONTACT_PTR)contact->data)->file_info );
		((CONTACT_PTR)contact->data)->file_info = NULL;

		return TRUE;
	}

	read( sock, (char*)(&packet_size) + 1, 1 );
	packet_size = GUINT16_FROM_LE (packet_size);

	packet = (BYTE *)g_malloc0( packet_size );
	read( sock, packet, packet_size );
	packet_print( packet, packet_size,
	              PACKET_TYPE_TCP | PACKET_DIRECTION_RECEIVE, "FILE(INIT(READ))" );

	switch( packet[ 0 ] )
	{
		case 0x01: /* We need to send XFER2 now */
			pak2a.id = 0x02;
			pak2a.zero = 0x00;
			Word_2_Chars( pak2a.fn_len, strlen( ((CONTACT_PTR)contact->data)->file_short_name ) + 1 );
			Word_2_Chars( pak2b.one, 0x0001 );
			pak2b.zero = 0;
			DW_2_Chars( pak2b.file_size, ((CONTACT_PTR)contact->data)->file_size );
			DW_2_Chars( pak2b.longzero, 0x00000000 );
			DW_2_Chars( pak2b.speed, 0x00000064 );
			
			packet_size = sizeof( xfer2a ) + sizeof( xfer2b ) + strlen( ((CONTACT_PTR)contact->data)->file_short_name ) + 1;
			index = 0;
			buffer = g_malloc0( packet_size + 2 );

			Word_2_Chars (buffer, packet_size);
			index += 2;
			memcpy( buffer + index, &pak2a, sizeof( xfer2a ) );
			index += sizeof( xfer2a );
			memcpy( buffer + index, ((CONTACT_PTR)contact->data)->file_short_name, strlen( ((CONTACT_PTR)contact->data)->file_short_name ) + 1 );
			index += strlen( ((CONTACT_PTR)contact->data)->file_short_name ) + 1;
			memcpy( buffer + index, &pak2b, sizeof( xfer2b ) );
			index += sizeof( xfer2b );

			packet_print( buffer+2, packet_size,
			              PACKET_TYPE_TCP | PACKET_DIRECTION_SEND, "FILE(INIT(2))" );
			write( ((CONTACT_PTR)contact->data)->file_sok, buffer, packet_size + 2 );
			g_free( buffer );
			break;

		case 0x03: /* Start sending the file :) */
		/* The best way to do this is to set up a timeout to send it so we don't
		   freeze the program, especially on large files */
			((CONTACT_PTR)contact->data)->file_offset = Chars_2_DW (&packet[1]);
			((CONTACT_PTR)contact->data)->file_fd = open( ((CONTACT_PTR)contact->data)->file_name, O_RDONLY,
			                               S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH );
			if( ((CONTACT_PTR)contact->data)->file_fd == -1 )
			{
				if( ((CONTACT_PTR)contact->data)->file_gdk_input )
				{
					gdk_input_remove( ((CONTACT_PTR)contact->data)->file_gdk_input );
					((CONTACT_PTR)contact->data)->file_gdk_input = 0;
				}
				sprintf( message, _("File transfer terminated:\n%s"), ((CONTACT_PTR)contact->data)->nick );
				OK_Box( message, ((CONTACT_PTR)contact->data)->file_short_name );
				close( sock );

				((CONTACT_PTR)contact->data)->file_sok = 0;
				((CONTACT_PTR)contact->data)->file_port = 0;
				((CONTACT_PTR)contact->data)->file_seq = 0;
				g_free( ((CONTACT_PTR)contact->data)->file_name );
				g_free( ((CONTACT_PTR)contact->data)->file_short_name );
				((CONTACT_PTR)contact->data)->file_name = NULL;
				((CONTACT_PTR)contact->data)->file_short_name = NULL;
				if( ((CONTACT_PTR)contact->data)->file_fd > 0 )
					close( ((CONTACT_PTR)contact->data)->file_fd );
				((CONTACT_PTR)contact->data)->file_fd = 0;
				((CONTACT_PTR)contact->data)->file_offset = 0;
				((CONTACT_PTR)contact->data)->file_size = 0;

				if( ((CONTACT_PTR)contact->data)->file_info != NULL &&
				    ((CONTACT_PTR)contact->data)->file_info->window != NULL )
					gtk_widget_destroy( ((CONTACT_PTR)contact->data)->file_info->window );
				g_free( ((CONTACT_PTR)contact->data)->file_info );
				((CONTACT_PTR)contact->data)->file_info = NULL;

				return TRUE;
			}

			g_free( packet );

			if( ((CONTACT_PTR)contact->data)->file_offset > 0 )
			{
				packet = (BYTE*)g_malloc0( ((CONTACT_PTR)contact->data)->file_offset );
				read( ((CONTACT_PTR)contact->data)->file_fd, packet, ((CONTACT_PTR)contact->data)->file_offset );
				g_free( packet );
			}

			((CONTACT_PTR)contact->data)->file_info = create_file_xfer_dialog(
			                           ((CONTACT_PTR)contact->data)->file_short_name,
			                           1, ((CONTACT_PTR)contact->data)->file_size,
			                           ((CONTACT_PTR)contact->data)->file_size );

			gtk_timeout_add( 5, (GtkFunction)TCPSendPartFile,
			                 GINT_TO_POINTER( ((CONTACT_PTR)contact->data)->uin ) );
			break;
	}

	return TRUE;
}

int TCPFileReadClient( GtkWidget *widget, int sock, GdkInputCondition cond )
{
	unsigned short packet_size;
	BYTE *packet;

	char message[256];
	int really_read;

	xfer1 pak1;
	BYTE *buf;
	int t;

	GSList *contact;

#ifdef TRACE_FUNCTION
	g_print( "TCPFileReadClient\n" );
#endif

	contact = Contacts;

	while( contact != NULL )
	{
		if( ((CONTACT_PTR)contact->data)->file_sok == sock )
		break;
		contact = contact->next;
	}

	if( contact == NULL )
		return FALSE;

	if( ( read( sock, (char*)(&packet_size), 1 ) ) <= 0 )
	{
		if( ((CONTACT_PTR)contact->data)->file_gdk_input )
		{
			gdk_input_remove( ((CONTACT_PTR)contact->data)->file_gdk_input );
			((CONTACT_PTR)contact->data)->file_gdk_input = 0;
		}
		sprintf( message, _("File transfer terminated:\n%s"), ((CONTACT_PTR)contact->data)->nick );
		OK_Box( message, ((CONTACT_PTR)contact->data)->file_short_name );
		close( sock );

		((CONTACT_PTR)contact->data)->file_sok = 0;
		((CONTACT_PTR)contact->data)->file_port = 0;
		((CONTACT_PTR)contact->data)->file_seq = 0;
		g_free( ((CONTACT_PTR)contact->data)->file_name );
		g_free( ((CONTACT_PTR)contact->data)->file_short_name );
		((CONTACT_PTR)contact->data)->file_name = NULL;
		((CONTACT_PTR)contact->data)->file_short_name = NULL;
		if( ((CONTACT_PTR)contact->data)->file_fd > 0 )
			close( ((CONTACT_PTR)contact->data)->file_fd );
		((CONTACT_PTR)contact->data)->file_fd = 0;
		((CONTACT_PTR)contact->data)->file_offset = 0;
		((CONTACT_PTR)contact->data)->file_size = 0;

		if( ((CONTACT_PTR)contact->data)->file_info != NULL &&
		    ((CONTACT_PTR)contact->data)->file_info->window != NULL )
			gtk_widget_destroy( ((CONTACT_PTR)contact->data)->file_info->window );
		g_free( ((CONTACT_PTR)contact->data)->file_info );
		((CONTACT_PTR)contact->data)->file_info = NULL;

		return TRUE;
	}

	read( sock, (char*)(&packet_size) + 1, 1 );
	packet_size = GUINT16_FROM_LE (packet_size);

	really_read = 0;
	packet = (BYTE *)g_malloc0( packet_size );

	really_read = read( sock, packet, packet_size );

	while( really_read != packet_size )
		really_read += read( sock, packet + really_read,
		packet_size - really_read );

	packet_print( packet, packet_size,
	              PACKET_TYPE_TCP | PACKET_DIRECTION_RECEIVE, "FILE PACKET" );

	switch( packet[ 0 ] )
	{
		case 0x00:
			if( ((CONTACT_PTR)contact->data)->file_info == NULL )
			{
				((CONTACT_PTR)contact->data)->file_info = create_file_xfer_dialog(
				                           ((CONTACT_PTR)contact->data)->file_short_name,
				                           Chars_2_DW( &packet[ 5 ] ),
				                           ((CONTACT_PTR)contact->data)->file_size,
				                           Chars_2_DW( &packet[ 9 ] ) );
				gtk_signal_connect( GTK_OBJECT( ((CONTACT_PTR)contact->data)->file_info->cancel ),
				                    "clicked", (GtkSignalFunc) cancel_transfer,
				                    GINT_TO_POINTER( ((CONTACT_PTR)contact->data)->uin ) );
			}
			pak1.id = 0x01;
			DW_2_Chars( pak1.speed, 0x00000064 );
			Word_2_Chars( pak1.nick_len, strlen( our_info->nick ) + 1 );
			buf = (BYTE*)g_malloc0( sizeof( xfer1 ) + strlen( our_info->nick ) + 1 );
			Word_2_Chars( pak1.psize, sizeof( xfer1 ) - 1 + strlen( our_info->nick ) );
			memcpy( buf, &pak1, sizeof( xfer1 ) );
			memcpy( &buf[ sizeof( xfer1 ) ], our_info->nick, strlen( our_info->nick ) + 1 );
			write( ((CONTACT_PTR)contact->data)->file_sok, buf, sizeof( xfer1 ) + strlen( our_info->nick ) + 1 );
			packet_print( &buf[2], sizeof( xfer1 ) + strlen( our_info->nick ) - 1,
			              PACKET_TYPE_TCP | PACKET_DIRECTION_SEND, "FILE(01)" );
			break;

		case 0x02:
			gtk_entry_set_text( GTK_ENTRY( ((CONTACT_PTR)contact->data)->file_info->current_file ),
			                    &packet[ 4 ] );
			sprintf( message, "%u", Chars_2_DW( &packet[ 8 + strlen( &packet[4] ) ] ) );
			gtk_entry_set_text( GTK_ENTRY( ((CONTACT_PTR)contact->data)->file_info->cf_size ),
			                    message );
			file_place( ((CONTACT_PTR)contact->data)->file_short_name, ((CONTACT_PTR)contact->data)->uin );
			break;
		case 0x06:
			if( ((CONTACT_PTR)contact->data)->file_fd <= 0 )
				break;
			write( ((CONTACT_PTR)contact->data)->file_fd, &packet[ 1 ], packet_size - 1 );

			((CONTACT_PTR)contact->data)->file_info->cf_bytes_sent += packet_size - 1;
			((CONTACT_PTR)contact->data)->file_info->b_bytes_sent += packet_size - 1;
			sprintf( message, "%d", ((CONTACT_PTR)contact->data)->file_info->cf_bytes_sent );
			gtk_entry_set_text( GTK_ENTRY( ((CONTACT_PTR)contact->data)->file_info->cf_trans ),
			                    message );
			sprintf( message, "%d", ((CONTACT_PTR)contact->data)->file_info->b_bytes_sent );
			gtk_entry_set_text( GTK_ENTRY( ((CONTACT_PTR)contact->data)->file_info->b_trans ),
			                    message );
			t = time( NULL ) - ((CONTACT_PTR)contact->data)->file_info->cf_starttime;
			sprintf( message, "%02d:%02d:%02d", t/3600, (t%3600)/60, t%60 );
			gtk_entry_set_text( GTK_ENTRY( ((CONTACT_PTR)contact->data)->file_info->cf_time ),
			                    message );
			t = time( NULL ) - ((CONTACT_PTR)contact->data)->file_info->b_starttime;
			sprintf( message, "%02d:%02d:%02d", t/3600, (t%3600)/60, t%60 );
			gtk_entry_set_text( GTK_ENTRY( ((CONTACT_PTR)contact->data)->file_info->b_time ),
			                    message );
			if( ((CONTACT_PTR)contact->data)->file_info->cf_bytes_sent && ((CONTACT_PTR)contact->data)->file_info->cf_starttime != time( NULL ) )
				sprintf( message, "%ld B/sec",
				         ((CONTACT_PTR)contact->data)->file_info->cf_bytes_sent /
				         ( time( NULL ) - ((CONTACT_PTR)contact->data)->file_info->cf_starttime ) );
			else
				strcpy( message, "0 B/sec" );
			gtk_entry_set_text( GTK_ENTRY( ((CONTACT_PTR)contact->data)->file_info->cf_bps ),
			                    message );
			if( ((CONTACT_PTR)contact->data)->file_info->b_bytes_sent && ((CONTACT_PTR)contact->data)->file_info->b_starttime != time( NULL ) )
				sprintf( message, "%ld B/sec",
				         ((CONTACT_PTR)contact->data)->file_info->b_bytes_sent /
				         ( time( NULL ) - ((CONTACT_PTR)contact->data)->file_info->b_starttime ) );
			else
				strcpy( message, "0 B/sec" );
			gtk_entry_set_text( GTK_ENTRY( ((CONTACT_PTR)contact->data)->file_info->b_bps ),
			                    message );
			if( ((CONTACT_PTR)contact->data)->file_info->cf_bytes_sent == 0 ||
			    ((CONTACT_PTR)contact->data)->file_info->cf_totbytes == 0 )
				gtk_progress_bar_update( GTK_PROGRESS_BAR( ((CONTACT_PTR)contact->data)->file_info->cf_progress ), 0.0F );
			else
				gtk_progress_bar_update( GTK_PROGRESS_BAR( ((CONTACT_PTR)contact->data)->file_info->cf_progress ),
				                         (float)( (float)((CONTACT_PTR)contact->data)->file_info->cf_bytes_sent / (float)((CONTACT_PTR)contact->data)->file_info->cf_totbytes ) );
			if( ((CONTACT_PTR)contact->data)->file_info->b_bytes_sent == 0 ||
			    ((CONTACT_PTR)contact->data)->file_info->b_totbytes == 0 )
				gtk_progress_bar_update( GTK_PROGRESS_BAR( ((CONTACT_PTR)contact->data)->file_info->b_progress ), 0.0F );
			else
				gtk_progress_bar_update( GTK_PROGRESS_BAR( ((CONTACT_PTR)contact->data)->file_info->b_progress ),
				                         (float)( (float)((CONTACT_PTR)contact->data)->file_info->b_bytes_sent / (float)((CONTACT_PTR)contact->data)->file_info->b_totbytes ) );

			if( packet_size < 2049 ) /* Last packet */
				close( ((CONTACT_PTR)contact->data)->file_fd );

			gdk_input_remove( ((CONTACT_PTR)contact->data)->file_gdk_input );
			while( gtk_events_pending() )
				gtk_main_iteration();
			((CONTACT_PTR)contact->data)->file_gdk_input = gdk_input_add( sock, GDK_INPUT_READ, (GdkInputFunction) TCPFileReadClient, NULL );

		break;
	}

	return TRUE;
}

int TCPSendPartFile( gpointer data )
{
	UIN_T uin;
	int bytes;
	xfer6 pak;
	char message[256];
	int t;
	GSList *contact;

	uin = GPOINTER_TO_INT( data );

	contact = Contacts;

	while( contact != NULL )
	{
		if( ((CONTACT_PTR)contact->data)->uin == uin )
			break;
		contact = contact->next;
	}

	if( contact == NULL )
		return FALSE;

	pak.id = 0x06;

	bytes = read( ((CONTACT_PTR)contact->data)->file_fd, pak.data, 2048 );

	Word_2_Chars( pak.psize, 1 + bytes );
	
	write( ((CONTACT_PTR)contact->data)->file_sok, &pak, 3 + bytes );
	packet_print( &pak.id, bytes + 1, PACKET_TYPE_TCP | PACKET_DIRECTION_SEND, "FILE PACKET" );

	if( bytes < 2048 )
	{
		if( ((CONTACT_PTR)contact->data)->file_gdk_input )
		{
			gdk_input_remove( ((CONTACT_PTR)contact->data)->file_gdk_input );
			((CONTACT_PTR)contact->data)->file_gdk_input = 0;
		}
		sprintf( message, _("File transfer complete:\n%s"), ((CONTACT_PTR)contact->data)->nick );
		OK_Box( message, ((CONTACT_PTR)contact->data)->file_short_name );
		close( ((CONTACT_PTR)contact->data)->file_sok );

		((CONTACT_PTR)contact->data)->file_sok = 0;
		((CONTACT_PTR)contact->data)->file_port = 0;
		((CONTACT_PTR)contact->data)->file_seq = 0;
		g_free( ((CONTACT_PTR)contact->data)->file_name );
		g_free( ((CONTACT_PTR)contact->data)->file_short_name );
		((CONTACT_PTR)contact->data)->file_name = NULL;
		((CONTACT_PTR)contact->data)->file_short_name = NULL;
		if( ((CONTACT_PTR)contact->data)->file_fd > 0 )
			close( ((CONTACT_PTR)contact->data)->file_fd );
		((CONTACT_PTR)contact->data)->file_fd = 0;
		((CONTACT_PTR)contact->data)->file_offset = 0;
		((CONTACT_PTR)contact->data)->file_size = 0;

		if( ((CONTACT_PTR)contact->data)->file_info != NULL &&
		    ((CONTACT_PTR)contact->data)->file_info->window != NULL )
			gtk_widget_destroy( ((CONTACT_PTR)contact->data)->file_info->window );
		g_free( ((CONTACT_PTR)contact->data)->file_info );
		((CONTACT_PTR)contact->data)->file_info = NULL;

		return FALSE;
	}

	((CONTACT_PTR)contact->data)->file_info->cf_bytes_sent += bytes;
	((CONTACT_PTR)contact->data)->file_info->b_bytes_sent += bytes;
	sprintf( message, "%d", ((CONTACT_PTR)contact->data)->file_info->cf_bytes_sent );
	gtk_entry_set_text( GTK_ENTRY( ((CONTACT_PTR)contact->data)->file_info->cf_trans ),
	                    message );
	sprintf( message, "%d", ((CONTACT_PTR)contact->data)->file_info->b_bytes_sent );
	gtk_entry_set_text( GTK_ENTRY( ((CONTACT_PTR)contact->data)->file_info->b_trans ),
	                    message );
	t = time( NULL ) - ((CONTACT_PTR)contact->data)->file_info->cf_starttime;
	sprintf( message, "%02d:%02d:%02d", t/3600, (t%3600)/60, t%60 );
	gtk_entry_set_text( GTK_ENTRY( ((CONTACT_PTR)contact->data)->file_info->cf_time ),
	                    message );
	t = time( NULL ) - ((CONTACT_PTR)contact->data)->file_info->b_starttime;
	sprintf( message, "%02d:%02d:%02d", t/3600, (t%3600)/60, t%60 );
	gtk_entry_set_text( GTK_ENTRY( ((CONTACT_PTR)contact->data)->file_info->b_time ),
	                    message );
	if( ((CONTACT_PTR)contact->data)->file_info->cf_bytes_sent && ((CONTACT_PTR)contact->data)->file_info->cf_starttime != time( NULL ) )
		sprintf( message, "%ld B/sec",
		         ((CONTACT_PTR)contact->data)->file_info->cf_bytes_sent /
		         ( time( NULL ) - ((CONTACT_PTR)contact->data)->file_info->cf_starttime ) );
	else
		strcpy( message, "0 B/sec" );
	gtk_entry_set_text( GTK_ENTRY( ((CONTACT_PTR)contact->data)->file_info->cf_bps ),
	                    message );
	if( ((CONTACT_PTR)contact->data)->file_info->b_bytes_sent && ((CONTACT_PTR)contact->data)->file_info->b_starttime != time( NULL ) )
		sprintf( message, "%ld B/sec",
		         ((CONTACT_PTR)contact->data)->file_info->b_bytes_sent /
		         ( time( NULL ) - ((CONTACT_PTR)contact->data)->file_info->b_starttime ) );
	else
		strcpy( message, "0 B/sec" );
	gtk_entry_set_text( GTK_ENTRY( ((CONTACT_PTR)contact->data)->file_info->b_bps ),
	                    message );
	if( ((CONTACT_PTR)contact->data)->file_info->cf_bytes_sent == 0 ||
	    ((CONTACT_PTR)contact->data)->file_info->cf_totbytes == 0 )
		gtk_progress_bar_update( GTK_PROGRESS_BAR( ((CONTACT_PTR)contact->data)->file_info->cf_progress ), 0.0F );
	else
		gtk_progress_bar_update( GTK_PROGRESS_BAR( ((CONTACT_PTR)contact->data)->file_info->cf_progress ),
		                         (float)( (float)((CONTACT_PTR)contact->data)->file_info->cf_bytes_sent / (float)((CONTACT_PTR)contact->data)->file_info->cf_totbytes ) );
	if( ((CONTACT_PTR)contact->data)->file_info->b_bytes_sent == 0 ||
	    ((CONTACT_PTR)contact->data)->file_info->b_totbytes == 0 )
		gtk_progress_bar_update( GTK_PROGRESS_BAR( ((CONTACT_PTR)contact->data)->file_info->b_progress ), 0.0F );
	else
		gtk_progress_bar_update( GTK_PROGRESS_BAR( ((CONTACT_PTR)contact->data)->file_info->b_progress ),
		                         (float)( (float)((CONTACT_PTR)contact->data)->file_info->b_bytes_sent / (float)((CONTACT_PTR)contact->data)->file_info->b_totbytes ) );
	return TRUE;
}

void file_place( char *prename, UIN_T uin )
{
	GtkWidget *filesel;
#ifdef TRACE_FUNCTION
	g_print( "file_place\n" );
#endif

	filesel = gtk_file_selection_new( _("GnomeICU: File Transfer") );

	gtk_file_selection_set_filename( GTK_FILE_SELECTION( filesel ), prename );
	gtk_signal_connect(GTK_OBJECT(filesel), "destroy", GTK_SIGNAL_FUNC(gtk_widget_destroy), NULL);
	gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(filesel)->ok_button),
	                   "clicked", GTK_SIGNAL_FUNC( save_incoming_file ), GINT_TO_POINTER( uin ) );
	gtk_signal_connect_object(GTK_OBJECT (GTK_FILE_SELECTION
	                   (filesel)->cancel_button), "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy),
	                   GTK_OBJECT( filesel ) );

	gtk_widget_show(filesel);
}

void save_incoming_file( GtkWidget *widget, gpointer data )
{
	UIN_T uin = GPOINTER_TO_INT( data );
	struct stat file_stat;
	xfer3 pak;

	GSList *contact = Contacts;
	
	while( contact != NULL )
	{
		if( ((CONTACT_PTR)contact->data)->uin == uin )
			break;
		contact = contact->next;
	}

	if( contact == NULL )
		return;

	((CONTACT_PTR)contact->data)->file_name = (char*)g_malloc0( strlen( gtk_file_selection_get_filename( GTK_FILE_SELECTION( widget->parent->parent->parent ) ) ) + 1 );
	strcpy( ((CONTACT_PTR)contact->data)->file_name, gtk_file_selection_get_filename( GTK_FILE_SELECTION( widget->parent->parent->parent ) ) );
	gtk_entry_set_text( GTK_ENTRY( ((CONTACT_PTR)contact->data)->file_info->local_filename ), ((CONTACT_PTR)contact->data)->file_name );
	gtk_widget_destroy( GTK_WIDGET( widget->parent->parent->parent ) );

	((CONTACT_PTR)contact->data)->file_info->cf_starttime = time( NULL );
	if( ((CONTACT_PTR)contact->data)->file_info->b_starttime == 0 )
		((CONTACT_PTR)contact->data)->file_info->b_starttime = time( NULL );
	((CONTACT_PTR)contact->data)->file_info->cf_bytes_sent = 0;

	((CONTACT_PTR)contact->data)->file_fd = open( ((CONTACT_PTR)contact->data)->file_name, O_WRONLY | O_CREAT | O_APPEND, S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP | S_IROTH | S_IWOTH );
	if( ((CONTACT_PTR)contact->data)->file_fd > 0 )
	{
		Word_2_Chars( pak.psize, 0x0d );
		pak.id = 0x03;
		if( stat( ((CONTACT_PTR)contact->data)->file_name, &file_stat ) == -1 )
			DW_2_Chars( pak.resume_loc, 0x00000000 );
		else
			DW_2_Chars( pak.resume_loc, file_stat.st_size );
		DW_2_Chars( pak.longzero, 0x00000000 );
		DW_2_Chars( pak.speed, 0x00000064 );
		write( ((CONTACT_PTR)contact->data)->file_sok, &pak, sizeof( pak ) );
	}

	return;
}

void cancel_transfer( GtkWidget *widget, gpointer data )
{
	UIN_T uin;

	GSList *contact;

	char message[256];

	uin = GPOINTER_TO_INT( data );

	contact = Contacts;
	
	while( contact != NULL )
	{
		if( ((CONTACT_PTR)contact->data)->uin == uin )
			break;
		contact = contact->next;
	}

	if( contact == NULL )
		return;
	
	if( ((CONTACT_PTR)contact->data)->file_gdk_input )
	{
		gdk_input_remove( ((CONTACT_PTR)contact->data)->file_gdk_input );
		((CONTACT_PTR)contact->data)->file_gdk_input = 0;
	}
	sprintf( message, _("File transfer terminated:\n%s"), ((CONTACT_PTR)contact->data)->nick );
	OK_Box( message, ((CONTACT_PTR)contact->data)->file_short_name );
	close( ((CONTACT_PTR)contact->data)->file_sok );

	((CONTACT_PTR)contact->data)->file_sok = 0;
	((CONTACT_PTR)contact->data)->file_port = 0;
	((CONTACT_PTR)contact->data)->file_seq = 0;
	g_free( ((CONTACT_PTR)contact->data)->file_name );
	g_free( ((CONTACT_PTR)contact->data)->file_short_name );
	((CONTACT_PTR)contact->data)->file_name = NULL;
	((CONTACT_PTR)contact->data)->file_short_name = NULL;
	if( ((CONTACT_PTR)contact->data)->file_fd > 0 )
		close( ((CONTACT_PTR)contact->data)->file_fd );
	((CONTACT_PTR)contact->data)->file_fd = 0;
	((CONTACT_PTR)contact->data)->file_offset = 0;
	((CONTACT_PTR)contact->data)->file_size = 0;

	if( ((CONTACT_PTR)contact->data)->file_info != NULL &&
	    ((CONTACT_PTR)contact->data)->file_info->window != NULL )
		gtk_widget_destroy( ((CONTACT_PTR)contact->data)->file_info->window );
	g_free( ((CONTACT_PTR)contact->data)->file_info );
	((CONTACT_PTR)contact->data)->file_info = NULL;

	return;
}

int TCPFinishConnection( GIOChannel *source, GIOCondition cond, gpointer data )
{
	GSList *contact;
	int sock;

#ifdef TRACE_FUNCTION
	g_print( "TCPFinishConnection(%d)\n", cond );
#endif

	if( cond & G_IO_ERR || cond & G_IO_HUP )
		return FALSE;

	contact = Contacts;

	while( contact != NULL )
	{
		if( ((CONTACT_PTR)contact->data)->gioc == source )
			break;
		contact = contact->next;
	}

	if( contact == NULL )
		return FALSE;

	sock = g_io_channel_unix_get_fd( source );

	TCPClearQueue( contact );

	((CONTACT_PTR)contact->data)->have_tcp_connection = TRUE;

	((CONTACT_PTR)contact->data)->giocw =
	g_io_add_watch( ((CONTACT_PTR)contact->data)->gioc,
	                G_IO_IN | G_IO_ERR | G_IO_HUP,
	                TCPReadPacket, NULL );

	return FALSE;
}

int TCPReadPacket( GIOChannel *source, GIOCondition cond, gpointer data )
{
	UIN_T uin;
	BYTE *packet;
	guint bytes_read;
	int sock;

	GSList *contact;

#ifdef TRACE_FUNCTION
	g_print( "TCPReadPacket(" );
	switch( cond )
	{
		case G_IO_IN:
			g_print( "G_IO_IN)\n" );
			break;
		case G_IO_HUP:
			g_print( "G_IO_HUP)\n" );
			break;
		case G_IO_ERR:
			g_print( "G_IO_ERR)\n" );
			break;
	default: 
			g_print( "UNKNOWN)\n" );
	}
#endif

	contact = Contacts;

	while( contact != NULL )
	{
		if( ((CONTACT_PTR)contact->data)->gioc == source )
			break;
		contact = contact->next;
	}

	if( contact == NULL )
		return FALSE;

	if( ((CONTACT_PTR)contact->data)->sok == 0 )
		return FALSE;

	if( cond != G_IO_IN )
		return FALSE; /* FIXME - This will be called if they close connection */

	sock = g_io_channel_unix_get_fd( source );

	if( ((CONTACT_PTR)contact->data)->tcp_buf_len == 0 )
	{
		g_io_channel_read( source, (gchar*)&((CONTACT_PTR)contact->data)->tcp_buf_len,
		                   2, &bytes_read );

		if( !bytes_read )
		{
			((CONTACT_PTR)contact->data)->have_tcp_connection = FALSE;
			((CONTACT_PTR)contact->data)->sok = 0;
			if( ((CONTACT_PTR)contact->data)->tcp_gdk_input )
			{
				gdk_input_remove( ((CONTACT_PTR)contact->data)->tcp_gdk_input );
				((CONTACT_PTR)contact->data)->tcp_gdk_input = 0;
			}
			close( sock );
			g_io_channel_close( source );
			((CONTACT_PTR)contact->data)->gioc = 0;
			return FALSE;
		}

		if( bytes_read == 1 )
		{
			bytes_read = 0;
			while( !bytes_read )
				g_io_channel_read( source, (gchar*)&((CONTACT_PTR)contact->data)->tcp_buf_len + 1,
				                   1, &bytes_read );
		}

		((CONTACT_PTR)contact->data)->tcp_buf_len =
		GINT16_FROM_LE
		  (((CONTACT_PTR)contact->data)->tcp_buf_len);
		((CONTACT_PTR)contact->data)->tcp_buf = (BYTE*)g_malloc0( ((CONTACT_PTR)contact->data)->tcp_buf_len );

		return TRUE;
	}

	packet = ((CONTACT_PTR)contact->data)->tcp_buf;

	if( packet == NULL )
		return TRUE;

	g_io_channel_read( source, packet + ((CONTACT_PTR)contact->data)->tcp_buf_read,
	                   ((CONTACT_PTR)contact->data)->tcp_buf_len - ((CONTACT_PTR)contact->data)->tcp_buf_read,
	                   &bytes_read );

	((CONTACT_PTR)contact->data)->tcp_buf_read += bytes_read;
	if( ((CONTACT_PTR)contact->data)->tcp_buf_read != ((CONTACT_PTR)contact->data)->tcp_buf_len )
		return FALSE;

	uin = Chars_2_DW (packet);

	contact = Contacts;

	while( contact != NULL )
	{
		if( ((CONTACT_PTR)contact->data)->uin == uin )
		{
			((CONTACT_PTR)contact->data)->sok = sock;
			break;
		}
		contact = contact->next;
	}

	TCPProcessPacket( packet, ((CONTACT_PTR)contact->data)->tcp_buf_len, sock );

	((CONTACT_PTR)contact->data)->tcp_buf_read = 0;
	((CONTACT_PTR)contact->data)->tcp_buf_len = 0;
	g_free( packet );
	((CONTACT_PTR)contact->data)->tcp_buf = NULL;

	return TRUE;
}

static int TCPTimeout( CONTACT_PTR contact )
{
	GtkWidget *dialog;
	GtkWidget *label;
	static int first = TRUE;

	if( first )
	{
		first = FALSE;
		return TRUE;
	}

	animate_off();

	if( contact->force == TRUE )
		return TRUE;

	label = gtk_label_new( "Unable to send message directly." );
	dialog = gnome_dialog_new( "GnomeICU: Question", "Retry", "Send Through Server", "Cancel", NULL );
	gtk_box_pack_start( GTK_BOX( GNOME_DIALOG( dialog )->vbox ), label,
	                    FALSE, FALSE, GNOME_PAD );

	gnome_dialog_set_close( GNOME_DIALOG( dialog ), TRUE );

	gnome_dialog_button_connect( GNOME_DIALOG(dialog), 0, timeout_continue, contact );
	gnome_dialog_button_connect( GNOME_DIALOG(dialog), 1, timeout_server, contact );
	gnome_dialog_button_connect( GNOME_DIALOG(dialog), 2, timeout_cancel, contact );
	gtk_widget_show( label );
	gtk_widget_show( dialog );

	return FALSE;
}

void timeout_continue( GtkWidget *widget, CONTACT_PTR contact )
{
	GSList *contact2;

	contact2 = Contacts;

	/* We do this in case the user was removed before the message
	   was sent.. */
	while( contact2 != NULL )
	{
		if( contact == (CONTACT_PTR)contact2->data )
			break;
		contact2 = contact2->next;
	}

	if( contact2 == NULL )
		return;

	contact->tcp_timeout = gtk_timeout_add( 10000, (GtkFunction)TCPTimeout, contact );
	animate_on();
}

void timeout_server( GtkWidget *widget, CONTACT_PTR contact )
{
	GSList *contact2;

	contact2 = Contacts;

	/* We do this in case the user was removed before the message
	   was sent.. */
	while( contact2 != NULL )
	{
		if( contact == (CONTACT_PTR)contact2->data )
			break;
		contact2 = contact2->next;
	}

	if( contact2 == NULL )
		return;

	if( contact->tcp_text_queue->data != NULL )
		icq_sendmsg( contact->uin, (char*)contact->tcp_text_queue->data, TRUE );

	if( contact->tcp_text_queue != NULL )
	{
		g_slist_foreach( contact->tcp_text_queue, (GFunc)g_free, NULL );
		g_slist_free( contact->tcp_text_queue );
		contact->tcp_text_queue = NULL;
	}

	if( contact->tcp_msg_queue != NULL )
	{
		g_slist_foreach( contact->tcp_msg_queue, (GFunc)g_free, NULL );
		g_slist_free( contact->tcp_msg_queue );
		contact->tcp_msg_queue = NULL;
	}

	contact->force = TRUE;
}

void timeout_cancel( GtkWidget *widget, CONTACT_PTR contact )
{
	GSList *contact2;

	contact2 = Contacts;

	/* We do this in case the user was removed before the message
	   was sent.. */
	while( contact2 != NULL )
	{
		if( contact == (CONTACT_PTR)contact2->data )
			break;
		contact2 = contact2->next;
	}

	if( contact2 == NULL )
		return;

	if( contact->tcp_text_queue != NULL )
	{
		g_slist_foreach( contact->tcp_text_queue, (GFunc)g_free, NULL );
		g_slist_free( contact->tcp_text_queue );
	}

	if( contact->tcp_msg_queue != NULL )
	{
		g_slist_foreach( contact->tcp_msg_queue, (GFunc)g_free, NULL );
		g_slist_free( contact->tcp_msg_queue );
	}
	contact->tcp_text_queue = contact->tcp_msg_queue = NULL;
}
