/* -*- Mode: C; tab-width: 4; indent-tabs-mode: t; c-basic-offset: 4 -*- */
/*
 * application/ps Bonobo bonobo_object.
 *
 * based on text/plain bonobo_object from Nat Friedman  <nat@gnome-support.com>
 *
 * Author:
 *   Jaka Mocnik  <jaka.mocnik@kiss.uni-lj.si>
 */

#include <config.h>
#include <gnome.h>
#include <libgnorba/gnorba.h>
#include <bonobo.h>

#include "gtkgs.h"
#include "prefs.h"

static GdkCursor *pan_cursor = NULL;

CORBA_Environment ev;
CORBA_ORB orb;

/*
 * Embeddable data
 */
typedef struct {
	BonoboEmbeddable *embeddable;

	gchar *tmp_name;
	gboolean loaded;

	GList *views;
} embeddable_data_t;

/*
 * View data
 */
typedef struct {
	embeddable_data_t *embeddable_data;

	GtkWidget        *gs;
	GtkObject        *hadj, *vadj;
	BonoboView *view;

	gint magstep;
	gboolean pan;
	gdouble prev_x, prev_y;
} view_data_t;


static void
view_system_exception_cb(BonoboView *view, CORBA_Object corba_object,
						 CORBA_Environment *ev, gpointer data)
{
	bonobo_object_destroy(BONOBO_OBJECT(view));
}

static void
set_page_item_sensitivity (view_data_t *view_data)
{
	BonoboUIHandler *uih = bonobo_view_get_ui_handler(view_data->view);
	GtkGS *gs = GTK_GS(view_data->gs);

	bonobo_ui_handler_menu_set_sensitivity(uih, "/Document/Next page",
										   gs->current_page < gs->doc->numpages - 1);
	bonobo_ui_handler_menu_set_sensitivity(uih, "/Document/Previous page",
										   gs->current_page > 0);
}

static void
view_next_page_cb  (BonoboUIHandler *uih, void *user_data, char *path)
{
	view_data_t *view_data = (view_data_t *)user_data;
	GtkGS *gs = GTK_GS(view_data->gs);

	if(view_data->embeddable_data->loaded)
        gtk_gs_goto_page(gs, gs->current_page + 1);

	set_page_item_sensitivity(view_data);
}

static void
view_prev_page_cb  (BonoboUIHandler *uih, void *user_data, char *path)
{
	view_data_t *view_data = (view_data_t *)user_data;
	GtkGS *gs = GTK_GS(view_data->gs);

	if(view_data->embeddable_data->loaded)
        gtk_gs_goto_page(gs, gs->current_page - 1);

	set_page_item_sensitivity(view_data);
}

static void
view_recenter_page_cb  (BonoboUIHandler *uih, void *user_data, char *path)
{
	view_data_t *view_data = (view_data_t *)user_data;
	GtkGS *gs = GTK_GS(view_data->gs);

	if(view_data->embeddable_data->loaded)
        gtk_gs_center_page(gs);
}


static void
view_merge_menus(view_data_t *view_data)
{
	BonoboUIHandler *uih = bonobo_view_get_ui_handler(view_data->view);
	Bonobo_UIHandler remote_uih;

	remote_uih = bonobo_view_get_remote_ui_handler(view_data->view);

	if(remote_uih == CORBA_OBJECT_NIL)
		return;

	bonobo_ui_handler_set_container(uih, remote_uih);

	bonobo_ui_handler_menu_new_subtree(uih, "/Document",
									   N_("Document"),
									   N_("Navigate the document"),
									   1,
									   BONOBO_UI_HANDLER_PIXMAP_NONE, NULL,
									   0, (GdkModifierType)0);

	bonobo_ui_handler_menu_new_item(uih, "/Document/Next page",
									N_("_Next Page"), N_("Show next page"),
									-1, BONOBO_UI_HANDLER_PIXMAP_NONE, NULL,
									0, (GdkModifierType)0,
									view_next_page_cb, (gpointer)view_data);
	bonobo_ui_handler_menu_new_item(uih, "/Document/Previous page",
									N_("_Previous Page"), N_("Show previous page"),
									-1, BONOBO_UI_HANDLER_PIXMAP_NONE, NULL,
									0, (GdkModifierType)0,
									view_prev_page_cb, (gpointer)view_data);
	bonobo_ui_handler_menu_new_item(uih, "/Document/Recenter page",
									N_("_Recenter page"), N_("Recenter page"),
									-1, BONOBO_UI_HANDLER_PIXMAP_NONE, NULL,
									0, (GdkModifierType)0,
									view_recenter_page_cb, (gpointer)view_data);
	set_page_item_sensitivity(view_data);
}

static void
view_remove_menus(view_data_t *view_data)
{
	BonoboUIHandler *uih = bonobo_view_get_ui_handler(view_data->view);

	bonobo_ui_handler_unset_container(uih);
}


static void
view_activate_cb(BonoboView *view, gboolean activate, view_data_t *view_data)
{
	bonobo_view_activate_notify(view, activate);

	/* merge menus */
	if(activate)
		view_merge_menus(view_data);
	else
		view_remove_menus(view_data);
}

static gboolean
view_destroy_cb(BonoboView *view, view_data_t *view_data)
{
	g_message("Destroying application-ps view\n");

	view_data->embeddable_data->views = g_list_remove(view_data->embeddable_data->views, view_data);

	gtk_object_unref(GTK_OBJECT(view_data->gs));

	gtk_object_destroy(view_data->hadj);
	gtk_object_destroy(view_data->vadj);

	g_free(view_data);

	return FALSE;
}

static void
reload_all_views(embeddable_data_t *embeddable_data)
{
	GList *l;
	GtkRequisition req;

	for(l = embeddable_data->views; l; l = l->next)
	{
		view_data_t *view_data = l->data;
		GtkGS *gs = GTK_GS(view_data->gs);

		if(embeddable_data->tmp_name == NULL ||
		   !gtk_gs_load(gs, embeddable_data->tmp_name))
			continue;

		gtk_gs_set_pagemedia(gs, -1, 0);

		if(GTK_WIDGET_REALIZED(gs))
			gtk_gs_goto_page(gs, gs->current_page);

		gtk_widget_size_request(GTK_WIDGET(gs), &req);
	}
}

static void
stream_read(Bonobo_Stream stream, embeddable_data_t *embeddable_data)
{
	Bonobo_Stream_iobuf *buffer;
	CORBA_Environment ev;
	CORBA_long bytes_read, bytes_written;

	if(embeddable_data->tmp_name == NULL) {
		embeddable_data->tmp_name = g_malloc(256);
		tmpnam(embeddable_data->tmp_name);
	}

	buffer = Bonobo_Stream_iobuf__alloc();

	CORBA_exception_init(&ev);

	Bonobo_Stream_copy_to(stream, embeddable_data->tmp_name,
						  -1, &bytes_read, &bytes_written, &ev);

	embeddable_data->loaded = TRUE;

	CORBA_exception_free(&ev);
}

/*
 * Loads text from a GNOME_Stream
 */
static int
load_ps_from_stream(BonoboPersistStream *ps, Bonobo_Stream stream,
					void *data)
{
	embeddable_data_t *embeddable_data = data;

	stream_read(stream, embeddable_data);

	reload_all_views(embeddable_data);

	return 0;
}

static void
view_button_press_cb(GtkWidget *widget, GdkEventButton *event, gpointer data)
{
        view_data_t *view_data = (view_data_t *)data;

        if(event->button == 1 && !view_data->pan) {
			gint wx = 0, wy = 0;
			
			gdk_window_get_pointer(widget->window, &wx, &wy, NULL);
			
			view_data->pan = TRUE;
			if(pan_cursor == NULL)
				pan_cursor = gdk_cursor_new(GDK_FLEUR);
			
			gtk_grab_add(widget);
			gdk_pointer_grab(widget->window, FALSE,
							 GDK_POINTER_MOTION_MASK |
							 GDK_BUTTON_RELEASE_MASK, NULL,
							 pan_cursor, GDK_CURRENT_TIME);
			view_data->prev_x = wx;
			view_data->prev_y = wy;
		}
		else if(event->button == 3)
			bonobo_view_popup_verbs(view_data->view);
}

static void
view_button_release_cb(GtkWidget *widget, GdkEventButton *event, gpointer data)
{
        view_data_t *v = (view_data_t *)data;

        if(event->button == 1 && v->pan) {
			v->pan = FALSE;
			gdk_pointer_ungrab(GDK_CURRENT_TIME);
			gtk_grab_remove(widget);
		}
}

static void
view_motion_cb(GtkWidget *widget, GdkEventMotion *event, gpointer data)
{
        view_data_t *v = (view_data_t *) data;

        if(v->pan) {
                gtk_gs_scroll(GTK_GS(v->gs), -event->x + v->prev_x, -event->y + v->prev_y);;
                v->prev_x = event->x;
                v->prev_y = event->y;
        }
}

static void
view_realize_cb(GtkWidget *w, view_data_t *view_data)
{
	GtkGS *gs = GTK_GS(w);

	if(gs->gs_filename) {
		GtkRequisition req;

		gtk_gs_set_pagemedia(gs, -1, 0);

		gtk_gs_goto_page(gs, gs->current_page);

		gtk_widget_size_request(GTK_WIDGET(gs), &req);
	}
}

static void
verb_next_page(BonoboView *view, const char *verb, view_data_t *data)
{
	GtkGS *gs = GTK_GS(data->gs);

	if(data->embeddable_data->loaded)
		gtk_gs_goto_page(gs, gs->current_page + 1);
}

static void
verb_prev_page(BonoboView *view, const char *verb, view_data_t *data)
{
	GtkGS *gs = GTK_GS(data->gs);

	if(data->embeddable_data->loaded)
		gtk_gs_goto_page(gs, gs->current_page - 1);
}

static const GnomeVerb verbs[] = {
	{ "next_page", "_Next Page", "Turn to next page" },
	{ "prev_page", "_Previous page", "Turn to previous page" },
    { NULL }
};

static BonoboView *
view_factory(BonoboEmbeddable *embeddable, const Bonobo_ViewFrame view_frame,
			 void *data)
{
	BonoboView *view;
	BonoboUIHandler *uih;
	embeddable_data_t *embeddable_data = data;
	view_data_t *view_data = g_new0 (view_data_t, 1);
	GtkGS *gs;

	view_data->embeddable_data = embeddable_data;
	view_data->hadj = gtk_adjustment_new(0.1, 0.0, 1.0, 1.0, 1.0, 0.5);
	view_data->vadj = gtk_adjustment_new(0.1, 0.0, 1.0, 1.0, 1.0, 0.5);
	view_data->pan = FALSE;
	view_data->magstep = 0; /* 1:1 */

	if(embeddable_data->tmp_name)
		view_data->gs = gtk_gs_new_from_file(GTK_ADJUSTMENT(view_data->hadj),
											 GTK_ADJUSTMENT(view_data->vadj),
											 embeddable_data->tmp_name);
	else
		view_data->gs = gtk_gs_new(GTK_ADJUSTMENT(view_data->hadj),
								   GTK_ADJUSTMENT(view_data->vadj));
	gs = GTK_GS(view_data->gs);

	gs->override_media = gs_override_media;
	gs->default_page_media = gs_def_media;
	gs->antialiased = gs_antialiased;

	gtk_widget_set_events(GTK_WIDGET(gs), 
						  GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK |
						  GDK_POINTER_MOTION_MASK);
	gtk_signal_connect(GTK_OBJECT(gs), "button_press_event",
					   GTK_SIGNAL_FUNC(view_button_press_cb), view_data);
	gtk_signal_connect(GTK_OBJECT(gs), "button_release_event",
					   GTK_SIGNAL_FUNC(view_button_release_cb), view_data);
	gtk_signal_connect(GTK_OBJECT(gs), "motion_notify_event",
					   GTK_SIGNAL_FUNC(view_motion_cb), view_data);
	gtk_signal_connect(GTK_OBJECT(gs), "realize",
					   GTK_SIGNAL_FUNC(view_realize_cb), view_data);

	gtk_widget_show(view_data->gs);

	view = bonobo_view_new(view_data->gs);


	bonobo_view_register_verb(view, "next_page",
							  BONOBO_VIEW_VERB_FUNC(verb_next_page),
							  view_data);
	bonobo_view_register_verb(view, "prev_page",
							  BONOBO_VIEW_VERB_FUNC(verb_prev_page),
							  view_data);

	gtk_signal_connect(GTK_OBJECT(view), "activate",
					   GTK_SIGNAL_FUNC(view_activate_cb), view_data);
	gtk_signal_connect(GTK_OBJECT(view), "system_exception",
					   GTK_SIGNAL_FUNC(view_system_exception_cb), view_data);
	gtk_signal_connect(GTK_OBJECT(view), "destroy",
					   GTK_SIGNAL_FUNC(view_destroy_cb), view_data);

	view_data->view = view;

	embeddable_data->views = g_list_prepend(embeddable_data->views,
											view_data);

	return view;
}

static void
embeddable_system_exception_cb(BonoboEmbeddable *embeddable, CORBA_Object corba_object,
							   CORBA_Environment *ev, gpointer data)
{
	bonobo_object_destroy(BONOBO_OBJECT(embeddable));
}

static gboolean
embeddable_destroy_cb(BonoboEmbeddable *embeddable, embeddable_data_t *data)
{
	g_message("Destroying application-ps embeddable\n");

	if(data->tmp_name) {
		unlink(data->tmp_name);
		g_free(data->tmp_name);
	}

	g_free(data);

	return FALSE;
}

static BonoboObject *
embeddable_factory(BonoboEmbeddableFactory *this, void *data)
{
	BonoboEmbeddable *embeddable;
	BonoboPersistStream *stream;
	embeddable_data_t *embeddable_data = data;

	embeddable_data = g_new0(embeddable_data_t, 1);
	if(!embeddable_data)
		return NULL;

	embeddable_data->tmp_name = NULL;
	embeddable_data->loaded = FALSE;

	/*
	 * Creates the Embeddable server
	 */
	embeddable = bonobo_embeddable_new(view_factory, embeddable_data);
	if(embeddable == NULL){
		g_free(embeddable_data);
		return NULL;
	}

	/*
	 * Interface BONOBO::PersistStream 
	 */
	stream = bonobo_persist_stream_new(load_ps_from_stream, NULL,
					  embeddable_data);

	if(stream == NULL) {
		gtk_object_unref(GTK_OBJECT(embeddable));
		g_free(embeddable_data);
		return NULL;
	}

	embeddable_data->embeddable = embeddable;

	/*
	 * Add the verbs
	 */
	bonobo_embeddable_add_verbs(embeddable, verbs);

	/*
	 * attach our embeddable_data to the Embeddable
	 */
	gtk_object_set_data(GTK_OBJECT(embeddable), "embeddable-data",
						embeddable_data);
	
	/*
	 * Bind the interfaces
	 */
	bonobo_object_add_interface(BONOBO_OBJECT(embeddable),
							   BONOBO_OBJECT(stream));

	gtk_signal_connect(GTK_OBJECT(embeddable), "system_exception",
					   GTK_SIGNAL_FUNC(embeddable_system_exception_cb),
					   embeddable_data);
	gtk_signal_connect_after(GTK_OBJECT(embeddable), "destroy",
							 GTK_SIGNAL_FUNC(embeddable_destroy_cb),
							 embeddable_data);

	return (BonoboObject *)embeddable;
}

static void
init_embeddable_application_ps_factory(void)
{
	BonoboEmbeddableFactory *factory;
	
	factory = bonobo_embeddable_factory_new("embeddable-factory:application-ps",
											embeddable_factory, NULL);
}

static void
init_server_factory(int argc, char **argv)
{
	gnome_CORBA_init_with_popt_table(
		"embeddable-application-ps", VERSION,
		&argc, argv, NULL, 0, NULL, GNORBA_INIT_SERVER_FUNC, &ev);

	orb = gnome_CORBA_ORB();
	if(bonobo_init(orb, NULL, NULL) == FALSE)
		g_error(_("I could not initialize Bonobo"));
}

int
main(int argc, char **argv)
{
	CORBA_exception_init(&ev);

	init_server_factory(argc, argv);
	init_embeddable_application_ps_factory();

	load_prefs("/bonobo-application-ps/");

	bonobo_main();

	CORBA_exception_free(&ev);

	return 0;
}
