/*****************************************************************************/
/*  misc.c - general purpose routines                                        */
/*  Copyright (C) 1998-1999 Brian Masney <masneyb@seul.org>                  */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either version 2 of the License, or        */
/*  (at your option) any later version.                                      */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA      */
/*****************************************************************************/

#include "ftp.h"

char *insert_commas (unsigned long number) {
   char *frompos, *topos, *src, *dest;
   unsigned long tempnum;
   int len, num, rem, i;
   
   len = 0;
   tempnum = number;
   while (tempnum > 0) {
      tempnum /= 10;
      len++;
   }
   if (len % 3 > 0) len++;
   len += len / 3;
   if (len <= 0) return (NULL);
   
   src = g_strdup_printf ("%ld", number);
   dest = g_malloc (len);

   num = strlen (src) / 3 - 1;
   rem = strlen (src) % 3;
   frompos = src;
   topos = dest;
   for (i=0; i<rem; i++) *topos++ = *frompos++;
   if (*frompos != '\0') {
      if (rem != 0) *topos++ = ',';
      while (num > 0) {
         for (i=0; i<3; i++) *topos++ = *frompos++;
         *topos++ = ',';
         num--;
      }
      for (i=0; i<3; i++) *topos++ = *frompos++;
   }
   *topos = '\0';
   g_free (src);
   return (dest);
}
/*****************************************************************************/
void add_local_files (struct ftp_window_data *wdata) {
   char tempstr[MAXSTR];

   wdata->local = 1;
   wdata->cached = 0;
   wdata->hdata->totalfiles = wdata->numselected = 0;
   wdata->sorted = 0;
   gftp_set_directory (wdata->hdata->ftpdata, getcwd (tempstr, sizeof (tempstr)));
   if ((wdata->hdata->files = get_local_files (NULL, &wdata->hdata->totalfiles)) == NULL) {
      wdata->local = -1;
      wdata->hdata->totalfiles = wdata->numselected = 0;
   }
   else sortrows (GTK_CLIST(wdata->listbox), wdata->sortcol, (gpointer) wdata);
   gtk_clist_select_row (GTK_CLIST (wdata->listbox), 0, 0);
   update_ftp_info (wdata);
}
/*****************************************************************************/
GList *get_local_files (char *path, int *total) {
   char curdir[MAXSTR];
   GList *files;
   FILE *dir;
   
   *total = 0;
   if (path != NULL) {
      if (getcwd (curdir, sizeof (curdir)) == NULL) return (NULL);
      if (chdir(path) == -1) return (NULL);
   }
   dir = popen ("/bin/ls -al", "r");
   files = parse_local_file (dir, 1, total);
   pclose (dir);
   if (path != NULL) chdir (curdir);
   return (files);
}
/*****************************************************************************/
GList *parse_local_file (FILE *fd, int local, int *total) {
   struct ftp_file_data *newfle;
   struct stat st;
   gftp_file fle;
   GList *files;

   files = NULL;
   while (gftp_get_next_file (window1.hdata->ftpdata, &fle, fd)) {
      if (fle.file == NULL) break;
      if (strcmp (fle.file, ".") == 0 || strcmp (fle.file, "..") == 0) continue;

      newfle = g_malloc0 (sizeof (struct ftp_file_data));
      memcpy (newfle, &fle, sizeof (gftp_file));
      if (local) {
         stat (newfle->file, &st);
         if (S_ISDIR (st.st_mode)) newfle->isdir = 1;
         if (newfle->attribs[0] == 'l') newfle->islink = 1;
         if ((strchr (newfle->attribs, 'x') != NULL) && !newfle->isdir
         	&& !newfle->islink) newfle->isexe = 1;
      }
      else {
         if (strchr (newfle->attribs, 'd') != NULL) newfle->isdir = 1;
         if (strchr (newfle->attribs, 'l') != NULL) newfle->islink = 1;
         if ((strchr (newfle->attribs, 'x') != NULL) && !newfle->isdir
         	&& !newfle->islink) newfle->isexe = 1;
      }
      files = g_list_append (files, newfle);
      (*total)++;
   }
   
   newfle = g_malloc0 (sizeof (struct ftp_file_data));
   newfle->file = g_malloc (3);
   strcpy (newfle->file, "..");
   newfle->user = g_malloc (1);
   *newfle->user = '\0';
   newfle->group = g_malloc (1);
   *newfle->group = '\0';
   newfle->attribs = g_malloc (1);
   *newfle->attribs = '\0';
   newfle->isdir = 1;
      
   files = g_list_prepend (files, newfle);
   (*total)++;
   return (files);
}
/*****************************************************************************/
void add_file_listbox (struct ftp_window_data *wdata, struct ftp_file_data *fle) {
   char *add_data[7] = {NULL, NULL, NULL, NULL, NULL, NULL, NULL};
   struct pix_ext *tempext;
   char *tempstr, *str;
   int clist_num;
   size_t stlen;

   if (!(!show_hidden_files && *fle->file == '.' && strcmp (fle->file, "..") != 0) &&
   	gtk_pattern_match_simple (wdata->filespec, fle->file)) {
      fle->shown = 1;
   }
   else {
      fle->shown = 0;
      fle->selected = 0;
      return;
   }
   stlen = strlen (fle->file);
   clist_num = gtk_clist_append (GTK_CLIST (wdata->listbox), add_data);
   if (fle->selected) {
      gtk_clist_select_row (GTK_CLIST (wdata->listbox), clist_num, 0);
      wdata->numselected++;
   }
   if (strcmp(fle->file, "..") == 0) {
      gtk_clist_set_pixmap (GTK_CLIST (wdata->listbox), clist_num, 0, dotdot_pixmap, dotdot_mask);
   }
   else if (fle->islink && fle->isdir) {
      gtk_clist_set_pixmap (GTK_CLIST (wdata->listbox), clist_num, 0, linkdir_pixmap, linkdir_mask);
   }
   else if (fle->islink) {
      gtk_clist_set_pixmap (GTK_CLIST (wdata->listbox), clist_num, 0, linkfile_pixmap, linkfile_mask);
   }
   else if (fle->isdir) {
      gtk_clist_set_pixmap (GTK_CLIST (wdata->listbox), clist_num, 0, dir_pixmap, dir_mask);
   }
   else if (fle->isexe) {
      gtk_clist_set_pixmap (GTK_CLIST (wdata->listbox), clist_num, 0, exe_pixmap, exe_mask);
   }
   else {
      tempext = registered_exts;
      while (tempext != NULL) {
         if (stlen >= tempext->stlen &&
            strcmp (&fle->file[stlen-tempext->stlen], tempext->ext) == 0) {

            if (toupper(*tempext->ascii_binary) == 'A') fle->ascii = 1;
            
            if (tempext->pixmap != NULL && *tempext->filename != '\0') {
               gtk_clist_set_pixmap (GTK_CLIST (wdata->listbox), clist_num, 0, tempext->pixmap, tempext->mask);
            }
            else tempext = NULL;
            break;
         }
         tempext = tempext->next;
      }
      if (tempext == NULL) {
         gtk_clist_set_pixmap (GTK_CLIST (wdata->listbox), clist_num, 0, doc_pixmap, doc_mask);
      }
   }
   gtk_clist_set_text (GTK_CLIST (wdata->listbox), clist_num, 1, fle->file);

   if (fle->attribs && (*fle->attribs == 'b' || *fle->attribs == 'c')) {
      tempstr = g_strdup_printf ("%d, %d", (int) fle->size >> 16, (int) fle->size & 0xFF);
   }
   else {
      tempstr = insert_commas (fle->size);
   }
   gtk_clist_set_text (GTK_CLIST (wdata->listbox), clist_num, 2, tempstr);
   g_free (tempstr);

   if (fle->user) {
      gtk_clist_set_text (GTK_CLIST (wdata->listbox), clist_num, 3, fle->user);
   }
   if (fle->group) {
      gtk_clist_set_text (GTK_CLIST (wdata->listbox), clist_num, 4, fle->group);
   }
   if ((str = ctime (&fle->datetime))) {
      gtk_clist_set_text (GTK_CLIST (wdata->listbox), clist_num, 5, str);
   }
   if (fle->attribs) {
      gtk_clist_set_text (GTK_CLIST (wdata->listbox), clist_num, 6, fle->attribs);
   }
}
/*****************************************************************************/
long file_countlf (int filefd, long endpos) {
   char tempstr[MAXSTR];
   long num = 0, mypos;
   ssize_t n;
   int i;
            
   lseek (filefd, 0, SEEK_SET);
   mypos = 0;
   while ((n = read (filefd, tempstr, sizeof (tempstr))) > 0) {
      for (i=0; i<n; i++) {
         if ((tempstr[i] == '\n') && (i > 0) && (tempstr[i-1] != '\r')
            && (endpos == 0 || mypos+i <= endpos)) ++num;
      }
      mypos += n;
   }
   lseek (filefd, 0, SEEK_SET);
   return (num);
}
/*****************************************************************************/
char *alltrim (char *str) {
   char *pos, *newpos;
   int diff;

   pos = str + strlen (str);
   while (*pos == ' ') *pos-- = '\0';
   
   pos = str;
   diff = 0;
   while (*pos++ == ' ') diff++;

   pos = str + diff;
   newpos = str;
   while (*pos != '\0') *newpos++ = *pos++;
   *newpos = '\0';
   return (str);
}
/*****************************************************************************/
char *expand_path (char *src) {
   char *str, *pos, *endpos, *prevpos, *newstr, *tempstr, tempchar;
   struct passwd *pw;
   
   pw = NULL;
   str = g_malloc (strlen (src) + 1);
   strcpy (str, src);

   if (*str == '~') {
      if (*(str + 1) == '/' || (pos = strchr (str, '/')) == NULL) {
	 pw = getpwuid (geteuid ());
      }
      else {
         *pos = '\0';
	 pw = getpwnam (str+1);
	 *pos = '/';
      }
   }   

   endpos = str;
   newstr = NULL;
   while ((pos = strchr (endpos, '/')) != NULL) {
      pos++;
      if ((endpos = strchr (pos, '/')) == NULL) endpos = strchr (pos, '\0');
      tempchar = *endpos;
      *endpos = '\0';
      if (strcmp (pos, "..") == 0) {
         *(pos-1) = '\0';
         if (newstr != NULL && (prevpos = strrchr (newstr, '/')) != NULL) *prevpos = '\0';
      }
      else if (strcmp (pos, ".") != 0) {
         if (newstr == NULL) {
            newstr = g_malloc (strlen (pos - 1) + 1);
            strcpy (newstr, pos - 1);
         }
         else {
            tempstr = g_strconcat (newstr, pos - 1, NULL);
            g_free (newstr);
            newstr = tempstr;
         }
      }
      *endpos = tempchar;
      if (*endpos == '\0') break;
      endpos = pos + 1;
   }

   if (newstr == NULL || *newstr == '\0') {
      if (newstr != NULL) g_free (newstr);
      newstr = g_malloc (2);
      *newstr = '/'; 
      *(newstr + 1) = '\0';
   }

   if (newstr != NULL) {
      g_free (str);
      str = newstr;
   }

   if (pw != NULL) {
      if ((pos = strchr (str, '/')) == NULL) {
         newstr = g_malloc (strlen (pw->pw_dir) + 1);
         strcpy (newstr, pw->pw_dir);
      }
      else {
         newstr = g_strconcat (pw->pw_dir, pos, NULL);
      }
      return (newstr);
   }
   else {
      newstr = g_malloc (strlen (str) + 1);
      strcpy (newstr, str);
   }
   g_free (str);
   return (newstr);
}
/*****************************************************************************/
void remove_double_slashes (char *string) {
   char *newpos, *oldpos;
   
   oldpos = newpos = string;
   while (*oldpos != '\0') {
      *newpos++ = *oldpos++;
      if (*oldpos == '\0') break;
      while (*(newpos-1) == '/' && *(oldpos) == '/') oldpos++;
   }
   *newpos = '\0';
   if (string[strlen (string) - 1] == '/') string[strlen (string) - 1] = '\0';
}
/*****************************************************************************/
char *make_temp_filename (char *ext) {
   char *tempstr, *destbuf;
   
   tempstr = expand_path (BASE_CONF_DIR "/temp");
   if (access (tempstr, F_OK) == -1) mkdir (tempstr, 0x1C0);

   srand (time (NULL));
   destbuf = NULL;
   do {
      if (destbuf) g_free (destbuf);
      destbuf = g_strdup_printf ("%s/temp%ld%s", tempstr, 1+(long) (99999999.0*rand()/(RAND_MAX+1.0)), ext == NULL ? "" : ext);
   } while (access (destbuf, F_OK) != -1);
   g_free (tempstr);
   return (destbuf);
}
/*****************************************************************************/
void free_file_list (GList *filelist) {
   struct ftp_file_data *tempfle;
   GList *templist;
   
   templist = filelist;
   while (templist != NULL) {
      tempfle = (struct ftp_file_data *) templist->data;
      free_fdata (tempfle);
      templist = templist->next;
   }
   g_list_free (filelist);
}
/*****************************************************************************/
void free_fdata (struct ftp_file_data *fle) {
   if (fle->file) g_free (fle->file);
   if (fle->user) g_free (fle->user);
   if (fle->group) g_free (fle->group);
   if (fle->attribs) g_free (fle->attribs);
   g_free (fle);
}
/*****************************************************************************/
void free_hdata (struct ftp_host_data *hdata) {
   free_file_list (hdata->files);
   gftp_request_destroy (hdata->ftpdata);
   g_free (hdata);
}
/*****************************************************************************/
struct ftp_host_data *new_hdata (void) {
   struct ftp_host_data *hdata;
   
   hdata = g_malloc0 (sizeof (struct ftp_host_data));
   hdata->files = NULL;
   hdata->ftpdata = gftp_request_new ();
   hdata->ftpdata->resolve_symlinks = resolve_symlinks;
   gftp_protocols[0].init (hdata->ftpdata);
   hdata->ftpdata->proxy_hosts = proxy_hosts;
   gftp_set_proxy_config (hdata->ftpdata, proxy_config);
   gftp_set_username (hdata->ftpdata, ANON_LOGIN);
   gftp_set_password (hdata->ftpdata, emailaddr);
   gftp_set_data_type (hdata->ftpdata, GFTP_TYPE_BINARY);
   gftp_logging (hdata->ftpdata, 1, ftp_log, NULL);
   GFTP_SET_TRANSFER_TYPE (hdata->ftpdata, 
   	passive_transfer ? gftp_transfer_passive : gftp_transfer_active);
   return (hdata);
}
/*****************************************************************************/
void free_tdata (struct ftp_transfer_data *tdata) {
   free_hdata (tdata->hdata);
   pthread_mutex_destroy (&tdata->mutex);
   g_free (tdata);
}
/*****************************************************************************/
struct ftp_transfer_data *new_tdata (void) {
   struct ftp_transfer_data *tdata;

   tdata = g_malloc0 (sizeof (struct ftp_transfer_data));
   tdata->hdata = new_hdata ();
   tdata->maxkbs = maxkbs;
   gftp_logging (tdata->hdata->ftpdata, 1, queue_log, NULL);
   tdata->dirs_to_be_made = g_malloc (sizeof (char *));
   tdata->dirs_to_be_made = NULL;
   tdata->num_dirs_to_be_made = 0;
   tdata->stalled = 1;
   pthread_mutex_init (&tdata->mutex, NULL);
   tdata->hdata->wdata = &window2;
   return (tdata);
}
/*****************************************************************************/
void copy_hdata_struct (struct ftp_host_data *hdata, struct ftp_host_data *newhdata) {
   char *str;

   if ((str = GFTP_GET_HOSTNAME (hdata->ftpdata))) {
      gftp_set_hostname (newhdata->ftpdata, str);
   }
   if ((str = GFTP_GET_USERNAME (hdata->ftpdata))) {
      gftp_set_username (newhdata->ftpdata, str);
   }
   if ((str = GFTP_GET_PASSWORD (hdata->ftpdata))) {
      gftp_set_password (newhdata->ftpdata, str);
   }
   if ((str = GFTP_GET_ACCOUNT (hdata->ftpdata))) {
      gftp_set_account (newhdata->ftpdata, str);
   }
   if ((str = GFTP_GET_DIRECTORY (hdata->ftpdata))) {
      gftp_set_directory (newhdata->ftpdata, str);
   }
   gftp_set_port (newhdata->ftpdata, GFTP_GET_PORT (hdata->ftpdata));
   if ((str = GFTP_GET_PROXY_HOSTNAME (hdata->ftpdata))) {
      gftp_set_proxy_hostname (newhdata->ftpdata, str);
   }
   if ((str = GFTP_GET_PROXY_USERNAME (hdata->ftpdata))) {
      gftp_set_proxy_username (newhdata->ftpdata, str);
   }
   if ((str = GFTP_GET_PROXY_PASSWORD (hdata->ftpdata))) {
      gftp_set_proxy_password (newhdata->ftpdata, str);
   }
   if ((str = GFTP_GET_PROXY_ACCOUNT (hdata->ftpdata))) {
      gftp_set_proxy_account (newhdata->ftpdata, str);
   }
   if ((str = GFTP_GET_PROXY_CONFIG (hdata->ftpdata))) {
      gftp_set_proxy_config (newhdata->ftpdata, str);
   }
   gftp_set_proxy_port (newhdata->ftpdata, GFTP_GET_PROXY_PORT (hdata->ftpdata));
   GFTP_SET_LOGGING (newhdata->ftpdata, GFTP_GET_LOGGING (hdata->ftpdata));
   gftp_set_data_type (newhdata->ftpdata, GFTP_GET_DATA_TYPE (hdata->ftpdata));
   newhdata->ftpdata->transfer_type = hdata->ftpdata->transfer_type;
   newhdata->ftpdata->logging_function = hdata->ftpdata->logging_function;
   newhdata->ftpdata->user_data = hdata->ftpdata->user_data;
}
/*****************************************************************************/
void copy_fdata_struct (struct ftp_file_data *fle, struct ftp_file_data *newfle) {
   memcpy (newfle, fle, sizeof (struct ftp_file_data ));
   if (fle->file) {
      newfle->file = g_malloc (strlen (fle->file) + 1);
      strcpy (newfle->file, fle->file);
   }
   if (fle->remote_file) {
      newfle->remote_file = g_malloc (strlen (fle->remote_file) + 1);
      strcpy (newfle->remote_file, fle->remote_file);
   }
   if (fle->user) {
      newfle->user = g_malloc (strlen (fle->user) + 1);
      strcpy (newfle->user, fle->user);
   }
   if (fle->group) {
      newfle->group = g_malloc (strlen (fle->group) + 1);
      strcpy (newfle->group, fle->group);
   }
   if (fle->attribs) {
      newfle->attribs = g_malloc (strlen (fle->attribs) + 1);
      strcpy (newfle->attribs, fle->attribs);
   }
   if (fle->remote_file) {
      newfle->remote_file = g_malloc (strlen (fle->remote_file) + 1);
      strcpy (newfle->remote_file, fle->remote_file);
   }
}
/*****************************************************************************/
int compare_hdata_structs (gftp_request *request1, gftp_request *request2, int compare_dirs) {
   char *strarr[3][2];
   int i, ret;

   ret = 1;
   if(GFTP_GET_PORT (request1) == GFTP_GET_PORT (request2)) {
      strarr[0][0] = GFTP_GET_HOSTNAME (request1);
      strarr[0][1] = GFTP_GET_HOSTNAME (request2);
      strarr[1][0] = GFTP_GET_USERNAME (request1);
      strarr[1][1] = GFTP_GET_USERNAME (request2);
      if (compare_dirs) {
         strarr[2][0] = GFTP_GET_DIRECTORY (request1);
         strarr[2][1] = GFTP_GET_DIRECTORY (request2);
      }
      else {
         strarr[2][0] = strarr[2][1] = "";
      }
      for (i=0; i<3; i++) {
         if ((strarr[i][0] && !strarr[i][1]) || (!strarr[i][0] && strarr[i][1])) {
            ret = 0;
            break;
         }

         if (strarr[i][0] && strarr[i][1] && strcmp (strarr[i][0], strarr[i][1]) != 0) {
            ret = 0;
            break;
         }
      }
   }
   else ret = 0;
   return (ret);
}
/*****************************************************************************/
GList *get_next_selected_filename (GList *filelist) {
   struct ftp_file_data *tempfle;
   GList *templist;
   
   templist = filelist;
   while (templist != NULL) {
      tempfle = (struct ftp_file_data *) templist->data;
      if (tempfle->shown && tempfle->selected) {
         return (templist);
      }
      templist = templist->next;
   }
   return (NULL);
}
/*****************************************************************************/
void open_xpm (char *filename, GtkWidget *parent, GdkPixmap **pixmap, GdkBitmap **mask, int quit_on_err) {
   char *exfile;
   GtkStyle *style;

   if (*filename == '\0') return;
   style = gtk_widget_get_style (parent);

   if ((exfile = get_xpm_path (filename, quit_on_err)) == NULL) return;
   
   *pixmap = gdk_pixmap_create_from_xpm (parent->window, mask, &style->bg[GTK_STATE_NORMAL], exfile);
   if (*pixmap == NULL) {
      if (!quit_on_err) return;
      printf (_("gFTP Error: Error opening file %s\n"), exfile);
      exit (-1);
   }
   g_free (exfile);
}
/*****************************************************************************/
char *get_xpm_path (char *filename, int quit_on_err) {
   char *tempstr, *exfile;
   
   tempstr = g_strconcat (BASE_CONF_DIR, "/", filename, NULL);
   exfile = expand_path (tempstr);
   g_free (tempstr);
   if (access (exfile, F_OK) != 0) {
      g_free (exfile);
      tempstr = g_strconcat (SHARE_DIR, "/", filename, NULL);
      exfile = expand_path (tempstr);
      g_free (tempstr);
      if (access (exfile, F_OK) != 0) {
         g_free (exfile);
         exfile = g_strconcat ("/usr/share/icons/", filename, NULL);
         if (access (exfile, F_OK) != 0) {
            g_free (exfile);
            if (!quit_on_err) return (NULL);
            printf (_("gFTP Error: Cannot find file %s in %s or %s\n"), filename, SHARE_DIR, BASE_CONF_DIR);
            exit (-1);
         }
      }
   }
   return (exfile);
}
/*****************************************************************************/
GtkWidget *toolbar_pixmap (GtkWidget *widget, char **xpmdata) {
   GtkWidget *pix;
   GdkPixmap *pixmap;
   GdkBitmap *bitmap;
   
   pixmap = gdk_pixmap_create_from_xpm_d (widget->window, &bitmap, 
  	&widget->style->bg[GTK_STATE_NORMAL], xpmdata);

   pix = gtk_pixmap_new (pixmap, bitmap);
   gtk_widget_show (pix);

   gdk_pixmap_unref (pixmap);
   gdk_pixmap_unref (bitmap);
   return (pix);
}
/*****************************************************************************/
int check_status(char *name, struct ftp_window_data *wdata, int only_one, 
	int at_least_one, int func) {

   if (wdata->hdata->stopable) {
      ftp_log (gftp_logging_misc, NULL, _("%s: Please hit the stop button first to do anything else\n"), name);
      return (0);
   }

   if ((wdata)->local == -1) {
      ftp_log (gftp_logging_misc, NULL, _("%s: Not connected to a remote site\n"), name);
      return (0);
   }

   if (!func) {
      ftp_log (gftp_logging_misc, NULL, _("%s: This feature is not available using this protocol\n"), name);
      return (0);
   }

   if (only_one && (wdata)->numselected != 1) {
      ftp_log (gftp_logging_misc, NULL, _("%s: You must only have one item selected\n"), name);
      return (0);
   }

   if (at_least_one && !only_one && (wdata)->numselected == 0) {
      ftp_log (gftp_logging_misc, NULL, _("%s: You must have at least one item selected\n"), name);
      return (0);
   }
   return (1);
}
/*****************************************************************************/
void make_nonnull (char **str) {
   if (*str == NULL) {
      *str = g_malloc (1);
      **str = '\0';
   }
}
/*****************************************************************************/
void create_item_factory (GtkItemFactory *ifactory, guint n_entries, GtkItemFactoryEntry *entries, gpointer callback_data) {
   GtkItemFactoryEntry dummy_item;
   int i;
   
   for (i=0; i<n_entries; i++) {
      memcpy (&dummy_item, entries + i, sizeof (GtkItemFactoryEntry));
      if (dummy_item.item_type) {
         dummy_item.item_type = _(entries[i].item_type);
      }
      dummy_item.path = _(entries[i].path);
      gtk_item_factory_create_item (ifactory, &dummy_item, callback_data, 1);
   }
}
/*****************************************************************************/
void add_history (GtkWidget *widget, GList **history, int *histlen, char *str) {
   GList *node, *delnode;
   char *tempstr;
   int i;
   
   if (str == NULL || *str == '\0') return;
   for (node = *history; node != NULL; node = node->next) {
      if (strcmp ((char *) node->data, str) == 0) break;
   }

   if (node == NULL) {
      if (*histlen >= MAX_HIST_LEN) {
         node = *history;
  	 for (i = 1; i < MAX_HIST_LEN; i++) node = node->next;
         node->prev->next = NULL;
         node->prev = NULL;
         delnode = node;
         while (delnode != NULL) {
            if (delnode->data) g_free (delnode->data);
            delnode = delnode->next;
	 }
         g_list_free (node);
      }
      tempstr = g_malloc (strlen (str) + 1);
      strcpy (tempstr, str);
      *history = g_list_prepend (*history, tempstr);
      ++*histlen;
   }
   else if (node->prev != NULL) {
      node->prev->next = node->next;
      if (node->next != NULL) node->next->prev = node->prev;
      node->prev = NULL;
      node->next = *history;
      if (node->next != NULL) node->next->prev = node;
      *history = node;
   }
   gtk_combo_set_popdown_strings (GTK_COMBO (widget), *history);
}
/*****************************************************************************/
