/*
 *  linux/ibcs/svr4.c
 *
 *  Copyright (C) 1995  Mike Jagdis
 *
 * $Id: svr4.c,v 1.6 1998/06/24 20:34:17 jaggy Exp $
 * $Source: /u/CVS/ibcs/iBCSemul/svr4.c,v $
 */

#include <linux/config.h>

#include <linux/module.h>
#include <linux/version.h>

#include <asm/uaccess.h>

#include <linux/types.h>
#include <linux/errno.h>
#include <linux/sched.h>
#include <linux/kernel.h>
#include <linux/mm.h>
#include <linux/stddef.h>
#include <linux/unistd.h>
#include <linux/ptrace.h>
#include <linux/fcntl.h>
#include <linux/time.h>

#include <asm/system.h>
#include <linux/fs.h>
#include <linux/sys.h>
#include <linux/malloc.h>

#include <ibcs/ibcs.h>
#include <ibcs/svr4.h>

#ifdef IBCS_TRACE
#include <ibcs/trace.h>
#endif


/* Interactive SVR4's /bin/sh calls access(... 011) but Linux returns
 * EINVAL if the access mode has any other bits than 007 set.
 */
int
svr4_access(char *path, int mode)
{
	return SYS(access)(path, mode & 007);
}


int svr4_getgroups(int n, unsigned long *buf)
{
	int i;

	if (n) {
		i = verify_area(VERIFY_WRITE, buf, sizeof(unsigned long) * n);
		if (i)
			return i;
	}
	for (i = 0; i < current->ngroups && i < n; i++) {
		put_user(current->groups[i], buf);
		buf++;
	}
	return(i);
}


int svr4_setgroups(int n, unsigned long *buf)
{
	int i;

	if (!capable(CAP_SETGID))
		return -EPERM;
	if (n > NGROUPS)
		return -EINVAL;
	/* FIXME: Yuk! What if we hit a bad address? */
	for (i = 0; i < n; i++, buf++) {
		get_user(current->groups[i], buf);
	}
	current->ngroups = n;
	return 0;
}


int svr4_waitid(int idtype, int id, struct svr4_siginfo *infop, int options)
{
	long result, kopt;
	mm_segment_t old_fs;
	int pid, status;

	switch (idtype) {
		case 0: /* P_PID */
			pid = id;
			break;

		case 1: /* P_PGID */
			pid = -id;
			break;

		case 7: /* P_ALL */
			pid = -1;
			break;

		default:
			return -EINVAL;
	}

	if (infop) {
		result = verify_area(VERIFY_WRITE, infop,
					sizeof(struct svr4_siginfo));
		if (result)
			return result;
	}

	kopt = 0;
	if (options & 0100) kopt |= WNOHANG;
	if (options & 4) kopt |= WUNTRACED;

	old_fs = get_fs();
	set_fs(get_ds());
	result = SYS(wait4)(pid, &status, kopt, NULL);
	set_fs(old_fs);
	if (result < 0)
		return result;

	if (infop) {
		unsigned long op, st;

		put_user(current->exec_domain->signal_map[SIGCHLD],
			&infop->si_signo);
		put_user(result,
			&infop->_data._proc._pid);

		if ((status & 0xff) == 0) {
			/* Normal exit. */
			op = SVR4_CLD_EXITED;
			st = status >> 8;
		} else if ((status & 0xff) == 0x7f) {
			/* Stopped. */
			st = (status & 0xff00) >> 8;
			op = (st == SIGSTOP || st == SIGTSTP)
				? SVR4_CLD_STOPPED
				: SVR4_CLD_CONTINUED;
			st = current->exec_domain->signal_invmap[st];
		} else {
			st = (status & 0xff00) >> 8;
			op = (status & 0200)
				? SVR4_CLD_DUMPED
				: SVR4_CLD_KILLED;
			st = current->exec_domain->signal_invmap[st];
		}
		put_user(op, &infop->si_code);
		put_user(st, &infop->_data._proc._pdata._cld._status);
	}
	return 0;
}
