/*
    interface.c -- parted binding glue to libext2resize
    Copyright (C) 1999 Lennert Buytenhek <buytenh@gnu.org>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/* VERSION: libext2resize 1.1.6 (by Lennert)
 * merged 1.1.11 changes (by Andrew)
 */

static const char _interface_c[] = "$Id: interface.c,v 1.6 1999/10/02 20:56:47 aclausen Exp $";

#include "config.h"

#include <parted/parted.h>
#include "ext2.h"
#include "parted_io.h"

struct ext2_dev_handle* ext2_make_dev_handle_from_parted_geometry(PedGeometry* geom);



int
_ext2_probe (const PedGeometry* geom);

PedFileSystem*
_ext2_open (const PedGeometry *geom);

PedFileSystem*
_ext2_create (const PedGeometry *geom);

int
_ext2_close (PedFileSystem *fs);

int
_ext2_check (PedFileSystem *fs);

int
_ext2_resize (PedFileSystem* fs, PedGeometry* geom);

PedSector
_ext2_get_min_size (const PedFileSystem* fs);

char
_ext2_get_system (const PedFileSystem* fs, const PedPartition* part,
		  PedDiskType* disk_type);






static PedFileSystemOps _ext2_ops = {
	probe:		_ext2_probe,
	open:		_ext2_open,
	create:         _ext2_create,
	close:		_ext2_close,
	check:          _ext2_check,
	resize:		_ext2_resize,
	copy:           NULL,
	get_min_size:	_ext2_get_min_size,
	get_system:	_ext2_get_system
};

static PedFileSystemType _ext2_type = {
       next:		 NULL,
       ops:		 &_ext2_ops,
       name:		 "ext2"
};

void ped_file_system_ext2_init ()
{
	ped_file_system_type_register (&_ext2_type);
}

void ped_file_system_ext2_done ()
{
	ped_file_system_type_unregister (&_ext2_type);
}



int
_ext2_probe (const PedGeometry* geom)
{
	struct ext2_super_block sb;

	if (!ped_geometry_read(geom, &sb, 2, 2)
	    || sb.s_magic != EXT2_SUPER_MAGIC)
		return 0;

	return 1;
}

PedFileSystem*
_ext2_open (const PedGeometry *geom)
{
	PedFileSystem*		fs;
	struct ext2_fs*		fs_info;
	struct ext2_dev_handle*	handle;

	fs = (PedFileSystem*) ped_malloc (sizeof (PedFileSystem));
	if (!fs) goto error;

	fs->type = &_ext2_type;
	fs->geom = ped_geometry_duplicate (geom);

	handle = ext2_make_dev_handle_from_parted_geometry(fs->geom);
	if (!handle) goto error_free_fs;

	fs_info = (struct ext2_fs*) ext2_open(handle, 0);
	if (!fs_info) goto error_free_handle;

	fs->type_specific = (void*) fs_info;
	fs_info->opt_verbose = 0;

	return fs;

error_free_handle:
	ext2_destroy_dev_handle(handle);
error_free_fs:
	ped_free(fs);
error:
	return NULL;
}

PedFileSystem*
_ext2_create (const PedGeometry *geom)
{
	PedFileSystem*		fs;
	struct ext2_fs*		fs_info;
	struct ext2_dev_handle*	handle;

	fs = (PedFileSystem*) ped_malloc (sizeof (PedFileSystem));
	if (!fs) goto error;

	fs->type = &_ext2_type;
	fs->geom = ped_geometry_duplicate (geom);

	handle = ext2_make_dev_handle_from_parted_geometry(fs->geom);
	if (!handle) goto error_free_fs;

	fs_info = ext2_mkfs (handle, -1, -1, -1, -1, -1, -1);
	if (!fs_info) goto error_free_handle;

	fs->type_specific = (void*) fs_info;
	fs_info->opt_verbose = 0;

	return fs;

error_free_handle:
	ext2_destroy_dev_handle(handle);
error_free_fs:
	ped_free(fs);
error:
	return NULL;
}

int
_ext2_close (PedFileSystem *fs)
{
	struct ext2_dev_handle* handle;

	handle = ((struct ext2_fs*)fs->type_specific)->devhandle;
	ext2_close(fs->type_specific);
	ext2_destroy_dev_handle(handle);

	ped_free(fs);
	return 1;
}

int
_ext2_check (PedFileSystem *fs)
{
	ped_exception_throw (PED_EXCEPTION_INFORMATION, PED_EXCEPTION_OK,
		_("The ext2 filesystem passed a basic check.  For a more "
		  "comprehensive check, use the e2fsck program."));
	return 1;
}

int
_ext2_resize (PedFileSystem* fs, PedGeometry* geom)
{
	struct ext2_fs* f;
	PedSector	old_length = fs->geom->length;

	if (fs->geom->disk != geom->disk)
	{
		fprintf(stderr, "huh? ungrokkable!\n");
		return 0;
	}

	if (fs->geom->start != geom->start)
	{
		ped_exception_throw (PED_EXCEPTION_NO_FEATURE,
		      PED_EXCEPTION_CANCEL,
		      _("Sorry, can't move the start of ext2 partitions yet!"));
		return 0;
	}

	f = (struct ext2_fs *) fs->type_specific;

/* ensure that the geometry contains the new and old geometry */
	if (old_length > geom->length) {
		if (!ext2_resize_fs(f, geom->length >> (f->logsize - 9)))
			goto error;

		fs->geom->length = geom->length;
		fs->geom->end = fs->geom->start + geom->length - 1;
	} else {
		fs->geom->length = geom->length;
		fs->geom->end = fs->geom->start + geom->length - 1;

		if (!ext2_resize_fs(f, geom->length >> (f->logsize - 9)))
			goto error;
	}
	return 1;

error:
	fs->geom->length = old_length;
	fs->geom->end = fs->geom->start + old_length - 1;
	return 0;
}

PedSector
_ext2_get_min_size (const PedFileSystem* fs)
{
	struct ext2_fs* f = (struct ext2_fs *) fs->type_specific;

	return f->sb.s_blocks_count - f->sb.s_free_blocks_count;
}

char
_ext2_get_system (const PedFileSystem* fs, const PedPartition* part,
		  PedDiskType* disk_type)
{
	if (part->hidden) {
		ped_exception_throw (PED_EXCEPTION_ERROR, PED_EXCEPTION_CANCEL,
				     _("Ext2 partitions can't be hidden."));
		return 0;
	}
	return 0x83;
}
