/*********************************************************************
 *
 * This is based on code created by Peter Harvey,
 * (pharvey@codebydesign.com).
 *
 * Modified and extended by Nick Gorham
 * (nick@easysoft.com).
 *
 * Any bugs or problems should be considered the fault of Nick and not
 * Peter.
 *
 * copyright (c) 1999 Nick Gorham
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 **********************************************************************
 *
 * $Id: SQLGetInfo.c,v 1.2 1999/12/10 01:50:35 harvey Exp $
 *
 * $Log: SQLGetInfo.c,v $
 * Revision 1.2  1999/12/10 01:50:35  harvey
 * Updated with current sources from unixODBC cvs.
 *
 * Revision 1.9  1999/11/13 23:40:59  ngorham
 *
 * Alter the way DM logging works
 * Upgrade the Postgres driver to 6.4.6
 *
 * Revision 1.8  1999/11/10 03:51:34  ngorham
 *
 * Update the error reporting in the DM to enable ODBC 3 and 2 calls to
 * work at the same time
 *
 * Revision 1.7  1999/10/29 21:07:40  ngorham
 *
 * Fix some stupid bugs in the DM
 * Make the postgres driver work via unix sockets
 *
 * Revision 1.6  1999/10/24 23:54:18  ngorham
 *
 * First part of the changes to the error reporting
 *
 * Revision 1.5  1999/09/21 22:34:25  ngorham
 *
 * Improve performance by removing unneeded logging calls when logging is
 * disabled
 *
 * Revision 1.4  1999/07/10 21:10:16  ngorham
 *
 * Adjust error sqlstate from driver manager, depending on requested
 * version (ODBC2/3)
 *
 * Revision 1.3  1999/07/04 21:05:07  ngorham
 *
 * Add LGPL Headers to code
 *
 * Revision 1.2  1999/06/30 23:56:55  ngorham
 *
 * Add initial thread safety code
 *
 * Revision 1.1.1.1  1999/05/29 13:41:07  sShandyb
 * first go at it
 *
 * Revision 1.1.1.1  1999/05/27 18:23:18  pharvey
 * Imported sources
 *
 * Revision 1.2  1999/04/30 16:22:47  nick
 * Another checkpoint
 *
 * Revision 1.1  1999/04/25 23:06:11  nick
 * Initial revision
 *
 *
 **********************************************************************/

#include "drivermanager.h"

static char const rcsid[]= "$RCSfile: SQLGetInfo.c,v $ $Revision: 1.2 $";

SQLRETURN SQLGetInfo( SQLHDBC connection_handle,
           SQLUSMALLINT info_type,
           SQLPOINTER info_value,
           SQLSMALLINT buffer_length,
           SQLSMALLINT *string_length )
{
    DMHDBC connection = (DMHDBC)connection_handle;
    SQLRETURN ret = SQL_SUCCESS;
    int type;
    char txt[ 30 ], *cptr;
    SQLPOINTER *ptr;
    SQLCHAR s0[ 20 ], s1[ 100 ];

    /*
     * check connection
     */

    if ( !__validate_dbc( connection ))
    {
        return SQL_INVALID_HANDLE;
    }

    function_entry( connection );

    if ( connection -> log_handle )
    {
        sprintf( connection -> msg, "\n\t\tEntry:\
            \n\t\t\tConnection = %p\
            \n\t\t\tInfo Type = %s\
            \n\t\t\tInfo Value = %p\
            \n\t\t\tBuffer Length = %d\
            \n\t\t\tStrLen = %p",
                connection,
                __info_as_string( s1, info_type ),
                info_value, 
                (int)buffer_length,
                (void*)string_length );

        dm_log_write( connection -> log_handle, 
                __get_pid( s0 ), 
                __FILE__, 
                __LINE__, 
                LOG_INFO, 
                LOG_INFO, 
                connection -> msg );
    }

    thread_protect( SQL_HANDLE_DBC, connection );

    if ( info_type != SQL_ODBC_VER &&
            connection -> state == STATE_C2 )
    {
        dm_log_write( connection -> log_handle, 
                __get_pid( s0 ), 
                __FILE__, 
                __LINE__, 
                LOG_INFO, 
                LOG_INFO, 
                "Error: 08003" );

        __post_internal_error( &connection -> error,
                ERROR_08003, NULL,
                connection -> environment -> requested_version );

        thread_release( SQL_HANDLE_DBC, connection );

        return function_return( connection, SQL_ERROR );
    }
    else if ( connection -> state == STATE_C3 )
    {
        dm_log_write( connection -> log_handle, 
                __get_pid( s0 ), 
                __FILE__, 
                __LINE__, 
                LOG_INFO, 
                LOG_INFO, 
                "Error: 08003" );

        __post_internal_error( &connection -> error,
                ERROR_08003, NULL,
                connection -> environment -> requested_version );

        thread_release( SQL_HANDLE_DBC, connection );

        return function_return( connection, SQL_ERROR );
    }

    if ( buffer_length < 0 )
    {
        dm_log_write( connection -> log_handle, 
                __get_pid( s0 ), 
                __FILE__, 
                __LINE__, 
                LOG_INFO, 
                LOG_INFO, 
                "Error: HY090" );

        __post_internal_error( &connection -> error,
                ERROR_HY090, NULL,
                connection -> environment -> requested_version );

        thread_release( SQL_HANDLE_DBC, connection );

        return function_return( connection, SQL_ERROR );
    }

    switch ( info_type )
    {
      case SQL_DATA_SOURCE_NAME:
        type = 1;
        cptr = connection -> dsn;
        break;

      case SQL_DM_VER:
        type = 1;
        sprintf( txt, "%02d.%02d.%04d.%04d",
                SQL_SPEC_MAJOR, SQL_SPEC_MINOR, 
                atoi( VERSION ), atoi( VERSION + 2 ));
        cptr = txt;
        break;

      case SQL_ODBC_VER:
        type = 1;
        sprintf( txt, "%02d.%02d",
                SQL_SPEC_MAJOR, SQL_SPEC_MINOR );
        cptr = txt;
        break;

      case SQL_DRIVER_HDBC:
        type = 2;
        ptr = (SQLPOINTER) connection -> driver_dbc;
        break;

      case SQL_DRIVER_HENV:
        type = 2;
        ptr = (SQLPOINTER) connection -> driver_env;
        break;

      case SQL_DRIVER_HDESC:
        {
            /*
             * TO_DO
             */
        }
        break;

      case SQL_DRIVER_HLIB:
        type = 2;
        ptr = connection -> dl_handle;
        break;

      case SQL_DRIVER_HSTMT:
        {
            DMHSTMT hstmt = *((DMHSTMT*) info_value); 

            type = 2;

            if ( __validate_stmt( hstmt ))
            {
                ptr = (SQLPOINTER) hstmt -> driver_stmt;
            }
            else
            {
                dm_log_write( connection -> log_handle, 
                        __get_pid( s0 ), 
                        __FILE__, 
                        __LINE__, 
                        LOG_INFO, 
                        LOG_INFO, 
                        "Error: HY024" );

                __post_internal_error( &connection -> error,
                        ERROR_HY024, NULL,
                        connection -> environment -> requested_version );

                thread_release( SQL_HANDLE_DBC, connection );

                return function_return( connection, SQL_ERROR );
            }
        }
        break;

      case SQL_XOPEN_CLI_YEAR:
        type = 1;
        cptr = "1994";
        break;

      default:
        /*
         * pass all the others on
         */
        if ( !CHECK_SQLGETINFO( connection ))
        {
            dm_log_write( connection -> log_handle, 
                    __get_pid( s0 ), 
                    __FILE__, 
                    __LINE__, 
                    LOG_INFO, 
                    LOG_INFO, 
                    "Error: IM001" );

            __post_internal_error( &connection -> error,
                    ERROR_IM001, NULL,
                    connection -> environment -> requested_version );

            thread_release( SQL_HANDLE_DBC, connection );

            return function_return( connection, SQL_ERROR );
        }

        ret = SQLGETINFO( connection,
                connection -> driver_dbc,
                info_type,
                info_value,
                buffer_length,
                string_length );

        if ( connection -> log_handle )
        {
            sprintf( connection -> msg, 
                    "\n\t\tExit:[%s]",
                        __get_return_status( ret ));

            dm_log_write( connection -> log_handle, 
                    __get_pid( s0 ), 
                    __FILE__, 
                    __LINE__, 
                    LOG_INFO, 
                    LOG_INFO, 
                    connection -> msg );
        }

        thread_release( SQL_HANDLE_DBC, connection );

        return function_return( connection, ret );
    }

    if ( type == 1 )
    {
        if ( string_length )
            *string_length = strlen( cptr );

        if ( info_value )
        {
            if ( buffer_length > strlen( cptr ) + 1 )
            {
                strcpy( info_value, cptr );
            }
            else
            {
                memcpy( info_value, cptr, buffer_length - 1 );
                ((char*)info_value)[ buffer_length - 1 ] = '\0';
                ret = SQL_SUCCESS_WITH_INFO;
            }
        }
    }
    else if ( type == 2 )
    {
        if ( info_value )
            *((int*)info_value) = (int)ptr;

        if ( string_length )
            *string_length = sizeof( SQLPOINTER );
    }

    if ( connection -> log_handle )
    {
        sprintf( connection -> msg, 
                "\n\t\tExit:[%s]",
                    __get_return_status( ret ));

        dm_log_write( connection -> log_handle, 
                __get_pid( s0 ), 
                __FILE__, 
                __LINE__, 
                LOG_INFO, 
                LOG_INFO, 
                connection -> msg );
    }

    thread_release( SQL_HANDLE_DBC, connection );

    return function_return( connection, ret );
}
