/*********************************************************************
 *
 * This is based on code created by Peter Harvey,
 * (pharvey@codebydesign.com).
 *
 * Modified and extended by Nick Gorham
 * (nick@easysoft.com).
 *
 * Any bugs or problems should be considered the fault of Nick and not
 * Peter.
 *
 * copyright (c) 1999 Nick Gorham
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 **********************************************************************
 *
 * $Id: SQLFreeHandle.c,v 1.2 1999/12/10 01:50:34 harvey Exp $
 *
 * $Log: SQLFreeHandle.c,v $
 * Revision 1.2  1999/12/10 01:50:34  harvey
 * Updated with current sources from unixODBC cvs.
 *
 * Revision 1.10  1999/11/17 21:11:59  ngorham
 *
 * Fix bug where the check for a valid handle was after the code had
 * used it.
 *
 * Revision 1.9  1999/11/13 23:40:59  ngorham
 *
 * Alter the way DM logging works
 * Upgrade the Postgres driver to 6.4.6
 *
 * Revision 1.8  1999/10/24 23:54:18  ngorham
 *
 * First part of the changes to the error reporting
 *
 * Revision 1.7  1999/10/09 00:15:58  ngorham
 *
 * Add mapping from SQL_TYPE_X to SQL_X and SQL_C_TYPE_X to SQL_C_X
 * when the driver is a ODBC 2 one
 *
 * Revision 1.6  1999/09/21 22:34:24  ngorham
 *
 * Improve performance by removing unneeded logging calls when logging is
 * disabled
 *
 * Revision 1.5  1999/08/03 21:47:39  shandyb
 * Moving to automake: changed files in DriverManager
 *
 * Revision 1.4  1999/07/10 21:10:16  ngorham
 *
 * Adjust error sqlstate from driver manager, depending on requested
 * version (ODBC2/3)
 *
 * Revision 1.3  1999/07/04 21:05:07  ngorham
 *
 * Add LGPL Headers to code
 *
 * Revision 1.2  1999/06/30 23:56:55  ngorham
 *
 * Add initial thread safety code
 *
 * Revision 1.1.1.1  1999/05/29 13:41:07  sShandyb
 * first go at it
 *
 * Revision 1.2  1999/06/07 01:29:31  pharvey
 * *** empty log message ***
 *
 * Revision 1.1.1.1  1999/05/27 18:23:17  pharvey
 * Imported sources
 *
 * Revision 1.5  1999/05/09 23:27:11  nick
 * All the API done now
 *
 * Revision 1.4  1999/05/03 19:50:43  nick
 * Another check point
 *
 * Revision 1.3  1999/04/30 16:22:47  nick
 * Another checkpoint
 *
 * Revision 1.2  1999/04/29 20:47:37  nick
 * Another checkpoint
 *
 * Revision 1.1  1999/04/25 23:06:11  nick
 * Initial revision
 *
 *
 **********************************************************************/

#include "drivermanager.h"

static char const rcsid[]= "$RCSfile: SQLFreeHandle.c,v $ $Revision: 1.2 $";

SQLRETURN __SQLFreeHandle( SQLSMALLINT handle_type,
        SQLHANDLE handle )
{
    switch( handle_type )
    {
      case SQL_HANDLE_ENV:
        {
            DMHENV environment = (DMHENV)handle;
            SQLCHAR s0[ 20 ];

            /*
             * check environment
             */

            if ( !__validate_env( environment ))
            {
                return SQL_INVALID_HANDLE;
            }
            
            function_entry( environment );

            if ( environment -> log_handle )
            {
                sprintf( environment -> msg,
                        "\n\t\tEntry:\n\t\t\tHandle Type = %d\n\t\t\tInput Handle = %p",
                        handle_type,
                        (void*)handle );

                dm_log_write( environment -> log_handle,
                        __get_pid( s0 ),
                        __FILE__,
                        __LINE__,
                        LOG_INFO,
                        LOG_INFO,
                        environment -> msg );
            }

            thread_protect( SQL_HANDLE_ENV, environment );

            /*
             * check states
             */
            if ( environment -> state != STATE_E1 )
            {
                dm_log_write( environment -> log_handle,
                        __get_pid( s0 ),
                        __FILE__,
                        __LINE__,
                        LOG_INFO,
                        LOG_INFO,
                        "Error: HY010" );

                __post_internal_error( &environment -> error,
                        ERROR_HY010, NULL,
                        environment -> requested_version );
                return function_return( environment, SQL_ERROR );
            }

            thread_release( SQL_HANDLE_ENV, environment );

            __release_env( environment );
            return SQL_SUCCESS;
        }
        break;

      case SQL_HANDLE_DBC:
        {
            DMHDBC connection = (DMHDBC)handle;
            DMHENV environment;
            SQLCHAR s0[ 20 ];

            /*
             * check connection
             */
            if ( !__validate_dbc( connection ))
            {
                return SQL_INVALID_HANDLE;
            }

            function_entry( connection );

            environment = connection -> environment;

            if ( connection -> log_handle )
            {
                sprintf( connection -> msg,
                        "\n\t\tEntry:\n\t\t\tHandle Type = %d\n\t\t\tInput Handle = %p",
                        handle_type,
                        (void*)handle );

                dm_log_write( connection -> log_handle,
                        __get_pid( s0 ),
                        __FILE__,
                        __LINE__,
                        LOG_INFO,
                        LOG_INFO,
                        connection -> msg );
            }

            thread_protect( SQL_HANDLE_ENV, environment );

            /*
             * check states
             */

            if ( connection -> state != STATE_C2 )
            {
                dm_log_write( connection->log_handle,
                        __get_pid( s0 ),
                        __FILE__,
                        __LINE__,
                        LOG_INFO,
                        LOG_INFO,
                        "Error: HY010" );

                __post_internal_error( &connection -> error,
                        ERROR_HY010, NULL,
                        connection -> environment -> requested_version );

                thread_release( SQL_HANDLE_ENV, environment);

                return function_return( environment, SQL_ERROR );
            }

            environment -> connection_count --;

            if ( environment -> connection_count == 0 )
            {
                environment -> state = STATE_E1;
            }

            environment = connection -> environment;

            __disconnect_part_one( connection );

            __release_dbc( connection );

            if ( environment -> log_handle )
            {
                sprintf( environment -> msg,
                        "\n\t\tExit:[SQL_SUCCESS]" );

                dm_log_write( environment -> log_handle,
                        __get_pid( s0 ),
                        __FILE__,
                        __LINE__,
                        LOG_INFO,
                        LOG_INFO,
                        environment -> msg );
            }

            thread_release( SQL_HANDLE_ENV, environment );

            return SQL_SUCCESS;
        }
        break;

      case SQL_HANDLE_STMT:
        {
            DMHSTMT statement = (DMHSTMT)handle;
            DMHDBC connection;
            SQLRETURN ret;
            SQLCHAR s0[ 20 ];

            /*
             * check statement
             */
            if ( !__validate_stmt( statement ))
            {
                return SQL_INVALID_HANDLE;
            }

            function_entry( statement );

            connection = statement -> connection;

            if ( statement -> log_handle )
            {
                sprintf( statement -> msg,
                        "\n\t\tEntry:\n\t\t\tHandle Type = %d\n\t\t\tInput Handle = %p",
                        handle_type,
                        (void*)handle );

                dm_log_write( statement -> log_handle,
                        __get_pid( s0 ),
                        __FILE__,
                        __LINE__,
                        LOG_INFO,
                        LOG_INFO,
                        statement -> msg );
            }

            thread_protect( SQL_HANDLE_STMT, statement );

            /*
             * check states
             */
            if ( statement -> state == STATE_S8 ||
                    statement -> state == STATE_S9 ||
                    statement -> state == STATE_S10 ||
                    statement -> state == STATE_S11 ||
                    statement -> state == STATE_S12 )
            {
                dm_log_write( connection->log_handle,
                        __get_pid( s0 ),
                        __FILE__,
                        __LINE__,
                        LOG_INFO,
                        LOG_INFO,
                        "Error: HY010" );

                __post_internal_error( &statement -> error,
                          ERROR_HY010, NULL,
                          statement -> connection -> environment -> requested_version );

                thread_release( SQL_HANDLE_STMT, statement );

                return function_return( statement, SQL_ERROR );
            }

            if ( !CHECK_SQLFREEHANDLE( statement -> connection ))
            {
                if ( !CHECK_SQLFREESTMT( statement -> connection ))
                {
                    dm_log_write( connection->log_handle,
                            __get_pid( s0 ),
                            __FILE__,
                            __LINE__,
                            LOG_INFO,
                            LOG_INFO,
                            "Error: IM001" );

                    __post_internal_error( &statement -> error,
                            ERROR_IM001, NULL,
                            statement -> connection -> environment -> requested_version );

                    thread_release( SQL_HANDLE_STMT, statement );

                    return function_return( statement, SQL_ERROR );
                }
                else
                {
                    ret = SQLFREESTMT( statement -> connection,
                        statement -> driver_stmt,
                        SQL_DROP );
                }
            }
            else
            {
                ret = SQLFREEHANDLE( statement -> connection,
                        handle_type,
                        statement -> driver_stmt );
            }

            if ( SQL_SUCCEEDED( ret ))
            {
                /*
                 * release the implicit descriptors
                 */
                if ( statement -> connection -> driver_act_ver == 3 )
                {
                    if ( statement -> implicit_ard )
                        __release_desc( statement -> implicit_ard );
                    if ( statement -> implicit_apd )
                        __release_desc( statement -> implicit_apd );
                    if ( statement -> implicit_ird )
                        __release_desc( statement -> implicit_ird );
                    if ( statement -> implicit_ipd )
                        __release_desc( statement -> implicit_ipd );
                }
                statement -> connection -> statement_count --;

                thread_release( SQL_HANDLE_STMT, statement );

                __release_stmt( statement );
            }
            else
            {
                thread_release( SQL_HANDLE_STMT, statement );
            }

            if ( connection -> log_handle )
            {
                sprintf( connection -> msg,
                        "\n\t\tExit:[SQL_SUCCESS]" );

                dm_log_write( connection -> log_handle,
                        __get_pid( s0 ),
                        __FILE__,
                        __LINE__,
                        LOG_INFO,
                        LOG_INFO,
                        connection -> msg );
            }

            return function_return( statement, ret ); 
        }
        break;

      case SQL_HANDLE_DESC:
        return SQL_ERROR;
        break;

      default:
        /*
         * there is nothing to report a error on
         */
        return SQL_ERROR;
    }
}

SQLRETURN SQLFreeHandle( SQLSMALLINT handle_type,
        SQLHANDLE handle )
{
    return __SQLFreeHandle( handle_type,
            handle );
}

