/* 
 *  Copyright (C) 1999-2000 Bernd Gehrmann
 *                          bernd@physik.hu-berlin.de
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the "Artistic License" which comes with this Kit. 
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * Artistic License for more details.
 *
 * You should have received a copy of the Artistic License with 
 * this Kit, in the file named "Artistic".  If not, I'll be glad
 * to provide one.
 */


#include <qlabel.h>
#include <qlineedit.h>
#include <qmessagebox.h>
#include <qheader.h>
#include <qlayout.h>

#include <kapp.h>
#include <kbuttonbox.h>
#include <klocale.h>
#include <kstddirs.h>

#include "peopleresults.h"
#include "people.h"


PeopleTopLevel::PeopleTopLevel(QWidget *parent, const char *name)
    : QWidget(parent, name)
{
    setAcceptDrops(true);

    man = new PeopleSiteManager();
    
    QBoxLayout *layout = new QVBoxLayout(this, 10, 4);
    
    QGridLayout *grid = new QGridLayout(3, 4, 4);
    layout->addLayout(grid, 1);
    grid->setColStretch(0, 0);
    grid->setColStretch(1, 10);
    grid->setColStretch(2, 0);
    grid->addColSpacing(2, 10);
    grid->addRowSpacing(1, 15);
    grid->setRowStretch(1, 0);
    
    QLabel *wordslabel = new QLabel(i18n("What:"), this);
    grid->addWidget(wordslabel, 0, 0);
    
    wordsedit = new QLineEdit(this);
    wordsedit->setFocus();
    grid->addWidget(wordsedit, 0, 1);
    
    QLabel *siteslabel = new QLabel(i18n("Where:"), this);
    grid->addWidget(siteslabel, 2, 0);
    
    sitebox = new QListView(this);
    sitebox->setFrameStyle( QFrame::Panel | QFrame::Sunken );  
    sitebox->header()->hide();
    sitebox->addColumn(i18n("Sites"));

    QStringList list = man->siteList();
    QStringList::Iterator it();
    for (QStringList::Iterator it = list.begin(); it != list.end(); ++it)
        new QCheckListItem(sitebox, *it, QCheckListItem::CheckBox);

    grid->addWidget(sitebox, 2, 1);
    
    KButtonBox *buttonbox = new KButtonBox(this, Vertical);
    searchbutton = buttonbox->addButton(i18n("&Search"));
    searchbutton->setDefault(true);
    connect( searchbutton, SIGNAL(clicked()), SLOT(slotSearch()) );
    cancelbutton = buttonbox->addButton(i18n("Cancel"));
    connect( cancelbutton, SIGNAL(clicked()), SLOT(slotCancel()) );
    cancelbutton->setEnabled(false);
    buttonbox->layout();
    grid->addMultiCellWidget(buttonbox, 0, 2, 3, 3);
    
    layout->addSpacing(10);
    
    foundlabel = new QLabel(i18n("Found:"), this);
    layout->addWidget(foundlabel);
    
    resultbox = new PeopleResultView(this);
    layout->addWidget(resultbox, 10);

    connect( resultbox, SIGNAL(emailActivated(const QString &)),
             this, SLOT(emailActivated(const QString &)) );
}


PeopleTopLevel::~PeopleTopLevel()
{}


void PeopleTopLevel::updateButtons()
{
    bool hasjobs = !jobs.isEmpty();
    searchbutton->setEnabled(!hasjobs);
    cancelbutton->setEnabled(hasjobs);
}


void PeopleTopLevel::slotSearch()
{
    int n;
    QString searchtext = wordsedit->text();
    while ( (n = searchtext.find('\n')) != -1) 
        searchtext.remove(n, 1);
    
    // Lookup checked sites
    QCheckListItem *cb = static_cast<QCheckListItem*>(sitebox->firstChild());
    for (; cb; cb =  static_cast<QCheckListItem*>(cb->nextSibling()))
        if (cb->isOn())
            {
                PeopleSearch *job = new PeopleSearch(man->find(cb->text()), searchtext);
                jobs.append(job);
                connect( job, SIGNAL(foundItem(const QString&,const QString&,const QString&,const QString&,const LdapItem&)),
                         this, SLOT(processItem(const QString&,const QString&,const QString&,const QString&,const LdapItem&)) );
                connect( job, SIGNAL(result(KIO::Job*) ),
                         this, SLOT(slotResult(KIO::Job*)) );
            }

    updateButtons();
}


void PeopleTopLevel::slotCancel()
{
    QListIterator<PeopleSearch> it(jobs);
    for (; it.current(); ++it)
        it.current()->kill();
}


void PeopleTopLevel::slotResult(KIO::Job *job)
{
    if (job->error())
        job->showErrorDialog(this);
    jobs.remove(static_cast<PeopleSearch*>(job));
    
    updateButtons();
}


void PeopleTopLevel::processItem(const QString &sitename, const QString &name,
                                 const QString &email, const QString &phone, const LdapItem &item)
{
    qDebug("Sitename: %s", sitename.latin1());
    resultbox->insertItem("" /*icon?*/, name, email, phone, item);
    foundlabel->setText(i18n("Found %1 matches").arg(resultbox->childCount()));
}


void PeopleTopLevel::emailActivated(const QString &email)
{
    qDebug( "Email activated %s", email.latin1());
    kapp->invokeMailer(email, QString::null);
}
