/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; version 2.                              *
 *                                                                         *
 ***************************************************************************/

#ifndef KSVDRAGLIST_H
#define KSVDRAGLIST_H

#pragma interface

#include <qstring.h>
#include <qdragobject.h>
#include <qcolor.h>
#include "Data.h"
#include "kndraglist.h"

class QPen;
class KSVAction;
class SimpleAction;
class KSVDragList;
class KPopupMenu;

class KSVItem : public QListViewItem
{
public:
  typedef enum {
	SortNumber = 0,
	Icon,
	ServiceName,
	FileName,

	// insert additional columns here

	Last
  } Columns;

  KSVItem (KListView*);
  KSVItem (const KSVItem&);
  explicit KSVItem (KSVDragList*, const KSVData&);
  KSVItem (KListView* view, QString file, QString path, QString label, Q_INT8 nr );
  virtual ~KSVItem ();

  virtual QString key (int col, bool ascending) const;

  inline bool isChanged() const { return mData->changed(); }
  
  inline bool isNumChanged() const { return mData->numberChanged(); }
  inline bool isLabelChanged() const { return mData->labelChanged(); }
  inline bool isNew() const { return mData->newEntry(); }

  inline const QString& oldFilename() const { return mData->oldFilename(); }
  inline const QString& filename() const { return mData->filename(); }
  inline const QString& oldLabel() const { return mData->oldLabel(); }
  inline const QString& label() const { return mData->label(); }
  inline const QString& runlevel() const { return mData->runlevel(); }
  
  inline QString filenameAndPath() const { return mData->path() + "/" + mData->filename(); }

  inline const QString& path() const { return mData->path(); }

  inline Q_INT8 number() const { return mData->number(); }
  inline Q_INT8 oldNumber() const { return mData->oldNumber(); }

  void copy (const KSVData&);

  virtual KSVItem* clone() const { return new KSVItem(*this); }
  KSVData* cloneData () const { return new KSVData(*mData); }

  KSVData* data () { return mData; }
  const KSVData* data () const { return mData; }

  QString toString () const;

  inline bool operator== (const KSVItem & rhs) const { return mData == rhs.mData; }

  inline const QColor& newNormalColor () const
  {
    return mNewNormalColor;
  }

  inline const QColor& changedNormalColor () const
  {
    return mChangedNormalColor;
  }

  inline QString originalRunlevel() const { return mData->originalRunlevel(); }  
  
  void setIcon (const QPixmap& icon);
  void setLabel (const QString& label);
  void setFilename (const QString& file);
  void setRunlevel (const QString& runlevel);
  void setNumber (Q_INT8 nr);
  void setPath (const QString& path);
  void setChanged (bool);
  void setNewNormalColor (const QColor&);
  void setChangedNormalColor (const QColor&);
  void setNew (bool);
  void setOriginalRunlevel (const QString&);

  inline void setNumberChanged (bool val) { mData->setNumberChanged (val); }

  /**
   * Overridden from @ref KDLItem
   */
  virtual QString tooltip () const;
  virtual void paintCell (QPainter* p, const QColorGroup& cg, int column, int width, int align);

protected:
  friend class KSVDragList;
  QPixmap paintDragIcon (const QFont& f, const QColorGroup& g) const;

private:
  friend class KSVDrag;

  KSVData* mData;

  QColor mNewNormalColor;
  QColor mChangedNormalColor;
};

class KSVDrag : public QDragObject
{
public:
  KSVDrag (const KSVData& item, QWidget* dragSource = 0L, const char* name = 0L);
  KSVDrag (const KSVItem& item, QWidget* dragSource = 0L, const char* name = 0L);
  virtual ~KSVDrag();

  virtual const char* format (int i) const;
  QByteArray encodedData (const char*) const;
  
  static bool decodeNative (const QMimeSource*, KSVData&);
  
  using QDragObject::drag;

private:
  enum
  {
	Native, Text, URL
  };

  QByteArray mNative;
  QString mText;
  QString mURL;
};

class KSVDragList : public K2ListView
{
  Q_OBJECT

public:
  KSVDragList ( QWidget* parent = 0, const char* name = 0 );
  virtual ~KSVDragList();

  virtual void initItem (QString file, QString path, QString name, Q_INT8 nr);
  QList<KSVData>& getDeletedItems() { return mRMList; }
  
  /**
   * \return whether this is the currently "selected" KSVDragList.
   */
  inline bool isOrigin() const { return mOrigin; }

  inline KSVItem* currentItem() { return dynamic_cast<KSVItem*> (K2ListView::currentItem()); }

  /**
   * Attention: relatively slow!
   */
  KSVItem* lastChild ();

  /**
   * \return true if the insertion was successful.
   */
  bool insert (const KSVData& data, const KSVData* above = 0L, const KSVData* below = 0L);

  bool insert (const KSVData& data, const KSVItem* where, KSVAction*& action);

  inline const QPixmap& defaultIcon() const { return mIcon; }

  bool removeFromRMList (const KSVData&);
  bool addToRMList (const KSVData&);
  void clearRMList();

  KSVItem* match (const KSVData&);

  QPopupMenu* itemContextMenu () const { return mItemMenu; }

public slots:
  void setNewNormalColor (const QColor &);
  void setChangedNormalColor (const QColor &);

  void setOrigin (bool);
  inline void slotNewOrigin() { setOrigin (false); }

  void setItemContextMenu (QPopupMenu* m);
  void setDefaultIcon (const QPixmap& icon);

  void drop (QDropEvent*, QListViewItem*);

protected:
  virtual void focusInEvent (QFocusEvent*);
  virtual bool acceptDrag (QDropEvent*) const;
  virtual KSVDrag* dragObject ();

  virtual void startDrag ();

  /**
   * Generates a sorting number for an item at
   * the given index by taking the average of the item
   * above and the item below.
   */
  Q_INT8 generateNumber (Q_INT8 high, Q_INT8 low);

private:
  QPixmap mIcon;
  QPopupMenu* mItemMenu; // context menu for items
  
  bool mOrigin;
  QList<KSVData> mRMList;

  QColor mNewNormalColor;
  QColor mChangedNormalColor;

  KPopupMenu* mDragMenu;
  KPopupMenu* mDragCopyMenu;
  KPopupMenu* mDragMoveMenu;
  
  typedef enum { Copy, Move } DragAction;

  KSVItem* mItemToDrag;

signals:
  void newOrigin();

  void cannotGenerateNumber();
  void changed();
  void undoAction(KSVAction*);
};

#endif
