/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; version 2.                              *
 *                                                                         *
 ***************************************************************************/

#pragma implementation

#include <stdlib.h>
#include <assert.h>

#include <qpainter.h>
#include <qdragobject.h>
#include <qdatastream.h>
#include <qheader.h>
#include <qpixmapcache.h>
#include <qbitmap.h>
#include <qcolor.h>
#include <qcstring.h>

#include <kapp.h>
#include <kpopupmenu.h>
#include <kglobal.h>
#include <klocale.h>
#include <kdebug.h>

#include "ksv_core.h"
#include "ksv_conf.h"
#include "ksvdraglist.h"
#include "ActionList.h"
#include "minmax.h"

#include <iostream>

#include <qlabel.h>

KSVItem::KSVItem (KListView* view)
  : QListViewItem (view),
// 	K2ListViewItem (),
	mData (new KSVData())
{
//   setPopupMenuAtIndex (1);
}

KSVItem::KSVItem (const KSVItem& item)
  : QListViewItem (item.listView()),
// 	K2ListViewItem (),
	mData (new KSVData(*item.mData))

{
  setText (SortNumber, mData->numberString());
  setText (ServiceName, label());
  setIcon (*item.pixmap(Icon));

//   setPopupMenuAtIndex (1);
//   setPopupMenuAtIndex (1);
}

KSVItem::KSVItem (KSVDragList* view, const KSVData& data)
  : QListViewItem (view),
// 	K2ListViewItem (),
	mData (new KSVData (data))
{
  setText (SortNumber, mData->numberString());
  setText (ServiceName, label());
//   setIcon (*item.pixmap(Icon));
  setIcon (view->defaultIcon());

//   setPopupMenuAtIndex (1);
}

void KSVItem::copy (const KSVData& item)
{
  *mData = item;

  setText (SortNumber, mData->numberString());
  setText (ServiceName, mData->label());
}

QString KSVItem::key (int, bool) const
{
  return mData->numberString() + mData->label();
}

KSVItem::KSVItem (KListView* view, QString file, QString path, QString label, Q_INT8 nr )
  : QListViewItem (view),
	mData (new KSVData (file, path, label, nr))
{
  setText (ServiceName, mData->label());
  setText (SortNumber, mData->numberString());

//   setPopupMenuAtIndex (1);
}

KSVItem::~KSVItem()
{
  delete mData;
}

void KSVItem::paintCell (QPainter* p, const QColorGroup& cg, int column, int width, int align)
{
  QColorGroup colors = cg;
  
  if (mData->newEntry())
	{
	  // FIXME
	  colors.setColor (QColorGroup::Text, blue);
	  colors.setColor (QColorGroup::HighlightedText, blue);
	}
  else if (mData->changed())
	{
	  // FIXME
	  colors.setColor (QColorGroup::Text, red);
	  colors.setColor (QColorGroup::HighlightedText, red);
	}

  QListViewItem::paintCell (p, colors, column, width, align);
}

QString KSVItem::toString() const
{
  return filenameAndPath();
}

QString KSVItem::tooltip () const
{
  QString result;

  if (KSVConfig::self()->showDescription())
    {
      if (!KSVCore::getServiceDescription(label(), result))
        result = toString();
      else
        {
          // split into nice chunks
          result = KSVCore::breakWords(result.simplifyWhiteSpace(), 50);
        }
    }
  else
    {
      result = toString();
    }

  return result;
}


void KSVItem::setNew ( bool val )
{
  mData->setNewEntry (val);
}


QPixmap KSVItem::paintDragIcon (const QFont& font, const QColorGroup& g) const
{
  QFontMetrics metric (font);
  QRect textRect = metric.boundingRect (label());
  const QPixmap& icon = *pixmap (Icon);
  const int margin = listView()->itemMargin();

  const int width = icon.width() + margin * 2 + textRect.width();
  const int height = max (icon.height(), textRect.height());

  QPixmap result (width, height);
  result.fill (white);

  QPainter p (&result);
  p.drawPixmap (0, 0, icon);
  p.setFont (font);

  p.drawText (icon.width() + margin, 0,
			  width, height,
			  AlignLeft | AlignVCenter,
			  label());
  p.end();

  QBitmap mask (width, height);
  p.begin (&mask);
  p.setFont (font);

  p.fillRect (0, 0, width, height, color0);

  p.setPen (color1);
  p.drawPixmap (0, 0, icon.createHeuristicMask());

  p.drawText (icon.width() + margin, 0,
			  width, height,
			  AlignLeft | AlignVCenter,
			  label());

  QBrush brush (color0);
  brush.setStyle(Dense5Pattern);
  p.fillRect (0, 0, width, height, brush);

  p.end();
  
  result.setMask(mask);
  result.setOptimization(QPixmap::BestOptim);
  return result;
}

void KSVItem::setIcon (const QPixmap& icon)
{
  setPixmap (Icon, icon);
}

void KSVItem::setLabel (const QString& label)
{
  mData->setLabel (label);

  setText (ServiceName, label);
}

void KSVItem::setRunlevel (const QString& runlevel)
{
  mData->setRunlevel (runlevel);
}

void KSVItem::setFilename (const QString& file)
{
  mData->setFilename (file);
}

void KSVItem::setNumber (Q_INT8 nr)
{
  mData->setNumber (nr);

  setText(SortNumber, mData->numberString());
}

void KSVItem::setPath (const QString& path)
{
  mData->setPath (path);
}

void KSVItem::setNewNormalColor (const QColor& col)
{
  mNewNormalColor = col;
}

void KSVItem::setChangedNormalColor (const QColor& col)
{
  mChangedNormalColor = col;
}

void KSVItem::setChanged (bool val)
{
  mData->setChanged (val);
}

void KSVItem::setOriginalRunlevel (const QString& rl)
{
  mData->setOriginalRunlevel (rl);
}



//-----------------------
// KSVDragList
//-----------------------

KSVDragList::KSVDragList ( QWidget* parent, const char* name )
  : K2ListView (parent, name),
	mDragMenu (new KPopupMenu (this)),
	mDragCopyMenu (new KPopupMenu (this)),
	mDragMoveMenu (new KPopupMenu (this)),
	mItemToDrag (0L)
{
  //DEBUG
  setSelectionModeExt (Konqueror);
  // DEBUG

  QHeader* h = header();
  h->setClickEnabled (false);

  addColumn (i18n("No")); // SortNumber
  setColumnWidthMode (KSVItem::SortNumber, Manual);
  addColumn (""); // Icon
//   setColumnWidthMode (KSVItem::Icon, Manual);
  addColumn (i18n("Name")); // ServiceName
//   setColumnWidthMode (KSVItem::ServiceName, Manual);
  
  h->setResizeEnabled (false, KSVItem::Icon);

  setShowSortIndicator (false);
  setAllColumnsShowFocus (true);

  // multiselection is to complicated due to the sorting numbers
  setMultiSelection(false);

  // enable tooltips
  setDisplayToolTips (true);

//   setPopupMenuAtIndex (1);

  // init DND menu
  mDragMenu->insertTitle (i18n("Drag Menu"));
  mDragMenu->insertItem ("&Copy", Copy);
  mDragMenu->insertItem ("&Move", Move);

  mDragCopyMenu->insertTitle (i18n("Drag Menu"));
  mDragCopyMenu->insertItem ("&Copy", Copy);

  mDragMoveMenu->insertTitle (i18n("Drag Menu"));
  mDragMoveMenu->insertItem ("&Move", Move);

  mRMList.setAutoDelete(true);

  // catch drops
  connect (this, SIGNAL (dropped (QDropEvent*, QListViewItem*)),
		   this, SLOT (drop (QDropEvent*, QListViewItem*)));
}

KSVDragList::~KSVDragList()
{
  delete mDragMenu;
  delete mDragCopyMenu;
  delete mDragMoveMenu;
}

void KSVDragList::initItem (QString file, QString path, QString name, Q_INT8 nr)
{
  KSVItem* tmp = new KSVItem(this, file, path, name, nr);
  tmp->setRunlevel(QObject::name());
  tmp->setOriginalRunlevel(QObject::name());

  tmp->setIcon (mIcon);
//   tmp->setContextMenu (mItemMenu);

  setUpdatesEnabled(false);

  // marked as new in insert, we dont want that
  tmp->setNew(false);
  tmp->setChanged (false);

  setUpdatesEnabled(true);
  repaint(false);
}

void KSVDragList::setDefaultIcon (const QPixmap& icon)
{
  mIcon = icon;
  
  for (QListViewItemIterator it (firstChild()); it.current(); ++it)
    {
	  static_cast <KSVItem*> (it.current())->setIcon (mIcon);
    }
}

void KSVDragList::setNewNormalColor ( const QColor& col )
{
  mNewNormalColor = col;

  KSVItem* item;
  for (QListViewItemIterator it (firstChild());
	   (item = dynamic_cast<KSVItem*>(it.current()));
	   ++it)
    {
      item->setNewNormalColor (mNewNormalColor);
    }
}


Q_INT8 KSVDragList::generateNumber (Q_INT8 high, Q_INT8 low)
{
  assert (high >= low || high == -1);

  Q_INT8 result = -1;
  
  if (low < 0)
	{
	  if (high == -1)
		result = 50;
	  else
		result = high / 2;
	}
  else if (high < 0)
	result = (100 - low) / 2 + low;
  else
	result = (high - low) / 2 + low;

  if (high == result || low == result)
	result = -1;

  return result;
}

void KSVDragList::setChangedNormalColor ( const QColor & col )
{
  mChangedNormalColor = col;
  
  KSVItem* item;
  for (QListViewItemIterator it (firstChild());
	   (item = dynamic_cast<KSVItem*> (it.current()));
	   ++it)
    {
      item->setChangedNormalColor (mChangedNormalColor);
    }
}

KSVItem* KSVDragList::match (const KSVData& data)
{
  KSVItem* res = 0L;

  for (QListViewItemIterator it (this);
	   (res = static_cast<KSVItem*> (it.current()));
	   ++it)
    {
      if (*res->data() == data)
		break;
      else
		res = 0L;
    }

  return res;
}

void KSVDragList::setOrigin (bool val)
{
  mOrigin = val;

  if (mOrigin)
    emit newOrigin();
  else
	clearSelection();
}

void KSVDragList::startDrag ()
{
  mItemToDrag = static_cast<KSVItem*> (currentItem());

  KSVDrag* d = dragObject();
 
  if (d)
	{
	  d->setPixmap (mItemToDrag->paintDragIcon (font(), colorGroup()));

	  if (d->dragMove())
		{
		}
	}
}

void KSVDragList::setItemContextMenu ( QPopupMenu* m )
{
  mItemMenu = m;

//   for (QListViewItemIterator it (firstChild()); it.current(); ++it)
//     {
// 	  static_cast <KSVItem*> (it.current())->setContextMenu (mItemMenu);
//     }
}

KSVDrag* KSVDragList::dragObject ()
{
//   KSVItem* item = dynamic_cast<KSVItem*> (i);

  if (mItemToDrag)
	{
	  return new KSVDrag (*mItemToDrag, this);
	}
  else
	return 0L;
}

bool KSVDragList::acceptDrag (QDropEvent* e) const
{
  e->acceptAction ();
//   kdDebug() << "accepting drag " << e <<": " << e->provides ("application/x-ksysv") << endl;
  return acceptDrops() && e->provides ("application/x-ksysv");
}

void KSVDragList::focusInEvent (QFocusEvent* e)
{
  K2ListView::focusInEvent(e);

  setOrigin(true);
}

void KSVDragList::clearRMList()
{
  mRMList.clear();
}

bool KSVDragList::removeFromRMList (const KSVData& item)
{
  KSVData* res = 0L;
  
  for (QListIterator<KSVData> it (mRMList);
	   it.current();
	   ++it)
	{
	  res = it.current();
	  
	  if (*res == item)
		break;
	  else
		res = 0L;
	}

  if (res)
	return mRMList.remove (res);
  else
	return false;
}

bool KSVDragList::insert (const KSVData& data, const KSVData* above, const KSVData* below)
{
  const int a = above ? above->number() : -1;
  const int b = below ? below->number() : -1;

  Q_INT8 nr = generateNumber (b, a);

  if (nr > -1)
	{
	  KSVData real (data);
	  real.setNumber (nr);

	  KSVItem* item = new KSVItem (this, real);
// 	  item->setContextMenu (mItemMenu);
	  item->setNew (true);

	  return true;
	}
  else
	emit cannotGenerateNumber ();

  return false;
}

bool KSVDragList::insert (const KSVData& data, const KSVItem* where, KSVAction*& action)
{
  const KSVData* above = 0L;
  const KSVData* below = 0L;
  
  if (where)
	{
	  above = where->data();
	  KSVItem* tmp = static_cast<KSVItem*> (where->nextSibling());
	  below = tmp ? tmp->data() : 0L;
	}

  bool success = false;
  KSVItem* exists = match (data);
  action = 0L;

  if (exists)
	{
	  if (exists->data() == above || exists->data() == below)
		return false;

	  Q_INT8 nr = generateNumber (below ? below->number() : -1, above ? above->number() : -1);
	  
	  if (nr == -1)
		{
		  emit cannotGenerateNumber();
		}
	  else
		{
		  KSVData oldState = *exists->data();
		  exists->setNumber (nr);
		  sort();

		  action = new ChangeAction (this, &oldState, exists->data());
		  success = true;
		}
	}
  else
	{
	  success = insert (data, above, below);
	  cerr << success;

	  if (success)
		action = new AddAction (this, match (data)->data());
	}

  return success;
}

void KSVDragList::drop (QDropEvent* e, QListViewItem* after)
{
  kdDebug() << "dropped " << e << " " << after << endl;
  KSVData data;
  KSVDragList* source = static_cast<KSVDragList*> (e->source());
  QPopupMenu* menu = 0L;

  if ((!source) || (!strcmp(source->name(), "Scripts")))
	menu = mDragCopyMenu;
  else if (source == this)
	menu = mDragMoveMenu;
  else
	menu = mDragMenu;

  if (KSVDrag::decodeNative (e, data))
	{
	  e->acceptAction();

	  int res = menu->exec (QCursor::pos(), 1);

	  if (res == -1) // operation cancelled
		return;

	  const bool move = res == Move;

	  KSVItem* tmp = static_cast<KSVItem*> (after);
	  if (!tmp)
		tmp = lastChild();

	  KSVAction* action = 0L;
	  if (insert (data, tmp, action))
		{
		  if (move && source != this)
			{
			  KSVAction* actions[2];
 			  actions [0] = new RemoveAction (source, &data);
 			  actions [1] = action;

			  action = new CompoundAction (actions, 2);
			  delete source->match (data);
			}

		  emit changed();
		  emit undoAction (action);
		}
	}
  else
	{
	  e->ignore();
	}
}

bool KSVDragList::addToRMList (const KSVData& item)
{
  KSVData* res = 0L;

  for (QListIterator<KSVData> it (mRMList);
	   it.current();
	   ++it)
	{
	  res = it.current();
	  
	  if (*res == item)
		break;
	  else
		res = 0L;
	}

  if (!res)
	{
	  mRMList.append (new KSVData(item));
	  return true;
	}
  else
	  return false;
}

KSVItem* KSVDragList::lastChild ()
{
  QListViewItem* result = 0L;

  for (QListViewItemIterator it (this);
	   it.current();
	   ++it)
	{
	  result = it.current();
	}

  return static_cast<KSVItem*> (result);
}

//------------------------
// KSVDrag
//------------------------

KSVDrag::KSVDrag (const KSVData& item, QWidget* source, const char* name)
  : QDragObject (source, name)
{
  QDataStream ds (mNative, IO_ReadWrite);
  ds << item;

  mText = item.toString();

  mURL = "file:" + item.path() + "/" + item.filename();
}

KSVDrag::KSVDrag (const KSVItem& item, QWidget* source, const char* name)
  : QDragObject (source, name)
{
  QDataStream ds (mNative, IO_ReadWrite);
  ds << *item.data();

  mText = item.toString();

  mURL = "file:" + item.path() + "/" + item.filename();
}

KSVDrag::~KSVDrag ()
{
}

const char* KSVDrag::format (int i) const
{
  switch (i)
	{
	case Native:
	  return "application/x-ksysv";
	  break;
	  
	case Text:
	  return "text/plain";
	  break;
	  
	case URL:
	  return "text/uri-list";
	  break;
	  
	default:
	  return 0L;
	}
}

QByteArray KSVDrag::encodedData (const char* format) const
{
  if (!strcmp (format, "application/x-ksysv"))
	{
	  return mNative;
	}
  else if (!strcmp (format, "text/plain"))
	{
	  QByteArray res;
	  QDataStream ds (res, IO_ReadWrite);
	  ds << mText;

	  return res;
	}
  else if (!strcmp (format, "text/uri-list"))
	{
	  QByteArray res;
	  QDataStream ds (res, IO_ReadWrite);
	  ds << mURL;

	  return res;
	}
  else
	{
	  return QByteArray();
	}
}

bool KSVDrag::decodeNative (const QMimeSource* mime, KSVData& data)
{
  if (mime && mime->provides ("application/x-ksysv"))
	{
	  QDataStream ds (mime->encodedData ("application/x-ksysv"), IO_ReadOnly);
	  ds >> data;

	  return true;
	}

  return false;
}
