/*
 *  This file is part of the KDE System Control Tool,
 *  Copyright (C)1999 Thorsten Westheider <twesthei@physik.uni-bielefeld.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 ****************************************************************************/

#include <stdio.h>

#include <qdir.h>
#include <qfile.h>
#include <qtextstream.h>

#include <kapp.h>
#include <klocale.h>
#include <kglobal.h>
#include <kstddirs.h>

#include <X11/Xlib.h>

#include "pcidevice.h"


PCIDevice::PCIDevice(uint bus, uint slot, uint func, uint vendorid, 
		     uint deviceid, baseclass bclass, 
		     pciclass pclass) : _vendorname(""),
		     			_devicename(""),
		     			_bus(bus),
		     			_slot(slot),
					_func(func),
					_vendorid(vendorid),
					_deviceid(deviceid)
{
  _pciclass = (pciclass) ((((uint) bclass) << 8)+((uint) pclass));
  
  lookupData();
 
  // On the long term PCIBaseClasses have to disappear in Device

  if (_pciclass == PCIClassStorageSCSI) _baseclass = BaseClassSCSIController;
  else                                  _baseclass = bclass;
}    


QString  PCIDevice::deviceClassName()
{
  switch (_pciclass)
  {
    case PCIClassUnclassifiedNonVGA    : return i18n("Non-VGA unclassified device");
    case PCIClassUnclassifiedVGA       : return i18n("VGA compatible unclassified device");

    case PCIClassStorageSCSI           : return i18n("SCSI storage controller");
    case PCIClassStorageIDE            : return i18n("IDE interface");
    case PCIClassStorageFloppy         : return i18n("Floppy disk controller");	  
    case PCIClassStorageIPI            : return i18n("IPI bus controller");
    case PCIClassStorageRAID           : return i18n("RAID bus controller");
    case PCIClassStorageOther          : return i18n("Unknown mass storage controller");

    case PCIClassNetworkEthernet       : return i18n("Ethernet controller");
    case PCIClassNetworkTokenring      : return i18n("Token ring network controller");
    case PCIClassNetworkFDDI 	       : return i18n("FDDI network controller");
    case PCIClassNetworkATM  	       : return i18n("ATM network controller");
    case PCIClassNetworkOther	       : return i18n("Network controller");

    case PCIClassDisplayVGA  	       : return i18n("VGA compatible controller");
    case PCIClassDisplayXGA  	       : return i18n("XGA compatible controller");
    case PCIClassDisplayOther	       : return i18n("Display controller");

    case PCIClassMultimediaVideo       : return i18n("Multimedia video controller");
    case PCIClassMultimediaAudio       : return i18n("Multimedia audio controller");
    case PCIClassMultimediaOther       : return i18n("Multimedia controller"); 

    case PCIClassMemoryRAM	       : return i18n("RAM memory");
    case PCIClassMemoryFlash 	       : return i18n("FLASH memory");
    case PCIClassMemoryOther 	       : return i18n("Memory");

    case PCIClassBridgeHost  	       : return i18n("Host bridge");
    case PCIClassBridgeISA	       : return i18n("ISA bridge");
    case PCIClassBridgeEISA  	       : return i18n("EISA bridge");
    case PCIClassBridgeMC	       : return i18n("MicroChannel bridge");
    case PCIClassBridgePCI	       : return i18n("PCI bridge");
    case PCIClassBridgePCMCIA	       : return i18n("PCMCIA bridge");
    case PCIClassBridgeNuBus 	       : return i18n("NuBus bridge");
    case PCIClassBridgeCardBus	       : return i18n("CardBus bridge");
    case PCIClassBridgeOther 	       : return i18n("Bridge");

    case PCIClassCommunicationSerial   : return i18n("Serial controller");
    case PCIClassCommunicationParallel : return i18n("Parallel controller");
    case PCIClassCommunicationOther    : return i18n("Communication controller");

    case PCIClassSystemPIC	       : return i18n("PIC");
    case PCIClassSystemDMA	       : return i18n("DMA controller");
    case PCIClassSystemTimer 	       : return i18n("Timer");
    case PCIClassSystemRTC	       : return i18n("Real time clock");
    case PCIClassSystemOther 	       : return i18n("System peripheral");
 
    case PCIClassInputKeyboard	       : return i18n("Keyboard controller");
    case PCIClassInputPen	       : return i18n("Digitizer Pen");
    case PCIClassInputMouse  	       : return i18n("Mouse controller");
    case PCIClassInputOther  	       : return i18n("Input device controller");

    case PCIClassDockingGeneric	       : return i18n("Generic Docking Station");
    case PCIClassDockingOther	       : return i18n("Docking Station");

    case PCIClassProcessor386	       : return i18n("386");
    case PCIClassProcessor486	       : return i18n("486");
    case PCIClassProcessorPentium      : return i18n("Pentium");
    case PCIClassProcessorAlpha	       : return i18n("Alpha");
    case PCIClassProcessorPowerPC      : return i18n("Power PC");
    case PCIClassProcessorCo 	       : return i18n("Co-processor");

    case PCIClassSerialFirewire	       : return i18n("FireWire (IEEE 1394)");
    case PCIClassSerialACCESS	       : return i18n("ACCESS Bus");
    case PCIClassSerialSSA	       : return i18n("SSA");
    case PCIClassSerialUSB	       : return i18n("USB Controller");
    case PCIClassSerialFiber 	       : return i18n("Fiber Channel");
    
    default                            : return i18n("Unknown device");
  }
}


Driver  *PCIDevice::deviceDriver()
{
  //if (deviceState() != DeviceStateReady) return 0L;
  
  switch (_pciclass)
  {
    case PCIClassDisplayVGA   : 
    case PCIClassDisplayXGA   : 
    case PCIClassDisplayOther : return serverInfo();
    				break;
    default                   : return 0L;
  }
}



QString  PCIDevice::deviceName()
{
  return _devicename;
}


Device::devstate  PCIDevice::deviceState()
{
  Resource  *res;

  /*
   * If the device requires I/O addresses, I guess there must be
   * a range of 2 bytes at least; so if range > 1 that indicates
   * those I/O addresse were found in /proc/ioports and thus the
   * driver for this device must have registered them, that is,
   * the driver is present - otherwise we don't know.
   ***************************************************************/
  
  for (res = firstResource(); res; res = nextResource())
    if ((res->resourceType() == Resource::ResIOAddress) && (res->resourceRange() > 1)) 
      return DeviceStateReady;
    
  return DeviceStateUnknown;
}


QString  PCIDevice::vendorName()
{
  return _vendorname;
}

/*
 * Private methods
 ******************/
 
void  PCIDevice::lookupData()
{
  QString        idfile = locate("data", "ksysctrl/pci.ids");
  QString        entry;
  char           buffer[256];
  FILE           *fhd;
  uint           vid, did;
  
  if ((fhd = fopen(idfile.data(), "r")))
  {
    while (fgets(buffer, sizeof(buffer), fhd))
    {
      entry = QString(buffer);
      
      if (entry.length() > 4)					// Line contains data
      {
        entry = entry.simplifyWhiteSpace();
	
        if ((entry.at(0) != ' ') && (entry.at(0) != '#'))  	// Line contains vendor information
	{
	  sscanf(buffer, "%04x", &vid);
	  
	  if (vid == _vendorid)					// Vendor matches
	  {
            _vendorname = entry.right(entry.length()-4);
	    _vendorname = _vendorname.stripWhiteSpace();
	    
	    while (fgets(buffer, sizeof(buffer), fhd))
            {
	      entry = QString(buffer);
              
	      if (entry.length() > 4)			      	// Line contains data
              {
 
                if (entry.at(0) == '\t')			// Line contains device information
                {
                  entry = entry.simplifyWhiteSpace();
	          entry = entry.stripWhiteSpace();
		  
		  sscanf(entry.data(), "%04x", &did);
		  
		  if (did == _deviceid)				// Device matches
		  {
                    _devicename = entry.right(entry.length()-4);
	            _devicename = _devicename.stripWhiteSpace();
		  
		    break;
		  }
		} 
		else 
		{
		  if ((entry.at(0) != ' ') && (entry.at(0) != '#')) 
		  {
		    _devicename = i18n("Unknown device");
		    break;
		  }
		}		 
	      }	      
	    }
	    
	    break;
	  }
	}
      }
    }
  
    fclose(fhd);
  } else perror("fopen");
}


XServer  *PCIDevice::serverInfo()
{
  Display  *display;
  XServer  *xserver;
  int      release;
  QString  name, relstr, vendor;
  QString  line;
  char     monthname[16];
  int      day, year;
  int      month;
  QDate    date;
  
  if (!(display = XOpenDisplay(0))) return 0L;

  release = VendorRelease(display);
  
  if ((vendor = ServerVendor(display)).contains("XFree86", true))
  {
    // Server is XFree86, so we can rely on some things
    
    name = "XFree86";
    relstr.sprintf("%i.%i.%i", release/100, (release%100)/10, release%10);
  
    QFile  xerr(QDir::homeDirPath()+"/.X.err");
    
    if (xerr.exists())
    {
      if (xerr.open(IO_ReadOnly))
      {
        QTextStream  s(&xerr);
        
        for (int i = 0; i < 5; i++) line = s.readLine(); 

        sscanf(line.data(), "Release Date: %s %i %i", monthname, &day, &year);
        
        for (month = 1; month < 13; month++) 
          if (date.monthName(month) == QString(monthname).left(3)) break;
          
        date = QDate(year, month, day);
      
        xerr.close();
      } 
    } 
  }
  
  xserver = new XServer(name, relstr, vendor, date);
  
  XCloseDisplay(display);

  return xserver;
}
