/*
 *   khexedit - Versatile hex editor
 *   Copyright (C) 1999  Espen Sand, espensa@online.no
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include <qheader.h>
#include <qlabel.h>
#include <qlayout.h>

#include <kapp.h>
#include <kglobal.h>
#include <klocale.h>

#include "fileinfodialog.h"
#include "listview.h"

CFileInfoDialog::CFileInfoDialog( QWidget *parent,const char *name,bool modal)
  :KDialogBase( Plain, i18n("Statistics"), Help|User1|Cancel, User1,
		parent, name, modal, true, i18n("&Update") ),
   mBusy(false), mDirty(false)
{
  setHelp( "khexedit/khexedit.html", QString::null );

  QString text;
  QVBoxLayout *topLayout = new QVBoxLayout( plainPage(), 0, spacingHint() );
  if( topLayout == 0 ) { return; }

  
  QGridLayout *gbox = new QGridLayout( 2, 2, spacingHint() );
  if( gbox == 0 ) { return; }
  topLayout->addLayout( gbox );
  gbox->setColStretch( 1, 10 );
  
  text = i18n("File name: ");
  QLabel *label = new QLabel( text, plainPage() );
  gbox->addWidget( label, 0, 0 );

  text = i18n("Size [bytes]: ");
  label = new QLabel( text, plainPage() );
  gbox->addWidget( label, 1, 0 );

  mFileNameLabel = new QLabel( plainPage() );
  mFileSizeLabel = new QLabel( plainPage() );
  gbox->addWidget( mFileNameLabel, 0, 1 );
  gbox->addWidget( mFileSizeLabel, 1, 1 );

  mFrequencyList = new CListView( plainPage(), "stringList" );
  mFrequencyList->setFont( KGlobal::fixedFont() );

  mFrequencyList->addColumn( i18n("Hexadecimal") );
  mFrequencyList->addColumn( i18n("Decimal") );
  mFrequencyList->addColumn( i18n("Octal") );
  mFrequencyList->addColumn( i18n("Binary") );
  mFrequencyList->addColumn( i18n("Text") );
  mFrequencyList->addColumn( i18n("Occurrence") );
  mFrequencyList->addColumn( i18n("Percent") );
  mFrequencyList->setAllColumnsShowFocus( true );
  mFrequencyList->setFrameStyle( QFrame::WinPanel + QFrame::Sunken );
  topLayout->addWidget( mFrequencyList, 10 );

  mDirtyLabel = new QLabel( plainPage() );
  mDirtyLabel->setFixedHeight( fontMetrics().height() );
  topLayout->addWidget( mDirtyLabel );

  setStatistics();
  setColumnWidth();
  mFrequencyList->setVisibleItem( 15 );

  //
  // Load the first set of data when this timer expires. I do it this 
  // way so that the dialog will be visible when the load operation starts.
  //
  startTimer( 0 );
}


CFileInfoDialog::~CFileInfoDialog( void )
{
}


void CFileInfoDialog::slotUser1( void ) // Update
{
  if( mBusy == true )
  {
    return;
  }
  
  SStatisticControl *sc = new SStatisticControl;
  if( sc == 0 ) { return; }
  
  mBusy = true;  
  emit collectStatistic( *sc );
  mBusy = false;  

  delete sc;
  
}


void CFileInfoDialog::setDirty( void )
{
  if( mDirty == true )
  {
    return;
  }
  
  mDirtyLabel->setText(
    i18n("Warning: Document has been modified since last update"));
  mDirty = true;
}


void CFileInfoDialog::setClean( void )
{
  if( mDirty == false )
  {
    return;
  }
  
  mDirtyLabel->setText("");
  mDirty = false;
}


const char *printBin( uint val )
{
  static char buf[9];
  for( int i = 0; i < 8; i++ )
  {
    buf[7-i] = (val&(1<<i)) ? '1' : '0';
  }
  buf[8] = 0;
  return( buf );
}


void CFileInfoDialog::setStatistics( void ) // Default
{
  setClean();
  mFrequencyList->clear();
  mFileNameLabel->setText("");
  mFileSizeLabel->setText("");

  QString u("?");
  QString d, h, o, b, c;
  QListViewItem *item = 0;

  char buf[10];
  memset( buf, 0, sizeof( buf ) );
  
  for( uint i=0; i<256; i++ )
  { 
    h.sprintf("0x%02x", i );
    d.sprintf("%03d", i );
    o.sprintf("%03o", i );
    b.sprintf("%s", printBin(i) );

    if( QChar((char)i).isPrint() == true )
    {
      c = QChar((char)i);
    }
    else
    {
      c = QChar('.');
    }

    item = new QListViewItem( mFrequencyList, item, h, d, o, b, c, u, u );
    if( i == 0 )
    {
      mFrequencyList->setSelected( item, true );
    }
  }
}



void CFileInfoDialog::setStatistics( SStatisticControl &sc )
{
  setClean();
  mFrequencyList->clear();
  mFileNameLabel->setText( sc.documentName );
  mFileSizeLabel->setText( QString("%1").arg(sc.documentSize));

  QString d, h, o, b, c, n, p;
  QListViewItem *item = 0;
  
  uint size, pre, i;

  for( i=size=0; i<256; i++ )
  { 
    if( sc.occurence[i] > size ) { size = sc.occurence[i]; }
  }
  for( pre = 1; size > 0 ; pre++ )
  {
    size /= 10;
  }

  for( i=0; i<256; i++ )
  { 
    h.sprintf("0x%02x", i );
    d.sprintf("%03d", i );
    o.sprintf("%03o", i );
    b.sprintf("%s", printBin(i) );

    n = QString("%1").arg( sc.occurence[i], pre );
    if( sc.documentSize == 0 )
    {
      p.sprintf("0.00" );
    }
    else
    {
      double val = 100.0*((double)sc.occurence[i]/(double)sc.documentSize);
      p = QString("%1").arg( val, 6, 'f', 2 );
    }

    if( QChar((char)i).isPrint() == true )
    {
      c = QChar((char)i);
    }
    else
    {
      c = QChar('.');
    }

    item = new QListViewItem( mFrequencyList, item, h, d, o, b, c, n, p );
    if( i == 0 )
    {
      mFrequencyList->setSelected( item, true );
    }
  }
}



void CFileInfoDialog::setColumnWidth( void )
{
  const QFontMetrics &fm = mFrequencyList->fontMetrics();
  int w0, w1, w2, w3, w4;

  w0 = -fm.minLeftBearing() - fm.minRightBearing() + 8 + fm.maxWidth();
  w3 = 0;

  w1  = fm.width( mFrequencyList->header()->label(0) ) + w0;
  w2  = fm.width('0')*6;
  w3 += w1 > w2 ? w1 : w2;
  mFrequencyList->setColumnWidth( 0, w1 > w2 ? w1 : w2 );
  
  w1  = fm.boundingRect( mFrequencyList->header()->label(1) ).width() + w0;
  w2  = fm.width('0')*5;
  w3 += w1 > w2 ? w1 : w2;
  mFrequencyList->setColumnWidth( 1, w1 > w2 ? w1 : w2 );

  w1  = fm.boundingRect( mFrequencyList->header()->label(2) ).width() + w0;
  w2  = fm.width('0')*5;
  w3 += w1 > w2 ? w1 : w2;
  mFrequencyList->setColumnWidth( 2, w1 > w2 ? w1 : w2 );

  w1  = fm.boundingRect( mFrequencyList->header()->label(3) ).width() + w0;
  w2  = fm.width('0')*10;
  w3 += w1 > w2 ? w1 : w2;
  mFrequencyList->setColumnWidth( 3, w1 > w2 ? w1 : w2 );

  w1  = fm.boundingRect( mFrequencyList->header()->label(4) ).width() + w0;
  w2  = fm.width('0')*3;
  w3 += w1 > w2 ? w1 : w2;
  mFrequencyList->setColumnWidth( 4, w1 > w2 ? w1 : w2 );
  
  w1  = fm.boundingRect( mFrequencyList->header()->label(5) ).width() + w0;
  w2  = fm.width('0')*10;
  w3 += w1 > w2 ? w1 : w2;
  mFrequencyList->setColumnWidth( 5, w1 > w2 ? w1 : w2 );

  w4 = mFrequencyList->viewport()->width() - w3;
  w1 = fm.boundingRect( mFrequencyList->header()->label(6) ).width() + w0;
  w2 = fm.width('0')*3;
  w1 = w1 > w2 ? w1 : w2;
  mFrequencyList->setColumnWidth( 6, w1 > w4 ? w1 : w4 );
}


void CFileInfoDialog::resizeEvent( QResizeEvent * )
{
  setColumnWidth();
}


void CFileInfoDialog::showEvent( QShowEvent *e )
{
  KDialogBase::showEvent(e);
  setColumnWidth();
  mFrequencyList->setFocus();
}


void CFileInfoDialog::timerEvent( QTimerEvent * )
{
  killTimers();
  slotUser1();
}



