/* This file implements the tab to edit the addresses.
 *
 * the KDE addressbook
 * copyright:  (C) Mirko Sucker, 1998, 1999, 2000
 * mail to:    Mirko Sucker <mirko@kde.org>
 * requires:   recent C++-compiler, at least Qt 2.0
 
 This library is free software; you can redistribute it and/or
 modify it under the terms of the GNU Library General Public
 License as published by the Free Software Foundation; either
 version 2 of the License, or (at your option) any later version.
 
 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Library General Public License for more details.
 
 You should have received a copy of the GNU Library General Public License
 along with this library; see the file COPYING.LIB.  If not, write to
 the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 Boston, MA 02111-1307, USA.
 
 * $Revision: 1.4 $
 */

#include "look_edit_tabaddresses.h"
#include <qcombobox.h>
#include <qpushbutton.h>
#include <qlayout.h>
#include <qlineedit.h>
#include <qlabel.h>
#include <qiconset.h>
#include <qstring.h>
#include <qtooltip.h>
#include <kmessagebox.h>
#include <kiconloader.h>
#include <klocale.h>
#include <kapp.h>

TabAddresses::TabAddresses(QWidget *parent)
  : TabBasic(parent),
    currentItem(0),
    modified(false)
{
  const int NoOfCols=3;
  int style; // currently unused
  int row=0;
  QHBoxLayout *hboxbuttons;
  QHBoxLayout *hboxposition;
  // -----
  layout=new QGridLayout(this, 100, NoOfCols, 3, 2);
  if(layout==0)
    {
      KMessageBox::sorry
	(this, i18n("Out of memory."),
	 i18n("General failure."));
      ::exit(-1);
    }
  for(int count=0; count<NoOfCols; ++count)
    { // make sure all columns get the same width:
      layout->setColStretch(count, 1);
    }
  // ----- create and add the buttons:
  hboxbuttons=new QHBoxLayout(layout);
  hboxbuttons->setDirection(QBoxLayout::LeftToRight);
  newAddress=new QPushButton(this);
  newAddress->setPixmap(BarIcon("filenew"));
  QToolTip::add(newAddress, i18n("Add another address"));
  hboxbuttons->addWidget(newAddress, row, 1);
  delAddress=new QPushButton(this);
  delAddress->setPixmap(BarIcon("delete"));
  QToolTip::add(delAddress, i18n("Delete this address"));
  hboxbuttons->addWidget(delAddress, row, 2);
  hboxbuttons->addStretch();
  layout->addMultiCell(hboxbuttons, row, row, 0, 2);
  ++row; // new row
  // ----- create and add the selector:
  selector=new QComboBox(/* true, */ this);
  connect(selector, SIGNAL(selected(int)), SLOT(highlighted(int)));
  layout->addMultiCellWidget(selector, row, row, 0, 2);
  ++row; // new row
  // ----- create and place the input fields according to country style selected:
  //       place the position fields:
  hboxposition=new QHBoxLayout(layout);
  l_position=new QLabel(i18n("Assignment:"), this);
  hboxposition->addWidget(l_position, 0, 0); // added to the layout directly
  position=new QLineEdit(this);
  connect(position, SIGNAL(textChanged(const QString&)),
	  SLOT(addressChanged(const QString &)));
  hboxposition->addWidget(position, 0, 1); // added to the hboxlayout
  orgdata=new QPushButton(this);
  orgdata->setPixmap(BarIcon("down"));
  hboxposition->addWidget(orgdata, 0, 2);
  QToolTip::add(orgdata, i18n("Edit organisational attributes"));
  /*
    hboxposition->setStretchFactor(orgdata, 1);
    hboxposition->setStretchFactor(l_position, 1);
    hboxposition->setStretchFactor(position, 10000);
  */
  layout->addMultiCell(hboxposition, row, row, 0, 2);
  ++row; // new row
  // ----- the address fields:
  switch(style)
    {
    default: // american style, city state zipcode
      l_street=new QLabel(i18n("Street:"), this);
      layout->addWidget(l_street, row, 0);
      street=new QLineEdit(this);
      layout->addMultiCellWidget(street, row, row, 1, 2);
      connect(street, SIGNAL(textChanged(const QString&)),
	      SLOT(addressChanged(const QString &)));
      ++row; // new row
      l_city=new QLabel(i18n("City"), this);
      layout->addWidget(l_city, row, 0, AlignBottom);
      l_state=new QLabel(i18n("State"), this);
      layout->addWidget(l_state, row, 1, AlignBottom);
      l_zip=new QLabel(i18n("Zipcode"), this);
      layout->addWidget(l_zip, row, 2, AlignBottom);
      city=new QLineEdit(this);
      ++row; // new row
      layout->addWidget(city, row, 0);
      connect(city, SIGNAL(textChanged(const QString&)),
	      SLOT(addressChanged(const QString &)));
      state=new QLineEdit(this);
      layout->addWidget(state, row, 1);
      connect(state, SIGNAL(textChanged(const QString&)),
	      SLOT(addressChanged(const QString &)));
      zip=new QLineEdit(this);
      layout->addWidget(zip, row, 2);
      connect(zip, SIGNAL(textChanged(const QString&)),
	      SLOT(addressChanged(const QString &)));
      ++row; // new row
      l_country=new QLabel(i18n("Country:"), this);
      layout->addWidget(l_country, row, 0);
      country=new QLineEdit(this);
      layout->addMultiCellWidget(country, row, row, 1, 2);
      connect(country, SIGNAL(textChanged(const QString&)),
	      SLOT(addressChanged(const QString &)));
    }
  connect(newAddress, SIGNAL(clicked()), SLOT(addAddress()));
  connect(delAddress, SIGNAL(clicked()), SLOT(deleteAddress()));
  connect(selector, SIGNAL(textChanged(const QString&)),
	  SLOT(headlineChanged(const QString&)));
}

void TabAddresses::storeContents(AddressBook::Entry& entry)
{
  if(modified) storeChanges();
  entry.addresses=addresses;
}

void TabAddresses::setContents(const AddressBook::Entry& entry)
{
  currentItem=0;
  addresses=entry.addresses;
  fillSelector();
  fillFields();
}

void TabAddresses::fillSelector()
{
  std::list<AddressBook::Entry::Address>::iterator pos;
  // -----
  selector->clear();
  for(pos=addresses.begin(); pos!=addresses.end(); ++pos)
    {
      selector->insertItem((*pos).headline);
    }
}

std::list<AddressBook::Entry::Address>::iterator TabAddresses::currentAddress()
{
  std::list<AddressBook::Entry::Address>::iterator pos;
  int index=currentItem;
  // ----- find the current address:
  if(addresses.size()==0)
    {
      return addresses.end();
    } else {
      index=selector->currentItem();
      pos=addresses.begin();
      advance(pos, index);
      return pos;
    }
}

void TabAddresses::fillFields()
{
  std::list<AddressBook::Entry::Address>::iterator pos;
  QLineEdit **lineedits[]= { &street, &zip, &city, &state, &country, &position };
  int Size=sizeof(lineedits)/sizeof(lineedits[0]);
  int count;
  // ----- find the current address:
  pos=currentAddress();
  // ----- fill the fields:
  if(pos==addresses.end())
    {
      for(count=0; count<Size; ++count)
	{
	  (*lineedits[count])->setEnabled(false);
	  (*lineedits[count])->setText("");
	}
    } else {
      QString *strings[]=
      {
	  &(*pos).address, &(*pos).zip, &(*pos).town,
	  &(*pos).state, &(*pos).country, &(*pos).position
      };
      for(count=0; count<Size; ++count)
	{
	  (*lineedits[count])->setEnabled(true);
	  (*lineedits[count])->setText(*strings[count]);
	}
    }
}

void TabAddresses::storeChanges()
{
  std::list<AddressBook::Entry::Address>::iterator pos;
  QLineEdit **lineedits[]= { &street, &zip, &city, &state, &country, &position };
  int Size=sizeof(lineedits)/sizeof(lineedits[0]);
  int count;
  // ----- find the current address:
  pos=currentAddress();
  // ----- store the fields:
  if(pos!=addresses.end())
    {
      QString *strings[]=
      {
	  &(*pos).address, &(*pos).zip, &(*pos).town,
	  &(*pos).state, &(*pos).country, &(*pos).position
      };
      for(count=0; count<Size; ++count)
	{
	  *strings[count]=(*lineedits[count])->text();
	}
    }
}

void TabAddresses::addAddress()
{
  if(modified) storeChanges();
  // WORK_TO_DO: provide menu of defined address types
  addresses.push_back(AddressBook::Entry::Address());
  fillSelector();
  selector->setCurrentItem(selector->count()-1);
  currentItem=selector->currentItem();
  fillFields();
  addressChanged();
}

void TabAddresses::deleteAddress()
{
  int index;
  std::list<AddressBook::Entry::Address>::iterator pos;  
  // -----
  if(addresses.size()==0)
    {
      kapp->beep();
      return;
    }
  if(KMessageBox::questionYesNo
     (this, i18n("Really delete this address?"),
      i18n("Question"))==KMessageBox::Yes)
    {
      index=selector->currentItem();
      if(index<0)
	{
	  kapp->beep();
	  return;
	}
      selector->removeItem(index);
      pos=addresses.begin();
      advance(pos, index);
      addresses.erase(pos);
      fillSelector();
      if(selector->count()>0)
	{
	  selector->setCurrentItem(QMAX(selector->count()-1, index));
	}
      fillFields();
    } else {
      kapp->beep();
    }
}

void TabAddresses::highlighted(int index)
{
  debug("TabAddresses::highlighted: %i (was %i).", index,
	currentItem);
  if(modified)
    {
      storeChanges();
      modified=false;
    }
  currentItem=selector->currentItem();
  fillFields();
}

void TabAddresses::addressChanged()
{
  modified=true;
  emit(changed());
  debug("TabAddresses::addressChanged: address changed.");
}

void TabAddresses::addressChanged(const QString &)
{
  addressChanged();
}

void TabAddresses::headlineChanged(const QString & string)
{
  int index=selector->currentItem();
  std::list<AddressBook::Entry::Address>::iterator pos;
  // -----
  if(index>=0)
    {
      pos=addresses.begin();
      advance(pos, index);
      (*pos).headline=string;
      // selector->changeItem(string, index);
    } else {
      kapp->beep();
    }
}

  

/*
  void TabAddresses::resizeEvent(QResizeEvent* e)
  {
  const int Grid=3;
  int tempx, tempy, x, y, w=width(), h=height();
  // -----
  tempx=newAddress->sizeHint().width();
  tempy=QMAX(newAddress->sizeHint().height(),
  selector->sizeHint().height());
  x=w-Grid-tempx;
  newAddress->setGeometry(x, Grid, tempx, tempy);
  selector->setGeometry(Grid, Grid, x-tempx-Grid, tempy);
  }
*/
