// -*- c++ -*-
/* This file is part of the KDE libraries
    Copyright (C) 1997, 1998 Richard Moore <rich@kde.org>
                  1998 Stephan Kulow <coolo@kde.org>
                  1998 Daniel Grana <grana@ie.iwi.unibe.ch>
		  1999 Carsten Pfeiffer <pfeiffer@kde.org>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
    Boston, MA 02111-1307, USA.
*/

// $Id: kfileviewitem.cpp,v 1.20 2000/04/25 16:51:29 bieker Exp $

#include "kfileviewitem.h"

#include <sys/types.h>
#include <dirent.h>
#include <grp.h>
#include <sys/stat.h>
#include <pwd.h>
#include <unistd.h>
#include <time.h>

#include <qfileinfo.h>
#include <qpixmap.h>
#include <qregexp.h>

#include <kapp.h>
#include <kdebug.h>
#include <kglobal.h>
#include <klocale.h>
#include <kmimetype.h>
#include <kurl.h>
#include "config-kfile.h"

template class QList<KFileViewItem>;

KFileViewItem::IntCache   * KFileViewItem::passwdCache = 0L;
KFileViewItem::IntCache   * KFileViewItem::groupCache  = 0L;
KFileViewItem::GroupCache * KFileViewItem::myGroupMemberships = 0L;

KFileViewItem::KFileViewItem(const QString& baseURL, const KIO::UDSEntry &e)
{
    myIsDir = false;
    myIsFile = true;
    myIsSymLink = false;
    myPermissions = 0755;
    mySize = 0;
    myPixmap = 0L;
    myBaseURL = baseURL;

    KIO::UDSEntry::ConstIterator it = e.begin();
    for( ; it != e.end(); it++ ) {
	switch (( *it ).m_uds) {
	case KIO::UDS_NAME:
	    myName = ( *it ).m_str;
	    break;
	case KIO::UDS_SIZE:
	    mySize = ( *it ).m_long;
	    break;
	case KIO::UDS_MODIFICATION_TIME:
	    myDate_t = ( *it ).m_long;
	    break;
	case KIO::UDS_USER:
	    myOwner = ( *it ).m_str;
	    break;
	case KIO::UDS_GROUP:
	    myGroup = ( *it ).m_str;
	    break;
	case KIO::UDS_ACCESS:
	    myPermissions = ( *it ).m_long;
	    break;
	case KIO::UDS_FILE_TYPE:
	    myIsDir = (( *it ).m_long & S_IFDIR) != 0;
	    myIsFile = !myIsDir;
	    break;
	case KIO::UDS_LINK_DEST:
	    myIsSymLink = (( *it ).m_str.length());
	    break;
	case KIO::UDS_MIME_TYPE:
	case KIO::UDS_ACCESS_TIME:
	case KIO::UDS_CREATION_TIME:
	    break;
	};
    }

    if (myName.at(myName.length() - 1) == '/')
	myName.truncate(myName.length() - 1);
    myIsReadable = true;

    init();
}

KFileViewItem::KFileViewItem(const QString& baseurl, const QString& name, bool delaystat)
{
    myPixmap = 0L;
    myName = name;
    myBaseURL = baseurl;

    myIsDir = false; // assumptions to get a nice default pixmap
    myIsFile = true;
    myIsSymLink = false;
    myIsReadable = true;
    ASSERT( myBaseURL.at(myBaseURL.length() - 1) == '/' || name.at(0) == '/' );

    if (!passwdCache) {
	passwdCache = new IntCache( 317 );
	groupCache = new IntCache( 317 );
	myGroupMemberships = new GroupCache;
	
	passwdCache->setAutoDelete( true );
	groupCache->setAutoDelete( true );
	
	readUserInfo();
	qAddPostRoutine( cleanup );
    }

    init();
    if (!delaystat)
	stat(false);
}

void KFileViewItem::stat(bool alreadyindir)
{
    struct stat buf;
    myIsSymLink = false;
    QCString local8;
    if (alreadyindir)
	local8 = myName.local8Bit();
    else {
	QString fullname = url();
	if (fullname.left(5) == QString::fromLatin1("file:"))
	    fullname = fullname.mid(5);
	local8 = fullname.local8Bit();
    }

    if (lstat(local8, &buf) == 0) {
	myIsDir = S_ISDIR(buf.st_mode) != 0;
        // check if this is a symlink to a directory
	if (S_ISLNK(buf.st_mode)) {
	  myIsSymLink = true;
	  struct stat st;
	  if (::stat(local8, &st) == 0) {
	      myIsDir = S_ISDIR(st.st_mode) != 0;
	  }
	  else {
	      myName = QString::null; // indicate, that the link is broken
	  }
	} else
	    myIsSymLink = false;

	myDate_t = buf.st_mtime;
	mySize = buf.st_size;
	myIsFile = !myIsDir;
	myIsReadable = testReadable( local8, buf );
	
	myPermissions = buf.st_mode;
	myOwner_t = buf.st_uid;
	myGroup_t = buf.st_gid;

	// guess we should update, now that we have the correct information
	if ( !myMimeType ) {
	    delete myPixmap;
	    myPixmap = 0L;
	}

    } else {
	// default
	myName.insert(0, '?');
	mySize = 0;
	myIsFile = false;
	myIsDir = false;
	myIsReadable = false;
	myPermissions = 0;
	parsePermissions(myPermissions);
    }
}

KFileViewItem::KFileViewItem(const KFileViewItem &i)
{
    *this = i;
}

KFileViewItem::~KFileViewItem()
{
    delete myPixmap;
}

KFileViewItem &KFileViewItem::operator=(const KFileViewItem &i)
{
    myName= i.myName;
    myBaseURL= i.myBaseURL;
    myAccess = i.myAccess;
    myDate= i.myDate;
    myDate_t = i.myDate_t;
    myOwner= i.myOwner;
    myGroup= i.myGroup;
    myIsDir = i.myIsDir;
    myIsFile = i.myIsFile;
    myIsSymLink = i.myIsSymLink;
    myPermissions= i.myPermissions;
    mySize= i.mySize;
    myIsReadable = i.myIsReadable;
    myFilePath = i.myFilePath;
    myPixmap = i.myPixmap;
    myMimeType = i.myMimeType;
    myPixmapDirty = i.myPixmapDirty;

    return *this;
}


void KFileViewItem::init()
{
    myNext = 0;
    myMimeType = 0L;
    myHidden = false;
    myPixmapDirty = false;
    myPixmapSize = KIcon::SizeSmall;
}


void KFileViewItem::parsePermissions(const char *perms)
{
    myPermissions = 0;
    char p[11] = {0, 0, 0, 0, 0,
		  0, 0, 0, 0, 0,
		  0};

    strncpy(p, perms, sizeof(p));

    myIsDir = (bool)(p[0] == 'd');
    myIsSymLink = (bool)(p[0] == 'l');
    myIsFile = !myIsDir;

    if(p[1] == 'r')
      myPermissions |= QFileInfo::ReadUser;

    if(p[2] == 'w')
      myPermissions |= QFileInfo::WriteUser;

    if(p[3] == 'x')
      myPermissions |= QFileInfo::ExeUser;

    if(p[4] == 'r')
      myPermissions |= QFileInfo::ReadGroup;

    if(p[5] == 'w')
      myPermissions |= QFileInfo::WriteGroup;

    if(p[6] == 'x')
      myPermissions |= QFileInfo::ExeGroup;

    if(p[7] == 'r')
      myPermissions |= QFileInfo::ReadOther;

    if(p[8] == 'w')
      myPermissions |= QFileInfo::WriteOther;

    if(p[9] == 'x')
      myPermissions |= QFileInfo::ExeOther;
}

QString KFileViewItem::parsePermissions(uint perm) const
{
    char p[] = "----------";

    if (myIsDir)
	p[0]='d';
    else if (myIsSymLink)
	p[0]='l';

    if (perm & QFileInfo::ReadUser)
	p[1]='r';
    if (perm & QFileInfo::WriteUser)
	p[2]='w';
    if (perm & QFileInfo::ExeUser)
	p[3]='x';

    if (perm & QFileInfo::ReadGroup)
	p[4]='r';
    if (perm & QFileInfo::WriteGroup)
	p[5]='w';
    if (perm & QFileInfo::ExeGroup)
	p[6]='x';

    if (perm & QFileInfo::ReadOther)
	p[7]='r';
    if (perm & QFileInfo::WriteOther)
	p[8]='w';
    if (perm & QFileInfo::ExeOther)
	p[9]='x';

    return QString::fromLatin1(p);
}

QString KFileViewItem::date() const {
    if (myDate.isNull())
	myDate = dateTime(myDate_t);
    return myDate;
}

QString KFileViewItem::dateTime(time_t _time) {
    QDateTime t;
    t.setTime_t(_time);
    return KGlobal::locale()->formatDateTime( t );
}

void KFileViewItem::readUserInfo()
{
    struct passwd *pass;
    while ( (pass = getpwent()) != 0L ) {
	passwdCache->insert(pass->pw_uid, qstrdup(pass->pw_name));
    }
    delete pass;
    endpwent();

    char *myUsername = passwdCache->find( getuid() );
    struct group *gr;
    while( (gr = getgrent()) != 0L ) {
	groupCache->insert(gr->gr_gid, qstrdup(gr->gr_name));

	int i = 0;
	char *member = 0L;
	while ( (member = gr->gr_mem[i++]) != 0L ) {
	    if ( myUsername && strcmp( myUsername, member ) == 0 ) {
	        myGroupMemberships->append( gr->gr_gid );
		break;
	    }
	}
    }
    delete gr;
    endgrent();
}


void KFileViewItem::cleanup()
{
    delete passwdCache;
    passwdCache = 0L;

    delete groupCache;
    groupCache = 0L;

    delete myGroupMemberships;
    myGroupMemberships = 0L;
}

QString KFileViewItem::url() const
{
    if (myFilePath.isNull())
	if (isDir())
	    myFilePath = myBaseURL + myName + '/';
	else
	    myFilePath = myBaseURL + myName;
    return myFilePath;
}

QString KFileViewItem::mimeType()
{
    if ( !myMimeType ) {
	myMimeType = KMimeType::findByURL( url(), 0, true );
	
	if ( myIsReadable ) // keep the "locked" icon for unreadable files/dirs
	    myPixmapDirty = true;
    }

    return myMimeType->name();
}


QPixmap KFileViewItem::pixmap( int size ) const
{
    if ( !myPixmapDirty ) {
	if ( myPixmapSize != size || !myPixmap ) {
	    delete myPixmap;
	    if ( !myMimeType )
		myPixmap = new QPixmap( KGlobal::iconLoader()->loadIcon(defaultIcon(),
			KIcon::Desktop, size) );
	    else
		myPixmap = new QPixmap( myMimeType->pixmap( url(), size ));
	}
    }

    else  { // pixmap dirty, we shall load the pixmap according to our mimetype
	QString icon = myMimeType->icon( url(), true );
	if ( icon != defaultIcon() || (size != myPixmapSize) || !myPixmap ) {
	
	    if ( icon.isEmpty() || !myIsReadable )
		icon = defaultIcon();
	
	    delete myPixmap;
	    KIconLoader *loader = KGlobal::iconLoader();
	    myPixmap = new QPixmap( loader->loadIcon(icon, KIcon::Desktop, size) );
	
	    // we either have found the correct pixmap, or there is none,
	    // anyway, we won't ever search for one again
	    myPixmapDirty = false;
	}
    }
    myPixmapSize = size;
    return *myPixmap;
}

QString KFileViewItem::defaultIcon() const
{
    // avoid creating a QString every time this function is called
    static QString folder = QString::fromLatin1("folder");
    static QString lockedfolder = QString::fromLatin1("lockedfolder");
    static QString unknown = QString::fromLatin1("mimetypes/unknown");
    static QString locked  = QString::fromLatin1("locked");
    static QString symlink = QString::fromLatin1("link");

    if ( myIsSymLink ) {
	if ( myIsReadable )
	    if ( myIsDir )
		return folder;
	    else
		return symlink;
	else
	    if ( myIsDir )
		return lockedfolder;
	    else
		return locked;
    }
    else if ( myIsDir ) {
	if ( myIsReadable )
	    return folder;
	else
	    return lockedfolder;
    }
    else {
	if ( myIsReadable )
	    return unknown;
	else
	    return locked;
    }

    return QString::null;
}

void KFileViewItem::setDeleted()
{
    myIsReadable = false;
    myAccess = "**********";
    myPixmapDirty = true; // next pixmap() call will load the "locked" icon
}


// FIXME: what should happen with those items in copy constructor and
// assignment operator???
void KFileViewItem::setViewItem( const KFileView *view, const void *item )
{
    viewItems[ view ] = item;
}

const void *KFileViewItem::viewItem( const KFileView *view ) const
{
    if ( !viewItems.contains( view ) )
	return 0;
    return viewItems[ view ];
}

QString KFileViewItem::access() const {
    if (myAccess.isNull())
      myAccess = parsePermissions(myPermissions);

    return myAccess;
}

QString KFileViewItem::owner() const
{
    static QString unknown = i18n("unknown");

    if (myOwner.isNull()) {
        myOwner = QString::fromLocal8Bit( passwdCache->find(myOwner_t) );

	if (myOwner.isNull())
	    myOwner = unknown;
    }
    return myOwner;
}


QString KFileViewItem::group() const
{
    static QString unknown = i18n("unknown");

    if ( myGroup.isNull() ) {
        myGroup = QString::fromLocal8Bit( groupCache->find(myGroup_t) );

	if (myGroup.isNull())
	    myGroup = unknown;
    }
    return myGroup;
}


// Tests if a file is readable. We don't just call ::access(), because we
// already have a stat-structure and know about the groups.
bool KFileViewItem::testReadable( const QCString& file, struct stat& buf )
{
    if (file.isEmpty())
	return false;

  // check the "others'" permissions first
  if ( (buf.st_mode & (S_IROTH | (myIsDir ? S_IXOTH : 0))) != 0 )
    return true;

  // check if user can read [execute dirs]
  if ( (buf.st_mode & (S_IRUSR | (myIsDir ? S_IXUSR : 0))) != 0 ) {
    if ( buf.st_uid == getuid() )
      return true;
  }

  // check the group permissions, if we have no user permissions
  if ( (buf.st_mode & (S_IRGRP | (myIsDir ? S_IXGRP : 0))) != 0 ) {
    if ( buf.st_gid == getgid() ||
	 myGroupMemberships->find( buf.st_gid ) != myGroupMemberships->end())
      return true;
  }

  return false;
}


/////////////

void KFileViewItemList::append( const KFileViewItem *item )
{
    if ( !item )
	return;

    KFileViewBaseList::append( item );
    dictdirty = true;
}


void KFileViewItemList::clear()
{
    KFileViewBaseList::clear();
    myDict.clear();
    dictdirty = false;
}

const KFileViewItem * KFileViewItemList::findByName( const QString& url ) const
{
    if (dictdirty) {
	KFileViewItemList *that = const_cast<KFileViewItemList*>(this);
	that->myDict.clear();

	QListIterator<KFileViewItem> it(*this);
	for ( ; it.current(); ++it)
	    that->myDict.insert( it.current()->name(), it.current() );
	that->dictdirty = false;
    }

    return myDict.find( url );
}
