/*****************************************************************

Copyright (c) 1996-2000 the kicker authors. See file AUTHORS.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN
AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************/

#include <qpainter.h>
#include <qpixmap.h>
#include <qimage.h>
#include <qapp.h>
#include <qstyle.h>

#include <kstddirs.h>
#include <kglobal.h>
#include <kdebug.h>
#include <kimageeffect.h>
#include <kglobalsettings.h>
#include <kcursor.h>
#include <kapp.h>
#include <kipc.h>

#include "panelbuttonbase.h"
#include "panelbuttonbase.moc"

PanelButtonBase::PanelButtonBase(QWidget *parent, const char *name)
  : QButton(parent, name)
  , _dir(Bottom)
  , _drawArrow(false)
  , _highlight(false)
  , _dohighlight(true)
  , _changeCursorOverItem(true)
  , _tile(QString::null)
{
  setBackgroundMode( PaletteBackground );
  oldCursor = cursor();

  slotSettingsChanged(KApplication::SETTINGS_MOUSE);
  connect(kapp, SIGNAL(settingsChanged(int)), SLOT(slotSettingsChanged(int)));
  kapp->addKipcEventMask(KIPC::SettingsChanged);
}

void PanelButtonBase::slotSettingsChanged(int category)
{
  if (category != KApplication::SETTINGS_MOUSE) return;

  _changeCursorOverItem = KGlobalSettings::changeCursorOverIcon();

    if(!_changeCursorOverItem)
      setCursor(oldCursor);
}

void PanelButtonBase::setTile(const QString& tile)
{
  _tile = tile;
  loadTiles();
  repaint();
}

void PanelButtonBase::setIcon(const QPixmap& p)
{
  QImage tmpImg;
  _iconh = p;

  tmpImg = _iconh.convertToImage();
  tmpImg = KImageEffect::desaturate(tmpImg, 0.6);
  //tmpImg = KImageEffect::toGray(tmpImg);
  _icon.convertFromImage(tmpImg);
  _icon.setMask(*_iconh.mask());

  repaint();
}

void PanelButtonBase::setDrawArrow(bool v)
{
  if (_drawArrow == v) return;
  _drawArrow = v;
  repaint();
}

void PanelButtonBase::setHighlightOnMouseOver(bool v)
{
  if (_dohighlight == v) return;

  _dohighlight = v;
  repaint();
}

void PanelButtonBase::setArrowDirection(Position dir)
{
  if (_dir == dir) return;
  _dir = dir;
  repaint();
}

void PanelButtonBase::resizeEvent(QResizeEvent*)
{
  // optimize: reload only when size really changes
  loadTiles();
  repaint();
}

void PanelButtonBase::enterEvent(QEvent *)
{
  if (_changeCursorOverItem)
    setCursor(KCursor().handCursor());

  _highlight = true;
  QPainter p(this);
  drawButtonLabel(&p);
}

void PanelButtonBase::leaveEvent(QEvent *)
{
  if (_changeCursorOverItem)
    setCursor(oldCursor);

  _highlight = false;
  QPainter p(this);
  drawButtonLabel(&p);
}

void PanelButtonBase::loadTiles()
{
  if(_tile == QString::null)
    _up = _down = QPixmap();

  QString uptile, downtile;

  if (height() < 42)
    {
      uptile = _tile + "_tiny_up.png";
      downtile = _tile + "_tiny_down.png";
    }
  else if (height() < 54)
    {
      uptile = _tile + "_normal_up.png";
      downtile = _tile + "_normal_down.png";
    }
  else
    {
      uptile = _tile + "_large_up.png";
      downtile = _tile + "_large_down.png";
    }

  uptile = KGlobal::dirs()->findResource("tiles", uptile);
  downtile = KGlobal::dirs()->findResource("tiles", downtile);

  QPixmap up(uptile);
  QPixmap down(downtile);

  if (!up.isNull())
    _up = up;
  else
    _up = QPixmap();

  if (!down.isNull())
    _down = down;
  else
    _down = QPixmap();
}

void PanelButtonBase::drawButton(QPainter *p)
{
  if(isDown()){

    if (!_down.isNull())     // draw down tile
      {
        int x = (width() - _down.width())/2;
        int y = (height() - _down.height())/2;
        p->drawPixmap(x, y, _down);
      }
    else // no tile loaded
      {
        p->fillRect(rect(), colorGroup().brush(QColorGroup::Mid));
        p->setPen(Qt::black);
        p->drawLine(0, 0, width()-1, 0);
        p->drawLine(0, 0, 0, height()-1);
        p->setPen(colorGroup().light());
        p->drawLine(0, height()-1, width()-1, height()-1);
        p->drawLine(width()-1, 0, width()-1, height()-1);
      }
  }
  else {

    if (!_up.isNull())  // draw up tile
      {
        int x = (width() - _up.width())/2;
        int y = (height() - _up.height())/2;
        p->drawPixmap(x, y, _up);
      }
    else // no tile loaded
      {
        // no fallback code needed
      }
  }
  drawButtonLabel(p);
}

void PanelButtonBase::drawButtonLabel(QPainter *p)
{
  bool hl;

  if (!_dohighlight)
    hl = true;
  else
    hl = _highlight;

  // draw icon
  if (!hl && !_icon.isNull())
    {
      int x = (width() - _icon.width())/2;
      int y = (height() - _icon.height())/2;
      if (isDown()) {
        x+=2;
        y+=2;
      }
      p->drawPixmap(x, y, _icon);
    }
  else if (hl && !_iconh.isNull())
    {
      int x = (width() - _iconh.width())/2;
      int y = (height() - _iconh.height())/2;
      if (isDown()) {
        x+=2;
        y+=2;
      }
      p->drawPixmap(x, y, _iconh);
    }

  int d = 0;
  if(isDown()) d = 2;

  // draw arrow
  if(_drawArrow && width() > 32 && height() > 32)
    {
      if(_dir == Top)
        QApplication::style().drawArrow(p, Qt::UpArrow, isDown(), 3+d, 3+d,
                                        8, 8, colorGroup(), true);
      else if (_dir == Bottom)
        QApplication::style().drawArrow(p, Qt::DownArrow, isDown(),
                                        3+d, height()-12+d, 8, 8,
                                        colorGroup(), true);
      else if (_dir == Right)
        QApplication::style().drawArrow(p, Qt::RightArrow, isDown(),
                                        width()-12+d, 3+d, 8, 8,
                                        colorGroup(), true);
      else
        QApplication::style().drawArrow(p, Qt::LeftArrow, isDown(), 3+d, 3+d,
                                        8, 8, colorGroup(), true);
    }
}
