/*
 *  desktop -- The 3dfx Desktop Demo 
 *  COPYRIGHT 3DFX INTERACTIVE, INC. 1999
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/* This is a model box */

#include "xforms.h"
#include "mathutil.h"
#include "global.h"
#include "flight.h"
#include "model.h"

#define VECTOR(point, x, y, z) \
  (point)[0] = x; \
  (point)[1] = y; \
  (point)[2] = z

/* Same as above but normalize */
#define NORMAL(normal, x, y, z) \
do \
{ \
  float len = sqrt ((x)*(x) + (y)*(y) + (z)*(z)); \
  (normal)[0] = x / len; \
  (normal)[1] = y / len; \
  (normal)[2] = z / len; \
} while (0)

XVertex keyframe[70][8];
int keytris[12][3] = {
  {0, 1, 2}, {0, 2, 3},
  {1, 5, 6}, {1, 6, 2},
  {5, 4, 7}, {5, 7, 6},
  {4, 0, 3}, {4, 3, 7},
  {4, 5, 1}, {4, 1, 0},
  {3, 2, 6}, {3, 6, 7}
};

static Matrix projection;

void
init_box ()
{
  XVertex base[8];
  int i, j;
  float axis[3];
  Matrix m;
  float theta = 0;

  /* Build a base box */
  VECTOR (base[0].point, -1,  1,  1);
  NORMAL (base[0].norm,  -1,  1,  1);
  VECTOR (base[1].point,  1,  1,  1);
  NORMAL (base[1].norm,   1,  1,  1);
  VECTOR (base[2].point,  1, -1,  1);
  NORMAL (base[2].norm,   1, -1,  1);
  VECTOR (base[3].point, -1, -1,  1);
  NORMAL (base[3].norm,  -1, -1,  1);

  VECTOR (base[4].point, -1,  1, -1);
  NORMAL (base[4].norm,  -1,  1, -1);
  VECTOR (base[5].point,  1,  1, -1);
  NORMAL (base[5].norm,   1,  1, -1);
  VECTOR (base[6].point,  1, -1, -1);
  NORMAL (base[6].norm,   1, -1, -1);
  VECTOR (base[7].point, -1, -1, -1);
  NORMAL (base[7].norm,  -1, -1, -1);

  /* Not really a normal vector but I want it normalized */
  NORMAL (axis, 1, 1, 1);

  /* Now build lots of keyframes */
  /* The velocity is 45 degrees = PI/4 radians per second */
  /* Keyframe rate is 8.75 keyframes per second */
  /* Therefore there should be (360 / 45) * 8.75 = 70 keyframes */
  for (i = 0; i < 70; i++)
  {
    /* Build a rotation matrix */
    RotateMat (m, theta, axis[0], axis[1], axis[2]);
    for (j = 0; j < 8; j++)
    {              
      /* Transform the vertices */
      MatMultVec4x4_3 (keyframe[i][j].point, m, base[j].point);

      /* Transform the normal */
      MatMultVec4x4_3 (keyframe[i][j].norm, m, base[j].norm);      

      /* Set the lighting data for each vertex */
      keyframe[i][j].diffuse = 1.0;

      /* Fixme set color data in GrVertex */
    }    

    theta += DEG_TO_RAD (360.f / 70.f);
  }
  

  InitializeXforms (NULL);
  
  //ViewportMat (viewport, 0.0f, 1280.f, 0.0f, 1024.f, 1.0f, 65535.f);
  /*SetClipVolume (0.0f, 1280.f, 0.0f, 1024.f, 1024.f, 1.0f, 65535.f);*/
  //PerspectiveMat (frustum, DEG_TO_RAD(60.f), 1280.f / 1024.f, 1.0f, 65535.f);
  //MatMultMat4x4 (projection, viewport, frustum);

  ScaleMat (projection, 12, 12, 12);
  //MatMultMat4x4 (projection, projection, scale);
  /* FIXME: I should setup some sort of camera to do this right */
  //TranslateMat (translate, 0, 0, -50);
  //MatMultMat4x4 (projection, scale, translate);
}

void
print_vertex (XVertex *vert)
{
   printf ("(%.2f %.2f %.2f %.2f) ", vert->xformed_vert.vertex.x,
	   vert->xformed_vert.vertex.y,
	   vert->xformed_vert.vertex.z,
	   1.f / vert->xformed_vert.vertex.oow
	   );
}

/* This function renders a model */
void
render_model (XVertex *verts, int nverts, int tris[][3], int ntris)
{
  int i;
  float fred;
  FxU8 red;
  /* FIXME: load texture */

  /* Solid red for now */

  for (i = 0; i < ntris; i++)
  {
    fred = verts[tris[i][0]].diffuse;
    red  = (FxU8)(fred * 255);
    grConstantColorValue (0xFF000000 | red);

    grDrawTriangle (&verts[tris[i][0]].xformed_vert.vertex,
		    &verts[tris[i][1]].xformed_vert.vertex,
		    &verts[tris[i][2]].xformed_vert.vertex);

#ifdef TRACE_VERTS
    print_vertex (&verts[tris[i][0]]);
    print_vertex (&verts[tris[i][1]]);
    print_vertex (&verts[tris[i][2]]);
    printf ("\n");
#endif
  }
}

#define LERP_VECTOR(result, a, b, t) \
do { \
  result[0] = a[0] * t + b[0] * (1 - t); \
  result[1] = a[1] * t + b[1] * (1 - t); \
  result[2] = a[2] * t + b[2] * (1 - t); \
} while (0)

#define LERP_VERTEX(result, a, b, t) \
do { \
  LERP_VECTOR (result.point, a.point, b.point, t); \
  LERP_VECTOR (result.norm,  a.norm,  b.norm,  t); \
  result.diffuse = a.diffuse * t + b.diffuse * (1 - t); \
  result.specular = a.specular * t + b.specular * (1 - t); \
} while (0)

/* This builds a frame of the geometry from keyframes and 
 * transforms it via the current view matrix and renders it.
 */
void
render_frame (Demo_State *state)
{
  int i;
  int frame1, frame2;
  float float_frame = 0;
  static float total_time = 0;
  float weight;
  Matrix xform, flight_mat;

  /* FIXME: I will have to dynamically allocate this to size */
  XVertex newframe[8];  

  /* Use the current time to determine what frame we are on */
  /* FIXME: This assumes that a frame gets rendered every display
   * frame or that if render_frame is not called then the animation
   * is paused.  Is this the desired behavior?
   */
  /* FIXME: I need an animation state that remembers its start time
   * and the last frame time and such so that it can determine the
   * real delta time for the animation.
   */
  total_time += (state->delta_time) / 1000000.f;

  /* Determine which frame I should be on. */
  /* FIXME: this assumes the keyframe rate is 8.75 frames per second.
   * This should be read from an animation state.
   */
  float_frame = total_time * 8.75f;
  /* Now truncate to an integer.  This gets the first of the two frame we
   * will use.
   */
  frame1 = (int)float_frame;

  /* Now use the fractional part as the weight for the other frame */
  weight = 1 - (float_frame - (float)frame1);

  /* If the frame runs over the max frame count then wraparround */
  /* FIXME: this assumes the animation loops.  It would probably be
   * be better to alert someone
   */
  /* FIXME: this assumes 70 keyframes */
  if (frame1 >= 70)
    frame1 = frame1 % 70;
  frame2 = frame1 + 1;
  if (frame2 >= 70)
    frame2 -= 70;

  /* Now Blend the keyframes together */
  for (i = 0; i < 8; i++)
  {
    LERP_VERTEX (newframe[i], keyframe[frame1][i], keyframe[frame2][i],weight);
  }


  /* FIXME: if I was not using matrix data then I could lerp the rotations
   * on the data.  But now I have to lerp the transformed positions.
   */

  /* I need to work in the translation to the current flight position */
  TranslateMat (flight_mat, state->flight->X[0], 
		state->flight->X[1], state->flight->X[2]);
  MatMultMat4x4 (xform, projection, flight_mat);

  /* Work in the projection */
  MatMultMat4x4 (xform, state->project, xform);
  BeginXforms ();
  SetXformsMatrix (xform);
  XformXVertex (newframe, 8);
  EndXforms ();

  render_model (newframe, 8, keytris, 12);
}



/* These are bounding box functions */
void
make_new_bbox (BBox *bbox, Vector v)
{
  /* Both verts become v */
  bbox->BLL[0] = bbox->FUR[0] = v[0];
  bbox->BLL[1] = bbox->FUR[1] = v[1];
  bbox->BLL[2] = bbox->FUR[2] = v[2];
}

void
add_point_to_bbox (BBox *bbox, Vector v)
{
  int i;

  for (i = 0; i < 3; i++)
  {
    if (v[i] < bbox->BLL[i])  
      bbox->BLL[i] = v[i];
    if (v[i] > bbox->FUR[i])
      bbox->FUR[i] = v[i];
  }
}

void
scale_model_to_bbox_matrix (Matrix scale, Model *m, BBox *desired)
{
  /* The dimensions of the model's bounding box */
  float mdeltax, mdeltay, mdeltaz;
  /* The dimensions of the desired bounding box */
  float ddeltax, ddeltay, ddeltaz;
  /* Scaling factors */
  float sx, sy, sz, s;

  /* Compute the deltas */
  mdeltax = m->bbox.FUR[0] - m->bbox.BLL[0];
  mdeltay = m->bbox.FUR[1] - m->bbox.BLL[1];
  mdeltaz = m->bbox.FUR[2] - m->bbox.BLL[2];

  ddeltax = desired->FUR[0] - desired->BLL[0];
  ddeltay = desired->FUR[1] - desired->BLL[1];
  ddeltaz = desired->FUR[2] - desired->BLL[2];

  if (mdeltax == 0)
    sx = 1;
  else
    sx = ddeltax / mdeltax;
  if (mdeltay == 0)
    sy = 1;
  else
    sy = ddeltay / mdeltay;
  if (mdeltaz == 0)
    sz = 1;
  else
    sz = ddeltaz / mdeltaz;

  /* which is smallest */
  s = sx;
  if (sy < s)
    s = sy;
  if (sz < 2)
    s = sz;

  /* Now build the scaling matrix */
  printf ("sx = %f, sy = %f, sz = %f\n", sx, sy, sz);
  ScaleMat (scale, s, s, s);
}

