/*
 *  desktop -- The 3dfx Desktop Demo 
 *  COPYRIGHT 3DFX INTERACTIVE, INC. 1999
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <string.h>
#include "gxffile.h"

#define PATH_SEPARATOR '/'

// fills in the IFF file chunk by using the data pointer
// returns the offset (in bytes) to increment data to point to the next chunk
int GetChunk(IFF_Chunk *chunk, unsigned char *data)
{
	memcpy(&chunk->header, data, sizeof(IFF_Header));
	chunk->data = data + sizeof(IFF_Header);

	return ((sizeof(IFF_Header) + chunk->header.data_size + 1) & ~1);
}


// **** GXFTexMap Functions ****

GXFTexMap::GXFTexMap(IFF_Chunk *chunk)
{
	unsigned char *data;
	int i;

	m_valid = m_num_texmaps = 0;

	m_texmap_chunk = chunk;
			
	data = chunk->data;

	m_num_texmaps = *(int *)data;
	data += 4;

	if (m_num_texmaps == 0) return;

	m_texmaps = new GXFTexMapRecord *[m_num_texmaps];
	if (!m_texmaps)
	{
		return;
	}

	for (i=0; i<m_num_texmaps; i++)
	{
		m_texmaps[i] = (GXFTexMapRecord *) data;	// record position of this texmap
		data += sizeof(GXFTexMapRecord);			// skip to next texmap
	}

	m_valid = 1;

}

// **** GXFMaterial Functions ****

GXFMaterial::GXFMaterial(IFF_Chunk *chunk)
{
	unsigned char *data;
	int i;

	m_valid = m_num_materials = 0;

	m_material_chunk = chunk;
			
	data = chunk->data;

	m_num_materials = *(int *)data;
	data += 4;

	if (m_num_materials == 0) return;

	m_materials = new GXFMaterialRecord *[m_num_materials];
	if (!m_materials)
	{
		return;
	}

	for (i=0; i<m_num_materials; i++)
	{
		m_materials[i] = (GXFMaterialRecord *) data;	// record position of this material
		data += sizeof(GXFMaterialRecord);				// skip to next material
	}

	m_valid = 1;

}

void GXFMaterial::GetDiffuseRGB(float *r, float *g, float *b, int index) // return the d RGB for the nth material
{
	*r = m_materials[index]->diffuse_r;
	*g = m_materials[index]->diffuse_g;
	*b = m_materials[index]->diffuse_b;
}

void GXFMaterial::GetSpecularRGB(float *r, float *g, float *b, int index) // return the s RGB for the nth material
{
	*r = m_materials[index]->specular_r;
	*g = m_materials[index]->specular_g;
	*b = m_materials[index]->specular_b;
}


float GXFMaterial::GetSpecularR(int index) // return the s red for the nth material
{
	return m_materials[index]->specular_r;
}



// **** GXFObjectNode Functions ****


GXFObjectNode::GXFObjectNode(IFF_Chunk *chunk)
{
	int i;
	unsigned char *data;

	m_valid = m_num_object_nodes = 0;

	m_object_node_chunk = chunk;
			
	data = chunk->data;

	m_num_object_nodes = *(int *)data;
	data += 4;
	if (!m_num_object_nodes)
	{
		return;
	}

	m_object_nodes = new GXFObjNodeRecord *[m_num_object_nodes];
	if (!m_object_nodes)
	{
		return;
	}

	for (i=0; i<m_num_object_nodes; i++)
	{
		m_object_nodes[i] = (GXFObjNodeRecord *) data;
		data += sizeof(GXFObjNodeRecord);     // collect pointers to object node records
	}

	m_valid = 1;
}


// **** GXFObject Functions ****


GXFObject::GXFObject(IFF_Chunk *chunk)
{
	int i,j;
	unsigned char *data;

	m_valid = m_num_objects = 0;

	m_object_chunk = chunk;
			
	data = chunk->data;

	m_num_objects = *(int *)data;
	data += 4;
	if (!m_num_objects)
	{
		return;
	}

	m_objects = new GXFObjectRecord[m_num_objects];
	if (!m_objects)
	{
		return;
	}

	for (i=0; i<m_num_objects; i++)
	{
		m_objects[i].object_header = (GXFObjectRecordHdr *) data;
		data += sizeof(GXFObjectRecordHdr);  // now point to vertices

		GXFObjectRecordHdr *t_obj_hdr = m_objects[i].object_header;

		m_objects[i].vertices =  (GXFVertex *) data;
		data += sizeof(GXFVertex) * t_obj_hdr->num_verts;

		m_objects[i].normals = (GXFVertex *) data;
		data += sizeof(GXFVertex) * t_obj_hdr->num_verts;

		if (t_obj_hdr->flags & 0x0ff)
		{
			m_objects[i].num_tverts = *(int *)data;		// read in number of texture vertices
			data += 4;
			m_objects[i].t_vertices =  (GXFUVRecord *) data;
			data += sizeof(GXFUVRecord) * m_objects[i].num_tverts;
		}
		else
		{
			m_objects[i].num_tverts = 0;
			m_objects[i].t_vertices =  NULL;
		}

		m_objects[i].num_meshes = *(int *)data;
		data += 4;

		if (!m_objects[i].num_meshes)
			return;

		m_objects[i].meshes = new GXFMeshRecord[m_objects[i].num_meshes];

		if (!m_objects[i].meshes)
			return;

		for (j = 0; j < m_objects[i].num_meshes; j++)
		{
			GXFMeshRecordHdr *t_head = (GXFMeshRecordHdr *) data;
			m_objects[i].meshes[j].mesh_header = (GXFMeshRecordHdr *) data;
			data += sizeof(GXFMeshRecordHdr); // now pointing to triangles

			int num_tris = m_objects[i].meshes[j].mesh_header->num_triangles;

			m_objects[i].meshes[j].triangles = (GXFTriangle *) data;

			data += sizeof(GXFTriangle) * num_tris;

			if (t_obj_hdr->flags & 0x0ff) // skip over texture indices
				data += sizeof(GXFTriangle) * num_tris;

			// we should be pointing to the next mesh at this time...

		}




	}

	m_valid = 1;
}

unsigned int GXFObject::GetFlags(int index)			// get object n's flags
{
	return m_objects[index].object_header->flags;
}

int GXFObject::GetNumVerts(int index)				// get number of vertices in object n
{
	return m_objects[index].object_header->num_verts;
}

int GXFObject::GetNumTVerts(int index)				// get number of texture vertices in object n
{
	return m_objects[index].num_tverts;
}

GXFVertex * GXFObject::GetVertexList(int index)		// return list of vertices for n'th object
{
	return m_objects[index].vertices;
}

GXFVertex * GXFObject::GetNormalList(int index)		// return list of normals for n'th object
{
	return m_objects[index].normals;
}

GXFUVRecord * GXFObject::GetTVertexList(int index)	// return list of texture vertices for n'th object
{
	return m_objects[index].t_vertices;
}


int GXFObject::GetNumMeshes(int index)				// number of meshes in this object
{
	return m_objects[index].num_meshes;
}


int GXFObject::GetMeshMaterialID(int index, int mesh_index)	// material id of nth mesh in nth object
{
	return m_objects[index].meshes[mesh_index].mesh_header->material_id;
}

int GXFObject::GetNumPrimitives(int index, int mesh_index) // number of primitives in...
{
	return m_objects[index].meshes[mesh_index].mesh_header->num_primitives;
}

int GXFObject::GetPrimitiveType(int index, int mesh_index, int prim_index) // type of primitives in...
{
	return m_objects[index].meshes[mesh_index].mesh_header->primitive_type;
}


int GXFObject::GetMeshNumTriangles(int index, int mesh_index) // number of triangles in...
{
	return m_objects[index].meshes[mesh_index].mesh_header->num_triangles;
}

GXFTriangle * GXFObject::GetMeshTriangleList(int index, int mesh_index) // only 1 primitive so this hacks
{
	return m_objects[index].meshes[mesh_index].triangles;
}


// **** GXFAnimation Functions ****

GXFAnimation::GXFAnimation(IFF_Chunk *chunk)
{
	unsigned char *data;
	int i;

	m_valid = m_num_frames = m_num_object_nodes = 0;

	m_animation_chunk = chunk;
			
	data = chunk->data;

	m_num_frames = *(int *)data;
	data += 4;

	m_num_object_nodes = *(int *)data;
	data += 4;

	if (m_num_frames == 0 || m_num_object_nodes == 0) return;

	m_frames = new GXFAnimationRecord *[m_num_frames];
	if (!m_frames)
	{
		return;
	}

	for (i=0; i<m_num_frames; i++)
	{
		m_frames[i] = (GXFAnimationRecord *) data;	// record position of this frame
		data += (sizeof(GXFAnimationRecord) * m_num_object_nodes); // skip to next frame

	}

	m_valid = 1;

}

GXFAnimation::~GXFAnimation()
{
}

float GXFAnimation::GetMatrixElement(int frame, int object_index, int element) // return mat[element] of the nth object's nth frame
{
	GXFAnimationRecord *t_anim = m_frames[frame] + object_index;
	return t_anim->mat[element];
}



// **** GXFNode Functions ****


GXFNode::GXFNode(IFF_Chunk *chunk)
{
	unsigned char *data;
	int i;
	IFF_Chunk sub_chunk;

	m_valid = m_num_nodes = 0;

	m__node_chunk = chunk;
			
	data = chunk->data;

	m_num_nodes = *(int *)data;
	data += 4;

	if (m_num_nodes == 0) return;

	m_nodes = new GXFNodeRecord[m_num_nodes];
	if (!m_nodes)
	{
		return;
	}

	for (i=0; i<m_num_nodes; i++)
	{
		m_nodes[i].node = (GXFNodeRecordHdr *) data;	// record position of this node
		data += sizeof(GXFNodeRecordHdr);				// lets see if there is anim data

		if (m_nodes[i].node->flags & HAS_ANIMATION_CHUNK) // yup!
		{
			data += GetChunk(&sub_chunk, data);
			m_nodes[i].animation = new GXFAnimation(&sub_chunk); // scary, huh?
			if (!m_nodes[i].animation->IsValid())
			{
				delete m_nodes[i].animation;
				return;
			}		
		}
	}

	m_valid = 1;

}

// **** GXFCamera Functions ****

GXFCamera::GXFCamera(IFF_Chunk *chunk)
{
	unsigned char *data;
	int i;

	m_valid = m_num_scene_frames = m_num_cameras = 0;

	m_camera_chunk = chunk;
			
	data = chunk->data;

	m_num_scene_frames = *(int *)data;
	data += 4;

	m_num_cameras = *(int *)data;
	data += 4;

	if (m_num_scene_frames == 0) return;

	if (m_num_cameras)
	{
		
		m_anim_cameras = new GXFAnimCamera *[m_num_cameras];
		if (!m_anim_cameras)
		{
			return;
		}
		
		for (i=0; i<m_num_cameras; i++)
		{
			m_anim_cameras[i] = (GXFAnimCamera *) data; // record position of this camera
			
			m_anim_cameras[i]->num_camera_frames = *(int *)data;
						
			data += 4;
			m_anim_cameras[i]->camera_frames = (GXFCameraFrame *) data;
			data += (sizeof(GXFCameraFrame) * m_anim_cameras[i]->num_camera_frames); // skip to next camera
			
		}
		
	}
	
	m_valid = 1;

}
int GXFCamera::GetNumAnimFrames(int index)	// number of animation frames in each camera
{
	return m_anim_cameras[index]->num_camera_frames;
}

float GXFCamera::GetFOV(int index, int frame_num)			// return the FOV for the nth frame of the nth camera
{

	return m_anim_cameras[index]->camera_frames[frame_num].fov;
}

float GXFCamera::GetDistance(int index, int frame_num)	// return the camera distance to target for the...
{
	return m_anim_cameras[index]->camera_frames[frame_num].dist;
}

void GXFCamera::GetU(GXFVector *v_u, int index, int frame_num)	// return the u vector for the...
{

	GXFVector3 *t_vec = &m_anim_cameras[index]->camera_frames[frame_num].u;


	SET_GXF_VECTOR(*v_u, (*t_vec)[0], (*t_vec)[1], (*t_vec)[2], 0.0f);
	
}

void GXFCamera::GetV(GXFVector *v_v, int index, int frame_num)	// return the v...
{

	GXFVector3 *t_vec = &m_anim_cameras[index]->camera_frames[frame_num].v;

	SET_GXF_VECTOR(*v_v, (*t_vec)[0], (*t_vec)[1], (*t_vec)[2], 0.0f);

}

void GXFCamera::GetN(GXFVector *v_n, int index, int frame_num)	// return the n...
{	

	GXFVector3 *t_vec = &m_anim_cameras[index]->camera_frames[frame_num].n;

	SET_GXF_VECTOR(*v_n, (*t_vec)[0], (*t_vec)[1], (*t_vec)[2], 0.0f);

}

void GXFCamera::GetPos(GXFVector *v_p, int index, int frame_num)// return the pos...
{

	GXFVector3 *t_vec = &m_anim_cameras[index]->camera_frames[frame_num].pos;

	SET_GXF_VECTOR(*v_p, (*t_vec)[0], (*t_vec)[1], (*t_vec)[2], 1.0f);

}


// **** GXFLight Functions ****


GXFLight::GXFLight(IFF_Chunk *chunk)
{
	unsigned char *data;
	int i;

	m_valid = m_num_lights = 0;

	m_lights_chunk = chunk;
			
	data = chunk->data;

	m_num_lights = *(int *)data;
	data += 4;

	if (m_num_lights)
	{
		
		m_lights = new GXFLightRecord *[m_num_lights];
		if (!m_lights)
		{
			return;
		}
		
		for (i=0; i<m_num_lights; i++)
		{
			m_lights[i] = (GXFLightRecord *) data;	// record position of this light
			data += sizeof(GXFLightRecord);		// skip to next light
		}
		
	}

	m_valid = 1;

}


int GXFLight::GetLightType(int index) // return the light type;
{
	return m_lights[index]->type;
}

void GXFLight::GetDiffuseRGB(float *r, float *g, float *b, int index) // return the RGB for the nth light
{

	*r = m_lights[index]->r;
	*r = m_lights[index]->g;
	*r = m_lights[index]->b;

}


float GXFLight::GetAttenStart(int index) // return the attenuation start for the nth light;
{
	return m_lights[index]->atten_start;
}

float GXFLight::GetAttenEnd(int index) // return the attenuation end for the nth light;
{
	return m_lights[index]->atten_end;
}


void GXFLight::GetU(GXFVector *v_u, int index)	// return the u vector for the...
{

	GXFVector3 *t_vec = &m_lights[index]->u;

	SET_GXF_VECTOR(*v_u, (*t_vec)[0], (*t_vec)[1], (*t_vec)[2], 0.0f);
	
}

void GXFLight::GetV(GXFVector *v_v, int index)	// return the v vector for the...
{

	GXFVector3 *t_vec = &m_lights[index]->v;

	SET_GXF_VECTOR(*v_v, (*t_vec)[0], (*t_vec)[1], (*t_vec)[2], 0.0f);
	
}

void GXFLight::GetN(GXFVector *v_n, int index)	// return the n vector for the...
{

	GXFVector3 *t_vec = &m_lights[index]->n;

	SET_GXF_VECTOR(*v_n, (*t_vec)[0], (*t_vec)[1], (*t_vec)[2], 0.0f);
	
}

void GXFLight::GetPos(GXFVector *v_pos, int index)	// return the pos vector for the...
{

	GXFVector3 *t_vec = &m_lights[index]->pos;

	SET_GXF_VECTOR(*v_pos, (*t_vec)[0], (*t_vec)[1], (*t_vec)[2], 1.0f);
	
}

GXFFile::GXFFile(const char *filename)
{
	FILE *fp;
	GXF_Header header;
	IFF_Chunk chunk;
	unsigned char *file_data, *data;
	int i, j;

	m_valid = 0;

	m_error_string = NULL;

	gxf_texmaps = NULL;
	gxf_materials = NULL;
	gxf_objects = NULL;
	gxf_object_nodes = NULL;
	gxf_nodes = NULL;
	gxf_cameras = NULL;
	gxf_lights = NULL;

	fp = fopen(filename, "rb");
	if (!fp)
	{
		m_error_string = "Couldn't open file";
		return;
	}

	fseek(fp, SEEK_SET, 0);

	// file header
	if (fread(&header, sizeof(GXF_Header), 1, fp) != 1)
	{
		fclose(fp);
		m_error_string = "Failed to read file";
		return;
	}

	if (header.form != FORM_ID || header.form_id != GXF_ID)
	{
		fclose(fp);
		m_error_string = "Not a recognized GXF file";
		return;
	}

	// the file size includes the form_id, which has already been read in
	header.file_size -= 4;

	m_error_string = "Not enough memory to allocate file";

	// allocate memory for the rest of the file
	file_data = new unsigned char[header.file_size];
	if (!file_data)
	{
		fclose(fp);
		return;
	}

	// read in the rest of the file
	if (fread(file_data, header.file_size, 1, fp) != 1)
	{
		delete [] file_data;
		fclose(fp);
		return;
	}

	// we're done with the file, close it
	fclose(fp);

	// find the path of the GXF file (if specified)
	// the same path is used when loading in textures
	file_path[0] = '\0';
	for (i=strlen(filename); i>=0; i--)
	{
		if (filename[i] == PATH_SEPARATOR)
		{
			// includes the last '\\'
			for (j=0; j<=i; j++)
			{
				file_path[j] = filename[j];
			}
			file_path[j] = '\0';
			break;
		}
	}

	// data points to the beginning of the first chunk
	data = file_data;

	while (data < file_data + header.file_size)
	{
		data += GetChunk(&chunk, data);

		switch (chunk.header.chunk_id)
		{
			case VERS_ID: // version string
				if (!ParseVersionString(&chunk))
				{
					m_error_string = "Wrong GXF version, need 1.92";
					goto ERROR_CLEANUP;
				}
				break;

			case MAPL_ID: // texture maps
				gxf_texmaps = new GXFTexMap(&chunk);
				if (!gxf_texmaps || !gxf_texmaps->IsValid())
				{
					m_error_string = "Error loading texture maps";
					goto ERROR_CLEANUP;
				}
				break;

			case MATL_ID: // materials
				gxf_materials = new GXFMaterial(&chunk);
				if (!gxf_materials || !gxf_materials->IsValid())
				{
					m_error_string = "Error loading materials";
					goto ERROR_CLEANUP;
				}

				break;

			case OBJL_ID: // objects
				gxf_objects = new GXFObject(&chunk);
				if (!gxf_objects || !gxf_objects->IsValid())
				{
					m_error_string = "Error loading objects";
					goto ERROR_CLEANUP;
				}
				break;

			case OBNL_ID: // nodes
				gxf_object_nodes = new GXFObjectNode(&chunk);
				if (!gxf_object_nodes || !gxf_object_nodes->IsValid())
				{
					m_error_string = "Error loading object nodes";
					goto ERROR_CLEANUP;
				}
				break;

			case NDEL_ID: // models
				gxf_nodes = new GXFNode(&chunk);
				if (!gxf_nodes || !gxf_nodes->IsValid())
				{
					m_error_string = "Error loading nodes";
					goto ERROR_CLEANUP;
				}
				break;

			case CAML_ID: // cameras
				gxf_cameras = new GXFCamera(&chunk);
				if (!gxf_cameras || !gxf_cameras->IsValid())
				{
					m_error_string = "Error loading cameras";
					goto ERROR_CLEANUP;
				}
				break;

			case LITL_ID: // lights
				gxf_lights = new GXFLight(&chunk);
				if (!gxf_lights || !gxf_lights->IsValid())
				{
					m_error_string = "Error loading lights";
					goto ERROR_CLEANUP;
				}
				break;

			default: // ???? unknown chunk ????
				m_error_string = "Unknown chunk type";
				goto ERROR_CLEANUP;
				break;
		}
	}

	m_error_string = "file succesfully loaded";

	m_valid = 1;
	return;

ERROR_CLEANUP:
	if (file_data)
	{
		delete [] file_data;
	}

	return;
}

int GXFFile::ParseVersionString(IFF_Chunk *chunk)
{
	unsigned char *data;

	data = chunk->data;

	if (strncmp((char *)data, "3dfx Export Version 1.92", chunk->header.data_size))
	{
		return 0;
	}

	return 1;
}
