/*
 *  desktop -- The 3dfx Desktop Demo 
 *  COPYRIGHT 3DFX INTERACTIVE, INC. 1999
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/* This file makes use of Bob's gxffile library to load gxf data */
#include "mathutil.h"
#include "gxffile.h"
#include "gxf.h"
#include "global.h"
#include "lighting.h"

#include <string.h>

Vector LightPos = {-0.4, 0.4, 0};
/* points at 0, 0, -5 */
Vector LightDir = {-0.07949, -0.07949, 0.99366};
Vector CameraDir = {0, 0, -1};

/* This structure is used for scaling texture
 * coordinates
 */
typedef struct TVertScale TVertScale;
struct TVertScale
{
  float s_scale;
  float t_scale;
};

/* This is the scale lookup table */
TVertScale tvscale[] = {
  /* 8x1 */
  {256.f,  32.f},
  {256.f,  64.f},
  {256.f, 128.f},
  {256.f, 256.f},
  {128.f, 256.f},
  { 64.f, 256.f},
  { 32.f, 256.f}
};

/* These two macros do the lookup, they try work with the
 * different defines for glide
 */
#if (defined GLIDE3_DEBUG  || !defined USE_GLIDE3)
#define s_scale(aspect) (tvscale[aspect].s_scale)
#define t_scale(aspect) (tvscale[aspect].t_scale)
#else
#define s_scale(aspect) (tvscale[-aspect - GR_ASPECT_LOG2_1x8].s_scale)
#define t_scale(aspect) (tvscale[-aspect - GR_ASPECT_LOG2_1x8].t_scale)
#endif

static void
build_bounding_boxes (Scene *scene);

static void
LoadTexture (Demo_State *state, char *name, Texture *texture)
{
  Gu3dfInfo info;
  int i = 0;
  FxU32 size;

  /* Assume the last 3 characters of the name are tga and change them
   * to 3df
   */
  int index = strlen (name) - 3;
  name[index++]='3';
  name[index++]='d';
  name[index++]='f';

  gu3dfGetInfo (name, &info);
  info.data = malloc (info.mem_required);

  gu3dfLoad (name, &info);

  /* Download the mip map */
#ifndef USE_GLIDE3
  texture->texinfo.smallLod = info.header.small_lod;
  texture->texinfo.largeLod = info.header.large_lod;
  texture->texinfo.aspectRatio = info.header.aspect_ratio;
  texture->texinfo.format = info.header.format;
  texture->texinfo.data = info.data;
  size = grTexCalcMemRequired (info.header.small_lod,
			       info.header.large_lod,
			       info.header.aspect_ratio,
			       info.header.format);
#else
  texture->texinfo.smallLodLog2 = info.header.small_lod;
  texture->texinfo.largeLodLog2 = info.header.large_lod;
  texture->texinfo.aspectRatioLog2 = info.header.aspect_ratio;
  texture->texinfo.format = info.header.format;
  texture->texinfo.data = info.data;

  size = grTexTextureMemRequired (GR_MIPMAPLEVELMASK_BOTH, &texture->texinfo);
#endif

  if (state->texloc + size < state->texmax)
  {
    grTexDownloadMipMap (GR_TMU0, state->texloc, GR_MIPMAPLEVELMASK_BOTH,
    			 &(texture->texinfo));
    texture->offset = state->texloc;

    state->texloc += size;
  }
  else
  {
    printf ("failed to download %d (%s): no room size = %ld, texloc = %ld, "
	    "min = %ld, max = %ld\n", 
	    i, name, size, state->texloc, state->texmin, state->texmax);
  }

  free (info.data);
}

static void
read_anim_matrix (Matrix m, GXFAnimation *anim, int frame, int obj_index)
{
  m[0][0] = anim->GetMatrixElement (frame, obj_index, 0);
  m[1][0] = anim->GetMatrixElement (frame, obj_index, 1);
  m[2][0] = anim->GetMatrixElement (frame, obj_index, 2);
  m[3][0] = 0;

  m[0][1] = anim->GetMatrixElement (frame, obj_index, 3);
  m[1][1] = anim->GetMatrixElement (frame, obj_index, 4);
  m[2][1] = anim->GetMatrixElement (frame, obj_index, 5);
  m[3][1] = 0;
      
  m[0][2] = anim->GetMatrixElement (frame, obj_index, 6);
  m[1][2] = anim->GetMatrixElement (frame, obj_index, 7);
  m[2][2] = anim->GetMatrixElement (frame, obj_index, 8);
  m[3][2] = 0;
  
  m[0][3] = anim->GetMatrixElement (frame, obj_index, 9);
  m[1][3] = anim->GetMatrixElement (frame, obj_index, 10);
  m[2][3] = anim->GetMatrixElement (frame, obj_index, 11);
  m[3][3] = 1;  
}

/* Read the GXF data in from a file into a Model */
Scene *
open_gxf (Demo_State *state, char *filename)
{
  GXFNode *node;
  GXFObjectNode *object_node;
  GXFObject *object;
  GXFMaterial *material;
  GXFTexMap *texmap;
  GXFAnimation *anim = NULL;
  char *path;
  char texname[200];

  int 
    scene_node_count,
    node_count,
    vert_count, 
    mesh_count, 
    tri_count,
    real_tri_count,
    frame_count,
    tex_count;
  int object_index;
  unsigned long int 
    nnodes,
    ntverts, 
    nverts, 
    ntris,
    ntexmaps;

  unsigned long int node_flags;
  GXFVertex *verts, *normals;
  GXFUVRecord *tverts;
  GXFTriangle *tris;
  XVertex *xverts;
  int material_id;
  Model *m;
  Scene *s;

  GXFFile *gxf = new GXFFile (filename);

  path = gxf->GetFilePath ();

  /* FIXME: check the validity of everything else */
  if (!gxf->IsValid())
  {
    printf ("could not open %s\n", filename);
    return (0);
  }
  s = (Scene*)malloc (sizeof (Scene));


  /* Get interfaces to the data */
  node = gxf->GetNodeInterface ();
  object_node = gxf->GetObjectNodeInterface ();
  object = gxf->GetObjectInterface ();
  material = gxf->GetMaterialInterface ();

  /* Process all of the nodes */
  nnodes = node->GetNumNodes ();
  s->nmodels = nnodes;
  s->models = (Model*) malloc (sizeof (Model) * nnodes);

  for (scene_node_count = 0, m = s->models; (unsigned int)scene_node_count < nnodes; scene_node_count++, m++)
  {   
    node_flags = node->GetNodeFlags (scene_node_count);
    if (node_flags & HAS_ANIMATION_CHUNK)
    {
      anim = node->GetGXFAnimation (scene_node_count);
      if (!anim->IsValid ())
	printf ("bad animation\n");

      /* Get the number of frames */
      m->nframes = anim->GetNumFrames ();
    }
    else
    {
      m->nframes = 0;
    }

    /* First Get all of the textures */
    texmap = gxf->GetTexMapInterface ();
    ntexmaps = texmap->GetNumTexMaps ();
    m->textures = (Texture*) malloc (sizeof (Texture) * ntexmaps);
    m->ntextures = ntexmaps;
    for (tex_count = 0; tex_count < (int)ntexmaps; tex_count++)
    {
      sprintf (texname, "%s%s", path, 
	       texmap->GetTexMapName (tex_count));
      LoadTexture (state, texname, &m->textures[tex_count]);
    }

    /* Allocate some space */
    m->nnodes = object_node->GetNumObjectNodes ();
    m->nodes = (Model_Node*)malloc (sizeof (Model_Node) * m->nnodes);

    /* Process each Object Node */
    for (node_count = 0; node_count < object_node->GetNumObjectNodes (); node_count++)
    {     
      m->nodes[node_count].parent_model = m;
      m->nodes[node_count].name = object_node->GetObjectNodeName (node_count);
      /* I don't care so much about the object nodes right now so I am
	 just grabbing the index of the object. */
      object_index = object_node->GetObjectIndex (node_count);
      
      /* Get the matrix */
      m->nodes[node_count].xform [0][0] = object_node->GetObjectMatrixElement (node_count, 0);
      m->nodes[node_count].xform [1][0] = object_node->GetObjectMatrixElement (node_count, 1);
      m->nodes[node_count].xform [2][0] = object_node->GetObjectMatrixElement (node_count, 2);
      m->nodes[node_count].xform [3][0] = 0;

      m->nodes[node_count].xform [0][1] = object_node->GetObjectMatrixElement (node_count, 3);
      m->nodes[node_count].xform [1][1] = object_node->GetObjectMatrixElement (node_count, 4);
      m->nodes[node_count].xform [2][1] = object_node->GetObjectMatrixElement (node_count, 5);
      m->nodes[node_count].xform [3][1] = 0;
      
      m->nodes[node_count].xform [0][2] = object_node->GetObjectMatrixElement (node_count, 6);
      m->nodes[node_count].xform [1][2] = object_node->GetObjectMatrixElement (node_count, 7);
      m->nodes[node_count].xform [2][2] = object_node->GetObjectMatrixElement (node_count, 8);
      m->nodes[node_count].xform [3][2] = 0;
      
      m->nodes[node_count].xform [0][3] = object_node->GetObjectMatrixElement (node_count, 9);
      m->nodes[node_count].xform [1][3] = object_node->GetObjectMatrixElement (node_count, 10);
      m->nodes[node_count].xform [2][3] = object_node->GetObjectMatrixElement (node_count, 11);
      m->nodes[node_count].xform [3][3] = 1;

      /* Get all of the animation matrices */
      if (node_flags & HAS_ANIMATION_CHUNK)
      {
	m->nodes[node_count].frames = (Matrix*) malloc (sizeof (Matrix) * m->nframes);
	for (frame_count = 0; frame_count < m->nframes; frame_count++)
	{
	  read_anim_matrix (m->nodes[node_count].frames[frame_count],
			    anim,
			    frame_count,
			    node_count);
	}      
      }
      else
      {
	m->nodes[node_count].frames = NULL;
      }

      /* Now grab the verts */
      nverts = object->GetNumVerts (object_index);
      ntverts = object->GetNumTVerts (object_index);
      
      m->nodes[node_count].verts = 
	(XVertex*)malloc (sizeof (XVertex) * nverts);
      m->nodes[node_count].tverts = 
	(TexCoord*)malloc (sizeof (TexCoord) * ntverts);

      /* Copy the vertex data into my vertices */
      verts   = object->GetVertexList (object_index);
      normals = object->GetNormalList (object_index);
      tverts  = object->GetTVertexList (object_index);
      
      m->nodes[node_count].nverts = nverts;
      m->nodes[node_count].ntverts = ntverts;
      /* Shorthand */
      xverts = m->nodes[node_count].verts;
      
      /* Now process all of the verts.  Build a bounding box while
       * reading the verts */
      for (vert_count = 0; vert_count < (int)nverts; vert_count++)
      {      
	/* Copy the position data */
	memcpy (xverts[vert_count].point, &verts[vert_count], 
		sizeof (float) * 3);
	/* Copy the normal data */
	memcpy (xverts[vert_count].norm, &normals[vert_count],
		sizeof (float) * 3);
      }

      /* Now process all of the texture coordinates */
      for (vert_count = 0; vert_count < (int)ntverts; vert_count++)
      {
	/* Copy the texture vertices */
	m->nodes[node_count].tverts[vert_count].s = (tverts[vert_count].s);
	/* TGA file are upside down, instead of the flipping them I just
	   flip my t coordinates */
	m->nodes[node_count].tverts[vert_count].t = (1 - tverts[vert_count].t); 
#ifdef TRACE_VERTS 
	printf ("%f %f\n", m->nodes[node_count].tverts[vert_count].s,
		m->nodes[node_count].tverts[vert_count].t); 
#endif 
      }
      
      
      /* Get the meshes */
      m->nodes[node_count].nmeshes =
	object->GetNumMeshes(object_index);    
      m->nodes[node_count].meshes = (Mesh*) malloc (sizeof (Mesh) *
						    m->nodes[node_count].nmeshes);
      for (mesh_count = 0; mesh_count < (int)m->nodes[node_count].nmeshes;
	   mesh_count++)
      {
	/* Get the triangles */
	ntris = object->GetMeshNumTriangles (object_index, mesh_count);
	/* Allocate space to hold the tris and texture tris */
	m->nodes[node_count].meshes[mesh_count].triangles = 
	  (Triangle*)malloc (sizeof (Triangle) * ntris);
	m->nodes[node_count].meshes[mesh_count].textris =
	  (Triangle*)malloc (sizeof (Triangle) * ntris);
	
	real_tri_count = 0;
	tris = object->GetMeshTriangleList (object_index, mesh_count);
	/* FIXME: how many texture tris are there? */
	for (real_tri_count = 0, tri_count = 0; 
	     real_tri_count < (int)ntris; 
	     real_tri_count++, tri_count++)
	{
	  
	  m->nodes[node_count].meshes[mesh_count].triangles[real_tri_count].a =
	    tris[tri_count].v0;
	  m->nodes[node_count].meshes[mesh_count].triangles[real_tri_count].b =
	    tris[tri_count].v1;
	  m->nodes[node_count].meshes[mesh_count].triangles[real_tri_count].c =
	    tris[tri_count].v2;
	  
	  if (object->GetFlags(object_index) & 0xff)
	  {	  
	    /* The next triangle specifies the texture coordinates */
	    tri_count++;
	    m->nodes[node_count].meshes[mesh_count].textris[real_tri_count].v =	  
	      1;
	    m->nodes[node_count].meshes[mesh_count].textris[real_tri_count].a =
	    tris[tri_count].v0;
	    m->nodes[node_count].meshes[mesh_count].textris[real_tri_count].b =
	      tris[tri_count].v1;
	    m->nodes[node_count].meshes[mesh_count].textris[real_tri_count].c =
	      tris[tri_count].v2;	  
	  }
	  else
	  {
	    m->nodes[node_count].meshes[mesh_count].textris[tri_count].v =	  
	      0;
	  }
	}
	m->nodes[node_count].meshes[mesh_count].ntriangles = real_tri_count;      

	
#ifdef TRACE_VERTS
	/* Print out all of the triangles */
	for (tri_count = 0; tri_count < ntris; tri_count++)
	{
	  printf ("tri: (%d %d %d)\n", 
		  m->nodes[node_count].meshes[mesh_count].triangles[tri_count].a,
		  m->nodes[node_count].meshes[mesh_count].triangles[tri_count].b,
		  m->nodes[node_count].meshes[mesh_count].triangles[tri_count].c);
	}
#endif
	
	/* Get the materials */
	material_id = 
	  object->GetMeshMaterialID(object_index, mesh_count);
	if (material_id < 0)
	{
	  printf ("bad material\n");
	  m->nodes[node_count].meshes[mesh_count].diffuse_map = -1;
	}
	else
	{
	  /* assume no texturing */
	  /*m->nodes[node_count].meshes[mesh_count].diffuse_map = -1;*/
	  
	  /* Get the Diffuse map */
	  m->nodes[node_count].meshes[mesh_count].diffuse_map = 
	    material->GetTextureID (material_id, DIFFUSE_TEXTURE);
	  if (m->nodes[node_count].meshes[mesh_count].diffuse_map == -1)
	  {
	    

	  }
	  else
	  {
	    m->nodes[node_count].meshes[mesh_count].diffuse_wrap = 
	      material->GetTextureWrapping (material_id, DIFFUSE_TEXTURE);

	    material->GetDiffuseRGB (&m->nodes[node_count].meshes[mesh_count].diffuse_r,
				     &m->nodes[node_count].meshes[mesh_count].diffuse_g,
				     &m->nodes[node_count].meshes[mesh_count].diffuse_b,
				     material_id);
	    material->GetSpecularRGB (&m->nodes[node_count].meshes[mesh_count].specular_r,
				     &m->nodes[node_count].meshes[mesh_count].specular_g,
				     &m->nodes[node_count].meshes[mesh_count].specular_b,
				     material_id);
	    m->nodes[node_count].meshes[mesh_count].specular_R = 
	      material->GetSpecularR (material_id);

	    /* Get the relevant material properties */
	  }
	}
      }
    }

    /* Now that all of the object nodes exist we can build the hierarchy */  
    /* Process each Object Node */
    /* 0 is the root so we can just start searching children */
    for (node_count = 0; node_count < object_node->GetNumObjectNodes (); node_count++)
    {
      int child;
      int nchildren = 0;

      /* This is a two pass operation.  First I go throug the children and count
       * them.  Then I allocate space to store the children.  Finally I 
       * enumerate the children and store.
       */
      /* 1: count */
      child = object_node->GetObjectFirstChild (node_count);
      if (child != -1)
      {
	nchildren++;
	while ((child = object_node->GetObjectSibling (child)) != -1)
	  nchildren++;
      }
      
      /* 2: Allocate */
      m->nodes[node_count].nchildren = nchildren;

      if (nchildren == 0)
      {
	m->nodes[node_count].children = NULL;
      }
      else
      {
	int i = 0;

	m->nodes[node_count].children = (Model_Node**)malloc (sizeof (Model_Node*) * nchildren);	

	/* 3: store */
	child = object_node->GetObjectFirstChild (node_count);	
	do
	{
	  m->nodes[node_count].children[i] = &m->nodes[child];
	  i++;
	} while ((child = object_node->GetObjectSibling (child)) != -1);
      }	
    }
  }

  /* Max's vertices are a bit weird.  It relies on the
   * transforms to keep everything together so the 
   * parts can be a bit spread out.  Therefore I transform
   * to the rest position before building a bounding box 
   */
  build_bounding_boxes (s);

  //exit (-1);
  return (s);
}

/* Destroy a Model */


/* Scale a gxf model to be the right size */


static void
bbox_node (Model *m, Model_Node *model_node, Matrix current, Matrix composite)
{
  Vector result;
  int i;

  if (m->nframes)
  {
    MatMultMat4x4 (composite, current, model_node->frames[0]);
  }
  else
  {
    MatMultMat4x4 (composite, current, model_node->xform);
  }

  /* Now transform all of the vertices by composite and 
   * add the result to the box 
   */
  for (i = 0; i < (int)model_node->nverts; i++)
  {
    /* Transform */
    MatMultVec4x4_3 (result, composite, model_node->verts[i].point);

    /* add the point */
    add_point_to_bbox (&m->bbox, result);
  }
}

static void
build_bounding_boxes_recursive (Model *m, Model_Node *model_node, Matrix current)
{
  Matrix composite;
  int i;

  /* Use this current matrix for the current node */
  bbox_node (m, model_node, current, composite);
  
  /* Now process the children with the composite matrix */
  if (model_node->children)
  {
    for (i = 0; i < (int)model_node->nchildren; i++)
    {
      build_bounding_boxes_recursive (m, model_node->children[i], composite);
    }
  }
}

static void
build_bounding_boxes (Scene *scene)
{
  Matrix I;
  int i;
  Vector result;

  /* Get an idetity matrix */
  IdentityMat (I);

  /* Do each model in the scene */
  for (i = 0; i < scene->nmodels; i++)
  {
    /* I need to initialize the box with a good point */
    MatMultVec4x4_4 (result, scene->models[i].nodes[0].xform, scene->models[i].nodes[0].verts[0].point); 
    make_new_bbox (&scene->models[i].bbox, result);

    /* Start the recursion */
    build_bounding_boxes_recursive (&scene->models[i], &scene->models[i].nodes[0], I);
  }
}

extern void
print_vertex (XVertex *vert);

static void
draw_gxf_node (Demo_State *state, Model *m, Model_Node *model_node, int *textured, Matrix current, Matrix composite, int frame)
{
  unsigned long int 
    meshes,
    tris;
  Mesh *mesh;
  Triangle *tri;
  Triangle *textri;
  Matrix xform, world;
  int verts;
  GrAspectRatio_t aspect;

  GrTextureClampMode_t clamp_mode[2] = {
    GR_TEXTURECLAMP_CLAMP, GR_TEXTURECLAMP_WRAP
  };

  //#define TRACE_VERTS
#ifdef TRACE_VERTS
  printf ("drawing %s %d verts\n", model_node->name, model_node->nverts);
#endif

  /* Transform and light the vertices */    
  if (frame < 0)
  {
    MatMultMat4x4 (composite, current, model_node->xform);
  }
  else
  {
    MatMultMat4x4 (composite, current, model_node->frames[frame]);
  }
  MatMultMat4x4 (world, state->model, composite);
  //MatMultMat4x4 (world, state->model, world);
  MatMultMat4x4 (xform, state->project, world);
  BeginXforms ();
  SetXformsMatrix (xform);
  XformXVertex (model_node->verts, model_node->nverts);
  EndXforms ();
  DirLightXforms (world, LightDir, CameraDir);

  //#define TRACE_VERTS
#ifdef TRACE_VERTS
  for (int j = 0; j < 4; j++)
  {
    for (int i = 0; i < 4; i++)
      printf ("%f ", xform[j][i]);
    printf ("\n");
  }

  for (verts = 0; verts < model_node->nverts; verts++)
  {
    Vector temp;

    printf ("%f %f %f\n", model_node->verts[verts].point[0],
	    model_node->verts[verts].point[1],
	    model_node->verts[verts].point[2]);
    
    MatMultVec4x4_3 (temp, world, model_node->verts[verts].point);
    printf ("pre-projection: %f %f %f\n", temp[0], temp[1], temp[2]);
    
    print_vertex (&(model_node->verts[verts]));
    
    printf ("\n");
  }
#endif

  /* Setup for iterated alpha */
  grAlphaCombine (GR_COMBINE_FUNCTION_LOCAL,
		  GR_COMBINE_FACTOR_NONE,
		  GR_COMBINE_LOCAL_ITERATED,
		  GR_COMBINE_OTHER_NONE,
		  FXFALSE);

  /* Set the diffuse color to the alpha chanel */
  for (verts = 0; verts < (int)model_node->nverts; verts++)
  {
    model_node->verts[verts].xformed_vert.vertex.a = 
      model_node->verts[verts].diffuse * 255;
  }

  /* Draw each mesh */
  for (meshes = 0, mesh = model_node->meshes; 
       meshes < model_node->nmeshes; meshes++, mesh++)
  {
    /* FIXME: Assume texture has already been downloaded to card */
    if (mesh->diffuse_map >= 0)
    {      
      /* Turn on texturing if it isn't already on */
      /* FIXME: make this work accross calls */
      if (!(*textured))
      {
	grTexCombine (GR_TMU0, GR_COMBINE_FUNCTION_LOCAL, GR_COMBINE_FACTOR_NONE,
		      GR_COMBINE_FUNCTION_LOCAL, GR_COMBINE_FACTOR_NONE,
		      FXFALSE, FXFALSE);
	/* Scale by alpha for lighting effect */
	grColorCombine (GR_COMBINE_FUNCTION_SCALE_OTHER, GR_COMBINE_FACTOR_LOCAL_ALPHA,
			GR_COMBINE_LOCAL_NONE, GR_COMBINE_OTHER_TEXTURE,
			  FXFALSE);
	*textured = 1;
      }
      
      grTexSource (GR_TMU0, m->textures[mesh->diffuse_map].offset,
		     GR_MIPMAPLEVELMASK_BOTH, 
		   &(m->textures[mesh->diffuse_map].texinfo));
      //printf ("offset = 0x%x\n", m->textures[mesh->diffuse_map].offset);
      grTexClampMode (GR_TMU0, clamp_mode[mesh->diffuse_wrap & 0x1],
		      clamp_mode[mesh->diffuse_wrap >> 1]);
#ifndef USE_GLIDE3
      aspect = m->textures[mesh->diffuse_map].texinfo.aspectRatio;
#else
      aspect = m->textures[mesh->diffuse_map].texinfo.aspectRatioLog2;
#endif

      /* Draw each triangle */
      for (tris = 0, tri = mesh->triangles, textri = mesh->textris; 
	   tris < mesh->ntriangles; 
	   tris++, tri++, textri++)
      {
	/* Get the appropriate texture coordinates */
	model_node->verts[tri->a].xformed_vert.vertex.tmuvtx[0].sow = 
	  model_node->tverts[textri->a].s * 
	  model_node->verts[tri->a].xformed_vert.vertex.oow *
	  s_scale (aspect);
	model_node->verts[tri->a].xformed_vert.vertex.tmuvtx[0].tow = 
	  model_node->tverts[textri->a].t *
	  model_node->verts[tri->a].xformed_vert.vertex.oow *
	  t_scale (aspect);
	model_node->verts[tri->b].xformed_vert.vertex.tmuvtx[0].sow = 
	  model_node->tverts[textri->b].s *
	  model_node->verts[tri->b].xformed_vert.vertex.oow *
	  s_scale (aspect);
	model_node->verts[tri->b].xformed_vert.vertex.tmuvtx[0].tow = 
	  model_node->tverts[textri->b].t *
	  model_node->verts[tri->b].xformed_vert.vertex.oow *
	  t_scale (aspect);
	model_node->verts[tri->c].xformed_vert.vertex.tmuvtx[0].sow = 
	  model_node->tverts[textri->c].s *
	  model_node->verts[tri->c].xformed_vert.vertex.oow *
	  s_scale (aspect);
	model_node->verts[tri->c].xformed_vert.vertex.tmuvtx[0].tow = 
	  model_node->tverts[textri->c].t *
	  model_node->verts[tri->c].xformed_vert.vertex.oow *
	  t_scale (aspect);

	/* Draw the triangle */
	grDrawTriangle (&(model_node->verts[tri->a].xformed_vert.vertex),
			&(model_node->verts[tri->b].xformed_vert.vertex),
			&(model_node->verts[tri->c].xformed_vert.vertex));
      }
    }
    else
    {
      if (*textured)
      {
	/* Setup alpha times iterated rgb */
	grColorCombine (GR_COMBINE_FUNCTION_SCALE_OTHER, 
			GR_COMBINE_FACTOR_LOCAL_ALPHA,
			GR_COMBINE_LOCAL_NONE,
			GR_COMBINE_OTHER_ITERATED,
			FXFALSE);
      }
      /* Setup iterated rgb mode? */
      *textured = 0;
     
      /* Draw each triangle */
      for (tris = 0, tri = mesh->triangles, textri = mesh->textris; 
	   tris < mesh->ntriangles; 
	   tris++, tri++, textri++)
      { 
	model_node->verts[tri->a].xformed_vert.vertex.r = 
	  mesh->diffuse_r;
	model_node->verts[tri->a].xformed_vert.vertex.g = 
	  mesh->diffuse_g;
	model_node->verts[tri->a].xformed_vert.vertex.b = 
	  mesh->diffuse_b;

	model_node->verts[tri->b].xformed_vert.vertex.r = 
	  mesh->diffuse_r;
	model_node->verts[tri->b].xformed_vert.vertex.g = 
	  mesh->diffuse_g;
	model_node->verts[tri->b].xformed_vert.vertex.b = 
	  mesh->diffuse_b;

	model_node->verts[tri->c].xformed_vert.vertex.r = 
	  mesh->diffuse_r;
	model_node->verts[tri->c].xformed_vert.vertex.g = 
	  mesh->diffuse_g;
	model_node->verts[tri->c].xformed_vert.vertex.b = 
	  mesh->diffuse_b;

	/* Draw the triangle */
	grDrawTriangle (&(model_node->verts[tri->a].xformed_vert.vertex),
			&(model_node->verts[tri->b].xformed_vert.vertex),
			&(model_node->verts[tri->c].xformed_vert.vertex));	  	
      }
    }   
  }  
}

/* Draw a Model */
void
draw_gxf (Demo_State *state, Model *m)
{
  unsigned long int 
    nodes;

  Model_Node *model_node;
  int textured = 0;
  Matrix t;
  static int frame = -1;

  if (m->nframes)
  {
    frame++;
    if (frame >= m->nframes)
      frame = 1;
    else if (frame <= 0)
      frame = 1;
  }

  /* Traverse all nodes */
  for (nodes = 0, model_node = m->nodes; nodes < m->nnodes; nodes++, model_node++)
  {
    draw_gxf_node (state, m, model_node, &textured, state->identity, t, frame);
  }
}

static void
draw_gxf_hierarchy_recursive (Demo_State *state, Model *m, Model_Node *model_node, int *textured, Matrix current, int frame)
{
  int i;
  Matrix t;

  /* First draw this node */
  draw_gxf_node (state, m, model_node, textured, current, t, frame);
  /* Now draw all of the children */
  if (model_node->children)
  {
    for (i = 0; i < (int)model_node->nchildren; i++)
    {
      draw_gxf_hierarchy_recursive (state, m, model_node->children[i], textured, t, frame);
    }
  }
}

void
draw_gxf_hierarchy (Demo_State *state, Model *m, int frame)
{
  int textured = 0;

  if (m->nframes)
  {
    frame++;
    if (frame >= m->nframes)
      frame = 1;
    else if (frame <= 0)
      frame = 1;
  }
  /* Draw the hierarchy recursively using node 0 as the root node */
  draw_gxf_hierarchy_recursive (state, m, &m->nodes[0], &textured, state->identity, frame);  
}

void
draw_scene (Demo_State *state, Scene *s, int frame)
{
  int i;

  for (i = 0; i < s->nmodels; i++)
  {
    //draw_gxf (state, &s->models[i]);
    draw_gxf_hierarchy (state, &s->models[i], frame);
  }
}

