/*****************************************************************************/
/*  file_transfer.c - contains the ui and network routines for file transfer */
/*  Copyright (C) 1998-1999 Brian Masney <masneyb@seul.org>                  */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either version 2 of the License, or        */
/*  (at your option) any later version.                                      */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA      */
/*****************************************************************************/

#include "ftp.h"

#define FTP_ASCII 	0
#define FTP_BINARY	1

struct recursive_transfer {
   struct ftp_transfer_data *tdata;
   char *path;
   char *dest;
   char *root_path;
   char *start_file;
};

static GtkWidget *dialog, *status;
static int skip_same_state;
static struct ftp_transfer_data *inittrans(int direction);
static void get_local_files_and_dirs(struct recursive_transfer *transdata);
static void asktrans(struct ftp_transfer_data *tdata);
static void trans_selectrow (GtkCList *clist, gint row, gint column, GdkEventButton *event, struct ftp_transfer_data *tdata);
static void trans_unselectrow (GtkCList *clist, gint row, gint column, GdkEventButton *event, struct ftp_transfer_data *tdata);
static void trans_selectall (GtkWidget *widget, struct ftp_transfer_data *tdata);
static void trans_unselectall (GtkWidget *widget, struct ftp_transfer_data *tdata);
static void overwrite (GtkWidget *widget, struct ftp_transfer_data *tdata);
static void resume (GtkWidget *widget, struct ftp_transfer_data *tdata);
static void skip (GtkWidget *widget, struct ftp_transfer_data *tdata);
static void ok(GtkWidget *widget, struct ftp_transfer_data *tdata);
static void cancel(GtkWidget *widget, struct ftp_transfer_data *tdata);
static void calc_kbs (struct ftp_transfer_data *tdata, ssize_t num_read);
static int set_file_transfer_mode (struct ftp_transfer_data *tdata);
static void goto_next_file (struct ftp_transfer_data *tdata);
static long get_transfer_size (struct ftp_transfer_data *tdata, int writefd);
static int get_status (struct ftp_transfer_data *tdata, ssize_t num_read);
static mode_t parse_local_attribs (char *attribs);
static mode_t parse_remote_attribs (char *attribs);
 
void retrCB (GtkWidget *widget, gpointer data) {
   struct ftp_transfer_data *tdata;
   
   if (!check_status (_("Retrieve Files"), &window2, 0, 1, 0)) return;
   if ((tdata = inittrans (1)) != NULL) dotrans (tdata);
}
/*****************************************************************************/
void putCB (GtkWidget *widget, gpointer data) {
   struct ftp_transfer_data *tdata;
   
   if (!check_status (_("Put Files"), &window1, 0, 1, 0)) return;
   if (window2.local == -1) {
      ftp_log (gftp_logging_misc, NULL, _("Put Files: Not connected to a remote site\n"));
      return;
   }
   if ((tdata = inittrans (0)) != NULL) dotrans (tdata);
}
/*****************************************************************************/
static struct ftp_transfer_data *inittrans (int direction) {
   struct ftp_file_data *newfle, *tempfle;
   struct recursive_transfer transdata;
   struct ftp_transfer_data *tdata;
   struct ftp_window_data *fromwdata;
   char *tempstr, *temp1str;
   GtkWidget *tempwid, *vbox;
   int oldsockfd;
   
   skip_same_state = 0;
   dialog = NULL;
   tdata = new_tdata ();
   if (direction) {
      tdata->flags |= TRANSFER_DIRECTION;
      fromwdata = &window2;
   }
   else fromwdata = &window1;
   copy_hdata_struct (window2.hdata, tdata->hdata);
   tdata->hdata->totalfiles = tdata->hdata->wdata->numselected;
   tdata->hdata->last = NULL;
   if ((tempfle = get_next_selected_filename (fromwdata->hdata->files)) == NULL) {
      ftp_log (gftp_logging_misc, NULL, "Internal gFTP Error: Could not find a selected file. This is probably a bug. Please email masneyb@seul.org about it\n");
      return (NULL);
   }
   while (tempfle != NULL) {
      if (strcmp (tempfle->file, "..") == 0) {
         ftp_log (gftp_logging_misc, NULL, _("Transfer Files: Skipping the transfer of the .. directory\n"));
         tempfle = get_next_selected_filename (tempfle->next);
         continue;
      }
      else if (tempfle->flags & FILE_ISDIR) {
         if (dialog == NULL) {
            dialog = gtk_window_new (GTK_WINDOW_DIALOG);
            gtk_grab_add (dialog);
            gtk_window_set_title (GTK_WINDOW (dialog), _("Getting directory listings"));
            gtk_window_set_position (GTK_WINDOW (dialog), GTK_WIN_POS_MOUSE);
            gtk_signal_connect_object (GTK_OBJECT (dialog), "destroy", GTK_SIGNAL_FUNC (gtk_widget_destroy), GTK_OBJECT (dialog));
    
            vbox = gtk_vbox_new (FALSE, 5);
            gtk_container_border_width (GTK_CONTAINER (vbox), 10);
            gtk_container_add (GTK_CONTAINER (dialog), vbox);
            gtk_widget_show (vbox);

            tempwid = gtk_label_new (_("Please wait while getting directory listings"));
            gtk_box_pack_start (GTK_BOX (vbox), tempwid, TRUE, TRUE, FALSE);
            gtk_widget_show (tempwid);

            status = gtk_label_new ("");
            gtk_box_pack_start (GTK_BOX (vbox), status, TRUE, TRUE, FALSE);
            gtk_widget_show (status);
            gtk_widget_show (dialog);
            fix_display ();
         }

         if (direction) {
            tempstr = g_strconcat (GFTP_GET_DIRECTORY (window2.hdata->ftpdata), "/", tempfle->file, NULL);
            temp1str = g_strconcat (GFTP_GET_DIRECTORY (window1.hdata->ftpdata), "/", tempfle->file, NULL);
         }
         else {
            temp1str = g_strconcat (GFTP_GET_DIRECTORY (window2.hdata->ftpdata), "/", tempfle->file, NULL);
            tempstr = g_strconcat (GFTP_GET_DIRECTORY (window1.hdata->ftpdata), "/", tempfle->file, NULL);
         }
         transdata.tdata = tdata;
         transdata.path = tempstr;
         transdata.dest = temp1str;
         transdata.root_path = g_malloc (strlen (tempstr) + 1);
         transdata.start_file = NULL;
         strcpy (transdata.root_path, tempstr);
         oldsockfd = GFTP_GET_CONTROL_FD (tdata->hdata->ftpdata);
         GFTP_GET_CONTROL_FD (tdata->hdata->ftpdata) = GFTP_GET_CONTROL_FD (window2.hdata->ftpdata);
         get_local_files_and_dirs(&transdata);
         GFTP_GET_CONTROL_FD (tdata->hdata->ftpdata) = oldsockfd;
      }
      else {
         newfle = g_malloc0 (sizeof (struct ftp_file_data));
         copy_fdata_struct (tempfle, newfle);

         if (newfle->file) g_free (newfle->file);
         if (newfle->remote_file) g_free (newfle->remote_file);
         newfle->file = g_strconcat (GFTP_GET_DIRECTORY (window1.hdata->ftpdata), "/", tempfle->file, NULL);
         remove_double_slashes (newfle->file);
         newfle->remote_file = g_strconcat (GFTP_GET_DIRECTORY (window2.hdata->ftpdata), "/", tempfle->file, NULL);
         remove_double_slashes (newfle->remote_file);

         newfle->flags = get_file_transfer_mode (newfle->file);
         newfle->next = NULL;
         if (tdata->hdata->last == NULL) tdata->hdata->files = newfle;
         else tdata->hdata->last->next = newfle;
         tdata->hdata->last = newfle;
      }
      tempfle = get_next_selected_filename (tempfle->next);
   }
   if (dialog != NULL) gtk_widget_destroy (dialog);

   if (tdata->hdata->files == NULL) {
      free_tdata (tdata);
      return (NULL);
   }
   tdata->curfle = tdata->prevfle = tdata->hdata->files;
   tdata->next = NULL;
   add_file_transfer (tdata);
   return (tdata);
}
/*****************************************************************************/
static void get_local_files_and_dirs (struct recursive_transfer *transdata) {
   struct ftp_file_data *newfle, *tempfle;
   char *newpath, *newdest, *pos;
   int i;
   
   for (i=0; i<transdata->tdata->num_dirs_to_be_made; i++) {
      if (strcmp (transdata->tdata->dirs_to_be_made[i], transdata->dest) == 0) {
         break;
      }
   }
   if (i == transdata->tdata->num_dirs_to_be_made) {
      transdata->tdata->num_dirs_to_be_made++;
      transdata->tdata->dirs_to_be_made = g_realloc (transdata->tdata->dirs_to_be_made, transdata->tdata->num_dirs_to_be_made * sizeof(char *));
      transdata->tdata->dirs_to_be_made[transdata->tdata->num_dirs_to_be_made-1] = g_malloc (strlen (transdata->dest) + 1);
      strcpy (transdata->tdata->dirs_to_be_made[transdata->tdata->num_dirs_to_be_made-1], transdata->dest);
   }
   if (transdata->tdata->flags & TRANSFER_DIRECTION) newfle = get_remote_files (transdata->tdata->hdata, transdata->path, &i, 1, GTK_LABEL (status));
   else newfle = get_local_files (transdata->path, &i);

   tempfle = newfle;
   if (transdata->start_file != NULL) {
      while (strcmp (tempfle->file, transdata->start_file) != 0) {
         tempfle = tempfle->next;
      }
      tempfle = tempfle->next;
      while (tempfle != NULL && (strcmp (tempfle->file, ".") == 0 || strcmp (tempfle->file, "..") == 0)) {
         tempfle = tempfle->next;
      }
      g_free (transdata->start_file);
      transdata->start_file = NULL;
   }
   while (tempfle != NULL) {
      if (tempfle->flags & FILE_ISDIR && strcmp (tempfle->file, ".") != 0 && strcmp (tempfle->file, "..") != 0) {
         newpath = g_strconcat (transdata->path, "/", tempfle->file, NULL);
         newdest = g_strconcat (transdata->dest, "/", tempfle->file, NULL);
         g_free (transdata->path);
         g_free (transdata->dest);
         transdata->path = newpath;
         transdata->dest = newdest;
         tempfle = tempfle->next;
         get_local_files_and_dirs (transdata);
         return;
      }
      else if (!(tempfle->flags & FILE_ISDIR)) {
         transdata->tdata->hdata->totalfiles++;
         newpath = g_strconcat (transdata->path, "/", tempfle->file, NULL);
         newdest = g_strconcat (transdata->dest, "/", tempfle->file, NULL);
         if (tempfle->file) g_free (tempfle->file);
         if (tempfle->remote_file) g_free (tempfle->remote_file);
         if (transdata->tdata->flags & TRANSFER_DIRECTION) {
            tempfle->file = newdest;
            tempfle->remote_file = newpath;
         }
         else {
            tempfle->file = newpath;
            tempfle->remote_file = newdest;
         }
         tempfle->flags = get_file_transfer_mode (tempfle->file);

         if (transdata->tdata->hdata->last == NULL) {
            transdata->tdata->hdata->files = tempfle;
            tempfle = tempfle->next;
            transdata->tdata->hdata->files->next = NULL;
            transdata->tdata->hdata->last = transdata->tdata->hdata->files;
         }
         else {
            transdata->tdata->hdata->last->next = tempfle;
            tempfle = tempfle->next;
            transdata->tdata->hdata->last = transdata->tdata->hdata->last->next;
            transdata->tdata->hdata->last->next = NULL;
         }
      }
      else tempfle = tempfle->next;
   }
   if (strcmp (transdata->path, transdata->root_path) != 0) {
      if ((pos = strrchr (transdata->path, '/')) != NULL) {
         pos++;
         transdata->start_file = g_malloc (strlen (pos) + 1);
         strcpy (transdata->start_file, pos);
         *(pos-1) = '\0';
         if ((pos = strrchr (transdata->dest, '/')) != NULL) *pos = '\0';
         get_local_files_and_dirs (transdata);
      }
   }
}
/*****************************************************************************/
void dotrans (struct ftp_transfer_data *tdata) {
   struct ftp_file_data *tempfle, *filelist;
   char *pos, origch, *filelist_dir;
   struct stat flestat;
   int same;
   
   same = 0;
   filelist = NULL;
   filelist_dir = NULL;
   pthread_mutex_lock (&tdata->mutex);
   while (tdata->curfle != NULL) {
      if (tdata->flags & TRANSFER_DIRECTION) {
         /* We are downloading these files */
         if (stat (tdata->curfle->file, &flestat) != -1) {
            tdata->curfle->remote_size = tdata->curfle->size;
            tdata->curfle->size = flestat.st_size;
            tdata->curfle->flags |= FILE_TRANS_EXISTS;
            same = 1;
         }
      }
      else {
         /* We are uploading these files */
         pos = strrchr (tdata->curfle->remote_file, '/');
         if (pos == NULL) pos = tdata->curfle->remote_file;
         origch = *pos;
         *pos = '\0';
         if (strcmp (tdata->curfle->remote_file, GFTP_GET_DIRECTORY (window2.hdata->ftpdata)) == 0) {
            tempfle = window2.hdata->files;
         }
         else {
/*FIXME            if (filelist == NULL || strcmp (tdata->curfle->remote_file, filelist_dir) != 0) {
               free_file_list (filelist);
               filelist_dir = tdata->curfle->remote_file;
               filelist = get_remote_files (tdata->hdata, filelist_dir, &total, 1, 0);
            }*/
            tempfle = filelist;
         }
         *pos++ = origch;
         while (tempfle != NULL) {
            if (strcmp (tempfle->file, pos) == 0) {
               tdata->curfle->remote_size = tempfle->size;
               tdata->curfle->flags |= FILE_TRANS_EXISTS;
               same = 1;
            }
            tempfle = tempfle->next;
         }
      }
      tdata->prevfle = tdata->curfle;
      tdata->curfle = tdata->curfle->next;
   }
   free_file_list (filelist);
   if (same == 0) {
      tdata->flags &= ~(TRANSFER_NEW | TRANSFER_SHOW | TRANSFER_UPDATE | TRANSFER_DONE);
      tdata->flags |= TRANSFER_SHOW;
      pthread_mutex_unlock (&tdata->mutex);
   }
   else {
      pthread_mutex_unlock (&tdata->mutex);
      asktrans (tdata);
   }
}
/*****************************************************************************/
static void asktrans (struct ftp_transfer_data *tdata) {
   char *dltitles[4] = {_("Filename"), _("Local Size"), _("Remote Size"), _("Action")};
   char *add_data[4] = {NULL, NULL, NULL, NULL};
   GtkWidget *tempwid, *scroll, *hbox;
   struct ftp_file_data *tempfle;
   char tempstr[20], *pos;
   size_t len;
   int i;

   dialog = gtk_dialog_new ();
   gtk_grab_add (dialog);
   gtk_window_set_title (GTK_WINDOW (dialog), (tdata->flags & TRANSFER_DIRECTION) ? _("Download Files") : _("Upload Files"));
   gtk_container_border_width (GTK_CONTAINER (GTK_DIALOG (dialog)->vbox), 10);
   gtk_container_border_width (GTK_CONTAINER (GTK_DIALOG (dialog)->action_area), 5);
   gtk_box_set_spacing (GTK_BOX (GTK_DIALOG (dialog)->vbox), 5);
   gtk_box_set_spacing (GTK_BOX (GTK_DIALOG (dialog)->action_area), 35);
   gtk_box_set_homogeneous (GTK_BOX (GTK_DIALOG (dialog)->action_area), TRUE);
   gtk_window_set_position (GTK_WINDOW (dialog), GTK_WIN_POS_MOUSE);
               
   tempwid = gtk_label_new (_("The following file(s) exist on both the local and remote computer\nPlease select what you would like to do"));
   gtk_box_pack_start (GTK_BOX (GTK_DIALOG(dialog)->vbox), tempwid, FALSE, FALSE, FALSE);
   gtk_widget_show (tempwid);

   scroll = gtk_scrolled_window_new (NULL, NULL);
   gtk_widget_set_usize (scroll, 450, 200);
   gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scroll),
   	GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
   tdata->clist = gtk_clist_new_with_titles (4, dltitles);
   gtk_container_add (GTK_CONTAINER (scroll), tdata->clist);
   gtk_clist_set_selection_mode (GTK_CLIST (tdata->clist), GTK_SELECTION_EXTENDED);
   gtk_clist_set_column_width (GTK_CLIST (tdata->clist), 0, 100);
   gtk_clist_set_column_justification (GTK_CLIST (tdata->clist), 1, GTK_JUSTIFY_RIGHT);
   gtk_clist_set_column_width (GTK_CLIST (tdata->clist), 1, 85);
   gtk_clist_set_column_justification (GTK_CLIST (tdata->clist), 2, GTK_JUSTIFY_RIGHT);
   gtk_clist_set_column_width (GTK_CLIST (tdata->clist), 2, 85);
   gtk_clist_set_column_width (GTK_CLIST (tdata->clist), 3, 85);
   gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), scroll, TRUE, TRUE, TRUE);
   gtk_signal_connect (GTK_OBJECT (tdata->clist), "select_row", GTK_SIGNAL_FUNC (trans_selectrow), (gpointer) tdata);
   gtk_signal_connect (GTK_OBJECT (tdata->clist), "unselect_row", GTK_SIGNAL_FUNC (trans_unselectrow), (gpointer) tdata);
   gtk_widget_show (tdata->clist);
   gtk_widget_show (scroll);

   tempfle = tdata->hdata->files;
   while (tempfle != NULL) {
      tempfle->flags &= ~(FILE_SELECTED);
      if (tempfle->flags & FILE_TRANS_EXISTS) {
         pos = tempfle->remote_file;
         len = strlen (GFTP_GET_DIRECTORY (window2.hdata->ftpdata));
         if (strncmp (pos, GFTP_GET_DIRECTORY (window2.hdata->ftpdata), len) == 0) {
            pos = tempfle->remote_file + len + 1;
         }
         add_data[0] = pos;
         add_data[3] = _("Overwrite");
         i = gtk_clist_append (GTK_CLIST (tdata->clist), add_data);
         
         insert_commas (tempfle->size, tempstr, sizeof (tempstr));
         gtk_clist_set_text (GTK_CLIST (tdata->clist), i, 1, tempstr);

         insert_commas (tempfle->remote_size, tempstr, sizeof (tempstr));
         gtk_clist_set_text (GTK_CLIST (tdata->clist), i, 2, tempstr);
      }
      tempfle = tempfle->next;
   }

   hbox = gtk_hbox_new (TRUE, 20);
   gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), hbox, TRUE, TRUE, TRUE);
   gtk_widget_show (hbox);

   tempwid = gtk_button_new_with_label (_("Overwrite"));
   gtk_box_pack_start (GTK_BOX (hbox), tempwid, TRUE, TRUE, FALSE);
   gtk_signal_connect (GTK_OBJECT (tempwid), "clicked", GTK_SIGNAL_FUNC (overwrite), (gpointer) tdata);
   gtk_widget_show (tempwid);

   tempwid = gtk_button_new_with_label (_("Resume"));
   gtk_box_pack_start (GTK_BOX (hbox), tempwid, TRUE, TRUE, FALSE);
   gtk_signal_connect (GTK_OBJECT (tempwid), "clicked", GTK_SIGNAL_FUNC (resume), (gpointer) tdata);
   if (tdata->hdata->protocol == http) gtk_widget_set_sensitive (tempwid, 0);
   gtk_widget_show (tempwid);

   tempwid = gtk_button_new_with_label (_("Skip File"));
   gtk_box_pack_start (GTK_BOX (hbox), tempwid, TRUE, TRUE, FALSE);
   gtk_signal_connect (GTK_OBJECT (tempwid), "clicked", GTK_SIGNAL_FUNC (skip), (gpointer) tdata);
   gtk_widget_show (tempwid);

   hbox = gtk_hbox_new (TRUE, 20);
   gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), hbox, TRUE, TRUE, TRUE);
   gtk_widget_show (hbox);

   tempwid = gtk_button_new_with_label (_("Select All"));
   gtk_box_pack_start (GTK_BOX (hbox), tempwid, TRUE, TRUE, FALSE);
   gtk_signal_connect (GTK_OBJECT (tempwid), "clicked", GTK_SIGNAL_FUNC (trans_selectall), (gpointer) tdata);
   gtk_widget_show (tempwid);

   tempwid = gtk_button_new_with_label (_("Deselect All"));
   gtk_box_pack_start (GTK_BOX (hbox), tempwid, TRUE, TRUE, FALSE);
   gtk_signal_connect (GTK_OBJECT (tempwid), "clicked", GTK_SIGNAL_FUNC (trans_unselectall), (gpointer) tdata);
   gtk_widget_show (tempwid);
   
   tempwid = gtk_button_new_with_label (_("OK"));
   GTK_WIDGET_SET_FLAGS (tempwid, GTK_CAN_DEFAULT);
   gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->action_area), tempwid, TRUE, TRUE, FALSE);
   gtk_signal_connect (GTK_OBJECT (tempwid), "clicked", GTK_SIGNAL_FUNC (ok), (gpointer) tdata);
   gtk_signal_connect_object (GTK_OBJECT (tempwid), "clicked", GTK_SIGNAL_FUNC (gtk_widget_destroy), GTK_OBJECT (dialog));
   gtk_widget_grab_default (tempwid);
   gtk_widget_show (tempwid);

   tempwid = gtk_button_new_with_label (_("  Cancel  "));
   GTK_WIDGET_SET_FLAGS (tempwid, GTK_CAN_DEFAULT);
   gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->action_area), tempwid, TRUE, TRUE, FALSE);
   gtk_signal_connect (GTK_OBJECT (tempwid), "clicked", GTK_SIGNAL_FUNC (cancel), (gpointer) tdata);
   gtk_signal_connect_object (GTK_OBJECT (tempwid), "clicked", GTK_SIGNAL_FUNC (gtk_widget_destroy), GTK_OBJECT (dialog));
   gtk_widget_show (tempwid);

   gtk_widget_show(dialog);
}
/*****************************************************************************/
static void trans_selectrow (GtkCList *clist, gint row, gint column, GdkEventButton *event, struct ftp_transfer_data *tdata) {
   struct ftp_file_data *tempfle;
   int i;
   
   i = 0;
   tempfle = tdata->hdata->files;
   while (tempfle != NULL) {
      if (tempfle->flags & FILE_TRANS_EXISTS) {
         if (i == row) {
            tempfle->flags |= FILE_SELECTED;
            break;
         }
         i++;
      }
      tempfle = tempfle->next;
   }
}
/*****************************************************************************/
static void trans_unselectrow (GtkCList *clist, gint row, gint column, GdkEventButton *event, struct ftp_transfer_data *tdata) {
   struct ftp_file_data *tempfle;
   int i;
   
   i = 0;
   tempfle = tdata->hdata->files;
   while (tempfle != NULL) {
      if (tempfle->flags & FILE_TRANS_EXISTS) {
         if (i == row) {
            tempfle->flags &= ~(FILE_SELECTED);
            break;
         }
         i++;
      }
      tempfle = tempfle->next;
   }
}
/*****************************************************************************/
static void trans_selectall (GtkWidget *widget, struct ftp_transfer_data *tdata) {
   struct ftp_file_data *tempfle;
   int i;
   
   i = 0;
   tempfle = tdata->hdata->files;
   while (tempfle != NULL) {
      if (tempfle->flags & FILE_TRANS_EXISTS) {
         gtk_clist_select_row (GTK_CLIST (tdata->clist), i++, 0);
      }
      tempfle = tempfle->next;
   }
}
/*****************************************************************************/
static void trans_unselectall (GtkWidget *widget, struct ftp_transfer_data *tdata) {
   struct ftp_file_data *tempfle;
   int i;
   
   i = 0;
   tempfle = tdata->hdata->files;
   while (tempfle != NULL) {
      if (tempfle->flags & FILE_TRANS_EXISTS) {
         gtk_clist_unselect_row (GTK_CLIST (tdata->clist), i++, 0);
      }
      tempfle = tempfle->next;
   }
}
/*****************************************************************************/
static void overwrite (GtkWidget *widget, struct ftp_transfer_data *tdata) {
   struct ftp_file_data *tempfle;
   int i;
   
   i = 0;
   tempfle = tdata->hdata->files;
   while (tempfle != NULL) {
      if (tempfle->flags & FILE_TRANS_EXISTS) {
         if (tempfle->flags & FILE_SELECTED) {
            tempfle->flags &= ~(FILE_RESTART | FILE_TRANS_SKIP);
            gtk_clist_set_text (GTK_CLIST (tdata->clist), i, 3, _("Overwrite"));
         }
         i++;
      }
      tempfle = tempfle->next;
   }
}
/*****************************************************************************/
static void resume (GtkWidget *widget, struct ftp_transfer_data *tdata) {
   struct ftp_file_data *tempfle;
   int i;
   
   i = 0;
   tempfle = tdata->hdata->files;
   while (tempfle != NULL) {
      if (tempfle->flags & FILE_TRANS_EXISTS) {
         if (tempfle->flags & FILE_SELECTED) {
            tempfle->flags &= ~(FILE_TRANS_SKIP);
            tempfle->flags |= FILE_RESTART;
            gtk_clist_set_text (GTK_CLIST (tdata->clist), i, 3, _("Resume"));
         }
         i++;
      }
      tempfle = tempfle->next;
   }
}
/*****************************************************************************/
static void skip (GtkWidget *widget, struct ftp_transfer_data *tdata) {
   struct ftp_file_data *tempfle;
   int i;
   
   i = 0;
   tempfle = tdata->hdata->files;
   while (tempfle != NULL) {
      if (tempfle->flags & FILE_TRANS_EXISTS) {
         if (tempfle->flags & FILE_SELECTED) {
            tempfle->flags &= ~(FILE_RESTART);
            tempfle->flags |= FILE_TRANS_SKIP;
            gtk_clist_set_text (GTK_CLIST (tdata->clist), i, 3, _("Skip"));
         }
         i++;
      }
      tempfle = tempfle->next;
   }
}
/*****************************************************************************/
static void ok (GtkWidget *widget, struct ftp_transfer_data *tdata) {
   struct ftp_file_data *curfle, *lastfle;
   
   curfle = lastfle = tdata->hdata->files;
   pthread_mutex_lock (&tdata->mutex);
   while (curfle != NULL) {
      if (curfle->flags & FILE_TRANS_SKIP) {
         if (curfle == lastfle) {
            tdata->hdata->files = tdata->hdata->files->next;
            free_fdata (curfle);
            curfle = lastfle = tdata->hdata->files;
         }
         else {
            lastfle->next = curfle->next;
            free_fdata (curfle);
            curfle = lastfle->next;
         }
         tdata->hdata->totalfiles--;
      }
      else {
         lastfle = curfle;
         curfle = curfle->next;
      }
   }
   tdata->flags &= ~(TRANSFER_NEW | TRANSFER_SHOW | TRANSFER_UPDATE | TRANSFER_DONE);
   if (tdata->hdata->files == NULL) tdata->flags |= TRANSFER_DONE;
   else tdata->flags |= TRANSFER_SHOW;
   pthread_mutex_unlock (&tdata->mutex);
}
/*****************************************************************************/
static void cancel (GtkWidget *widget, struct ftp_transfer_data *tdata) {
   pthread_mutex_lock (&tdata->mutex);
   tdata->flags &= ~(TRANSFER_NEW | TRANSFER_SHOW | TRANSFER_UPDATE | TRANSFER_DONE);
   tdata->flags |= TRANSFER_DONE;
   pthread_mutex_unlock (&tdata->mutex);
}
/*****************************************************************************/
void *ftp_get_files(void *ptr) {
   char buf[8192];
   int i, j, dlmode, success, writefd;
   struct ftp_transfer_data *tdata;
   ssize_t num_read;
   long startsize;

   tdata = (struct ftp_transfer_data *) ptr;
   pthread_detach (pthread_self ());
   gftp_logging (tdata->hdata->ftpdata, 1, queue_log, NULL);
   g_snprintf (tdata->progressstr, sizeof(tdata->progressstr), _("Connecting..."));
   if (ftp_connect (tdata->hdata, 1)) {
      for (i=0; i<tdata->num_dirs_to_be_made; i++) {
         mkdir (tdata->dirs_to_be_made[i], 488);
         g_free (tdata->dirs_to_be_made[i]);
      }
      g_free (tdata->dirs_to_be_made);

      while (!(tdata->flags & TRANSFER_CANCEL) && tdata->curfle != NULL) {
         success = 1;
         dlmode = set_file_transfer_mode (tdata);

         if ((writefd = open (tdata->curfle->file, tdata->curfle->flags & FILE_RESTART ? O_APPEND | O_CREAT | O_WRONLY : O_CREAT | O_TRUNC | O_WRONLY, S_IRUSR | S_IWUSR)) == -1) {
            queue_log (gftp_logging_error, NULL, _("Cannot create %s: %s\n"), 
            	tdata->curfle->file , g_strerror (errno));
            goto_next_file (tdata);
            continue;
         }
         if (tdata->curfle->flags & FILE_RESTART) {
            startsize = tdata->curfle->size;
            if (dlmode == FTP_ASCII && !(tdata->curfle->flags & TRANSFER_AUTO_RESTART)) {
               startsize += file_countlf (writefd, startsize);
            }
         }
         else startsize = 0;

         if (tdata->hdata->protocol == ftp ?
         	gftp_get_file (tdata->hdata->ftpdata, tdata->curfle->remote_file, startsize) != 0 :
         	!http_get_file (tdata->hdata, tdata->curfle->remote_file)) {
            queue_log (gftp_logging_error, NULL, _("Cannot download file %s from %s\n"), 
            	tdata->curfle->file , GFTP_GET_HOSTNAME (tdata->hdata->ftpdata));
            goto_next_file (tdata);
            continue;
         }
         
         pthread_mutex_lock (&tdata->mutex);
         tdata->current_file_retries++;
         tdata->current_file_number++;
         tdata->curfle->size = get_transfer_size (tdata, writefd);
         gettimeofday (&tdata->starttime, NULL);
         memcpy (&tdata->lasttime, &tdata->starttime, sizeof (tdata->lasttime));
         tdata->curtrans = startsize;
         tdata->sttrans = startsize;
         pthread_mutex_unlock (&tdata->mutex);

         num_read = 0;
         while (!(tdata->flags & TRANSFER_CANCEL) && 
         	(num_read = read (GFTP_GET_DATA_FD (tdata->hdata->ftpdata), &buf, sizeof (buf))) > 0) {
            calc_kbs (tdata, num_read);
            
            if (dlmode == FTP_ASCII) { 
               for (i=0, j=0; i<num_read; i++) {
                  if (buf[i] != '\r') buf[j++] = buf[i];
               }
            }
            else j = 0;
            
            if (write (writefd, buf, dlmode == FTP_ASCII ? j : num_read) == -1) {
               queue_log (gftp_logging_error, NULL, _("Error writing %s to disk: %s...Aborting entire transfer...\n"), 
               		tdata->curfle->file, g_strerror (errno));
               success = 0;
               break;
            }
         }
         if (success && !(tdata->flags & TRANSFER_CANCEL)) {
            switch (get_status (tdata, num_read)) {
               case -1: success = 0;
               case  1: continue;
               default: success = 1;
            }
         }
         else {
            if (tdata->hdata->protocol == ftp) gftp_disconnect (tdata->hdata->ftpdata);
            else close (GFTP_GET_DATA_FD (tdata->hdata->ftpdata));
            break;
         }

         if (tdata->hdata->protocol == ftp) {
            gftp_end_transfer (tdata->hdata->ftpdata);
            if (success) {
               queue_log (gftp_logging_misc, NULL, _("Successfully downloaded %s\n"), 
               		tdata->curfle->remote_file);
            }
            else {
               queue_log (gftp_logging_error, NULL, _("Could not download %s from %s\n"), 
               		tdata->curfle->remote_file, GFTP_GET_HOSTNAME (tdata->hdata->ftpdata));
            }
            if (preserve_attribs && tdata->curfle->attribs) {
               chmod (tdata->curfle->file, parse_local_attribs (tdata->curfle->attribs));
            }
         }
         else close (GFTP_GET_DATA_FD (tdata->hdata->ftpdata));
         if (!success) gftp_disconnect (tdata->hdata->ftpdata);
         goto_next_file (tdata);
      }
   }

   pthread_mutex_lock (&tdata->mutex);
   tdata->flags &= ~(TRANSFER_NEW | TRANSFER_SHOW | TRANSFER_UPDATE | TRANSFER_DONE);
   tdata->flags |= TRANSFER_DONE;
   pthread_mutex_unlock (&tdata->mutex);
   pthread_exit (NULL);
}
/*****************************************************************************/
void *ftp_put_files(void *ptr) {
   char buf[8192], *newbuf;
   int i, j, dlmode, success, readfd;
   struct ftp_transfer_data *tdata;
   long startsize, newsize;
   ssize_t num_read;

   tdata = (struct ftp_transfer_data *) ptr;
   pthread_detach (pthread_self ());
   gftp_logging (tdata->hdata->ftpdata, 1, queue_log, NULL);
   g_snprintf (tdata->progressstr, sizeof(tdata->progressstr), _("Connecting..."));
   if (ftp_connect (tdata->hdata, 1)) {
      for (i=0; i<tdata->num_dirs_to_be_made; i++) {
         gftp_make_directory (tdata->hdata->ftpdata, tdata->dirs_to_be_made[i]);
         g_free (tdata->dirs_to_be_made[i]);
      }
      g_free (tdata->dirs_to_be_made);

      while (!(tdata->flags & TRANSFER_CANCEL) && tdata->curfle != NULL) {
         success = 1;
         dlmode = set_file_transfer_mode (tdata);

         if (!(readfd = open (tdata->curfle->file, O_RDONLY))) {
            queue_log (gftp_logging_error, NULL, _("Error: Cannot open local file %s: %s\n"), 
            	tdata->curfle->file, g_strerror (errno));
            goto_next_file (tdata);
            continue;
         }

         if (tdata->curfle->flags & FILE_RESTART) {
            startsize = gftp_get_file_size (tdata->hdata->ftpdata, tdata->curfle->remote_file);
            if (dlmode == FTP_ASCII && !(tdata->curfle->flags & TRANSFER_AUTO_RESTART)) {
               startsize += file_countlf(readfd, startsize);
            }
            lseek (readfd, startsize, SEEK_SET);
         }
         else startsize = 0;
         
         if (gftp_put_file (tdata->hdata->ftpdata, tdata->curfle->remote_file, startsize) != 0) {
            queue_log (gftp_logging_error, NULL, _("Cannot put file %s to %s\n"), 
            	tdata->curfle->remote_file, GFTP_GET_HOSTNAME (tdata->hdata->ftpdata));
            goto_next_file (tdata);
            continue;
         }

         pthread_mutex_lock (&tdata->mutex);
         tdata->current_file_retries++;
         tdata->current_file_number++;
         gettimeofday (&tdata->starttime, NULL);
         memcpy (&tdata->lasttime, &tdata->starttime, sizeof (tdata->lasttime));
         tdata->curtrans = startsize;
         tdata->sttrans = startsize;
         pthread_mutex_unlock (&tdata->mutex);

         num_read = 0;
         while (!(tdata->flags & TRANSFER_CANCEL) && 
         	(num_read = read (readfd, buf, sizeof (buf))) > 0) {

            calc_kbs (tdata, num_read);

            if (dlmode == FTP_ASCII) { 
               newsize = 0;
               for (i=0; i<num_read; i++) {
                  newsize++;
                  if(i > 0 && buf[i] == '\n' && buf[i-1] != '\r') newsize++;
               }
               newbuf = g_malloc (newsize);
               for (i=0,j=0; i<num_read; i++) {
                  if (i > 0 && buf[i] == '\n' && buf[i-1] != '\r') newbuf[j++] = '\r';
                  newbuf[j++] = buf[i];
               }
            }
            else {
               newbuf = buf;
               newsize = num_read;
            }

            if (!write (GFTP_GET_DATA_FD (tdata->hdata->ftpdata), newbuf, newsize)) {
               queue_log (gftp_logging_error, NULL, _("Cannot write %s to %s. Aborting entire transfer...\n"), 
               		tdata->curfle->file, GFTP_GET_HOSTNAME (tdata->hdata->ftpdata));
               if (dlmode == FTP_ASCII) g_free (newbuf);
               success = 0;
               continue;
            }
            if (dlmode == FTP_ASCII) g_free (newbuf);
         }   
         close (readfd);
         gftp_end_transfer (tdata->hdata->ftpdata);
         if (success) {
            switch (get_status (tdata, num_read)) {
               case -1: success = 0;
               case  1: continue;
               default: success = 1;
            }
         }
         else break;

         if (success) {
            queue_log (gftp_logging_misc, NULL, _("Successfully uploaded %s to %s\n"), 
            	tdata->curfle->remote_file, GFTP_GET_HOSTNAME (tdata->hdata->ftpdata));
            if (preserve_attribs && tdata->curfle->attribs) {
               gftp_chmod (tdata->hdata->ftpdata, tdata->curfle->remote_file, parse_remote_attribs (tdata->curfle->attribs));
            }
         }
         else {
            queue_log (gftp_logging_error, NULL, _("Could not upload %s to %s\n"), 
            	tdata->curfle->remote_file, GFTP_GET_HOSTNAME (tdata->hdata->ftpdata));
            gftp_disconnect (tdata->hdata->ftpdata);
         }
         goto_next_file (tdata);
      }
   }

   pthread_mutex_lock (&tdata->mutex);
   tdata->flags &= ~(TRANSFER_NEW | TRANSFER_SHOW | TRANSFER_UPDATE | TRANSFER_DONE);
   tdata->flags |= TRANSFER_DONE;
   pthread_mutex_unlock (&tdata->mutex);
   pthread_exit (NULL);
}
/*****************************************************************************/
static void calc_kbs (struct ftp_transfer_data *tdata, ssize_t num_read) {
   unsigned long waitusecs, difftime;
   char gotstr[100], ofstr[100];
   struct timeval tv;
   float curkbs;
   
   gettimeofday (&tv, NULL);
   insert_commas (tdata->curfle->size, ofstr, sizeof (ofstr));/*FIXME*/
   pthread_mutex_lock (&tdata->mutex);
   difftime = (tv.tv_sec - tdata->starttime.tv_sec) * 1000000 + tv.tv_usec - tdata->starttime.tv_usec;
   if (difftime == 0) tdata->kbs = 0;
   else tdata->kbs = ((float) (tdata->curtrans - tdata->sttrans) / 1024) / ((float) difftime / 1000000);
   tdata->curtrans += num_read;
   insert_commas (tdata->curtrans, gotstr, sizeof (gotstr));
   g_snprintf (tdata->progressstr, sizeof(tdata->progressstr), 
   	_("Recv %s of %s at %.2fKB/s (%d of %d)"), gotstr, ofstr, 
   	tdata->kbs, tdata->current_file_number, tdata->hdata->totalfiles);
   tdata->flags |= TRANSFER_NEED_UPDATED;
   pthread_mutex_unlock (&tdata->mutex);

   difftime = (tv.tv_sec - tdata->lasttime.tv_sec) * 1000000 + tv.tv_usec - tdata->lasttime.tv_usec;
   if (difftime == 0) curkbs = (float) num_read / 1024;
   else curkbs = ((float) num_read / 1024) / ((float) difftime / 1000000);
   if (maxkbs > 0 && curkbs > maxkbs) {
      waitusecs = ((float) num_read / 1024) / maxkbs * 1000000 - difftime;
      usleep (waitusecs);
   }
   gettimeofday (&tdata->lasttime, NULL);
}
/*****************************************************************************/
static int set_file_transfer_mode (struct ftp_transfer_data *tdata) {
   int dlmode;

   if (tdata->hdata->protocol == ftp) {
      if (tdata->curfle->flags & FILE_ASCII) {
         dlmode = FTP_ASCII;
         gftp_set_data_type (tdata->hdata->ftpdata, gftp_type_ascii);
      }
      else {
         dlmode = FTP_BINARY;
         gftp_set_data_type (tdata->hdata->ftpdata, gftp_type_binary);
      }
   }
   else dlmode = FTP_BINARY;
   return (dlmode);
}
/*****************************************************************************/
static void goto_next_file (struct ftp_transfer_data *tdata) {
   pthread_mutex_lock (&tdata->mutex);
   tdata->curfle->flags |= FILE_TRANS_DONE;
   tdata->flags |= TRANSFER_ON_NEXT_FILE;
   tdata->prevfle = tdata->curfle;
   tdata->curfle = tdata->curfle->next;
   tdata->current_file_retries = 0;
   pthread_mutex_unlock (&tdata->mutex);
}
/*****************************************************************************/
static long get_transfer_size (struct ftp_transfer_data *tdata, int writefd) {
   char *pos, *resp, *tempstr, buf[8192];
   ssize_t n;
   long size;

   size = 0;
   if (tdata->hdata->protocol == ftp) {
      resp = GFTP_GET_LAST_RESPONSE (tdata->hdata->ftpdata) + 4;
      if ((pos = strrchr (resp, '(')) == NULL) {
         pos = resp;
         while (!isdigit (*pos) && *pos != '\0') pos++;
      }
      else pos++;
      size = strtol (pos, NULL, 10);
   }
   else {
      tempstr = NULL;
      /* Remove proxy header */
      while ((n = read (GFTP_GET_DATA_FD (tdata->hdata->ftpdata), &buf, sizeof (buf)-1)) > 0) {
         if (tempstr == NULL) {
            tempstr = g_malloc (strlen (buf) + 1);
            strcpy (tempstr, buf);
         }
         else {
            tempstr = g_realloc (tempstr, strlen (buf) + strlen (tempstr) + 1);
            strcat (tempstr, buf);
         }
         if ((pos = strstr (buf, "Content-Length: ")) != NULL) {
            size = strtol (pos+16, NULL, 10);
         }
         if ((pos = strstr (buf, "\r\n\r\n")) != NULL) {
            pos += 4;
            write (writefd, pos, n - (pos - buf));
            g_free (tempstr);
            break;
         }
      }
   }
   return (size);
}
/*****************************************************************************/
static int get_status (struct ftp_transfer_data *tdata, ssize_t num_read) {
   struct timeval tv;
   
   if ((num_read < 0 || tdata->curtrans < tdata->curfle->size) && tdata->hdata->protocol == ftp) {
      if (tdata->current_file_retries >= retries) {
         queue_log (gftp_logging_error, NULL, _("Error: Remote site %s disconnected. Max retries reached...giving up\n"), 
         	GFTP_GET_HOSTNAME (tdata->hdata->ftpdata));
         return (-1);
      }
      else {
         queue_log (gftp_logging_error, NULL, _("Error: Remote site %s disconnected. Will reconnect in %d seconds\n"), 
         	GFTP_GET_HOSTNAME (tdata->hdata->ftpdata), sleep_time);
      }
      gftp_disconnect (tdata->hdata->ftpdata);
      tv.tv_sec = sleep_time;
      tv.tv_usec = 0;              
      select (0, NULL, NULL, NULL, &tv);
      if (ftp_connect (tdata->hdata, 1)) {
         close (GFTP_GET_DATA_FD (tdata->hdata->ftpdata));
         pthread_mutex_lock (&tdata->mutex);
         tdata->curfle->flags |= FILE_RESTART | TRANSFER_AUTO_RESTART;
         tdata->curfle->size = tdata->curtrans;
         pthread_mutex_unlock (&tdata->mutex);
         tdata->current_file_number--; /* Decrement this because it will 
               			be incremented when we continue in the loop */
         return (1);
      }
   }
   else if (num_read < 0 && tdata->hdata->protocol == http) return (-1);
   
   return (0);
}
/*****************************************************************************/
static mode_t parse_local_attribs (char *attribs) {
   mode_t mode;
   
   mode = 0;
   if (tolower (attribs[3]) == 's') mode |= S_ISUID;
   if (tolower (attribs[6]) == 's') mode |= S_ISGID;
   if (attribs[1] == 'r') mode |= S_IRUSR;
   if (attribs[2] == 'w') mode |= S_IWUSR;
   if (attribs[3] == 'x') mode |= S_IXUSR;
   if (attribs[4] == 'r') mode |= S_IRGRP;
   if (attribs[5] == 'w') mode |= S_IWGRP;
   if (attribs[6] == 'x') mode |= S_IXGRP;
   if (attribs[7] == 'r') mode |= S_IROTH;
   if (attribs[8] == 'w') mode |= S_IWOTH;
   if (attribs[9] == 'x') mode |= S_IXOTH;
   return (mode);
}
/*****************************************************************************/
static mode_t parse_remote_attribs (char *attribs) {
   mode_t mode;
   int cur;
   
   mode = 0;
   cur = 0;
   if (tolower (attribs[3]) == 's') cur += 4;
   if (tolower (attribs[6]) == 's') cur += 2;
   mode = (mode * 10) + cur;

   cur = 0;
   if (attribs[1] == 'r') cur += 4;
   if (attribs[2] == 'w') cur += 2;
   if (attribs[3] == 'x') cur += 1;
   mode = (mode * 10) + cur;

   cur = 0;
   if (attribs[4] == 'r') cur += 4;
   if (attribs[5] == 'w') cur += 2;
   if (attribs[6] == 'x') cur += 1;
   mode = (mode * 10) + cur;

   cur = 0;
   if (attribs[7] == 'r') cur += 4;
   if (attribs[8] == 'w') cur += 2;
   if (attribs[9] == 'x') cur += 1;
   mode = (mode * 10) + cur;
   return (mode);
}
/*****************************************************************************/
