/*****************************************************************************/
/*  bookmarks.c - routines for the bookmarks                                 */
/*  Copyright (C) 1998-1999 Brian Masney <masneyb@seul.org>                 */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either version 2 of the License, or        */
/*  (at your option) any later version.                                      */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA      */
/*****************************************************************************/

#include "ftp.h"

static void try_bookmark_again (GtkWidget *widget, struct dialog_data *data);
static void doadd_bookmark (GtkWidget *widget, struct dialog_data *data);
static void bm_apply_changes (GtkWidget *widget, gpointer backup_data);
static struct conn_categories *copy_bookmarks (struct conn_categories *bookmarks);
static void bm_close_dialog (GtkWidget *widget, GtkWidget *dialog);
static void after_move (GtkCTree *ctree, GtkCTreeNode *child, GtkCTreeNode *parent,
	GtkCTreeNode *sibling, gpointer data);
static gint bm_dblclick (GtkWidget *widget, GdkEventButton *event, gpointer data);
static gint bm_enter (GtkWidget *widget, GdkEventKey *event, gpointer data);
static gint hash_compare (gconstpointer path1, gconstpointer path2);
static guint hash_function (gconstpointer key);
static void new_folder_entry (gpointer data);
static void do_make_new (GtkWidget *widget, struct dialog_data *data);
static void new_item_entry (gpointer data);
static void edit_entry (gpointer data);
static void delete_entry (gpointer data);
static void entry_apply_changes (GtkWidget *widget, struct conn_categories *entry);
static void set_userpass_visible(GtkWidget *checkbutton, GtkWidget *entry);
static void build_bookmarks_tree (void);
static void clear_bookmarks_tree (void);

static GtkWidget *bm_hostedit, *bm_portedit, *bm_diredit, *bm_useredit, 
	*bm_passedit, *bm_acctedit, *anon_chk, *firewall_check, *bm_pathedit, 
	*tree;
static struct conn_categories *new_bookmarks;
static GHashTable *htable, /* Lookups by node */
                  *new_bookmarks_htable; /* Lookups by path for new bookmarks */
static GtkItemFactory *edit_factory;

void run_bookmark (gpointer data) {
   struct conn_categories *tempentry;
   
   tempentry = (struct conn_categories *) 
   	g_hash_table_lookup (bookmarks_htable, (char *) data);
   
   if (tempentry == NULL) {
      ftp_log (gftp_logging_misc, NULL, _("Internal gFTP Error: Could not look up bookmark entry. This is definately a bug. Please email masneyb@seul.org about it. Please be sure to include the version number and how you can reproduce it\n"));
      return;
   }
   else if (tempentry->hostname == NULL || *tempentry->hostname == '\0' ||
   	tempentry->user == NULL || *tempentry->user == '\0') {
      ftp_log (gftp_logging_misc, NULL, _("Bookmarks Error: There are some missing entries in this bookmark. Make sure you have a hostname and username\n"));
      return;
   }
   else if (tempentry->pass == NULL || *tempentry->pass == '\0') {
      MakeEditDialog (_("Enter Password"), _("Please enter your password for this site"), NULL, 0, 0,
      	_("Connect"), try_bookmark_again, data, _("  Cancel  "), NULL, NULL);
      return;
   }
   
   if (window2.local != -1) disconnect (&window2);
   gftp_set_username (window2.hdata->ftpdata, tempentry->user);
   if (strncmp (tempentry->pass, "@EMAIL@", 7) == 0) {
      gftp_set_password (window2.hdata->ftpdata, emailaddr);
   }
   else {
      gftp_set_password (window2.hdata->ftpdata, tempentry->pass);
   }
   if (tempentry->acct != NULL) {
      gftp_set_account (window2.hdata->ftpdata, tempentry->acct);
   }
   gftp_set_hostname (window2.hdata->ftpdata, tempentry->hostname);
   gftp_set_directory (window2.hdata->ftpdata, tempentry->dir);
   gftp_set_port (window2.hdata->ftpdata, strtol(tempentry->port, NULL, 10));
   if (*tempentry->firewall == '1' && *proxy_config != '\0') {
      if (firewall_host == NULL || *firewall_host == '\0') {
         ftp_log (gftp_logging_error, NULL, _("Connect Error: You must specify a proxy hostname in the options dialog\n"));
         return;
      }
      if (strcmp (proxy_config, "http") == 0) {
         gftp_set_proxy_config (window2.hdata->ftpdata, "");
         window2.hdata->protocol = http;
      }
      else {
         gftp_set_proxy_config (window2.hdata->ftpdata, proxy_config);
      }  
   }
   else {
      gftp_set_proxy_config (window2.hdata->ftpdata, "");
      if (strcmp (proxy_config, "http") == 0) window2.hdata->protocol = ftp;
   }
   ftp_connect (window2.hdata, 1);
}
/*****************************************************************************/
static void try_bookmark_again (GtkWidget *widget, struct dialog_data *data) {
   struct conn_categories *tempentry;
   gpointer d;
   char *str;
   
   tempentry = (struct conn_categories *) 
   	g_hash_table_lookup (bookmarks_htable, (char *) data->data);
   str = gtk_entry_get_text (GTK_ENTRY (data->edit));
   if (tempentry->pass) g_free (tempentry->pass);
   tempentry->pass = g_malloc (strlen (str) + 1);
   strcpy (tempentry->pass, str);
   tempentry->save_password = 0;
   d = data->data;
   data->data = NULL;
   gtk_widget_destroy (data->dialog);
   run_bookmark (d);
}
/*****************************************************************************/
void add_bookmark (gpointer data) {
   char *edttxt;
   
   if (!check_status (_("Add Bookmark"), &window2, 0, 0, 0)) return;
   
   edttxt = gtk_entry_get_text (GTK_ENTRY (hostedit));
   if (*edttxt == '\0') {
      ftp_log (gftp_logging_error, NULL, _("Add Bookmark: You must enter a hostname\n"));
      return;
   }

   MakeEditDialog (_("Add Bookmark"), _("Enter the name of the bookmark you want to add\nYou can separate items by a / to put it into a submenu\n(ex: Linux Sites/Debian)"), NULL, 1, 1,
   	_("Change"), doadd_bookmark, data,
   	_("  Cancel  "), NULL, NULL);
}
/*****************************************************************************/
static void doadd_bookmark (GtkWidget *widget, struct dialog_data *data) {
   GtkItemFactoryEntry test = {NULL, NULL, run_bookmark, 0};
   struct conn_categories *tempentry;
   char *edttxt;
   
   edttxt = gtk_entry_get_text (GTK_ENTRY (data->edit));
   if (*edttxt == '\0') {
      ftp_log (gftp_logging_error, NULL, _("Add Bookmark: You must enter a name for the bookmark\n"));
      return;
   }
   
   if (g_hash_table_lookup (bookmarks_htable, edttxt) != NULL) {
      ftp_log (gftp_logging_error, NULL, _("Add Bookmark: Cannot add bookmark %s because that name already exists\n"), edttxt);
      return;
   }
   
   if (hostcat->children == NULL) {
      hostcat->children = g_malloc0 (sizeof (struct conn_categories));
      tempentry = hostcat->children;
   }
   else {
      tempentry = hostcat->children;
      while (tempentry->next != NULL) tempentry = tempentry->next;
      tempentry->next = g_malloc0 (sizeof (struct conn_categories));
      tempentry = tempentry->next;
   }
   tempentry->prev = hostcat;

   tempentry->path = g_malloc (strlen (edttxt) + 1);
   strcpy (tempentry->path, edttxt);

   edttxt = gtk_entry_get_text (GTK_ENTRY (hostedit));
   tempentry->hostname = g_malloc (strlen (edttxt) + 1);
   strcpy (tempentry->hostname, edttxt);

   edttxt = gtk_entry_get_text (GTK_ENTRY (portedit));
   tempentry->port = g_malloc (strlen (edttxt) + 1);
   strcpy (tempentry->port, edttxt);

   edttxt = gtk_entry_get_text (GTK_ENTRY (GTK_COMBO (window2.combo)->entry));
   tempentry->dir = g_malloc (strlen (edttxt) + 1);
   strcpy (tempentry->dir, edttxt);

   if ((edttxt = gtk_entry_get_text (GTK_ENTRY (useredit))) == NULL) {
      tempentry->user = g_malloc (strlen (ANON_LOGIN) + 1);
      strcpy (tempentry->user, ANON_LOGIN);
      
      tempentry->pass = g_malloc (strlen (emailaddr) + 1);
      strcpy (tempentry->pass, emailaddr);
   }
   else {
      tempentry->user = g_malloc (strlen (edttxt) + 1);
      strcpy (tempentry->user, edttxt);

      edttxt = gtk_entry_get_text (GTK_ENTRY (passedit));
      tempentry->pass = g_malloc (strlen (edttxt) + 1);
      strcpy (tempentry->pass, edttxt);
   }

   tempentry->firewall = g_malloc (1);
   *tempentry->firewall = GTK_CHECK_MENU_ITEM (firewall_btn)->active;

   g_hash_table_insert (bookmarks_htable, tempentry->path, tempentry);

   test.path = g_strconcat ("/Bookmarks/", tempentry->path, NULL);
   gtk_item_factory_create_item (factory, &test, (gpointer) tempentry->path, 1);
   g_free (test.path);
   write_config_file ();
}
/*****************************************************************************/
void edit_bookmarks (gpointer data) { 
   GtkWidget *dialog, *scroll, *tempwid;
   GtkItemFactory *ifactory;
   GtkItemFactoryEntry dummy_item, menu_items[] = {
      {"/_File",		NULL,   0,      	0, 	"<Branch>"},
      {"/File/tearoff",  	NULL,   0,		0,	"<Tearoff>"},
      {"/File/New Folder...", 	NULL, 	new_folder_entry,0},
      {"/File/New Item...", 	NULL, 	new_item_entry,0},
      {"/File/Delete", 		NULL, 	delete_entry,	0},
      {"/File/Properties...", 	NULL, 	edit_entry,	0},
      {"/File/sep",  		NULL,   0,		0,	"<Separator>"},
      {"/File/Close",		NULL,	gtk_widget_destroy,	0}};
   int i;
   
   htable = g_hash_table_new (hash_function, hash_compare);
   new_bookmarks = copy_bookmarks (hostcat);
   new_bookmarks_htable = build_hash_table (new_bookmarks);
   
   dialog = gtk_dialog_new ();
   gtk_window_set_title (GTK_WINDOW (dialog), _("Edit Bookmarks"));
   gtk_box_set_spacing (GTK_BOX (GTK_DIALOG (dialog)->action_area), 15);
   gtk_window_set_position (GTK_WINDOW (dialog), GTK_WIN_POS_MOUSE);
   
   ifactory = gtk_item_factory_new (GTK_TYPE_MENU_BAR, "<main>", NULL);
   create_item_factory (ifactory, 7, menu_items, NULL);
   create_item_factory (ifactory, 1, menu_items + 7, dialog);
   gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), ifactory->widget, FALSE, FALSE, FALSE);
   gtk_widget_show (ifactory->widget);
   
   scroll = gtk_scrolled_window_new (NULL, NULL);
   gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scroll), 
   	GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
   gtk_widget_set_usize (scroll, 150, 200);
   gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), scroll, TRUE, TRUE, TRUE);
   gtk_container_border_width (GTK_CONTAINER (scroll), 3);
   gtk_widget_show (scroll);

   edit_factory = gtk_item_factory_new (GTK_TYPE_MENU, "<edit_bookmark>", NULL);
   for (i=2; i<8; i++) {
      memcpy (&dummy_item, menu_items + i, sizeof (GtkItemFactoryEntry));
      dummy_item.path = strrchr (_(dummy_item.path), '/');
      gtk_item_factory_create_item (edit_factory, &dummy_item, dialog, 1);
   }
   
   tree = gtk_ctree_new (1, 0);
   gtk_clist_set_selection_mode (GTK_CLIST (tree), GTK_SELECTION_SINGLE);
   gtk_clist_set_reorderable (GTK_CLIST (tree), 1);
   gtk_scrolled_window_add_with_viewport (GTK_SCROLLED_WINDOW (scroll), tree);
   gtk_signal_connect_after (GTK_OBJECT (tree), "key_press_event", GTK_SIGNAL_FUNC (bm_enter), (gpointer) tree);
   gtk_signal_connect_after (GTK_OBJECT (tree), "tree_move", GTK_SIGNAL_FUNC (after_move), NULL);
   gtk_signal_connect_after (GTK_OBJECT (tree), "button_press_event", GTK_SIGNAL_FUNC (bm_dblclick), (gpointer) tree);
   gtk_widget_show (tree);
   
   build_bookmarks_tree ();

   tempwid = gtk_button_new_with_label (_("OK"));
   gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->action_area), tempwid, TRUE, TRUE, FALSE);
   gtk_signal_connect (GTK_OBJECT (tempwid), "clicked", GTK_SIGNAL_FUNC (bm_apply_changes), (gpointer) NULL);
   gtk_signal_connect (GTK_OBJECT (tempwid), "clicked", GTK_SIGNAL_FUNC (bm_close_dialog), (gpointer) dialog);
   GTK_WIDGET_SET_FLAGS (tempwid, GTK_CAN_DEFAULT);
   gtk_widget_show (tempwid);

   tempwid = gtk_button_new_with_label (_("  Cancel  "));
   gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->action_area), tempwid, TRUE, TRUE, FALSE);
   gtk_signal_connect (GTK_OBJECT (tempwid), "clicked", GTK_SIGNAL_FUNC (bm_close_dialog), (gpointer) dialog);
   GTK_WIDGET_SET_FLAGS (tempwid, GTK_CAN_DEFAULT);
   gtk_widget_grab_focus (tempwid);
   gtk_widget_show (tempwid);

   tempwid = gtk_button_new_with_label (_("Apply"));
   gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->action_area), tempwid, TRUE, TRUE, FALSE);
   gtk_signal_connect (GTK_OBJECT (tempwid), "clicked", GTK_SIGNAL_FUNC (bm_apply_changes), (gpointer) 1);
   GTK_WIDGET_SET_FLAGS (tempwid, GTK_CAN_DEFAULT);
   gtk_widget_show (tempwid);

   gtk_widget_show (dialog);
}
/*****************************************************************************/
static void bm_apply_changes (GtkWidget *widget, gpointer backup_data) {
   struct conn_categories *tempentry, *delentry;
   char tempstr[MAXSTR];
   
   tempentry = hostcat->children;
   while (tempentry != NULL) {
      if (tempentry->path) {
         if (!tempentry->isfolder) {
            g_snprintf (tempstr, sizeof (tempstr), _("/Bookmarks/%s"), tempentry->path);
            tempstr[sizeof (tempstr) - 1] = '\0';

/* The gtk_item_factory_delete_item () has a bug in it. This will be fixed in
   version 1.2.4 of gtk+. I will uncomment these in my next version */

/*FIXME            gtk_item_factory_delete_item (factory, tempstr);*/
            gtk_widget_destroy (gtk_item_factory_get_item (factory, tempstr));
            g_free (tempentry->path);
         }
      }
      if (tempentry->hostname) g_free (tempentry->hostname);
      if (tempentry->dir) g_free (tempentry->dir);
      if (tempentry->user) g_free (tempentry->user);
      if (tempentry->pass) g_free (tempentry->pass);
      if (tempentry->acct) g_free (tempentry->acct);
      if (tempentry->port) g_free (tempentry->port);
      if (tempentry->firewall) g_free (tempentry->firewall);

      if (tempentry->children != NULL) {
         tempentry = tempentry->children;
      }
      else {
         if (tempentry->next == NULL) {
            while (tempentry->next == NULL && tempentry->prev != NULL) {
               delentry = tempentry;
               tempentry = tempentry->prev;
               if (delentry->isfolder) {
                  g_snprintf (tempstr, sizeof (tempstr), _("/Bookmarks/%s"), delentry->path);
                  tempstr[sizeof (tempstr) - 1] = '\0';
/*FIXME                  gtk_item_factory_delete_item (factory, tempstr);*/
                  gtk_widget_destroy (gtk_item_factory_get_item (factory, tempstr));
                  g_free (delentry->path);
               }
               g_free (delentry);
            }
            delentry = tempentry;
            tempentry = tempentry->next;
            if (delentry->isfolder && tempentry != NULL) {
               g_snprintf (tempstr, sizeof (tempstr), _("/Bookmarks/%s"), delentry->path);
               tempstr[sizeof (tempstr) - 1] = '\0';
/*FIXME               gtk_item_factory_delete_item (factory, tempstr);*/
               gtk_widget_destroy (gtk_item_factory_get_item (factory, tempstr));
               g_free (delentry->path);
               g_free (delentry);
            }
         }
         else {
            delentry = tempentry;
            tempentry = tempentry->next;
            g_free (delentry);
         }
      }
   }
   g_free (hostcat);
   g_hash_table_destroy (bookmarks_htable);
   hostcat = new_bookmarks;
   bookmarks_htable = new_bookmarks_htable;
   if (backup_data) {
      new_bookmarks = copy_bookmarks (hostcat);
      new_bookmarks_htable = build_hash_table (new_bookmarks);
   }
   else {
      new_bookmarks = NULL;
      new_bookmarks_htable = NULL;
   }
   build_bookmarks_menu ();
}
/*****************************************************************************/
static struct conn_categories *copy_bookmarks (struct conn_categories *bookmarks) {
   struct conn_categories *new_bm, *preventry, *tempentry, *sibling, 
      *newentry, *tentry;
   
   new_bm = g_malloc0 (sizeof (struct conn_categories));
   new_bm->path = g_malloc (1);
   *new_bm->path = '\0';
   new_bm->isfolder = bookmarks->isfolder;
   preventry = new_bm;
   tempentry = bookmarks->children;
   sibling = NULL;
   while (tempentry != NULL) {
      newentry = g_malloc0 (sizeof (struct conn_categories));
      newentry->isfolder = tempentry->isfolder;
      if (tempentry->path) {
         newentry->path = g_malloc (strlen (tempentry->path) + 1);
         strcpy (newentry->path, tempentry->path);
      }
      if (tempentry->hostname) {
         newentry->hostname = g_malloc (strlen (tempentry->hostname) + 1);
         strcpy (newentry->hostname, tempentry->hostname);
      }
      if (tempentry->dir) {
         newentry->dir = g_malloc (strlen (tempentry->dir) + 1);
         strcpy (newentry->dir, tempentry->dir);
      }
      if (tempentry->user) {
         newentry->user = g_malloc (strlen (tempentry->user) + 1);
         strcpy (newentry->user, tempentry->user);
      }
      if (tempentry->pass) {
         newentry->pass = g_malloc (strlen (tempentry->pass) + 1);
         strcpy (newentry->pass, tempentry->pass);
      }
      if (tempentry->acct) {
         newentry->acct = g_malloc (strlen (tempentry->acct) + 1);
         strcpy (newentry->acct, tempentry->acct);
      }
      if (tempentry->port) {
         newentry->port = g_malloc (strlen (tempentry->port) + 1);
         strcpy (newentry->port, tempentry->port);
      }
      if (tempentry->firewall) {
         newentry->firewall = g_malloc (1);
         *newentry->firewall = *tempentry->firewall;
      }
      if (sibling == NULL) {
         if (preventry->children == NULL) preventry->children = newentry;
         else {
            tentry = preventry->children;
            while (tentry->next != NULL) tentry = tentry->next;
            tentry->next = newentry;
         }
      }
      else sibling->next = newentry;
      newentry->prev = preventry;

      if (tempentry->children != NULL) {
         preventry = newentry;
         sibling = NULL;
         tempentry = tempentry->children;
      }
      else {
         if (tempentry->next == NULL) {
            sibling = NULL;
            while (tempentry->next == NULL && tempentry->prev != NULL) {
               tempentry = tempentry->prev;
               preventry = preventry->prev;
            }
            tempentry = tempentry->next;
         }
         else {
            sibling = newentry;
            tempentry = tempentry->next;
         }
      }
   }
   return (new_bm);
}
/*****************************************************************************/
static void bm_close_dialog (GtkWidget *widget, GtkWidget *dialog) {
   struct conn_categories *tempentry;
   
   if (new_bookmarks_htable) g_hash_table_destroy (new_bookmarks_htable);
   tempentry = new_bookmarks;
   while (tempentry != NULL) {
      tempentry->cnode = NULL;
      if (tempentry->children != NULL) {
         tempentry = tempentry->children;
         continue;
      }
      while (tempentry->next == NULL && tempentry->prev != NULL) tempentry = tempentry->prev;
      tempentry = tempentry->next;
   }
   new_bookmarks = NULL;
   new_bookmarks_htable = NULL;
   gtk_widget_destroy (dialog);
}
/*****************************************************************************/
static void after_move (GtkCTree *ctree, GtkCTreeNode *child, GtkCTreeNode *parent,
	GtkCTreeNode *sibling, gpointer data) {
  
   struct conn_categories *childentry, *siblingentry, *parententry, *tempentry;
   char tempstr[MAXSTR], *pos, *stpos;
   
   childentry = (struct conn_categories *) g_hash_table_lookup (htable, child);
   parententry = (struct conn_categories *) g_hash_table_lookup (htable, parent);
   siblingentry = (struct conn_categories *) g_hash_table_lookup (htable, sibling);

   tempentry = childentry->prev->children;
   if (tempentry == childentry) {
      childentry->prev->children = childentry->prev->children->next;
   }
   else {
      while (tempentry->next != childentry) tempentry = tempentry->next;
      tempentry->next = childentry->next;
   }
   childentry->prev = parententry;
   if (!parententry->isfolder) {
      childentry->next = parententry->children;
      parententry->children = childentry;
      gtk_ctree_move (ctree, child, parententry->prev->cnode, parententry->cnode);
   }
   else {
      if (siblingentry == NULL || parententry->children == siblingentry) {
         childentry->next = parententry->children;
         parententry->children = childentry;
      }
      else {
         tempentry = parententry->children;
         while (tempentry->next != siblingentry) tempentry = tempentry->next;
         childentry->next = tempentry->next;
         tempentry->next = childentry;
      }

      tempentry = childentry;
      while (tempentry != NULL) {
         g_hash_table_remove (new_bookmarks_htable, tempentry->path);
         if ((pos = strrchr (tempentry->path, '/')) == NULL) pos = tempentry->path;
         else pos++;
         g_snprintf (tempstr, sizeof (tempstr), "%s/%s", tempentry->prev->path, pos);
         tempstr[sizeof (tempstr) - 1] = '\0';
         for (stpos = tempstr; *stpos == '/'; stpos++);
         g_free (tempentry->path);
         tempentry->path = g_malloc (strlen (stpos) + 1);
         strcpy (tempentry->path, stpos);
         g_hash_table_insert (new_bookmarks_htable, tempentry->path, tempentry);
         if (tempentry->children != NULL) tempentry = tempentry->children;
         else {
            if (tempentry->next == NULL) {
               if (tempentry->prev == childentry) break;
               else {
                  while (tempentry->next == NULL && tempentry->prev != NULL) {
                     tempentry = tempentry->prev;
                  }
               }
            }
            tempentry = tempentry->next;
         }
      }
   }
}
/*****************************************************************************/
static gint bm_dblclick (GtkWidget *widget, GdkEventButton *event, gpointer data) {
   if (event->button == 3) {
      gtk_item_factory_popup (edit_factory, (guint) event->x_root, (guint) event->y_root, 1, 0);
   }
   else if (event->type == GDK_2BUTTON_PRESS) {
      edit_entry (NULL);
      return (FALSE);
   }
   return (TRUE);
}
/*****************************************************************************/
static gint bm_enter (GtkWidget *widget, GdkEventKey *event, gpointer data) {
   if (event->type == GDK_KEY_PRESS) {
      if (event->keyval == GDK_Return || event->keyval == GDK_KP_Enter) {
         edit_entry (NULL);
         return (FALSE);
      }
      else if (event->keyval == GDK_KP_Delete || event->keyval == GDK_Delete) {
         delete_entry (NULL);
         return (FALSE);
      }
   }
   return (TRUE);
}
/*****************************************************************************/
static gint hash_compare (gconstpointer path1, gconstpointer path2) {
   return (path1 == path2);
}
/*****************************************************************************/
static guint hash_function (gconstpointer key) {
   return ((guint) key * 33);
}
/*****************************************************************************/
static void new_folder_entry (gpointer data) {
   MakeEditDialog (_("New Folder"), _("Enter the name of the new folder to create"), NULL, 1, 1,
   	_("Create"), do_make_new, (void *) 1,
   	_("  Cancel  "), NULL, NULL);
}
/*****************************************************************************/
static void new_item_entry (gpointer data) {
   MakeEditDialog (_("New Folder"), _("Enter the name of the new item to create"), NULL, 1, 1,
   	_("Create"), do_make_new, 0,
   	_("  Cancel  "), NULL, NULL);
}
/*****************************************************************************/
static void do_make_new (GtkWidget *widget, struct dialog_data *data) {
   struct conn_categories *tempentry, *newentry;
   GtkCTreeNode *sibling;
   char *str, *pos, *text[2];
   
   str = gtk_entry_get_text (GTK_ENTRY (data->edit));
   pos = str;
   while ((pos = strchr (pos, '/')) != NULL) *pos++ = ' ';

   newentry = g_malloc0 (sizeof (struct conn_categories));
   newentry->path = g_malloc (strlen (str) + 1);
   strcpy (newentry->path, str);
   newentry->prev = new_bookmarks;
   if (data) newentry->isfolder = 1;
   
   if (new_bookmarks->children == NULL) {
      new_bookmarks->children = newentry;
      sibling = NULL;
   }
   else {
      tempentry = new_bookmarks->children;
      while (tempentry->next != NULL) tempentry = tempentry->next;
      tempentry->next = newentry;
      sibling = tempentry->cnode;
   }

   text[0] = text[1] = newentry->path;
   if (newentry->isfolder) {
      newentry->cnode = gtk_ctree_insert_node (GTK_CTREE (tree), 
      		new_bookmarks->cnode, NULL, text, 5, dir_pixmap, dir_mask, 
      		open_dir_pixmap, open_dir_mask, FALSE, FALSE);
   }
   else {
      newentry->cnode = gtk_ctree_insert_node (GTK_CTREE (tree), 
   		new_bookmarks->cnode, NULL, text, 5, NULL, NULL, 
   		NULL, NULL, FALSE, FALSE);
   }
   g_hash_table_insert (htable, newentry->cnode, newentry);
   g_hash_table_insert (new_bookmarks_htable, newentry->path, newentry);
}
/*****************************************************************************/
static void edit_entry (gpointer data) {
   GtkWidget *table, *tempwid, *dialog;
   struct conn_categories *entry;
   char *pos;
   
   if (GTK_CLIST (tree)->selection == NULL) return;
   entry = (struct conn_categories *) g_hash_table_lookup (htable, GTK_CLIST (tree)->selection->data);
   if (entry == NULL || entry == new_bookmarks) return;

   dialog = gtk_dialog_new ();
   gtk_window_set_title (GTK_WINDOW (dialog), _("Edit Entry"));
   gtk_box_set_spacing (GTK_BOX (GTK_DIALOG (dialog)->action_area), 15);
   gtk_container_border_width (GTK_CONTAINER (GTK_DIALOG (dialog)->vbox), 10);
   gtk_window_set_position (GTK_WINDOW (dialog), GTK_WIN_POS_MOUSE);
   
   tempwid = gtk_frame_new (NULL);
   gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), tempwid, TRUE, TRUE, FALSE);
   gtk_widget_show (tempwid);
   
   table = gtk_table_new (10, 2, FALSE);
   gtk_container_border_width (GTK_CONTAINER (table), 5);
   gtk_table_set_row_spacings (GTK_TABLE (table), 5);
   gtk_table_set_col_spacings (GTK_TABLE (table), 5);
   gtk_container_add (GTK_CONTAINER (tempwid), table);
   gtk_widget_show (table);
   
   tempwid = gtk_label_new (_("Description:"));
   gtk_misc_set_alignment (GTK_MISC (tempwid), 1, 0.5);
   gtk_table_attach_defaults (GTK_TABLE (table), tempwid, 0, 1, 0, 1);
   gtk_widget_show (tempwid);

   bm_pathedit = gtk_entry_new ();
   gtk_table_attach_defaults (GTK_TABLE (table), bm_pathedit, 1, 2, 0, 1);
   if ((pos = strrchr (entry->path, '/')) == NULL) pos = entry->path;
   else pos++;
   if (pos) gtk_entry_set_text (GTK_ENTRY (bm_pathedit), pos);
   gtk_widget_show (bm_pathedit);

   tempwid = gtk_label_new (_("Hostname:"));
   gtk_misc_set_alignment (GTK_MISC (tempwid), 1, 0.5);
   gtk_table_attach_defaults (GTK_TABLE (table), tempwid, 0, 1, 1, 2);
   gtk_widget_show (tempwid);

   bm_hostedit = gtk_entry_new ();
   gtk_table_attach_defaults (GTK_TABLE (table), bm_hostedit, 1, 2, 1, 2);
   if (entry->isfolder) gtk_widget_set_sensitive (bm_hostedit, 0);
   else if (entry->hostname) gtk_entry_set_text (GTK_ENTRY (bm_hostedit), entry->hostname);
   gtk_widget_show (bm_hostedit);

   tempwid = gtk_label_new (_("Port:"));
   gtk_misc_set_alignment (GTK_MISC (tempwid), 1, 0.5);
   gtk_table_attach_defaults (GTK_TABLE (table), tempwid, 0, 1, 2, 3);
   gtk_widget_show (tempwid);

   bm_portedit = gtk_entry_new ();
   gtk_table_attach_defaults (GTK_TABLE (table), bm_portedit, 1, 2, 2, 3);
   if (entry->isfolder) gtk_widget_set_sensitive (bm_portedit, 0);
   else if (entry->port) gtk_entry_set_text (GTK_ENTRY (bm_portedit), entry->port);
   gtk_widget_show (bm_portedit);

   tempwid = gtk_label_new (_("Directory:"));
   gtk_misc_set_alignment (GTK_MISC (tempwid), 1, 0.5);
   gtk_table_attach_defaults (GTK_TABLE (table), tempwid, 0, 1, 3, 4);
   gtk_widget_show (tempwid);

   bm_diredit = gtk_entry_new ();
   gtk_table_attach_defaults (GTK_TABLE (table), bm_diredit, 1, 2, 3, 4);
   if (entry->isfolder) gtk_widget_set_sensitive (bm_diredit, 0);
   else if (entry->dir) gtk_entry_set_text (GTK_ENTRY (bm_diredit), entry->dir);
   gtk_widget_show (bm_diredit);

   tempwid = gtk_hseparator_new ();
   gtk_table_attach_defaults (GTK_TABLE (table), tempwid, 0, 2, 4, 5);
   gtk_widget_show (tempwid);

   tempwid = gtk_label_new (_("Username:"));
   gtk_misc_set_alignment (GTK_MISC (tempwid), 1, 0.5);
   gtk_table_attach_defaults (GTK_TABLE (table), tempwid, 0, 1, 5, 6);
   gtk_widget_show (tempwid);

   bm_useredit = gtk_entry_new ();
   gtk_table_attach_defaults (GTK_TABLE (table), bm_useredit, 1, 2, 5, 6);
   if (entry->isfolder) gtk_widget_set_sensitive (bm_useredit, 0);
   else if (entry->user) gtk_entry_set_text (GTK_ENTRY (bm_useredit), entry->user);
   gtk_widget_show (bm_useredit);

   tempwid = gtk_label_new (_("Password:"));
   gtk_misc_set_alignment (GTK_MISC (tempwid), 1, 0.5);
   gtk_table_attach_defaults (GTK_TABLE (table), tempwid, 0, 1, 6, 7);
   gtk_widget_show (tempwid);

   bm_passedit = gtk_entry_new ();
   gtk_table_attach_defaults (GTK_TABLE (table), bm_passedit, 1, 2, 6, 7);
   gtk_entry_set_visibility (GTK_ENTRY (bm_passedit), FALSE);
   if (entry->isfolder) gtk_widget_set_sensitive (bm_passedit, 0);
   else if (entry->pass) gtk_entry_set_text (GTK_ENTRY (bm_passedit), entry->pass);
   gtk_widget_show (bm_passedit);

   tempwid = gtk_label_new (_("Account:"));
   gtk_misc_set_alignment (GTK_MISC (tempwid), 1, 0.5);
   gtk_table_attach_defaults (GTK_TABLE (table), tempwid, 0, 1, 7, 8);
   gtk_widget_show (tempwid);

   bm_acctedit = gtk_entry_new ();
   gtk_table_attach_defaults (GTK_TABLE (table), bm_acctedit, 1, 2, 7, 8);
   if (entry->isfolder) gtk_widget_set_sensitive (bm_acctedit, 0);
   else if (entry->acct) gtk_entry_set_text (GTK_ENTRY (bm_acctedit), entry->acct);
   gtk_widget_show (bm_acctedit);

   anon_chk = gtk_check_button_new_with_label (_("Log in as ANONYMOUS"));
   gtk_table_attach_defaults (GTK_TABLE (table), anon_chk, 0, 2, 8, 9);
   if (entry->isfolder) gtk_widget_set_sensitive (anon_chk, 0);
   else {
      gtk_signal_connect (GTK_OBJECT (anon_chk), "toggled", GTK_SIGNAL_FUNC (set_userpass_visible), NULL);
      gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (anon_chk), entry->user && strcmp (entry->user, ANON_LOGIN) == 0);
   }
   gtk_widget_show (anon_chk);

   firewall_check = gtk_check_button_new_with_label (_("Connect through FTP Proxy Server"));
   gtk_table_attach_defaults (GTK_TABLE (table), firewall_check, 0, 2, 9, 10);
   if (*proxy_config != '\0' && !entry->isfolder) {
      gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (firewall_check), entry->firewall && entry->firewall[0] == '1');
   }
   else {
      gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (firewall_check), 1);
      gtk_widget_set_sensitive (firewall_check, 0);
   }
   gtk_widget_show (firewall_check);
   
   tempwid = gtk_button_new_with_label (_("OK"));
   gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->action_area), tempwid, TRUE, TRUE, FALSE);
   gtk_signal_connect (GTK_OBJECT (tempwid), "clicked", GTK_SIGNAL_FUNC (entry_apply_changes), (gpointer) entry);
   gtk_signal_connect_object (GTK_OBJECT (tempwid), "clicked", GTK_SIGNAL_FUNC (gtk_widget_destroy), GTK_OBJECT (dialog));
   GTK_WIDGET_SET_FLAGS (tempwid, GTK_CAN_DEFAULT);
   gtk_widget_show (tempwid);

   tempwid = gtk_button_new_with_label (_("  Cancel  "));
   gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->action_area), tempwid, TRUE, TRUE, FALSE);
   gtk_signal_connect_object (GTK_OBJECT (tempwid), "clicked", GTK_SIGNAL_FUNC (gtk_widget_destroy), GTK_OBJECT (dialog));
   GTK_WIDGET_SET_FLAGS (tempwid, GTK_CAN_DEFAULT);
   gtk_widget_grab_focus (tempwid);
   gtk_widget_show (tempwid);

   tempwid = gtk_button_new_with_label (_("Apply"));
   gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->action_area), tempwid, TRUE, TRUE, FALSE);
   gtk_signal_connect (GTK_OBJECT (tempwid), "clicked", GTK_SIGNAL_FUNC (entry_apply_changes), (gpointer) entry);
   GTK_WIDGET_SET_FLAGS (tempwid, GTK_CAN_DEFAULT);
   gtk_widget_show (tempwid);
   
   gtk_widget_show (dialog);
}
/*****************************************************************************/
static void delete_entry (gpointer data) {
   struct conn_categories *entry, *tempentry, *delentry;
   
   if (GTK_CLIST (tree)->selection == NULL) return;
   entry = (struct conn_categories *) g_hash_table_lookup (htable, GTK_CLIST (tree)->selection->data);
   if (entry == NULL || entry->prev == NULL) return;

   g_hash_table_remove (htable, entry->cnode);
   g_hash_table_remove (new_bookmarks_htable, entry->path);
   gtk_ctree_remove_node (GTK_CTREE (tree), entry->cnode);
   if (entry->prev->children == entry) {
      entry->prev->children = entry->prev->children->next;
   }
   else {
      tempentry = entry->prev->children;
      while (tempentry->next != entry) tempentry = tempentry->next;
      tempentry->next = entry->next;
   }

   entry->prev = NULL;
   entry->next = NULL;
   tempentry = entry;
   while (tempentry != NULL) {
      if (tempentry->path) g_free (tempentry->path);
      if (tempentry->hostname) g_free (tempentry->hostname);
      if (tempentry->dir) g_free (tempentry->dir);
      if (tempentry->user) g_free (tempentry->user);
      if (tempentry->pass) g_free (tempentry->pass);
      if (tempentry->port) g_free (tempentry->port);
      if (tempentry->firewall) g_free (tempentry->firewall);

      if (tempentry->children != NULL) {
         tempentry = tempentry->children;
         continue;
      }
      else if (tempentry->next == NULL && tempentry->prev != NULL) {
         delentry = tempentry->prev;
         g_free (tempentry);
         tempentry = delentry->next;
         if (delentry != entry) g_free (delentry);
      }
      else tempentry = tempentry->next;
   }
   g_free (entry);
}
/*****************************************************************************/
static void entry_apply_changes (GtkWidget *widget, struct conn_categories *entry) {
   struct conn_categories *tempentry, *nextentry;
   char *str, *pos, *newpath, tempchar;
   int oldpathlen;

   oldpathlen = strlen (entry->path);
   if ((pos = strrchr (entry->path, '/')) == NULL) {
      pos = entry->path;
      tempchar = *entry->path;
      *entry->path = '\0';
   }
   else {
      tempchar = *pos;
      *pos = '\0';
   }
   newpath = g_strconcat (entry->path, "/", gtk_entry_get_text (GTK_ENTRY (bm_pathedit)), NULL);
   remove_double_slashes (newpath);
   for (; *newpath == '/'; newpath++);
   *pos = tempchar;

   str = gtk_entry_get_text (GTK_ENTRY (bm_hostedit));
   if (entry->hostname) g_free (entry->hostname);
   entry->hostname = g_malloc (strlen (str) + 1);
   strcpy (entry->hostname, str);
   
   str = gtk_entry_get_text (GTK_ENTRY (bm_portedit));
   if (entry->port) g_free (entry->port);
   entry->port = g_malloc (strlen (str) + 1);
   strcpy (entry->port, str);
   
   str = gtk_entry_get_text (GTK_ENTRY (bm_diredit));
   if (entry->dir) g_free (entry->dir);
   entry->dir = g_malloc (strlen (str) + 1);
   strcpy (entry->dir, str);
   
   if (GTK_TOGGLE_BUTTON (anon_chk)->active) str = ANON_LOGIN;
   else str = gtk_entry_get_text (GTK_ENTRY (bm_useredit));
   if (entry->user) g_free (entry->user);
   entry->user = g_malloc (strlen (str) + 1);
   strcpy (entry->user, str);
   
   if (GTK_TOGGLE_BUTTON (anon_chk)->active) str = "@EMAIL@";
   else str = gtk_entry_get_text (GTK_ENTRY (bm_passedit));
   if (entry->pass) g_free (entry->pass);
   entry->pass = g_malloc (strlen (str) + 1);
   strcpy (entry->pass, str);
   entry->save_password = *entry->pass != '\0';
   
   str = gtk_entry_get_text (GTK_ENTRY (bm_acctedit));
   if (entry->acct) g_free (entry->acct);
   entry->acct = g_malloc (strlen (str) + 1);
   strcpy (entry->acct, str);
   
   if (!entry->firewall) entry->firewall = g_malloc (1);
   *entry->firewall = GTK_TOGGLE_BUTTON (firewall_check)->active ? '1' : '0';

   if (strcmp (entry->path, newpath) != 0) {
      tempentry = entry;
      nextentry = entry->next;
      entry->next = NULL;
      while (tempentry != NULL) {
         g_hash_table_remove (new_bookmarks_htable, tempentry->path);
         str = g_strconcat (newpath, "/", tempentry->path + oldpathlen, NULL);
         remove_double_slashes (str);
         g_free (tempentry->path);
         tempentry->path = str;
         g_hash_table_insert (new_bookmarks_htable, tempentry->path, tempentry);
         if (tempentry->children != NULL) {
            tempentry = tempentry->children;
            continue;
         }
         while (tempentry->next == NULL && tempentry != entry && 
      		tempentry->prev != NULL) tempentry = tempentry->prev;
         tempentry = tempentry->next;
      }
      entry->next = nextentry;
      clear_bookmarks_tree ();
      build_bookmarks_tree ();
   }
}
/*****************************************************************************/
static void set_userpass_visible(GtkWidget *checkbutton, GtkWidget *entry) {
   gtk_widget_set_sensitive (bm_useredit, !GTK_TOGGLE_BUTTON (anon_chk)->active);
   gtk_widget_set_sensitive (bm_passedit, !GTK_TOGGLE_BUTTON (anon_chk)->active);
}
/*****************************************************************************/
void build_bookmarks_menu (void) {
   GtkItemFactoryEntry test = {NULL, NULL, run_bookmark, 0};
   struct conn_categories *tempentry;
   
   tempentry = hostcat->children;
   while (tempentry != NULL) {
      test.path = g_strconcat (_("/Bookmarks/"), tempentry->path, NULL);
      if (tempentry->isfolder) {
         test.item_type = "<Branch>";
         test.callback = NULL;
      }
      else {
         test.item_type = "";
         test.callback = run_bookmark;
      }
      gtk_item_factory_create_item (factory, &test, (gpointer) tempentry->path, 1);
      g_free (test.path);
      if (tempentry->children != NULL) {
         tempentry = tempentry->children;
         continue;
      }
      while (tempentry->next == NULL && tempentry->prev != NULL) tempentry = tempentry->prev;
      tempentry = tempentry->next;
   }
}   
/*****************************************************************************/
static void build_bookmarks_tree (void) {
   struct conn_categories *tempentry, *preventry; 
   char *pos, *prevpos, *text[2], *str;
   GtkCTreeNode *parent;
 
   text[0] = text[1] = _("Bookmarks");
   parent = gtk_ctree_insert_node (GTK_CTREE (tree), NULL, NULL, text, 5, 
   	dir_pixmap, dir_mask, open_dir_pixmap, open_dir_mask, FALSE, TRUE);
   new_bookmarks->cnode = parent;
   g_hash_table_insert (htable, new_bookmarks->cnode, new_bookmarks);
   
   tempentry = new_bookmarks->children;
   while (tempentry != NULL) {
      if (tempentry->children != NULL) {
         tempentry = tempentry->children;
         continue;
      }
      else {
         pos = tempentry->path;
         while ((pos = strchr (pos, '/')) != NULL) {
            *pos = '\0';
            str = g_malloc (strlen (tempentry->path) + 1);
            strcpy (str, tempentry->path);
            *pos = '/';
            preventry = (struct conn_categories *) 
            	g_hash_table_lookup (new_bookmarks_htable, str);
            if (preventry->cnode == NULL) {
               if ((prevpos = strrchr (str, '/')) == NULL) prevpos = str;
               else prevpos++;
               text[0] = text[1] = prevpos;
               preventry->cnode = gtk_ctree_insert_node (GTK_CTREE (tree), 
               		preventry->prev->cnode, NULL, text, 5, dir_pixmap, dir_mask, 
               		open_dir_pixmap, open_dir_mask, FALSE, FALSE);
               g_hash_table_insert (htable, preventry->cnode, preventry);
            }
            g_free (str);
            pos++;
         }
      }

      if ((pos = strrchr (tempentry->path, '/')) == NULL) pos = tempentry->path;
      else pos++;
 
      text[0] = text[1] = pos;
      tempentry->cnode = gtk_ctree_insert_node (GTK_CTREE (tree), 
      		tempentry->prev->cnode, NULL, text, 5, NULL, NULL, 
      		NULL, NULL, FALSE, FALSE);
      g_hash_table_insert (htable, tempentry->cnode, tempentry);

      while (tempentry->next == NULL && tempentry->prev != NULL) tempentry = tempentry->prev;
      tempentry = tempentry->next;
   }
}
/*****************************************************************************/
static void clear_bookmarks_tree (void) {
   struct conn_categories *tempentry;
 
   tempentry = new_bookmarks->children;
   while (tempentry != NULL) {
      if (tempentry->children != NULL) {
         tempentry = tempentry->children;
         continue;
      }
      while (tempentry->next == NULL && tempentry->prev != NULL) {
         gtk_ctree_remove_node (GTK_CTREE (tree), tempentry->cnode);
         tempentry->cnode = NULL;
         tempentry = tempentry->prev;
      }
      gtk_ctree_remove_node (GTK_CTREE (tree), tempentry->cnode);
      tempentry->cnode = NULL;
      tempentry = tempentry->next;
   }
   g_hash_table_destroy (htable);
   htable = g_hash_table_new (hash_function, hash_compare);
}
/*****************************************************************************/
